# Copyright 2004, $Last Software, Inc.
# The code related to click 3 points was reused here from an original
# example titled rectangle.rb by $Last Software, Inc. 
# To it I added the rooftool code. cheers! Tim 
# Name:           RoofTool
# Description:    Create a roof  
# Usage:          click 3 points, create a roof at any angle over a defined horizontal
#                 surface or sloped surface
# Date:           2004,10,25 
# Type:           Tool
# Problems:       2004,10,31 user may have to reverse surfaces depending on where
#                 he/she chose to pick the start point. 
#                 2004,11,03 changed roof from reading degress to pitch
#                 2004,11,10 fixed a math problem related to pitch 
#-----------------------------------------------------------------------------
require 'sketchup.rb'

class RoofTool

def initialize
    $ip = Sketchup::InputPoint.new
    $ip1 = Sketchup::InputPoint.new
    reset
    
    # sets the default roof settings
    $rrdepth = 25.cm if not $rrdepth
    $rrpitch = 6.0 if not $rrpitch
    $rroh = 0.0.cm if not $rroh
     
    # Dialog box
    prompts = ["Depth", "Pitch", "Overhang "]
    values = [$rrdepth, $rrpitch, $rroh]
    results = inputbox prompts, values, "Roof parameters"

    return if not results
    $rrdepth, $rrpitch, $rroh = results
end

def reset
    $pts = []
    $state = 0
    $ip1.clear
    $drawn = false
    Sketchup::set_status_text "", SB_VCB_LABEL
    Sketchup::set_status_text "", SB_VCB_VALUE
    Sketchup::set_status_text "Click anywhere to start"
    $shift_down_time = Time.now
end

def activate
    self.reset
end

def deactivate(view)
    view.invalidate if $drawn
end

def set_current_point(x, y, view)
    if( !$ip.pick(view, x, y, $ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = $ip.tooltip
        
    # Compute points
    case $state
    when 0
        $pts[0] = $ip.position
        $pts[4] = $pts[0]
        need_draw = $ip.display? || $drawn
    when 1
        $pts[1] = $ip.position
        $length = $pts[0].distance $pts[1]
        Sketchup::set_status_text $length.to_s, SB_VCB_VALUE
    when 2
        pt1 = $ip.position
        pt2 = pt1.project_to_line $pts
        vec = pt1 - pt2

        $pt1 = $ip.position
        $pt2 = pt1.project_to_line $pts
        $vec = $pt1 - $pt2


        $width = vec.length
        if( $width > 0 )
            # test for a square
            square_point = pt2.offset(vec, $length)
            if( view.pick_helper.test_point(square_point, x, y) )
                $width = $length
                $pts[2] = $pts[1].offset(vec, $width)
                $pts[3] = $pts[0].offset(vec, $width)
                view.tooltip = "Square"
            else
                $pts[2] = $pts[1].offset(vec)
                $pts[3] = $pts[0].offset(vec)
            end
        else
            $pts[2] = $pts[1]
            $pts[3] = $pts[0]
        end
        Sketchup::set_status_text $width.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end

def onMouseMove(flags, x, y, view)
    self.set_current_point(x, y, view)
end

def create_rectangle
#--------------------------------------------------------------------------
    model = Sketchup.active_model
    model.start_operation "Create Roof"
    entities = model.active_entities
    group=entities.add_group
    entities=group.entities
        
    # creates a new point from p1 in the direction of p2-p3 with length d
    # params are Point3d, 2 vertices, a length, returns a Point3d
    def translate(p1, p2, p3, d)
        v = p3 - p2
        v.length = d
        trans = Geom::Transformation.translation(v)
        return p1.transform(trans)
    end  

        
    #------plan of rectangle for checking purposes
    #base = entities.add_face($pts[0], $pts[1], $pts[2], $pts[3])
    
    #------create a new set of points from the original 3 point pick
    $pt0=Geom::Point3d.new($pts[0][0], $pts[0][1], $pts[0][2])
    $pt1=Geom::Point3d.new($pts[1][0], $pts[1][1], $pts[1][2])
    $pt2=Geom::Point3d.new($pts[2][0], $pts[2][1], $pts[2][2])
    $pt3=Geom::Point3d.new($pts[3][0], $pts[3][1], $pts[3][2])
    
    #------create a new mid-point $pt5 btw. $pt0 & $pt1
    $pt5=translate($pt0, $pt0, $pt1, $length/2)
    #------create a new mid-point $pt6 btw. $pt3 & $pt4
    $pt6=translate($pt3, $pt3, $pt2, $length/2)
    
    #------formula defines similar triangles & Pythagoras 
    $rrheight=($rrpitch/12)*($length/2)
    $rrlength=Math::sqrt(($length/2)*($length/2)+$rrheight*$rrheight)
    
    
    #------formula defines plump cut of roof by Similar triangles    
    $rrcut=$rrdepth*($rrlength/($length/2))
    
    #------formula defines overhang of roof by Similar triangles
    $rrslope=$rroh*($rrlength/($length/2))
    
        
    #------points related to Near first left & right Rafter planes
    $pt55=Geom::Point3d.new($pt5[0], $pt5[1], $pt5[2]+$rrheight)
    $pt555=Geom::Point3d.new($pt55[0], $pt55[1], $pt55[2]+$rrcut)
    $pt00=Geom::Point3d.new($pt0[0], $pt0[1], $pt0[2]+$rrcut)
    $pt11=Geom::Point3d.new($pt1[0], $pt1[1], $pt1[2]+$rrcut)
    $pt7=translate($pt0, $pt55, $pt0, $rrslope)
    $pt77=translate($pt00, $pt555, $pt00, $rrslope)
    $pt8=translate($pt1, $pt55, $pt1, $rrslope)
    $pt88=translate($pt11, $pt555, $pt11, $rrslope)
    
    #------point related to Far left & right Rafter planes  
    $pt66=Geom::Point3d.new($pt6[0],$pt6[1],$pt6[2]+$rrheight)
    $pt666=Geom::Point3d.new($pt66[0], $pt66[1], $pt66[2]+$rrcut)
    $pt33=Geom::Point3d.new($pt3[0], $pt3[1], $pt3[2]+$rrcut)
    $pt22=Geom::Point3d.new($pt2[0], $pt2[1], $pt2[2]+$rrcut)
    $pt9=translate($pt3, $pt66, $pt3, $rrslope)
    $pt99=translate($pt33, $pt666, $pt33, $rrslope)
    $pt10=translate($pt2, $pt66, $pt2, $rrslope)
    $pt1010=translate($pt22, $pt666, $pt22, $rrslope)
    
    #------Do Roof Planes
    if $rroh == 0 
    #-----Do Roof Planes without overhang
    base = entities.add_face($pt0, $pt00, $pt555, $pt55)
    base = entities.add_face($pt2, $pt22, $pt666, $pt66)
    base = entities.add_face($pt1, $pt11, $pt555, $pt55)
    base = entities.add_face($pt3, $pt33, $pt666, $pt66)
  
    base = entities.add_face($pt00, $pt33, $pt666, $pt555)
    base = entities.add_face($pt11, $pt22, $pt666, $pt555)
    base = entities.add_face($pt0, $pt55, $pt66, $pt3)
    base = entities.add_face($pt1, $pt55, $pt66, $pt2)
  
    base = entities.add_face($pt0, $pt00, $pt33, $pt3)
    base = entities.add_face($pt1, $pt11, $pt22, $pt2)
    else 
    #------Do Roof Planes with overhang
    base = entities.add_face($pt7, $pt77, $pt555, $pt55)
    base = entities.add_face($pt8, $pt88, $pt555, $pt55)
    base = entities.add_face($pt9, $pt66, $pt666, $pt99)
    base = entities.add_face($pt10, $pt1010, $pt666, $pt66) 
    
    base = entities.add_face($pt77, $pt99, $pt666, $pt555)
    base = entities.add_face($pt88, $pt1010, $pt666, $pt555)
    base = entities.add_face($pt7, $pt55, $pt66, $pt9)
    base = entities.add_face($pt8, $pt55, $pt66, $pt10)
    
    base = entities.add_face($pt7, $pt77, $pt99, $pt9)
    base = entities.add_face($pt8, $pt88, $pt1010, $pt10)
    end
 #-----------------------------------------------------------------------------
    #view = Sketchup.active_model.active_view 
    #view.zoom_extents 
#----------------------------------------------------------------------------- 
    model.commit_operation
    self.reset
end

def increment_state
    $state += 1
    case $state
    when 1
        $ip1.copy! $ip
        Sketchup::set_status_text "Click for second point"
        Sketchup::set_status_text "Length", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 2
        $ip1.clear
        Sketchup::set_status_text "Click for third point"
        Sketchup::set_status_text "Width", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 3
        self.create_rectangle
    end
end

def onLButtonDown(flags, x, y, view)
    self.set_current_point(x, y, view)
    self.increment_state
    view.lock_inference
end

def onCancel(flag, view)
    view.invalidate if $drawn
    self.reset
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
    
    case $state
    when 1
        # update the width
        vec = $pts[1] - $pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            $pts[1] = $pts[0].offset(vec)
            view.invalidate
            self.increment_state
        end
    when 2
        # update the height
        vec = $pts[3] - $pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            $pts[2] = $pts[1].offset(vec)
            $pts[3] = $pts[0].offset(vec)
            self.increment_state
        end
    end
end

def getExtents
    bb = Geom::BoundingBox.new
    case $state
    when 0
        # We are getting the first point
        if( $ip.valid? && $ip.display? )
            bb.add $ip.position
        end
    when 1
        bb.add $pts[0]
        bb.add $pts[1]
    when 2
        bb.add $pts
    end
    bb
end

def draw(view)
    $drawn = false
    
    # Show the current input point
    if( $ip.valid? && $ip.display? )
        $ip.draw(view)
        $drawn = true
    end

    # show the rectangle
    if( $state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line($ip1, $ip)
        inference_locked = view.inference_locked?
        view.line_width = 3 if inference_locked
        view.draw(GL_LINE_STRIP, $pts[0], $pts[1])
        view.line_width = 1 if inference_locked
        $drawn = true
    elsif( $state > 1 )
        # draw the curve
        view.drawing_color = "black"
        view.draw(GL_LINE_STRIP, $pts)
        $drawn = true
    end
end

def onKeyDown(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY && rpt == 1 )
        $shift_down_time = Time.now
        
        # if we already have an inference lock, then unlock it
        if( view.inference_locked? )
            view.lock_inference
        elsif( $state == 0 )
            view.lock_inference $ip
        elsif( $state == 1 )
            view.lock_inference $ip, $ip1
        end
    end
end

def onKeyUp(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY &&
        view.inference_locked? &&
        (Time.now - $shift_down_time) > 0.5 )
        view.lock_inference
    end
end
    
end # class RoofTool

#=============================================================================
def RoofTool
    Sketchup.active_model.select_tool RoofTool.new
end

#if( not file_loaded?("rooftool.rb") )
#    UI.menu("Plugins").add_item("RoofTool") { Sketchup.active_model.select_tool RoofTool.new }
#end
#-----------------------------------------------------------------------------
file_loaded("rooftool.rb")


