# Copyright 2005, Didier Bur.
# The code related to click 3 points was reused here from an original
# example titled rectangle.rb by @Last Software, Inc. 
# To it I added the angle dimension code.
# Name:           dim_angle
# Description:    Create an angle dimension with arc and text in a group
# Usage:          click 3 points counterclockwise
# Date:           2005,11,28 
# Type:           Tool     
#-----------------------------------------------------------------------------
require 'sketchup.rb'


class DimAngleTool

def initialize
    @ip = Sketchup::InputPoint.new
    @ip1 = Sketchup::InputPoint.new
    reset
end

def reset
    @pts = []
    @state = 0
    @ip1.clear
    @drawn = false
    Sketchup::set_status_text "", SB_VCB_LABEL
    Sketchup::set_status_text "", SB_VCB_VALUE
    Sketchup::set_status_text "Point 1: first end of mesured angle (trigonometric order)"
    @shift_down_time = Time.now
end

def activate
    self.reset
end

def deactivate(view)
    view.invalidate if @drawn
end

def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        need_draw = @ip.display? || @drawn
    when 1
        @pts[1] = @ip.position
        @length = @pts[0].distance @pts[1]
        Sketchup::set_status_text @length.to_s, SB_VCB_VALUE
    when 2
        @pts[2] = @ip.position
        pt1 = @ip.position
        pt2 = pt1.project_to_line @pts
        vec = pt1 - pt2

        @pt1 = @ip.position
        @pt2 = pt1.project_to_line @pts
        @vec = @pt1 - @pt2

        @width = vec.length
        
        Sketchup::set_status_text @width.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end

def onMouseMove(flags, x, y, view)
    self.set_current_point(x, y, view)
end

def create_angle_dim
    model = Sketchup.active_model
    model.start_operation "Angular dimension"
    ents = model.active_entities

#Do Grouping: 
#Place a # before each of the 2 following lines to ungroup the house
    group = ents.add_group
    ents = group.entities
        
    # creates a new point from p1 in the direction of p2-p3 with length d
    # params are Point3d, 2 vertices, a length, returns a Point3d
    def translate(p1, p2, p3, d)
        v = p3 - p2
        v.length = d
        trans = Geom::Transformation.translation(v)
        return p1.transform(trans)
    end   
    
    # returns the mid point (Point3d) of two vertices
  def mid( p1, p2 )
  vec = p2 - p1
  return Geom::Point3d.new(p1.x + vec[0]/2, p1.y + vec[1]/2, p1.z + vec[2]/2)
  end

    # Points
    #puts "1   " + @pts[0].to_s
    #puts "2   " + @pts[1].to_s
    #puts "3   " + @pts[2].to_s
  
    #Add the face to the entities in the model
    face = ents.add_face @pts
    segs = face.edges
    $plane = face.plane
    $vec_Z = face.normal
    face.erase!
    e1v1, e1v2 = segs[0].vertices
    e2v1, e2v2 = segs[1].vertices 
    if ((e1v1 == e2v1) || (e1v1 == e2v2)) then $pt = e1v1 # pt = shared vertex
      else $pt = e1v2 
    end
    segs.each{|s|
      if not (s.vertices.include?($pt))
        s.erase!
      end
    }
    
    # Angles data
    $vec1 = @pts[1].vector_to @pts[0]
    $vec2 = @pts[1].vector_to @pts[2]
    $angle1 = $vec1.angle_between($vec2) 
    $angle2 = $vec2.angle_between($vec1)
    #puts "A1:  " + $angle1.radians.to_s + "\n"
    #puts "A2:  " + $angle2.radians.to_s + "\n"
    
    # Find the middle point of each edge
    $pt1_middle = (mid @pts[1], @pts[0])
    $pt2_middle = (mid @pts[1], @pts[2])
    
    # Draw the arc and insert text
    ang1_between = $angle1
    ang2_between = $angle2
    #text1 = $angle1.radians.to_s[0,6] + ""
    text2 = $angle2.radians.to_s[0,6] + ""
    #$arc1 = ents.add_arc $pt.position, $vec1, $vec_Z, $vec1.length/2.0, 0, $angle1, 30
    $arc2 = ents.add_arc $pt.position, $vec2, $vec_Z, $vec1.length/2.0, 0, $angle2, 30
    #leader_point1 = $arc1[15].start.position
    leader_point2 = $arc2[15].start.position
    #t = element.add_text reference,pt,[leader_length,leader_length,leader_length]
    #t1 = ents.add_text text1,leader_point1,[$vec1.length,$vec1.length,$vec1.length]
    t2 = ents.add_text text2,leader_point2,[$vec1.length/2.0,$vec1.length/2.0,$vec1.length/2.0]
    #t1.leader_type = 1
    t2.leader_type = 1
    
#----- -----------------------------------------------------------------------
    #view = Sketchup.active_model.active_view 
    #view.zoom_extents 
#-----------------------------------------------------------------------------    
    self.reset
end

def increment_state
    @state += 1
    case @state
    when 1
        @ip1.copy! @ip
        Sketchup::set_status_text "Point 2: angle end of mesured angle (trigonometric order)"
        Sketchup::set_status_text "Length", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 2
        @ip1.copy! @ip
        Sketchup::set_status_text "Point 3: second end of mesured angle (trigonometric order)"
        Sketchup::set_status_text "Length", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 3
        @ip1.clear
        self.create_angle_dim
    end
end

def onLButtonDown(flags, x, y, view)
    self.set_current_point(x, y, view)
    self.increment_state
    view.lock_inference
end

def onCancel(flag, view)
    view.invalidate if @drawn
    self.reset
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
    
    case @state
    when 1
        # update the width
        vec = @pts[1] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[1] = @pts[0].offset(vec)
            view.invalidate
            self.increment_state
        end
    when 2
        # update the height
        vec = @pts[3] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[2] = @pts[1].offset(vec)
            @pts[3] = @pts[0].offset(vec)
            self.increment_state
        end
    end
end

def draw(view)
    @drawn = false
    
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end

    # show the rectangle
    if( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
        inference_locked = view.inference_locked?
        view.line_width = 3 if inference_locked
        view.draw(GL_LINE_STRIP, @pts[0], @pts[1])
        view.line_width = 1 if inference_locked
        @drawn = true
    elsif( @state > 1 )
        # draw the curve
        view.drawing_color = "black"
        view.draw(GL_LINE_STRIP, @pts)
        @drawn = true
    end
end

def onKeyDown(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY && rpt == 1 )
        @shift_down_time = Time.now
        
        # if we already have an inference lock, then unlock it
        if( view.inference_locked? )
            view.lock_inference
        elsif( @state == 0 )
            view.lock_inference @ip
        elsif( @state == 1 )
            view.lock_inference @ip, @ip1
        end
    end
end

def onKeyUp(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY &&
        view.inference_locked? &&
        (Time.now - @shift_down_time) > 0.5 )
        view.lock_inference
    end
end
    
end # class DimAngleTool

#=============================================================================
def dim_angle_tool
    Sketchup.active_model.select_tool DimAngleTool.new
end

if( not file_loaded?("dim_angle.rb") )
    UI.menu("Tools").add_item("Ƕ ") { dim_angle_tool }
end
#-----------------------------------------------------------------------------
file_loaded("dim_angle.rb")


