# Name :          Mirror 2.5
# Description :   mirror geometry at plane, line or point
# Author :        Frank Wiesner
# Usage :         1. select geometry
#                 2. select mirror tool
#                 3. pick one point and hit RETURN to mirror at point
#                 4. pick a second and hit RETURN to mirror at line
#                 5. pick a third point to mirror at plane
# Date :          4.Jul.2oo5
# Type :          script
# Todo :          - preserve materials & colors (impossible to do in version SU4)
# History:        
#                 2.5  (8.Jul.2oo5)  - bugfix: faces with holes reduce now to a single face
#                                    - major code rewrite (dropped @clondedObject hash, heavy 
#                                      use of groups during construction,...
#                                    - more redundant constructions, but seems to be more robust
#                                    - WARNING: POLYGONS DO NOT WORK VERY WELL
#                 2.4  (5.Jul.2oo5)  - all unnecessary edges created by the add_faces_from_mesh() 
#                                      method are erased! faces with holes mirror now perfectly.
#                 2.3  (4.Jul.2oo5)  - fixed "mirror at plane" error
#                 2.2  (2.Jul.2oo5)  - works in group editting mode
#                                    - prevents picking of identical points
#                 2.1  (1.Jul.2oo5)  - fixed "undo group" bug
#                                    - code cleanup
#                 2.0  (30.Jun.2oo5) - new tool-like interface
#                 1.4  (29.Jun.2oo5) - arc bug fixed
#                                    - code simplified (clone_selection() removed)
#                 1.3  (24.Jun.2oo5) - can handle faces with holes in it (although more edges 
#                                      are drawn than nessesary)
#                                    - correct orientation of mirrored faces 
#                                    - supports arcs (at least works for circles), curves and groups
#                 1.2  (20.Jun.2oo5) - axis and origin selection algorithm does not rely on tool tips any more.
#                 1.1  (19.Jun.2oo5) - fix coplanar point bug
#                                    - better tooltips
#                                    - supports mirror at lines (edges, axis, construction lines)
#                                    - mirror-at-plane code simplified
#                 1.o  (10.Jun.2oo5) - first version

require 'sketchup.rb'

class MirrorTool

    def initialize
        @MIRROR_AT_POINT = 1
        @MIRROR_AT_LINE = 2
        @MIRROR_AT_PLANE = 3
        @numPrecision = 0.00001
    end

    def reset
        @pts = []
        @state = 0
        @transformationType = nil
        @trans = nil
        @point = nil
        @line = nil
        @linev = nil
        @plane = nil
        @planeNormal = nil
        @pointOnPlane = nil
        @curvesDone = Hash.new()
        @cloneGroup = nil
        @ip = Sketchup::InputPoint.new
        @ip1 = Sketchup::InputPoint.new
        @ip2 = Sketchup::InputPoint.new
        @ip3 = Sketchup::InputPoint.new
        @ip = Sketchup::InputPoint.new
        Sketchup::set_status_text "Click for first point"
        @drawn = false
    end

    def activate
        Sketchup.send_action("showRubyPanel:")
        self.reset
        if (Sketchup.active_model.selection.empty?)
    		UI.messagebox("First select geometry, then reselect mirror tool.")
            Sketchup.active_model.select_tool nil
        end
    end

    def deactivate(view)
        view.invalidate if @drawn
        @ip1 = nil
        @ip2 = nil
        @ip3 = nil
    end

    def onMouseMove(flags, x, y, view)
        case @state
        when 0 # getting the first end point
            @ip.pick view, x, y
            if( @ip.valid? && @ip != @ip1 )
                @ip1.copy! @ip
            end
            view.invalidate
            view.tooltip = @ip.tooltip if @ip.valid?
        when 1 # getting the second end point
            @ip.pick view, x, y, @ip1
            if( @ip.valid? && @ip != @ip2 )
                @ip2.copy! @ip
                @pts[1] = @ip2.position
            end
            view.invalidate
            view.tooltip = @ip.tooltip if @ip.valid?
        when 2 # getting the third end point
            @ip.pick view, x, y, @ip2
            if( @ip.valid? && @ip != @ip3 )
                @ip3.copy! @ip
                @pts[1] = @ip3.position
            end
            view.invalidate
            view.tooltip = @ip.tooltip if @ip.valid?
        end
    end

    def onLButtonDown(flags, x, y, view)
        @ip.pick view, x, y
        if( @ip.valid? )
            case @state
            when 0
                if( @ip.valid? )
                    @pts[0] = @ip.position
                    Sketchup::set_status_text "Hit RETURN to mirror at point or click for point 2"
                    @state = 1
                end
            when 1
                if( @ip.valid? && @ip != @ip1 )
                    @pts[1] = @ip.position
                    @state = 2
                    Sketchup::set_status_text "Hit RETURN to mirror at line or click for point 3"
                end
            when 2
                if( @ip.valid? && @ip != @ip1 && @ip != @ip2 )
                    @pts[2] = @ip.position
                    @state = 3
                    mirror()
                end
            end
        end
    end

    def onCancel(flag, view)
        view.invalidate if @drawn
        reset
    end

    def draw(view)
        if( @ip.valid? && @ip.display? )
            @ip.draw(view)
            @drawn = true
        end
        if( @state == 1 )
            view.set_color_from_line(@ip1, @ip)
            view.draw(GL_LINE_STRIP, @ip1.position, @ip.position)
            @drawn = true
        elsif( @state == 2 )
            view.drawing_color = "gray"
            view.draw(GL_LINE_STRIP, @ip1.position, @ip2.position)
            view.set_color_from_line(@ip2, @ip)
            view.draw(GL_LINE_STRIP, @ip2.position, @ip.position)
            @drawn = true
        end
    end

    def onKeyDown(key, repeat, flags, view)
        if( key == CONSTRAIN_MODIFIER_KEY && repeat == 1 )
            @shift_down_time = Time.now
            if( view.inference_locked? )
                view.lock_inference
            elsif( @ip.valid? )
                view.lock_inference @ip
            end
        end
    end

    def onKeyUp(key, repeat, flags, view)
        if( key == CONSTRAIN_MODIFIER_KEY && view.inference_locked? && (Time.now - @shift_down_time) > 0.5 )
            view.lock_inference
        end
        if( key == 13)
            mirror()
            self.reset
        end
    end

    def mirror()
        case @state
        when 1
            @transformationType = @MIRROR_AT_POINT
            @point = @ip1.position.clone
            @trans = Geom::Transformation.scaling(@point, -1.0)
            Sketchup.active_model.start_operation "Mirror at Point"
        when 2
            @transformationType = @MIRROR_AT_LINE
            @line = [@ip1.position, @ip2.position] 
            @linev = Geom::Vector3d.new(@ip2.position.x-@ip1.position.x, @ip2.position.y-@ip1.position.y, @ip2.position.z-@ip1.position.z)
            Sketchup.active_model.start_operation "Mirror at Line"
        when 3
            @transformationType = @MIRROR_AT_PLANE
            @plane = Geom.fit_plane_to_points(@ip1.position, @ip2.position, @ip3.position) 
            @planeNormal = Geom::Vector3d.new(@plane[0], @plane[1], @plane[2])
            @planeNormal.normalize!
            @pointOnPlane = @ip1.position
            Sketchup.active_model.start_operation "Mirror at Plane"
        end
        @cloneGroup = Sketchup.active_model.active_entities.add_group
        clone_entities(Sketchup.active_model.selection,@cloneGroup)
        @cloneGroup.explode
    	Sketchup.active_model.commit_operation
        self.reset
    end

    def number_of_faces(g)
        faceCounter = 0
        g.entities.each {|o| 
            if (o.kind_of? Sketchup::Face) 
                faceCounter = faceCounter+1
            end
        }
        return faceCounter
    end

    def clone_face(g,f)
        temp_g = g.entities.add_group
        # use add_face() where possible (no holes) to avoid triangulation
        if (f.loops.length <= 1)
            new_edges = Array.new();
            f.edges.each {|e| 
                new_edges.push(clone_edge(temp_g,e))
            }
            new_f = temp_g.entities.add_face(new_edges)
        else
            # all following operations we are doing in a separate group
            new_mesh = Geom::PolygonMesh.new(f.mesh.count_points, f.mesh.count_polygons)
            pts = f.mesh.points 
            pts.each {|o| 
                new_mesh.add_point(clone_point3d(o))
            }
            f.mesh.polygons.each {|o| 
                new_mesh.add_polygon(o)
            }
            temp_g.entities.add_faces_from_mesh(new_mesh)
            # add_faces_from_mesh() triangulates even if all points 
            # are coplanar, so we check for edges coplanar separating
            # 2 coplanar faces and delete them.
            # we repeat this, until we have got one single face again.
            faceCounter = number_of_faces(temp_g)
            while (faceCounter > 1) do
                temp_g.entities.each {|new_e| 
                    if (new_e.kind_of? Sketchup::Edge)
                        if (new_e.faces.length == 2) # if created during triangualtion only 2 faces are connected by an edge
                            if (new_e.faces[0].normal == new_e.faces[1].normal) # faces coplanar?
                                new_e.erase!
                            end
                        end
                    end
                }
                newfaceCounter = number_of_faces(temp_g)
                # avoid infinite loop
                if (faceCounter == newfaceCounter) # no progress!
                    UI.messagebox("Warning: Could not reduce mesh to one face!")
                    break
                end
                faceCounter = newfaceCounter
            end
            # now we copy the properties of the original edges
            temp_g.entities.each {|e| 
                if (e.kind_of? Sketchup::Edge) 
                    f.edges.each {|old_e|
                        old_start = mirror_point(old_e.start.position.clone)
                        old_end = mirror_point(old_e.end.position.clone)
                        if ( ( (e.start.position == old_start) && (e.end.position == old_end) ) || ( (e.start.position == old_end) && (e.end.position == old_start) ) )
                            copy_edge_properties(old_e, e)
                            break
                        end
                    }
                end
            }
            # find the face in the temp group
            temp_g.entities.each {|o| 
                if (o.kind_of? Sketchup::Face)
                    new_f = o
                    break
                end
            }
        end
        reverse_face_if_nessesary(f, new_f)
        copy_face_properties(f, new_f)
        temp_g.explode
        return new_f
    end
    
    def copy_face_properties(old_f, new_f)
        new_f.back_material = old_f.back_material if (old_f.back_material != nil)
        # new_f.front_material = old_f.front_material
        # new_f.position_material = old_f.position_material
    end

    def reverse_face_if_nessesary(f, new_f)
        if (@transformationType == @MIRROR_AT_PLANE)
            d = f.normal.dot(@planeNormal)
            if ( d*d > @numPrecision )
                v = f.normal.dot(@planeNormal)-new_f.normal.dot(@planeNormal.reverse)
                if ( v*v > @numPrecision )
                    new_f.reverse!
                end
            else
                if (!(f.normal.samedirection?(new_f.normal)))
                    new_f.reverse!
                end
            end
        elsif (@transformationType == @MIRROR_AT_LINE)
            d = f.normal.dot(@linev)
            if ( d*d > @numPrecision )
                v = @linev.dot(f.normal)-@linev.dot(new_f.normal)
                if ( v*v > @numPrecision )
                    new_f.reverse!
                end
            else
                if (f.normal.samedirection?(new_f.normal))
                    new_f.reverse!
                end
            end
        elsif (@transformationType == @MIRROR_AT_POINT)
            if (!(f.normal == new_f.normal.reverse))
                new_f.reverse!
            end
        end
    end
    
    def clone_edge(g,e)
        new_e = g.entities.add_edges(clone_point3d(e.start.position), clone_point3d(e.end.position)).first
        copy_edge_properties(e, new_e)
        return new_e
    end

    def copy_edge_properties(old_e, new_e)
        new_e.soft = old_e.soft?
        new_e.smooth = old_e.smooth?
    end

    def clone_point3d(p)
        new_p = p.clone
        mirror_point(new_p)
        return new_p
    end
   
    def clone_arcCurve(g,a)
        puts("arcCurve: count_edges = "+a.count_edges.to_s)
        temp_g = g.entities.add_group
        new_center = a.center.clone
        mirror_point(new_center)
        
        new_xaxis_e = Geom::Point3d.new(a.center.x+a.xaxis.x, a.center.y+a.xaxis.y, a.center.z+a.xaxis.z)
        mirror_point(new_xaxis_e)
        new_xaxis = Geom::Vector3d.new(new_xaxis_e.x-new_center.x, new_xaxis_e.y-new_center.y, new_xaxis_e.z-new_center.z)

        new_normal_e = Geom::Point3d.new(a.center.x+a.normal.x, a.center.y+a.normal.y, a.center.z+a.normal.z)
        mirror_point(new_normal_e)
        new_normal = Geom::Vector3d.new(new_normal_e.x-new_center.x, new_normal_e.y-new_center.y, new_normal_e.z-new_center.z)
        if ( (@transformationType == @MIRROR_AT_POINT) || (@transformationType == @MIRROR_AT_PLANE) )
            new_normal.reverse!
        end

        new_a = temp_g.entities.add_arc(new_center, new_xaxis, new_normal, a.radius, a.start_angle, a.end_angle, a.count_edges) 
        temp_g.explode
        return new_a
    end

    def clone_curve(g,c)
        if (@curvesDone[c.id] != true)
            if (c.kind_of? Sketchup::ArcCurve)
                new_c = clone_arcCurve(g,c)
            else
                new_points = Array.new();
                c.vertices.each {|v|
                    new_points.push(clone_point3d(v.position))
                }
                new_points.flatten!
                new_c = g.entities.add_curve(new_points)
            end
            @curvesDone[c.id] = true
        end
        return new_c
    end
    
    def clone_group(super_g,g)
        new_g = super_g.entities.add_group
        clone_entities(g.entities, new_g)
        # groups get dislocated when mirrored (don't know why this is)
        # to correct this we move the center of the new group to the 
        # mirrored point of the center of the original group 
        new_center = g.bounds.center.clone
        mirror_point(new_center)
        transVec = Geom::Vector3d.new(new_center.x-new_g.bounds.center.x, new_center.y-new_g.bounds.center.y, new_center.z-new_g.bounds.center.z) 
        t = Geom::Transformation.translation(transVec)
        new_g.transform!(t)
        return new_g
    end
    
    def clone_entities(entis, new_g) 
        entis.each {|o| 
            clone_edge(new_g,o) if o.kind_of? Sketchup::Edge
            clone_face(new_g,o) if o.kind_of? Sketchup::Face
            clone_group(new_g,o) if o.kind_of? Sketchup::Group
        }
        entis.each {|e| 
            if (e.kind_of? Sketchup::Edge)
                if (e.curve != nil)
                    clone_curve(new_g,e.curve)
                end
            end
        }
    end

    def mirror_point(new_p)
        if (@transformationType == @MIRROR_AT_PLANE)
            if (!new_p.on_plane?(@plane))
                mirrorp = new_p.project_to_plane(@plane)
                @trans = Geom::Transformation.scaling(mirrorp, -1.0)
                new_p.transform!(@trans)
            end
        elsif (@transformationType == @MIRROR_AT_LINE)
            if (!new_p.on_line?(@line))
                mirrorp = new_p.project_to_line(@line)
                @trans = Geom::Transformation.scaling(mirrorp, -1.0)
                new_p.transform!(@trans)
            end
        elsif (@transformationType == @MIRROR_AT_POINT)
            if (!(new_p == @point))
                new_p.transform!(@trans)
            end
        end
    end

end # class MirrorTool

if( not file_loaded?("mirror.rb") )
  UI.add_context_menu_handler do |menu|
                menu.add_separator
 		menu.add_item("") { Sketchup.active_model.select_tool MirrorTool.new }
    end
end
file_loaded("mirror.rb")

