# Name :          Layer_manager
# Description :   Create and manage your layers configuration in your SU models
# Author :        Didier Bur
# Usage :         Select Tools/Layer Manager/Option
# All layers visible: unhide all layers
# Invert all layers: layers visible become invisible and vice-versa, except current layer
# Isolate current layer: hide all layers except active layer
# Isolate selection: hide all layers except all layers of selected objects
# Hide selection: hide all layers of selected objects, except current layer
# Save layer configuration: save the current status of all layers. When used the first time, enter a config name, otherwise either select a config name to update an existing config, or enter a new config name to create a new one.
# Load layers configuration: select a config name in the list, SU reads it and switches layers on/off
# Display layers config: select a config name in the list and see the message.
# KNOWN BUG:      SU layers window doesn't reflect changes when loading a config. Help @Last !
# Date :          07.March.2oo5
# Type :          tool
# History:        1.0 (07.March.2oo5) beta version (needs an external SLM file)
#                 1.1 (09.March.2oo5) no more external file, data stored within the SKP
#-----------------------------------------------------------------------------

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------

require 'sketchup.rb'

# ----------------------------------------------------- Turns all layers visible
def lm_all_visible
model=Sketchup.active_model
layers = model.layers
layers.each do |lay|
  lay.visible = true
end
end

# ----------------------------------------------------- Turns all layers invisible
def lm_all_invisible
model=Sketchup.active_model
layers = model.layers
layers.each do |lay|
    if lay.visible? and lay.name != model.active_layer.name
      lay.visible = false
    end
end
end

# ----------------------------------------------------- Turns all layers invisible, except current
def lm_none_but_current
model=Sketchup.active_model
layers = model.layers
current = model.active_layer
layers.each do |lay|
  if( lay.name != current )
    lay.visible = false
  end
end
end

# ----------------------------------------------------- Turns all layers of selection visible
def lm_isolate_selection
model = Sketchup.active_model
ss = model.selection
layers = model.layers
selection_layers = []
if ss.empty? 
  UI.messagebox("No selection. Like some coffee ?")
  return nil
end
ss.each do |element|
  selection_layers.push(element.layer)
end
selection_layers.uniq!
layers.each do |lay|
  if(selection_layers.include?(lay))
    lay.visible = true
    else
    lay.visible = false
  end
end
return nil
end

# ----------------------------------------------------- Hides all layers of selection visible
def lm_hide_selection
model = Sketchup.active_model
ss = model.selection
layers = model.layers
selection_layers = []
if ss.empty? 
  UI.messagebox("No selection. Like some coffee ?")
  return nil
end
ss.each do |element|
  selection_layers.push(element.layer)
end
selection_layers.uniq!
layers.each do |lay|
  if(selection_layers.include?(lay))
    lay.visible = false
  end
end
return nil
end

# ----------------------------------------------------- Invert all layers 
def lm_invert
model = Sketchup.active_model
ss = model.selection
layers = model.layers

layers.each do |lay|
  if(lay.visible? == true)
    lay.visible = false
    else
    lay.visible = true
  end
end
return nil
end

# ----------------------------------------------------- Displays a config in a message box
def lm_display_config
model=Sketchup.active_model
layers = model.layers

# Get all the existing lm_configs
config_list = (collect_lm_config)
# Supress the default config
config_list = config_list.slice(7 .. config_list.length)
if( config_list.length != 0 )
# Dialog
  dropdown = [config_list.join("|")]
  prompts=["Display config:           "]
  values=[config_list[0]]
  results = inputbox prompts, values, dropdown, "Layers Manager config display"
  return nil if not results
  config_name = results[0] 
  # Get the corresponding LMAD
  lm_ad = model.attribute_dictionary("lm_" + config_name)
  # Construct message
  message = "Layers config: " + config_name + "\n\n"
  lm_ad.each_key {|k|
    if( lm_ad[k] == true )
      message = message + k + ": Visible\n"
      else
      message = message + k + ": Invisible\n"
    end
  }
  UI.messagebox(message)
else
  # No LMAD's in the model
  UI.messagebox("No layers configuration saved in this model. Like some coffee ?")
end
return nil
end

# ----------------------------------------------------- Restores the selected config
def lm_restore_config(config)
model = Sketchup.active_model
layers = model.layers

if(config != "All layers visible") and (config != "Isolate current layer   ") and (config != "Isolate selection") and (config != "") and (config != "Invert all layers") and (config != "Hide selection") and (config != "All layers invisible" )
  # Get the keys and values of this config
  lm_ad = model.attribute_dictionary("lm_" + config)
  # Browse layers, find the corresponding pair key-value and here we go
  layers.each do |lay|
    lay.visible = lm_ad[lay.name]
  end
  else
  # default config selected
  if (config == "All layers visible")
    (lm_all_visible)
  end
  if (config == "All layers invisible")
    (lm_all_invisible)
  end
  if (config == "Invert all layers")
    (lm_invert)
  end
  if (config == "Isolate current layer   ")
    (lm_none_but_current)
  end
  if (config == "Isolate selection")
    (lm_isolate_selection)
  end
  if (config == "Hide selection")
    (lm_hide_selection)
  end
  if (config == "")
    UI.messagebox("Not a valid layers config.")
  end
end
return nil
end

# ----------------------------------------------------- Saves or overwrites a lm_config_name key-value = layer-status
def lm_save_layers_config
model=Sketchup.active_model
layers = model.layers
# Get all the existing lm_configs
config_list = (collect_lm_config)

if( config_list.length == 0 )
  # Give the first config a name
  prompts=["Layers config name:  "]
  values=[""]
  results = inputbox prompts, values, "Layers Manager config saver"
  return nil if not results
  # Creates the LMAD
  lm_ad = model.attribute_dictionary("lm_" + results[0], true)
  #Add pairs keys-values: layer-status
  layers.each do |lay|
    lm_ad[lay.name] = lay.visible?
  end
  else
  # There's at least one LMAD in the model
  dropdown = [config_list.join("|")]
  prompts=["Overwrite layers config:  ", "New config:  "]
  values=[config_list[0], ""]
  results = inputbox prompts, values, dropdown, "Layer Manager config saver"
  return nil if not results
  existing_config = results[0]
  new_config = results[1]
  if( new_config != "")
    # new config
    (lm_write_config new_config)
    Sketchup.set_status_text("Layers config " + new_config + " saved.")
    else
    # Existing config needs overwrite, so first clear it
    (lm_delete_config existing_config)
    # Then write it
    (lm_write_config existing_config)
    Sketchup.set_status_text("Layers config " + existing_config + " updated.")
  end
end
end

# ----------------------------------------------------- Loads a lm_config_name and restores it
def lm_load_layers_config
model=Sketchup.active_model
layers = model.layers
# Get all the existing lm_configs
config_list = (collect_lm_config)

if( config_list.length != 0 )
# Dialog
  dropdown = [config_list.join("|")]
  prompts=["Load config:           "]
  values=[config_list[0]]
  results = inputbox prompts, values, dropdown, "Layers Manager config loader"
  return nil if not results
  config_name = results[0] 
  # Restore selected config
  (lm_restore_config config_name)
  Sketchup.set_status_text("Layers config " + config_name + " restored.")
else
  # No LMAD's in the model
  UI.messagebox("No layers configuration saved in this model.")
end
end

# ----------------------------------------------------- List existing configs in AD's
def collect_lm_config
model=Sketchup.active_model
ad = model.attribute_dictionaries
config_array = []
# Adds the default configs
config_array.push("All layers visible")
config_array.push("All layers invisible" )
config_array.push("Isolate current layer   ")
config_array.push("Isolate selection")
config_array.push("Hide selection")
config_array.push("Invert all layers")
config_array.push("")
# Browse Ad's to find LMAD's
if( ad != nil )
  # At least an AD in the model
  ad.each do |dict|
    if (dict.name["lm_"])
      config_array.push(dict.name.split("lm_")[1])
    end
  end
end
return config_array
end

# ----------------------------------------------------- Writes a lm_config_name key-value = layer-status
def lm_write_config( config_name )
  model=Sketchup.active_model
  layers = model.layers
  # Creates the LMAD if needed
  lm_ad = model.attribute_dictionary("lm_" + config_name, true)
  # Add pairs keys-values: layer-status
  layers.each do |lay|
    lm_ad[lay.name] = lay.visible?
  end
end

# ----------------------------------------------------- Deletes all pairs key-value of a lm_config_name
def lm_clear_config( config_name )
  model=Sketchup.active_model
  layers = model.layers
  # Get the LMAD
  lm_ad = model.attribute_dictionary("lm_" + config_name)
  #Add pairs keys-values: layer-status
  lm_ad.each {|k, v| 
    lm_ad.delete_key(k)
  }
end

# ----------------------------------------------------- Deletes a lm_config_name
def lm_delete_config( config_name )
  model=Sketchup.active_model
  model_dicts = model.attribute_dictionaries
  lm_ad = model.attribute_dictionary("lm_" + config_name)
  model_dicts.delete(lm_ad)
end

# ---------------------------------------------------------------------------
# Menu items:
#----------------------------------------------------------------------------
if( not $layer_manager_menu_loaded )
    add_separator_to_menu("Tools")
    utilities_menu = UI.menu("Tools").add_submenu("ͼ")
    utilities_menu.add_item("ȫͼɼ    ") { (lm_all_visible) }
    utilities_menu.add_item("ȫͼ㲻ɼ    ") { (lm_all_invisible) }
    utilities_menu.add_item("תͼʾ") { (lm_invert) }
    utilities_menu.add_item("ǰͼɼ    ") { (lm_none_but_current) }
    utilities_menu.add_item("ֻʾѡͼ  ") { (lm_isolate_selection) }
    utilities_menu.add_item("ѡͼ  ") { (lm_hide_selection) }
    utilities_menu.add_item("---------------------") { UI.messagebox("No command here !") }
    utilities_menu.add_item("ͼ") { (lm_save_layers_config) }
    utilities_menu.add_item("ʾͼ") { (lm_display_config) }
    utilities_menu.add_item("ȡͼ") { (lm_load_layers_config) }
    $layer_manager_menu_loaded = true
end

file_loaded("layer_manager.rb")