# Copyright 2004, Didier Bur, Inc.

# This software is provided as an example of using the Ruby interface
# to SketchUp.

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------
# Name        :   su2pov 2.1.
# Description :   A converter to export SU models to POV ray-tracer.
# Menu Item   :   Plugins->Export to POV-Ray
# Context Menu:   None
# Usage       :   See PDF User Guide
# Date        :   01/25/2005
# Type        :   Converter - Tool
#-----------------------------------------------------------------------------
# Supported entites and features:
# Camera (perspective, orthographic, cylindrical, spherical...), as many as pages
# Faces, groups, components (nested or not), images (un-exploded)
# Materials: colors, textures, transparencies...
# Rendering options: ambient, Background, ground, scene type, fog, lens flare, glow effects...
# Lights: Sun, pointlights, realpointlights, spot lights, box and cone media lights...
#-----------------------------------------------------------------------------

require 'sketchup.rb'

# ------------------------------------------------------------------------
# ------------------------------------ GEOMETRY RELATED FUNCTIONS
# ------------------------------------------------------------------------

# ----------------------------------- Switch from radians to degrees
def rad2deg ( a )
return ((a*180.0)/Math::PI)
end
# ----------------------------------- Given a SU spotlight component instance, returns a Point3D that it points at
def get_spotlight_point_at( component )
trans = component.transformation
target = Geom::Point3d.new(0,0,-100).transform(trans)
return target
end

# ----------------------------------- Given a SU arealight component instance, returns a Point3D that it points at
def get_arealight_point_at( component )
trans = component.transformation
target = Geom::Point3d.new(0,0,100).transform(trans)
return target
end

# ----------------------------------- Given a SU arealight component instance, returns the X and Y vectors
def get_arealight_vectors( component )
trans = component.transformation
p1 = Geom::Point3d.new(-50.cm,0,0).transform(trans)
p2 = Geom::Point3d.new(50.cm,0,0).transform(trans)
p3 = Geom::Point3d.new(0,50.cm,0).transform(trans)
p4 = Geom::Point3d.new(0,-50.cm,0).transform(trans)
x_vec = p2 - p1
y_vec = p4 - p3
return [x_vec, y_vec]
end

# ------------------------------------------------------------------------
# ------------------------------------ TREES RELATED FUNCTIONS
# ------------------------------------------------------------------------

# ----------------------------------- Set the parameters for a 3d tree macro call
def set_tree_params
model=Sketchup.active_model
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("Tree not selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_tree" )
#default values
dict_key_season = "Summer" if not dict_key_season
dict_key_age = "Adult" if not dict_key_age
dict_key_species = "Mapple" if not dict_key_species
dict_key_bearing = "Normal" if not dict_key_bearing
dict_key_fruit = "None" if not dict_key_fruit
dict_key_husk =  "Medium" if not dict_key_husk
# Retrieve parameters if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_season = e.get_attribute(dict_name, "season")
  dict_key_age = e.get_attribute(dict_name, "age")
  dict_key_species = e.get_attribute(dict_name, "species")
  dict_key_bearing = e.get_attribute(dict_name, "bearing")
  dict_key_husk = e.get_attribute(dict_name, "husk")
  dict_key_fruit = e.get_attribute(dict_name, "fruit")
  else
  UI.messagebox("No parameters set for the tree.")
end

#Dialog box:
seasons_list = %w[Spring Summer Autumn Winter].join("|")
ages_list = %w[Very_young Young Adult Old Very_old].join("|")
species_list = %w[Mapple Prunus Chestnut_tree].join("|")
bearings_list = %w[Normal Twisty Straight Slender Weeping].join("|")
husks_list = %w[Light_maroon Light Medium Dark].join("|")
fruits_list = %w[None Several_white_flowers Several_pink_flowers Several_berries Several_oranges Several_fruits Lot_of_white_flowers Lot_of_pink_flowers Lot_of_berries Lot_of_oranges Lot_of_fruits].join("|")

array_of_dropdowns = [seasons_list, ages_list, species_list, bearings_list, husks_list, fruits_list]
prompts=["Season  ", "Age  ", "Species  ","Bearing  ", "Husks  ", "Fruits  "]
values=[dict_key_season, dict_key_age, dict_key_species, dict_key_bearing, dict_key_husk, dict_key_fruit]
dict_key_season, dict_key_age, dict_key_species, dict_key_bearing, dict_key_husk, dict_key_fruit  = inputbox prompts, values, array_of_dropdowns, "Tree parameters"

dict_name="su2pov"
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"season", dict_key_season)
e.set_attribute(dict_name,"age",dict_key_age)
e.set_attribute(dict_name,"species",dict_key_species)
e.set_attribute(dict_name,"bearing", dict_key_bearing)
e.set_attribute(dict_name,"husk", dict_key_husk)
e.set_attribute(dict_name,"fruit", dict_key_fruit)
else
  UI.messagebox("This is not a tree.")
end
end

# ----------------------------------- Get the parameters for a 3d tree macro call
def get_tree_params( tree )
model=Sketchup.active_model

# Retrieve parameters if any
dict_name="su2pov"
dicts=tree.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_season = tree.get_attribute(dict_name, "season")
  dict_key_age = tree.get_attribute(dict_name, "age")
  dict_key_species = tree.get_attribute(dict_name, "species")
  dict_key_bearing = tree.get_attribute(dict_name, "bearing")
  dict_key_husk = tree.get_attribute(dict_name, "husk")
  dict_key_fruit = tree.get_attribute(dict_name, "fruit")
end

#default values if not set by the user
dict_key_season = "Summer" if not dict_key_season
dict_key_age = "Adult" if not dict_key_age
dict_key_species = "Mapple" if not dict_key_species
dict_key_bearing = "Normal" if not dict_key_bearing
dict_key_husk =  "Medium" if not dict_key_husk
dict_key_fruit = "None" if not dict_key_fruit

ret = [dict_key_season, dict_key_age, dict_key_species, dict_key_bearing, dict_key_husk, dict_key_fruit]
return ret
end

# ----------------------------------- Transforms the option tree param to a Pov-ray param
def trans_tree_param( p )
case p
  #Seasons
  when "Spring"
    $season_leafprob = 7
    $season_leafscale = 7
  when "Summer"
    $season_leafprob = 9.5
    $season_leafscale = 10
  when "Autumn"
    $season_leafprob = 3
    $season_leafscale = 5
  when "Winter"
    $season_leafprob = 0  
    $season_leafscale = 0
  # Age:
  when "Very_young"
    $recurs = 2
    $divisions = [4,8,0,0,0,0]
    $roots = "init_roots (2)"
    $simplify = "init_simplify (1,1)"
    $branchl = "init_branchl (1.8,0.7,0.6)"
    $branchw = "init_branchw (0.1,0.003,0.4)"
    $twigprob = "init_twigprob (0.3,0.6,0.6)"
    $leafnoise = "#declare leafnoise = array[6] {1,1,1,1,1,1}"
    $age_leafscale = 5
    $age_leafprob = 5
  when "Young"
    $tree_recurs = 3
    $divisions = [4,5,2,0,0,0]
    $roots = "init_roots (2)"
    $simplify = "init_simplify (1,1)"
    $branchl = "init_branchl (2.5,1,0.6)"
    $branchw = "init_branchw (0.2,0.003,0.4)"
    $twigprob = "init_twigprob (0.45,0.85,0.85)"
    $leafnoise = "#declare leafnoise = array[6] {1,1,1,1,1,1}"
    $age_leafscale = 7
    $age_leafprob = 7
  when "Adult"
    $tree_recurs = 4
    $divisions = [4,2,3,5,0,0]
    $roots = "init_roots (2)"
    $simplify = "init_simplify (2,2)"
    $branchl = "init_branchl (3.8,0.3,0.6)"
    $branchw = "init_branchw (0.3,0.003,0.4)"
    $twigprob = "init_twigprob (0.7,0.8,0.95)"
    $leafnoise = "#declare leafnoise = array[6] {1,1,1,1,1,1}"
    $age_leafscale = 10
    $age_leafprob = 10
  when "Old"
    $tree_recurs = 5
    $divisions = [4,3,2,2,4,0]
    $roots = "init_roots (3)"
    $simplify = "init_simplify (3,3)"
    $branchl = "init_branchl (4.2,1.2,0.6)"
    $branchw = "init_branchw (0.45,0.003,0.4)"
    $twigprob = "init_twigprob (0.4,0.4,0.9)"
    $leafnoise = "#declare leafnoise = array[6] {1,1,1,1,1,1}"
    $age_leafscale = 11
    $age_leafprob = 12
  when "Very_old"
    $tree_recurs= 6
    $divisions = [5,4,2,3,2,3]
    $roots = "init_roots (3)"
    $simplify = "init_simplify (4,4)"
    $branchl = "init_branchl (4.9,1.3,0.6)"
    $branchw = "init_branchw (0.55,0.003,0.4)"
    $twigprob = "init_twigprob (0.3,0.3,0.8)"
    $leafnoise = "#declare leafnoise = array[6] {1,1,1,1,1,1}"
    $age_leafscale = 14
    $age_leafprob = 13
  when "Slender"
    $bearing = "#declare tendancy = <0.5,0.5,0.5>;"
    $rdev = "init_rdev (35, 35, 1)"
    $splinedev = "init_splinedev (0.2,0.2)"
  when "Weeping"
    $bearing = "#declare tendancy = <-0.5,-0.75,-0.5>;"
    $rdev = "init_rdev (80, 80, 1)"
    $splinedev = "init_splinedev (0.2,0.2)"
  when "Normal"
    $bearing = "#declare tendancy = <-0,-0.1,0>;"
    $rdev = "init_rdev (55, 55, 1)"
    $splinedev = "init_splinedev (0.4,0.4)"
  when "Twisty"
    $bearing = "#declare tendancy = <-0,-0.1,0>;"
    $rdev = "init_rdev (45, 35, 1)"
    $splinedev = "init_splinedev (0.9,0.9)"
  when "Straight"
    $bearing = "#declare tendancy = <-0,-0.1,0>;"
    $rdev = "init_rdev (55, 55, 1)"
    $splinedev = "init_splinedev (0.15,0.15)"
  when "Mapple"
    $leafs = ["maple1","maple2","maple3","maple1","maple2","maple3"]
  when "Prunus"
    $leafs = ["pissardi1","pissardi2","pissardi3","pissardi1","pissardi2","pissardi3"]
  when "Chestnut_tree"
    $leafs = ["chestnut1","chestnut2","chestnut3","chestnut1","chestnut2","chestnut3"]
  when "Light_maroon"
    $husk = "#declare branchtexture = texture { verylight_branch }"
  when "Light"
    $husk = "#declare branchtexture = texture { light_branch }"
  when "Medium"
    $husk = "#declare branchtexture = texture { standard_branch }"
  when "Dark"
    $husk = "#declare branchtexture = texture { dark_branch }"
  when "Several_white_flowers"
    $leafs[1] = "whiteflower"
  when "Several_pink_flowers"
    $leafs[1] = "pinkflower"
  when "Several_berries"
    $leafs[1] = "berries"
  when "Several_oranges"
    $leafs[1] = "orange1"
  when "Several_fruits"
    $leafs[1] = "fruit"
  when "Lot_of_white_flowers"
    $leafs[0] = "whiteflower"
  when "Lot_of_pink_flowers"
    $leafs[0] = "pinkflower"
  when "Lot_of_berries"
    $leafs[0] = "berries"
  when "Lot_of_oranges"
    $leafs[0] = "orange2"
  when "Lot_of_fruits"
    $leafs[0] = "fruit"
end
#m.collect! {|x| x + 10 }
end
# ----------------------------------- Outputs the trees macros
def write_trees( export_file, report_file)
model=Sketchup.active_model
entities=model.entities
params = []
entities.each do |e|
if( e.typename == "ComponentInstance" )
  name = e.definition.name
  if (name == "su2pov_tree")
    # get its params
    params = (get_tree_params e)
    #age:
    (trans_tree_param params[1])
    # bearing
    (trans_tree_param params[3])
    # species
    (trans_tree_param params[2])
    # season
    (trans_tree_param params[0])
    # Leaves scale:
    ls = ( $season_leafscale * $age_leafscale ) / 330.0
    # Leaves probability
    lp = [(( $season_leafprob * $age_leafprob * $divisions[0]) / 350.0), (( $season_leafprob * $age_leafprob * $divisions[1]) / 400.0), (( $season_leafprob * $age_leafprob * $divisions[2]) / 400.0), (( $season_leafprob * $age_leafprob * $divisions[3]) / 300.0), (( $season_leafprob * $age_leafprob * $divisions[4]) / 300), (( $season_leafprob * $age_leafprob * $divisions[5]) / 300.0)]
    #husk
    (trans_tree_param params[4])
    # fruits and blossoms
    (trans_tree_param params[5])
    #location
    trans = e.transformation.to_a
    x = trans[12].to_cm
    z = trans[13].to_cm
    y = trans[14].to_cm
    scale_x = trans[0]*100.0
    scale_y = trans[5]*100.0
    scale_z = trans[10]*100.0
    export_file.puts( "// " + params[2] + " tree: ")
    export_file.puts( "#declare recursionmax = " + $tree_recurs.to_s + ";" )
    export_file.puts( "#declare divisions = array[6] {" + $divisions[0].to_s + "," + $divisions[1].to_s + "," + $divisions[2].to_s + "," + $divisions[3].to_s + "," + $divisions[4].to_s + "," + $divisions[5].to_s + "}" )
    export_file.puts( "init_rand (" + (rand*200.0).to_i.to_s + "," + (rand*200.0).to_i.to_s + "," + (rand*200.0).to_i.to_s + ")" )
    export_file.puts( "init_tdev (0.4, 0.8, 0.9)" )
    export_file.puts( $rdev )
    export_file.puts( $twigprob )
    export_file.puts( $branchl )
    export_file.puts( $branchw )
    export_file.puts( $roots )
    export_file.puts( $splinedev )
    export_file.puts( $simplify )
    export_file.puts( $bearing )
    #export_file.puts( $leafs )
    export_file.puts( "#declare leafarray = array[6] {" + $leafs[0] + "," + $leafs[1] + "," + $leafs[2] + "," + $leafs[3] + "," + $leafs[4] + "," + $leafs[5] + "}" )
    
    export_file.puts( "#declare leafprob = array[6] {" + lp[0].to_s + "," + lp[1].to_s + "," + lp[2].to_s + "," + lp[3].to_s + "," + lp[4].to_s + "," + lp[5].to_s + "}"  )
    export_file.puts( "#declare leafscale = array[6] {" + ls.to_s + "," + ls.to_s + "," + ls.to_s + "," + ls.to_s + "," + ls.to_s + ","  + ls.to_s + "}"  )
    export_file.puts( $leafnoise )
    export_file.puts( $husk )
    
    # Location and scale
    export_file.puts( "// Tree location" )
    export_file.puts( "#declare global_transform = transform {" )
    export_file.puts( "    scale <" + scale_x.to_s + "," + scale_y.to_s + "," + scale_z.to_s + ">" )
    export_file.puts( "    rotate <0,0,0>" )
    export_file.puts( "    translate <" + x.to_s + "," + y.to_s + "," + z.to_s + ">" )
    export_file.puts( "}" )
    export_file.puts( "// Provide feedback" )
    export_file.puts( "#declare feedback = 2;" )
    export_file.puts( "// Call tree macro" )
    export_file.puts( "splinetree(0,0)" )
    export_file.puts( "" )
    $n_trees = $n_trees + 1
  end
end
end 
report_file.puts( "Tree(s): " + $n_trees.to_s )
end
# ------------------------------------------------------------------------
# ------------------------------------ LIGHT RELATED FUNCTIONS
# ------------------------------------------------------------------------

# ----------------------------------- Declare a bulb shape for visible light sources
def declare_bulb (export_file)
export_file.puts( "#declare Lightbulb = sphere {" )
export_file.puts( "    <0,0,0>,3" )
export_file.puts( "    texture { pigment {color rgb <1, 1, 0.8>}}" )
export_file.puts( "    finish {ambient 10}" )
export_file.puts( "    }" )
export_file.puts( "" )
end

# ----------------------------------- Outputs default ambient light when no radiosity in the render type
def write_ambient_type (s_type,export_file, r_type)
if( r_type == "Ray-tracing" )
  # Ray-tracing selected
  export_file.puts( "// Ambient light source" )
  case s_type
    when "Outdoor_Day"
      export_file.puts( "ambient_light <0.5,0.5,0.48>" )
    when "Outdoor_Night"
      export_file.puts( "ambient_light <0.1,0.1,0.2>" )
    when "Indoor_Day"
      export_file.puts( "ambient_light <1,1,0.85>" )
    when "Indoor_Night"
      export_file.puts( "ambient_light <0.1,0.1,0.1>" )
  end
  else
  # Radiosity selected
  export_file.puts( "" )
  export_file.puts( "// No ambient light source because of radiosity algorithm." )
  export_file.puts( "" )
end

end

# ----------------------------------- Get the sun position, outputs string formated for macro sunpos.inc 
def get_shadow_info ( export_file, report_file )
$shadow = Sketchup.active_model.shadow_info.to_a
location = $shadow[0][1]
country = $shadow[1][1]
latitude = $shadow[9][1]
longitude = $shadow[11][1]
north = $shadow[12][1]
date = $shadow[13][1]
tzoffset = $shadow[16][1]
# Unknown problem: must add 1 to get correct time zone (summer hour ?)
tzoffset = tzoffset + 2
sun_col = ""
found = false

time = date.to_a
sec = time[0]
min = time[1]
hour = time[2]
day = time[3]
month = time[4]
year = time [5]

# Search for the sun color and power
model=Sketchup.active_model
entities = model.entities
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    if( e.definition.name == "su2pov_sun")
      found = true
      dict_name="su2pov"
      dicts=e.attribute_dictionaries
      if( dicts != nil and dicts[dict_name] )
        $sun_power = e.get_attribute(dict_name, "power").to_f
        $sun_color = e.get_attribute(dict_name, "color")
        
      end
    end
  end
end

# default values if no sun found
if( found == false)
  $sun_power = 1.0
  $sun_color = "White"
  sun_col = ["rgb <1.0,1.0,1.0>*1.0"]
end
# default values if sun found but not set by the user
if( found == true) and ($sun_power == 0.0)
  sun_col = ["rgb <1.0,1.0,1.0>*1.0"]
  else
  col = Sketchup::Color.new $sun_color
        red = (col.red / 255.0)
        green = (col.green / 255.0)
        blue = (col.blue / 255.0)
        sun_col = "rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + ">*" + $sun_power.to_s
end

# Macro      SunPos(Year, Month, Day, Hour, Minute, Lstm, LAT, LONG)
export_file.puts( "light_source {" )
export_file.puts( "    SunPos(" + year.to_s + "," + month.to_s + "," + day.to_s + "," + hour.to_s + "," + min.to_s + "," + (tzoffset*15.0).to_s + "," + latitude.to_s + "," + longitude.to_s + ")" )
#export_file.puts( "    rgb 1" )
export_file.puts( sun_col )
export_file.puts( "}" )
export_file.puts( "" )

report_file.puts( "Sunlight parameters: Location: " + country + " - " + location + " - Coordinates: " + latitude.to_s + " * " + longitude.to_s )
report_file.puts( "Time zone: " + " " + (tzoffset-2).to_s + " - Current time and date: " + date.to_s)
(underline export_file, report_file)
end

# ----------------------------------- Outputs omni-directionnal light sources (point_lights)
def write_point_lights(export_file)
model=Sketchup.active_model
entities=model.entities
params = []
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if (name == "su2pov_pointlight")
      # get its color, power and visibility
      params = (get_light_params e)
      # Light status is "On"
      if( params[3] == "On" )
        #location
        trans = e.transformation.to_a
        x = trans[12].to_cm
        z = trans[13].to_cm
        y = trans[14].to_cm
        export_file.puts( "// Point light: " + params[0])
        export_file.puts( "light_source {" )
        export_file.puts( "<" + x.to_s + "," + y.to_s + "," + z.to_s + ">, " + params[4] + "*" + params[1].to_s)
        if( params[5] == "On" )
          export_file.puts( "    shadowless" )
        end
        # Light fading
        export_file.puts( "fade_distance " + $fade_distance.to_s )
        export_file.puts( "fade_power " + $fade_power.to_s )
        # Visible in the image or not
        if(params[2] == "On")
          export_file.puts( "looks_like { Lightbulb }" )
        end
        export_file.puts( "}" )
        # Glow effect, if any
        glow = (get_light_glow_effect e)
        if( glow[0] != nil )
          export_file.puts( "// Glow effect for pointlight " + params[0] )
          export_file.puts( "#declare camera_location = " + $camera_location + ";" )
          export_file.puts( "#declare camera_look_at = " + $camera_look_at + ";" )
          export_file.puts( "#declare effect_location = " + "<" + x.to_s + "," + y.to_s + "," + z.to_s + ">" + ";" )
          export_file.puts( "#declare effect_type = \"" + glow[0] + "\";" )
          export_file.puts( "#declare effect_scale = " + glow[1] + ";" )
          export_file.puts( "#declare effect_brightness = " + glow[2] + ";" )
          export_file.puts( "#declare effect_intensity = " + glow[3] + ";" )
          export_file.puts( "#declare effect_always_on_top = true;" )
          export_file.puts( "#include \"Lens.inc\"" )
          export_file.puts( "" )
          $n_pointlights = $n_pointlights + 1
        end
      else
        # Light status is "Off"
        export_file.puts( "// Point light: " + params[0] + " is off." )
      end
    end
  end
end
end

# ----------------------------------- Outputs omni-directionnal light sources with spectrum and flux (real point_lights)
def write_real_point_lights(export_file)
model=Sketchup.active_model
entities=model.entities
params = []
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if (name == "su2pov_realpointlight")
      # get its spectrum, power(lm) and visibility
      params = (get_real_light_params e)
      # Light status is "On"
      if( params[4] == "On" )
        #location
        trans = e.transformation.to_a
        x = trans[12].to_cm
        z = trans[13].to_cm
        y = trans[14].to_cm
        export_file.puts( "// Real Point light: " + params[0].to_s )
        export_file.puts( "object {" )
        export_file.puts( "    Light( " )
        export_file.puts( "    EmissiveSpectrum( " + params[1] + " )," )
        export_file.puts( "    " + params[2] + "," )
        export_file.puts( "    x*25, z*25, 4, 4, on" )
        export_file.puts( "    )" )
        export_file.puts( "translate <" + x.to_s + "," + y.to_s + "," + z.to_s + ">" )
        export_file.puts( "}" )
        
        # Visible in the image or not
        if(params[3] == "On")
          export_file.puts( "sphere {" )
          export_file.puts( "    <" + x.to_s + "," + y.to_s + "," + z.to_s + ">,3.5" )
          export_file.puts( "    texture { pigment {color rgb EmissiveSpectrum(" + params[1] + ") }}" )
          export_file.puts( "    finish {ambient 10}" )
          export_file.puts( "}" )
        end
        # Glow effect, if any
        glow = (get_light_glow_effect e)
        if( glow[0] != nil )
          export_file.puts( "// Glow effect for pointlight " + params[0] )
          export_file.puts( "#declare camera_location = " + $camera_location + ";" )
          export_file.puts( "#declare camera_look_at = " + $camera_look_at + ";" )
          export_file.puts( "#declare effect_location = " + "<" + x.to_s + "," + y.to_s + "," + z.to_s + ">" + ";" )
          export_file.puts( "#declare effect_type = \"" + glow[0] + "\";" )
          export_file.puts( "#declare effect_scale = " + glow[1] + ";" )
          export_file.puts( "#declare effect_brightness = " + glow[2] + ";" )
          export_file.puts( "#declare effect_intensity = " + glow[3] + ";" )
          export_file.puts( "#declare effect_always_on_top = true;" )
          export_file.puts( "#include \"Lens.inc\"" )
          export_file.puts( "" )
          $n_realpointlights = $n_realpointlights + 1
        end
      else
        # Light status is "Off"
        export_file.puts( "// Real Point light: " + params[0] + " is off." )
      end
    end
  end
end
end
# ----------------------------------- Outputs spotlight sources
def write_spotlights(export_file)
model=Sketchup.active_model
entities=model.entities
params = []
rad_fall_tight = []
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if (name == "su2pov_spotlight")
      # Get its params
      params = (get_light_params e)
      # Light status is "On"
      if( params[3] == "On" )      
        # get its radius, falloff and tightness parameters
        rad_fall_tight = (get_spotlight_rad_fall_tight e)
        #location
        trans = e.transformation.to_a
        x = trans[12].to_cm
        z = trans[13].to_cm
        y = trans[14].to_cm

        export_file.puts( "// Spot light: " + params[0])
        export_file.puts( "light_source {" )
        export_file.puts( "    <" + x.to_s + "," + y.to_s + "," + z.to_s + ">, " + params[4] + "*" + params[1].to_s)
        if( params[5] == "On" )
          export_file.puts( "    shadowless" )
        end
        export_file.puts( "    spotlight" )
        export_file.puts( "        radius " + rad_fall_tight[0].to_s)
        export_file.puts( "        falloff " + rad_fall_tight[1].to_s)
        export_file.puts( "        tightness " + rad_fall_tight[2].to_s)
        # Light fading
        export_file.puts( "        fade_distance " + $fade_distance.to_s )
        export_file.puts( "        fade_power " + $fade_power.to_s )
        # target:
        target = (get_spotlight_point_at e)
        export_file.puts( "        point_at <"+ target.x.to_cm.to_f.to_s + "," + target.z.to_cm.to_f.to_s + "," + target.y.to_cm.to_f.to_s + ">" )
        # Visible in the image or not
        if(params[2] == "On")
          export_file.puts( "        looks_like { Lightbulb }" )
        end
        export_file.puts( "    }" )
        # Glow effect, if any
        glow = (get_light_glow_effect e)
        if( glow[0] != nil )
          export_file.puts( "// Glow effect for spotlight " + params[0] )
          export_file.puts( "#declare camera_location = " + $camera_location + ";" )
          export_file.puts( "#declare camera_look_at = " + $camera_look_at + ";" )
          export_file.puts( "#declare effect_location = " + "<" + x.to_s + "," + y.to_s + "," + z.to_s + ">" + ";" )
          export_file.puts( "#declare effect_type = \"" + glow[0] + "\";" )
          export_file.puts( "#declare effect_scale = " + glow[1] + ";" )
          export_file.puts( "#declare effect_brightness = " + glow[2] + ";" )
          export_file.puts( "#declare effect_intensity = " + glow[3] + ";" )
          export_file.puts( "#declare effect_always_on_top = true;" )
          export_file.puts( "#include \"Lens.inc\"" )
          export_file.puts( "" )
        end
        $n_spotlights = $n_spotlights + 1
        
        else
          # Light status is "Off"
          export_file.puts( "// Spot light: " + params[0] + " is off." )
      end
    end
  end
end
end

# ----------------------------------- Outputs spotlight sources
def write_arealights(export_file)
model=Sketchup.active_model
entities=model.entities
params = []
rad_fall_tight = []
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if (name == "su2pov_arealight")
      # Get its params [name, pow, areatype, nx, ny, status, return_col, shadowless]
      params = (get_arealight_params e)
      # Light status is "On"
      if( params[5] == "On" )
        #location
        trans = e.transformation.to_a
        x = trans[12].to_cm
        z = trans[13].to_cm
        y = trans[14].to_cm
        x_scale = trans[0]
        y_scale = trans[5]
        z_scale = trans[10]
        target = (get_arealight_point_at e)
        vectors = (get_arealight_vectors e)
        export_file.puts( "// Area light: " + params[0])
        export_file.puts( "light_source {" )
        export_file.puts( "    <" + x.to_s + "," + y.to_s + "," + z.to_s + ">, " + params[6] + "*" + params[1].to_s)
        if( params[7] == "On" )
          export_file.puts( "    shadowless" )
        end
        export_file.puts( "    area_light <" + vectors[0][0].to_cm.to_s + "," + vectors[0][2].to_cm.to_s + "," + vectors[0][1].to_cm.to_s + ">, <" + vectors[1][0].to_cm.to_s + "," + vectors[1][2].to_cm.to_s + "," + vectors[1][1].to_cm.to_s + ">," + params[3].to_s + "," + params[4].to_s )
        export_file.puts( "    adaptive 1" )
        export_file.puts( "    jitter" )
        
        if( params[2] == "Spotlight" )
          export_file.puts( "    spotlight" )
          export_file.puts( "    point_at <" + target[0].to_cm.to_f.to_s + "," + target[2].to_cm.to_f.to_s + "," + target[1].to_cm.to_f.to_s + ">" )
          export_file.puts( "    tightness 0" )
          export_file.puts( "    radius 30" )
          export_file.puts( "    falloff 40" )
        end
        # Light fading
        export_file.puts( "        fade_distance " + $fade_distance.to_s )
        export_file.puts( "        fade_power " + $fade_power.to_s )
        export_file.puts( "    }" )
        $n_arealights = $n_arealights + 1
        else
          # Light status is "Off"
          export_file.puts( "// Area light: " + params[0] + " is off." )
      end
    end
  end
end
end

# ----------------------------------- Outputs box media lights
def write_boxmedialights(export_file)
model=Sketchup.active_model
entities=model.entities
color_power_vis = []
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if (name == "su2pov_boxmedialight")
      # get its bounding box
      bb = e.bounds
      corner_1 = bb.min
      corner_2 = bb.max
      # parameters
      dict_name="su2pov"
      dicts=e.attribute_dictionaries
      dict_key_scattering = e.get_attribute(dict_name, "scattering")
      dict_key_samples = e.get_attribute(dict_name, "samples")
      dict_key_extinction = e.get_attribute(dict_name, "extinction")
      # default values if not set by the user
      dict_key_scattering = "1" if not dict_key_scattering
      dict_key_samples = "30" if not dict_key_samples
      dict_key_extinction = "0.01" if not dict_key_extinction
      
      export_file.puts( "// Box media light effect" )
      export_file.puts( "box" )
      export_file.puts( " { <" + corner_1.x.to_f.to_cm.to_s + "," + corner_1.z.to_f.to_cm.to_s + "," + corner_1.y.to_f.to_cm.to_s + ">, <" + corner_2.x.to_f.to_cm.to_s + "," + corner_2.z.to_f.to_cm.to_s + "," + corner_2.y.to_f.to_cm.to_s + ">" )
      export_file.puts( "    pigment {rgbt <0.98,0.98,0.45,1.0> } hollow" )
      export_file.puts( "    interior" )
      export_file.puts( "    { media" )
      export_file.puts( "        { scattering { "+ dict_key_scattering + ", 0.002 extinction " + dict_key_extinction + " }" )
      export_file.puts( "          samples " + dict_key_samples + ",100" )
      export_file.puts( "    }" )
      export_file.puts( "          }" )
      export_file.puts( "}" )
      export_file.puts( "" )
     $n_boxmedialights = $n_boxmedialights + 1
    end
  end
end
end

# ----------------------------------- Outputs cone media lights
def write_conemedialights(export_file)
model=Sketchup.active_model
entities=model.entities
color_power_vis = []
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if (name == "su2pov_conemedialight")
      # get its bounding box
      bb = e.bounds
      corner_1 = bb.min
      corner_2 = bb.max
      diam = corner_2.x.to_f.to_cm - corner_1.x.to_f.to_cm
      #location
      trans = e.transformation.to_a
      x = trans[12].to_cm
      y = trans[13].to_cm
      z = trans[14].to_cm
      x_scale = trans[0]
      y_scale = trans[5]
      z_scale = trans[10]
      diam = [(x_scale * 200.0), (y_scale * 200.0)] # base radius is 50.0...
      # target:
      target = (get_spotlight_point_at e)
      x_target = target.x.to_cm.to_f
      y_target = target.y.to_cm.to_f
      z_target = target.z.to_cm.to_f
        
      # parameters
      dict_name="su2pov"
      dicts=e.attribute_dictionaries
      dict_key_scattering = e.get_attribute(dict_name, "scattering")
      dict_key_samples = e.get_attribute(dict_name, "samples")
      dict_key_extinction = e.get_attribute(dict_name, "extinction")
      # default values if not set by the user
      dict_key_scattering = "1" if not dict_key_scattering
      dict_key_samples = "30" if not dict_key_samples
      dict_key_extinction = "0.01" if not dict_key_extinction
      
      export_file.puts( "// Cone media light effect" )
      export_file.puts( "cone" )
      #export_file.puts( " { <" + x.to_f.to_s + "," + corner_1.z.to_f.to_s + "," + y.to_f.to_s + ">, " + diam.to_s + ", <" + x.to_f.to_s + "," + z.to_f.to_s + "," + y.to_f.to_s + ">, 0.0" )
      
      export_file.puts( " { <" + x_target.to_s + "," + z_target.to_s + "," + y_target.to_s + ">, " + diam.max.to_s + ", <" + x.to_f.to_s + "," + z.to_f.to_s + "," + y.to_f.to_s + ">, 0.0" )
      export_file.puts( "    pigment {rgbt <0.98,0.98,0.45,1.0> } hollow" )
      export_file.puts( "    interior" )
      export_file.puts( "    { media" )
      export_file.puts( "        { scattering { "+ dict_key_scattering + ", 0.002 extinction " + dict_key_extinction + " }" )
      export_file.puts( "          samples " + dict_key_samples + ",100" )
      export_file.puts( "    }" )
      export_file.puts( "          }" )
      export_file.puts( "}" )
      export_file.puts( "" )
     $n_conemedialights = $n_conemedialights + 1
    end
  end
end
end

# ----------------------------------- Get the params of light (except arealights)
# ----------------------------------- returns an array
def get_light_params( light )
model=Sketchup.active_model
dict_name="su2pov"
dicts=light.attribute_dictionaries
ret = []
if( dicts != nil and dicts[dict_name] )
  light_name = light.get_attribute(dict_name, "name")
  pow = light.get_attribute(dict_name, "power").to_f
  vis = light.get_attribute(dict_name, "visibility")
  status = light.get_attribute(dict_name, "status")
  color = light.get_attribute(dict_name, "color")
  shadowless = light.get_attribute(dict_name, "shadowless")
  else
  ret = ["noname", "1", "On", "On", "rgb <1.0,1.0,1.0>", "Off"]
end

# Component found, real values or default values if not set by the user
if( color != nil)
  col = Sketchup::Color.new color
  red = (col.red / 255.0)
  green = (col.green / 255.0)
  blue = (col.blue / 255.0)
  return_col = "rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + ">"
  ret = [light_name, pow, vis, status, return_col, shadowless]
  else
  ret = ["noname", "1", "On", "On", "rgb <1.0,1.0,1.0>", "Off"]
end
return ret
end

# ----------------------------------- Get the params of real light, returns an array
def get_real_light_params( light )
model=Sketchup.active_model
dict_name="su2pov"
dicts=light.attribute_dictionaries
#ret = []
if( dicts != nil and dicts[dict_name] )
  light_name = light.get_attribute(dict_name, "name")
  spectrum = light.get_attribute(dict_name, "spectrum")
  flux = light.get_attribute(dict_name, "flux")
  vis = light.get_attribute(dict_name, "visibility")
  status = light.get_attribute(dict_name, "status")
  ret = [light_name, spectrum, flux, vis, status]
  else
  ret = ["noname", "1000", "Lm_Incandescent_60w", "On", "On"]
end
return ret
end
  
# ----------------------------------- Get the params of arealights, returns an array
def get_arealight_params( light )
model=Sketchup.active_model
dict_name="su2pov"
dicts=light.attribute_dictionaries
ret = []
if( dicts != nil and dicts[dict_name] )
  name = light.get_attribute(dict_name, "name")
  pow = light.get_attribute(dict_name, "power").to_f
  areatype = light.get_attribute(dict_name, "areatype")
  nx = light.get_attribute(dict_name, "nx")
  ny = light.get_attribute(dict_name, "ny")
  status = light.get_attribute(dict_name, "status")
  color = light.get_attribute(dict_name, "color")
  shadowless = light.get_attribute(dict_name, "shadowless")
  col = Sketchup::Color.new color
  red = (col.red / 255.0)
  green = (col.green / 255.0)
  blue = (col.blue / 255.0)
  return_col = "rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + ">"
  ret = [name, pow, areatype, nx, ny, status, return_col, shadowless]
  else
  ret = ["noname", "1", "Omnidirectionnal", "4", "4", "Off", "rgb <1.0,1.0,1.0>", "Off"]
end
return ret
end

# ----------------------------------- Get the radius, falloff and tightness parameters of a spotlight
def get_spotlight_rad_fall_tight( light )
dict_name="su2pov"
dicts=light.attribute_dictionaries
ret = []
if( dicts != nil and dicts[dict_name] )
  spotlight_rad = light.get_attribute(dict_name, "radius").to_f
  fall = light.get_attribute(dict_name, "falloff").to_f
  tight = light.get_attribute(dict_name, "tightness").to_f
  else
  ret = ["20", "40", "20"]
end
# default values if not set by the user
if( spotlight_rad == nil )
  ret = ["20", "40", "20"]
  else
  ret = [spotlight_rad, fall, tight]
end
return ret
end

# ----------------------------------- Set the color and power of selected sun
def set_sunlight_params
model=Sketchup.active_model
#all_colors = Sketchup::Color.names
(create_colors_dropdowns)
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("Sun not selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_sun" )
#default values
dict_key_power = "1" if not dict_key_power
dict_key_color_1 = "AliceBlue" if not dict_key_color_1
dict_key_color_2 = "-" if not dict_key_color_2
dict_key_color_3 = "-" if not dict_key_color_3
# Retrieve color and power if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_power = e.get_attribute(dict_name, "power")
  dict_key_color = e.get_attribute(dict_name, "color")
  else
  UI.messagebox("No parameters set for the sun.")
end
dict_key_colors = (set_dict_key_colors dict_key_color)
dict_key_color_1 = dict_key_colors[0]
dict_key_color_2 = dict_key_colors[1]
dict_key_color_3 = dict_key_colors[2]

#Dialog box: power, color
#colors_list = [all_colors.join("|")]
colors_list_1 = [$colors_ag.join("|")]
colors_list_2 = [$colors_ho.join("|")]
colors_list_3 = [$colors_pz.join("|")]
array_of_dropdowns = [colors_list_1, colors_list_2, colors_list_3]
prompts=["Sun color [A-G]  ", "Sun color [H-O]  ", "Sun color [P-Z]  ","Sun power  "]
values=[dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_power]
dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_power  = inputbox prompts, values, array_of_dropdowns, "Sun parameters"

dict_name="su2pov"
dict_key_color = (select_color dict_key_color_1, dict_key_color_2, dict_key_color_3)
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"name", "sun")
e.set_attribute(dict_name,"power",dict_key_power)
e.set_attribute(dict_name,"color",dict_key_color)
e.set_attribute(dict_name,"visibility", "Off")
e.set_attribute(dict_name,"status", "On")
else
  UI.messagebox("This is not the sun.")
end
end

# ----------------------------------- Set the color and power of selected pointlight
def set_pointlight_params
model=Sketchup.active_model
#all_colors = Sketchup::Color.names
(create_colors_dropdowns)
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("No pointlight selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_pointlight" )
#default values
dict_key_name = "Pointlight 1" if not dict_key_name
dict_key_color_1 = "AliceBlue" if not dict_key_color_1
dict_key_color_2 = "-" if not dict_key_color_2
dict_key_color_3 = "-" if not dict_key_color_3
dict_key_power = "1" if not dict_key_power
dict_key_visibility = "On" if not dict_key_visibility
dict_key_status = "On" if not dict_key_status
dict_key_shadowless = "Off" if not dict_key_shadowless
# Retrieve color and power if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_name = e.get_attribute(dict_name, "name")
  dict_key_power = e.get_attribute(dict_name, "power")
  dict_key_visibility = e.get_attribute(dict_name, "visibility")
  dict_key_status = e.get_attribute(dict_name, "status")
  dict_key_color = e.get_attribute(dict_name, "color")
  dict_key_shadowless = e.get_attribute(dict_name, "shadowless")
  else
  UI.messagebox("No parameters set for this pointlight.")
end
dict_key_colors = (set_dict_key_colors dict_key_color)
dict_key_color_1 = dict_key_colors[0]
dict_key_color_2 = dict_key_colors[1]
dict_key_color_3 = dict_key_colors[2]

#Dialog box 1: Name, power
prompts=["Light name  ", "Light power  "]
values=[dict_key_name, dict_key_power]
dict_key_name, dict_key_power = inputbox prompts, values, "Light main parameters"

#Dialog box 2: other params
#colors_list = [all_colors.join("|")]
colors_list_1 = [$colors_ag.join("|")]
colors_list_2 = [$colors_ho.join("|")]
colors_list_3 = [$colors_pz.join("|")]
vis_list = %w[On Off].join("|")
status_list = %w[On Off].join("|")
shadowless_list = %w[On Off].join("|")
dropdowns = [vis_list, status_list, colors_list_1, colors_list_2, colors_list_3, shadowless_list]
prompts=["Ligth visibility ", "Status  ", "Light color [A_G]  ", "Light color [H_O]  ", "Light color [P_Z]  ", "Shadowless  "]
values=[dict_key_visibility, dict_key_status, dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_shadowless]
dict_key_visibility, dict_key_status, dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_shadowless = inputbox prompts, values, dropdowns, "Light other parameters"

dict_name="su2pov"
dict_key_color = (select_color dict_key_color_1, dict_key_color_2, dict_key_color_3)
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"name",dict_key_name)
e.set_attribute(dict_name,"power",dict_key_power)
e.set_attribute(dict_name,"visibility",dict_key_visibility)
e.set_attribute(dict_name,"status",dict_key_status)
e.set_attribute(dict_name,"color",dict_key_color)
e.set_attribute(dict_name,"shadowless",dict_key_shadowless)
else
  UI.messagebox("This is not a pointlight.")
end
end

# ----------------------------------- Set the color and power of selected realpointlight
def set_realpointlight_params
model=Sketchup.active_model
#all_colors = Sketchup::Color.names
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("No realpointlight selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_realpointlight" )
# Retrieve color and power if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_name = e.get_attribute(dict_name, "name")
  dict_key_spectrum = e.get_attribute(dict_name, "spectrum")
  dict_key_flux = e.get_attribute(dict_name, "flux")
  dict_key_visibility = e.get_attribute(dict_name, "visibility")
  dict_key_status = e.get_attribute(dict_name, "status")
  else
  #default values
  dict_key_name = "Realpointlight 1" if not dict_key_name
  dict_key_spectrum = "ES_GE_SW_Incandescent_60w" if not dict_key_spectrum
  dict_key_flux = "1000" if not dict_key_flux
  dict_key_visibility = "On" if not dict_key_visibility
  dict_key_status = "On" if not dict_key_status
  UI.messagebox("No parameters set for this realpointlight.")
end

#Dialog box 1:
prompts=["Light name  ", "Light flux  "]
values=[dict_key_name, dict_key_flux]
results = inputbox prompts, values, "Real light name"
dict_key_name = results[0]
dict_key_flux = results[1]

#Dialog box 2:
spectrum_list = %w[ES_GE_SW_Incandescent_60w ES_GE_SW_Incandescent_100w ES_Nikon_SB16_XenonFlash ES_GTE_341_Warm ES_GTE_341_Cool ES_Phillips_PLS_11w ES_Phillips_Mastercolor_3K ES_Phillips_HPS ES_Osram_CoolBeam_20w ES_Osram_CoolFluor_36w ES_Solux_Halog4700K ES_Cornell_Box_Light ES_Sunlight ES_Extraterrestrial_Sun ES_Daylight_Fluor ES_Mitsubishi_Metal_Halide ES_Mitsubishi_Daylight_Fluorescent ES_Mitsubishi_Moon_Fluorescent ES_Mitsubishi_Standard_Fluorescent].join("|")
vis_list = %w[On Off].join("|")
status_list = %w[On Off].join("|")
dropdowns = [spectrum_list, vis_list, status_list]
prompts=["Light spectrum  ", "Ligth visibility ", "Status  "]
values=[dict_key_spectrum, dict_key_visibility, dict_key_status]
dict_key_spectrum, dict_key_visibility, dict_key_status = inputbox prompts, values, dropdowns, "Real light parameters"

dict_name="su2pov"
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"name",dict_key_name)
e.set_attribute(dict_name,"spectrum",dict_key_spectrum)
e.set_attribute(dict_name,"flux",dict_key_flux)
e.set_attribute(dict_name,"visibility",dict_key_visibility)
e.set_attribute(dict_name,"status",dict_key_status)
else
  UI.messagebox("This is not a realpointlight.")
end
end

# ----------------------------------- Set the parameters of selected spotlight
def set_spotlight_params
model=Sketchup.active_model
#all_colors = Sketchup::Color.names
(create_colors_dropdowns)
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("No spotlight selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_spotlight" )
#default values
dict_key_name = "Spotlight 1" if not dict_key_name
dict_key_color_1 = "AliceBlue" if not dict_key_color_1
dict_key_color_2 = "-" if not dict_key_color_2
dict_key_color_3 = "-" if not dict_key_color_3
dict_key_power = "1" if not dict_key_power
dict_key_visibility = "On" if not dict_key_visibility
dict_key_status = "On" if not dict_key_status
dict_key_radius = "15" if not dict_key_radius
dict_key_falloff = "20" if not dict_key_falloff
dict_key_tightness = "10" if not dict_key_tightness
dict_key_shadowless = "Off" if not dict_key_shadowless

# Retrieve parameters if already set
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_name = e.get_attribute(dict_name, "name")
  dict_key_power = e.get_attribute(dict_name, "power")
  dict_key_visibility = e.get_attribute(dict_name, "visibility")
  dict_key_status = e.get_attribute(dict_name, "status")
  dict_key_color = e.get_attribute(dict_name, "color")
  dict_key_radius = e.get_attribute(dict_name, "radius")
  dict_key_falloff = e.get_attribute(dict_name, "falloff")
  dict_key_tightness = e.get_attribute(dict_name, "tightness")
  dict_key_shadowless = e.get_attribute(dict_name, "shadowless")
  else
  UI.messagebox("No parameters set for this spotlight.")
end
dict_key_colors = (set_dict_key_colors dict_key_color)
dict_key_color_1 = dict_key_colors[0]
dict_key_color_2 = dict_key_colors[1]
dict_key_color_3 = dict_key_colors[2]

#Dialog box 1:
#colors_list = [all_colors.join("|")]
prompts=["Light name  ", "Ligth power ", "Radius  ", "Falloff  ", "Tightness  "]
values=[dict_key_name, dict_key_power, dict_key_radius, dict_key_falloff, dict_key_tightness]
dict_key_name, dict_key_power, dict_key_radius, dict_key_falloff, dict_key_tightness = inputbox prompts,values, "Spotlight main parameters"

#Dialog box 2:
vis_list = %w[On Off].join("|")
status_list = %w[On Off].join("|")
colors_list_1 = [$colors_ag.join("|")]
colors_list_2 = [$colors_ho.join("|")]
colors_list_3 = [$colors_pz.join("|")]
shadowless_list = %w[On Off].join("|")
dropdowns = [vis_list, status_list, colors_list_1, colors_list_2, colors_list_3, shadowless_list]
prompts=["Light visibility  ",  "Status  ", "Light color [A-G]  ", "Light color [H-O]  ", "Light color [P-Z]  ", "Shadowless  "]
values=[dict_key_visibility, dict_key_status, dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_shadowless]
dict_key_visibility, dict_key_status, dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_shadowless = inputbox prompts,values, dropdowns, "Spotlight other parameters"
    
dict_name="su2pov"
dict_key_color = (select_color dict_key_color_1, dict_key_color_2, dict_key_color_3)
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"name",dict_key_name)
e.set_attribute(dict_name,"power",dict_key_power)
e.set_attribute(dict_name,"visibility",dict_key_visibility)
e.set_attribute(dict_name,"status",dict_key_status)
e.set_attribute(dict_name,"color",dict_key_color)
e.set_attribute(dict_name,"radius",dict_key_radius)
e.set_attribute(dict_name,"falloff",dict_key_falloff)
e.set_attribute(dict_name,"tightness",dict_key_tightness)
e.set_attribute(dict_name,"shadowless",dict_key_shadowless)
else
  UI.messagebox("This is not a spotlight.")
end
end

# ----------------------------------- Set the medialight parameters
def set_medialight_params
model=Sketchup.active_model
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("Medialight not selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_boxmedialight" ) or ( name == "su2pov_conemedialight" )
#default values
dict_key_scattering = "1" if not dict_key_scattering
dict_key_samples = "30" if not dict_key_samples
dict_key_extinction = "0.01" if not dict_key_extinction

# Retrieve old parameters if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_scattering = e.get_attribute(dict_name, "scattering")
  dict_key_samples = e.get_attribute(dict_name, "samples")
  dict_key_extinction = e.get_attribute(dict_name, "extinction")
  else
  UI.messagebox("No parameters set for the medialight.")
end

#Dialog box:

prompts=["Scattering [0..5] (method) ", "Samples [1...100] (minimum number, the higher the better)  ", "Extinction [0..1] (the lower the more realistic)  "]
values=[dict_key_scattering, dict_key_samples, dict_key_extinction]
dict_key_scattering, dict_key_samples, dict_key_extinction  = inputbox prompts, values, "Medialight parameters"

dict_name="su2pov"
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"scattering",dict_key_scattering)
e.set_attribute(dict_name,"samples",dict_key_samples)
e.set_attribute(dict_name,"extinction",dict_key_extinction)
else
  UI.messagebox("This is not a medialight.")
end
end

# ----------------------------------- Set the parameters of selected arealight
def set_arealight_params
model=Sketchup.active_model
(create_colors_dropdowns)
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("No arealight selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_arealight" )
#default values
dict_key_name = "Arealight 1" if not dict_key_name
dict_key_power = "1" if not dict_key_power
dict_key_nx = "4" if not dict_key_nx
dict_key_ny = "4" if not dict_key_ny
dict_key_power = "1" if not dict_key_power
dict_key_areatype = "Omnidirectionnal" if not dict_key_areatype
dict_key_color_1 = "AliceBlue" if not dict_key_color_1
dict_key_color_2 = "-" if not dict_key_color_2
dict_key_color_3 = "-" if not dict_key_color_3
dict_key_status = "On" if not dict_key_status
dict_key_shadowless = "Off" if not dict_key_shadowless
# Retrieve parameters if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_name = e.get_attribute(dict_name, "name")
  dict_key_power = e.get_attribute(dict_name, "power")
  dict_key_areatype = e.get_attribute(dict_name, "areatype")
  dict_key_nx = e.get_attribute(dict_name, "nx")
  dict_key_ny = e.get_attribute(dict_name, "ny")
  dict_key_status = e.get_attribute(dict_name, "status")
  dict_key_color = e.get_attribute(dict_name, "color")
  dict_key_shadowless = e.get_attribute(dict_name, "shadowless")
  else
  UI.messagebox("No parameters set for this arealight.")
end
dict_key_colors = (set_dict_key_colors dict_key_color)
dict_key_color_1 = dict_key_colors[0]
dict_key_color_2 = dict_key_colors[1]
dict_key_color_3 = dict_key_colors[2]

#Dialog box 1: Name, power
prompts=["Light name  ", "Light power  ", "Number od lights along the width  ", "Number od lights along the height  "]
values=[dict_key_name, dict_key_power, dict_key_nx, dict_key_ny]
dict_key_name, dict_key_power, dict_key_nx, dict_key_ny = inputbox prompts, values, "Arealight main parameters"

#Dialog box 2: other params
#colors_list = [all_colors.join("|")]
colors_list_1 = [$colors_ag.join("|")]
colors_list_2 = [$colors_ho.join("|")]
colors_list_3 = [$colors_pz.join("|")]
status_list = %w[On Off].join("|")
areatype_list = %w[Omnidirectionnal Spotlight].join("|")
shadowless_list = %w[On Off].join("|")
dropdowns = [status_list, areatype_list, colors_list_1, colors_list_2, colors_list_3, shadowless_list]
prompts=["Status  ", "Type  ", "Light color [A_G]  ", "Light color [H_O]  ", "Light color [P_Z]  ", "Shadowless  "]
values=[dict_key_status, dict_key_areatype, dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_shadowless]
dict_key_status, dict_key_areatype, dict_key_color_1, dict_key_color_2, dict_key_color_3, dict_key_shadowless = inputbox prompts, values, dropdowns, "Arealight other parameters"

dict_name="su2pov"
dict_key_color = (select_color dict_key_color_1, dict_key_color_2, dict_key_color_3)
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"name",dict_key_name)
e.set_attribute(dict_name,"power",dict_key_power)
e.set_attribute(dict_name, "areatype", dict_key_areatype)
e.set_attribute(dict_name,"nx",dict_key_nx)
e.set_attribute(dict_name,"ny",dict_key_ny)
e.set_attribute(dict_name,"status",dict_key_status)
e.set_attribute(dict_name,"color",dict_key_color)
e.set_attribute(dict_name,"shadowless",dict_key_shadowless)
else
  UI.messagebox("This is not an arealight.")
end
end

# ----------------------------------- Given a selected spotlight, rotate-translate the selected conemedialight accordingly
def assign_conemedialight_to_spot
model=Sketchup.active_model
ss = model.selection
if( ss.length != 2 )
  UI.messagebox("Select a spotlight and a conemedialight.")
  return nil
end

if( ss[0].typename != "ComponentInstance" ) or ( ss[1].typename != "ComponentInstance" )
  UI.messagebox("Selected objects are not components.")
  return nil
end

if( ss[0].typename != "ComponentInstance" ) or ( ss[1].typename != "ComponentInstance" )
  UI.messagebox("Selected objects are not components.")
  return nil
end

name_1 = ss[0].definition.name
name_2 = ss[1].definition.name
if( name_1 == "su2pov_spotlight") and ( name_2 == "su2pov_conemedialight")
  spot = ss[0]
  cone = ss[1]
  else
  spot = ss[1]
  cone = ss[0]
end

cone_org = cone.transformation.origin
translate = Geom::Transformation.new(cone_org).inverse
spot_trans = spot.transformation 
cone.transform!(translate)
cone.transform!(spot_trans)

end

# ----------------------------------- Set a glow effect to a pointlight or a spotlight
def set_light_glow_effect
model=Sketchup.active_model
light = model.selection[0]
if light
  type = light.typename
  else
  UI.messagebox("No light selected")
  return nil
end
if(type == "ComponentInstance" )
  name = light.definition.name
end

if ( name == "su2pov_pointlight" ) or ( name == "su2pov_realpointlight" ) or ( name == "su2pov_spotlight" )

# Retrieve old parameters if any
dict_name="su2pov"
dicts=light.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_type = light.get_attribute(dict_name, "type")
  dict_key_scale = light.get_attribute(dict_name, "scale")
  dict_key_brightness = light.get_attribute(dict_name, "brightness")
  dict_key_intensity = light.get_attribute(dict_name, "intensity")
  else
  UI.messagebox("No parameters set for the glow effect.")
end

#default values
  dict_key_type = "Sparkle" if not dict_key_type
  dict_key_scale = "1" if not dict_key_scale
  dict_key_brightness = "1" if not dict_key_brightness
  dict_key_intensity = "1" if not dict_key_intensity
  
#Dialog box:
lens_effects_list = %w[Camera Camera2 Candle Diamond Headlight Headlight2 Rainbow Rainbow2 SoftGlow Sparkle Sparkle2 Spotlight Spotlight2 Star Star2 Sun Sun2 Sun3 Undersea].join("|")
prompts=["Type  ", "Scale  ", "Brightness  ", "Intensity  "]
values=[dict_key_type, dict_key_scale, dict_key_brightness, dict_key_intensity]
dropdown = [lens_effects_list]
dict_key_type, dict_key_scale, dict_key_brightness, dict_key_intensity  = inputbox prompts, values, dropdown, "Light glow parameters"

dict_name="su2pov"
light.attribute_dictionary(dict_name, true)
light.set_attribute(dict_name,"type",dict_key_type)
light.set_attribute(dict_name,"scale",dict_key_scale)
light.set_attribute(dict_name,"brightness",dict_key_brightness)
light.set_attribute(dict_name,"intensity",dict_key_intensity)
else
  UI.messagebox("This is not a pointlight or a spotlight.")
end
end

# ----------------------------------- Set a glow effect to a pointlight or a spotlight
def delete_light_glow_effect
model=Sketchup.active_model
light = model.selection[0]
if light
  type = light.typename
  else
  UI.messagebox("No light selected")
  return nil
end
if(type == "ComponentInstance" )
  name = light.definition.name
end

if ( name == "su2pov_pointlight" ) or ( name == "su2pov_spotlight" ) or ( name == "su2pov_realpointlight" )
  dict_name="su2pov"
  light.attribute_dictionary(dict_name, true)
  light.set_attribute(dict_name,"type", nil)
  light.set_attribute(dict_name,"scale",nil)
  light.set_attribute(dict_name,"brightness",nil)
  light.set_attribute(dict_name,"intensity",nil)
  UI.messagebox("Glow effect deleted.")
else
  UI.messagebox("This is not a pointlight or a spotlight.")
end
end

# ----------------------------------- Get a glow effect of a pointlight or a spotlight
def get_light_glow_effect( light )
dict_name="su2pov"
dicts=light.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_type = light.get_attribute(dict_name, "type")
  dict_key_scale = light.get_attribute(dict_name, "scale")
  dict_key_brightness = light.get_attribute(dict_name, "brightness")
  dict_key_intensity = light.get_attribute(dict_name, "intensity")
  ret = [dict_key_type, dict_key_scale, dict_key_brightness, dict_key_intensity]
  else
  ret = [nil, nil, nil, nil]
end
return ret
end
# ------------------------------------------------------------------------
# ------------------------------------ SKY and BACKGROUND RELATED FUNCTIONS
# ------------------------------------------------------------------------

# ----------------------------------- Let the user select the weather conditions
def set_weather_settings
#default values
$sky_type = "Clear" if not $sky_type
#$snow_use = "No_snow" if not $snow_use

skies_list = %w[Clear Gradient Clouds Stars].join("|")
#fog_list = %w[No_fog Add_fog].join("|")
#snow_list = %w[No_snow Add_snow].join("|")
dropdowns = [skies_list]
prompts = ["Sky type  "]
values = [$sky_type]
results = inputbox prompts, values, dropdowns, "Weather settings"
$sky_type = results[0]
end

# ----------------------------------- Let the user select the clouds types
def set_clouds_settings
clouds_list = %w[Blue_Scattered Blue_Wide_Scattered Blue_Grey_Cumulonimbus Blue_Grey_Puffy Grey_Stormy High_Cirrus Middle_Cirrus Low_Cirrus Scattered Overcast Rainy Custom].join("|")
dropdown = [clouds_list]
#default value
$clouds_type = "Blue_Scattered" if not $clouds_type
prompts = ["Clouds type  "]
values = [$clouds_type]
$clouds_type = inputbox prompts,values, dropdown, "Clouds parameters"
$clouds_type = $clouds_type[0]
end

# ----------------------------------- Outputs background depending on "type": clear, gradient, clouds.
def write_background (export_file, report_file)
# Init for clouds
array_Blue_Scattered = ["bozo", "2", "0.0", "SkyBlue", "0.3", "0.5", "rgb 0.6", "1.0", "SkyBlue", "0.3", "20", "0.65", "2", "<1.2, 0.6, 1>"]
array_Blue_Wide_Scattered = ["marble", "1.2", "0.0", "SkyBlue", "0.4", "0.5", "rgb 0.8", "1.0", "SkyBlue", "0.3", "30", "0.65", "2", "<4, 3.6, 2.8>"]
array_Blue_Grey_Cumulonimbus = ["bozo", "2", "0.0", "SkyBlue", "0.8", "0.5", "rgb 0.8", "1.0", "SkyBlue", "0.3", "30", "0.65", "2", "<1, 0.7, 3>"]
array_Blue_Grey_Puffy = ["bozo", "1", "0.0", "SkyBlue", "1", "0.5", "rgb 0.95", "1.0", "SkyBlue", "1", "10", "0.7", "2", "<1, 1, 1>"]
array_Grey_Stormy = ["bozo", "3", "0.0", "SkyBlue", "1", "0.5", "LightGrey", "1.0", "SkyBlue", "1", "20", "0.55", "2.5", "<1, 1, 1>"]

# default values if not set by the user
$fog_use = (get_rendering_option "DisplayFog")
$sky_type = "Clear" if not $sky_type
#$fog_use = "No_fog" if not $fog_use
$snow_use = "No_snow" if not $snow_use

Sketchup.set_status_text("Exporting background...")

export_file.puts( "// Background" )
case $sky_type
  when "Clear"
    #get the SU solid color background
    solid_color = (get_rendering_option "BackgroundColor")
    col_r = solid_color.red / 255.0
    col_g = solid_color.green / 255.0
    col_b = solid_color.blue / 255.0
    export_file.puts( "sky_sphere {pigment {rgb <" + col_r.to_s + "," + col_g.to_s + "," + col_b.to_s + ">}}" )
  when "Gradient"
    #get the SU gradient sky background
    sky_color = (get_rendering_option "SkyColor")
    s_red = sky_color.red / 255.0
    s_green = sky_color.green / 255.0
    s_blue = sky_color.blue / 255.0
    s_col = "rgb <" + s_red.to_s + "," + s_green.to_s + "," + s_blue.to_s + ">"
    # get the color of horizon: always black ?
    hor_color = (get_rendering_option "HorizonColor")
    #h_red = hor_color.red / 255.0
    #h_green = hor_color.green / 255.0
    #h_blue = hor_color.blue / 255.0
    h_col = "rgb <" + (s_red*2).to_s + "," + (s_green*2).to_s + "," + (s_blue*2).to_s + ">"
    export_file.puts( "sky_sphere {" )
    export_file.puts( "  pigment {" )
    export_file.puts( "    gradient y" )
    export_file.puts( "    color_map {" )
    export_file.puts( "      [0 color rgb <" + s_red.to_s + "," + s_green.to_s + "," + s_blue.to_s + ">]" )
    export_file.puts( "      [1 color rgb 1.0]" )
    #export_file.puts( "      [1 color rgb <" + (s_red*2).to_s + "," + (s_green*2).to_s + "," + (s_blue*2).to_s + ">]" )
    export_file.puts( "    }" )
    export_file.puts( "    scale -1.3" )
    export_file.puts( "    translate -1" )
    export_file.puts( "  }" )
    export_file.puts( "}" )  
    # Report Sky
    report_file.puts( "SU gradient sky exported.")
  when "Clouds"
    #get the clouds type
    if $clouds_type
      export_file.puts( "//Sky: " + $clouds_type)
      # Report Sky
      report_file.puts( "Sky type: " + $clouds_type)
      #write a sky_sphere
      case $clouds_type
        when "Blue_Scattered"
          (write_sky_sphere_clouds export_file, array_Blue_Scattered)
        when "Blue_Wide_Scattered"
          (write_sky_sphere_clouds export_file, array_Blue_Wide_Scattered)
        when "Blue_Grey_Cumulonimbus"
          (write_sky_sphere_clouds export_file, array_Blue_Grey_Cumulonimbus)
        when "Blue_Grey_Puffy"
          (write_sky_sphere_clouds export_file, array_Blue_Grey_Puffy)
        when "Grey_Stormy"
          (write_sky_sphere_clouds export_file, array_Grey_Stormy)
        when "High_Cirrus"
          export_file.puts( "sky_sphere { S_Cloud1 }" )
        when "Middle_Cirrus"
          export_file.puts( "sky_sphere { S_Cloud3 }" )
        when "Low_Cirrus"
          export_file.puts( "sky_sphere { S_Cloud2 }" )
        when "Scattered"
          export_file.puts( "sky_sphere { S_Cloud4 }" )
        when "Overcast"
          export_file.puts( "sky_sphere { S_Cloud5 }" )
        when "Rainy"
          (write_rainy_clouds export_file)
        when "Custom"
          #(write_custom_clouds export_file)
          (write_fastsky export_file)
      end
      else
      #write a default Blue_Scattered cloudy sky
      export_file.puts( "//Default sky type: Blue_Scattered")
      (write_sky_sphere_clouds array_Blue_Scattered)
      # Report Sky
          report_file.puts( "Default sky type: Blue_Scattered")
    end
  when "Stars"
          export_file.puts( "sky_sphere {" )
          export_file.puts( "pigment {" )
          export_file.puts( "    granite" )
          export_file.puts( "    turbulence 0.1" )
          export_file.puts( "    frequency 1" )
          export_file.puts( "    color_map {" )
          export_file.puts( "        [0.82 rgb 0.00 ]" )
          export_file.puts( "        [0.82 rgb 0.20 ]" )
          export_file.puts( "        [0.75 rgb 0.40 ]" )
          export_file.puts( "        [0.78 rgb 0.60 ]" )
          export_file.puts( "        [0.81 rgb 0.80 ]" )
          export_file.puts( "        [0.85 rgb 0.95 ]" )
          export_file.puts( "        [0.91 rgb <1.0,1.0,0.3> ]" )
          export_file.puts( "        [0.91 rgb 0.00 ]" )
          export_file.puts( "    }" )
          export_file.puts( "scale .02" )
          export_file.puts( "}" )
          export_file.puts( "}" )
end
   
# Output fog, if set by the user
if( $fog_use == true)
  (write_su_fog export_file)
  # Report Fog
  report_file.puts( "SU fog added.")
end

# Output snow, if set by the user
if( $snow_use != "No_snow" )
  # Report Snow
  report_file.puts( "Snow added.")
end

# Report Background
report_file.puts( "Background type: " + $sky_type)

#
end

# ----------------------------------- Outputs sky sphere for clouds
def write_sky_sphere_clouds (export_file, array_cloud)
export_file.puts( "// Clouds sphere" )
export_file.puts( "sky_sphere {" )
export_file.puts( "    pigment {" )
export_file.puts( "    " + array_cloud[0])
export_file.puts( "    turbulence " + array_cloud[1])
export_file.puts( "    color_map {")
export_file.puts( "       [" + array_cloud[2] + " color " + array_cloud[3] + " filter " + array_cloud[4] + "]")
export_file.puts( "       [" + array_cloud[5] + " color " + array_cloud[6] + "]")
export_file.puts( "       [" + array_cloud[7] + " color " + array_cloud[8] + " filter " + array_cloud[9] + "]")
export_file.puts( "       }" )
export_file.puts( "    octaves " + array_cloud[10])
export_file.puts( "    omega " + array_cloud[11])
export_file.puts( "    lambda " + array_cloud[12])
export_file.puts( "    scale " + array_cloud[13])
export_file.puts( "    }" )
export_file.puts( "}" )
end

# ----------------------------------- Write rainy sky
def write_rainy_clouds(export_file)
export_file.puts( "// Clouds sphere" )
export_file.puts( "sky_sphere {" )
export_file.puts( "pigment {" )
export_file.puts( "gradient y" )
export_file.puts( "color_map {" )
export_file.puts( "[0.000 0.002 color rgb <1.0, 0.5, 0.5>" )
export_file.puts( "color rgb <1.0, 0.5, 0.5>]" )
export_file.puts( "[0.002 0.200 color rgb <0.5, 0.5, 0.5>" )
export_file.puts( "color rgb <0.5, 0.5, 0.5>]" )
export_file.puts( "}" )
export_file.puts( "scale 2" )
export_file.puts( "translate -1" )
export_file.puts( "}" )
export_file.puts( "pigment {" )
export_file.puts( "bozo" )
export_file.puts( "turbulence 0.8" )
export_file.puts( "octaves 26" )
export_file.puts( "omega 0.7" )
export_file.puts( "lambda 2" )
export_file.puts( "color_map {" )
export_file.puts( "[0.0 0.1 color rgb <0.85, 0.85, 0.85>" )
export_file.puts( "color rgb <0.75, 0.75, 0.75>]" )
export_file.puts( "[0.1 0.5 color rgb <0.75, 0.75, 0.75>" )
export_file.puts( "color rgbt <1, 1, 1, 1>]" )
export_file.puts( "[0.5 1.0 color rgbt <1, 1, 1, 1>" )
export_file.puts( "color rgbt <1, 1, 1, 1>]" )
export_file.puts( "}" )
export_file.puts( "//scale <0.2, 0.5, 0.2>" )   
export_file.puts( "scale <3,1,1>" )
export_file.puts( "}" )
export_file.puts( "rotate -135*x" )
export_file.puts( "}" )
end

# ----------------------------------- Sets the custom clouds parameters, returns an array to feed the macro: macro clouds(dc,lc,h,cl,np,pd,ptr,pturb,pocta,plamb,pomeg,td,s,df,b,fs,fe)
def set_custom_clouds
# default values if not set
$cc_dark_color = "SkyBlue" if not $cc_dark_color
$cc_light_color = "Grey20"  if not $cc_light_color
$cc_height = "5000" if not $cc_height
$cc_coverage = "0.5" if not $cc_coverage
$cc_nplanes = "60" if not $cc_nplanes
$cc_dplanes = "300" if not $cc_dplanes
$cc_pigment = "bozo" if not $cc_pigment
$cc_tdistance = "30" if not $cc_tdistance
$cc_scale = "1" if not $cc_scale
$cc_dfactor = "0.9" if not $cc_dfactor
$cc_transition = "0.1" if not $cc_transition
$cc_sfilter = "0.7" if not $cc_sfilter
$cc_efilter = "0.8" if not $cc_efilter

#Dialox box 1:
dark_colors_list = %w[Aquamarine Black Blue BlueViolet CadetBlue CornflowerBlue DarkSlateBlue Gray10 Gray20 Gray30 Gray40 Gray50 Gray60 Gray70 Gray80 Gray90 LightBlue LightSteel MediumBlue MidnightBlue NavyBlue NeonBlue RichBlue SkyBlue SlateBlue SteelBlue SummerSky Turquoise White ].join("|")
light_colors_list = %w[Aquamarine Black Blue BlueViolet CadetBlue CornflowerBlue DarkSlateBlue Gray10 Gray20 Gray30 Gray40 Gray50 Gray60 Gray70 Gray80 Gray90 LightBlue LightSteel MediumBlue MidnightBlue NavyBlue NeonBlue RichBlue SkyBlue SlateBlue SteelBlue SummerSky Turquoise White].join("|")
pigment_list = %w[agate bozo bumps dents granite marble ripples spotted waves wrinkles].join("|")
dropdowns = [dark_colors_list, light_colors_list, pigment_list]
prompts=["Dark color  ", "Light color  ", "Clouds pattern  "]
values=[$cc_dark_color, $cc_light_color, $cc_pigment]
$cc_dark_color, $cc_light_color, $cc_pigment = inputbox prompts, values, dropdowns, "Custom clouds main parameters"

#Dialox box 2:
prompts=["Height of lower clouds layer (m)  ", "Clouds coverage (0=none, 1=full) ", "Number of clouds layers  ", "Distance between layers (m)  ", "Distance between succesive textures  ", "Scale  ", "Texture darkening factor  ", "Width of transition zone  ", "Filter starts at (0..1)  ", "Filter ends at (0..1)  "]
values=[$cc_height, $cc_coverage, $cc_nplanes, $cc_dplanes, $cc_tdistance, $cc_scale, $cc_dfactor, $cc_transition, $cc_sfilter,$cc_efilter]
$cc_height, $cc_coverage, $cc_nplanes, $cc_dplanes, $cc_tdistance, $cc_scale, $cc_dfactor, $cc_transition, $cc_sfilter,$cc_efilter  = inputbox prompts, values, "Custom clouds other parameters"
end

# ----------------------------------- Sets the custom clouds parameters, returns an array to feed the macro call
def set_fastsky
# default values if not set
$cc_color = "White" if not $cc_color
$cc_type = "bozo"  if not $cc_type
$cc_shadows = "No"  if not $cc_shadows
$cc_altitude = "5000" if not $cc_altitude
$cc_turbulence = "1.0" if not $cc_turbulence
$cc_brightness = "1.0" if not $cc_brightness

#Dialox box 1:
cc_colors_list = %w[Aquamarine Black Blue BlueViolet CadetBlue CornflowerBlue DarkSlateBlue Gray10 Gray20 Gray30 Gray40 Gray50 Gray60 Gray70 Gray80 Gray90 LightBlue LightSteel MediumBlue MidnightBlue NavyBlue NeonBlue RichBlue SkyBlue SlateBlue SteelBlue SummerSky Turquoise White ].join("|")
cc_type_list = %w[agate bozo crackles dents granite ripples spotted waves wood wrinkles].join("|")
shadows_list = %w[Yes No].join("|")
dropdowns = [cc_colors_list, cc_type_list, shadows_list]
prompts=["Clouds main color  ", "Type  ", "Clouds cast shadows on the ground",  "Altitude (m)  ", "Turbulence  ", "Clouds brightness  "]
values=[$cc_color, $cc_type, $cc_shadows, $cc_altitude, $cc_turbulence, $cc_brightness]
$cc_color, $cc_type, $cc_shadows, $cc_altitude, $cc_turbulence, $cc_brightness = inputbox prompts, values, dropdowns, "Custom clouds parameters"
end

# ----------------------------------- Outputs the custom clouds parameters as an object call
def write_fastsky( export_file )
# default values if not set by the user
$cc_color = "White" if not $cc_color
$cc_type = "bozo"  if not $cc_type
$cc_altitude = "1000" if not $cc_altitude
$cc_turbulence = "1" if not $cc_turbulence
$cc_brightness = "1.0" if not $cc_brightness
# The greater the altitude, the lower the planes, the gretaer the scale
plane_alt = $cc_altitude.to_f / 10.0
scale_alt = 0.125 / ( $cc_altitude.to_f / 1000.0 )
transmit_1 = 0.5 + ($cc_brightness.to_f / 10.0)
transmit_2 = 0.4 + ($cc_brightness.to_f / 10.0)
transmit_3 = 0.3 + ($cc_brightness.to_f / 10.0)
# Output clouds
export_file.puts( "// Custom clouds" )
export_file.puts( "#declare Custom_clouds =" )
export_file.puts( "union {" )
export_file.puts( "   plane {" )
export_file.puts( "      y, 260" )
# first layer
export_file.puts( "      texture {" )
export_file.puts( "         finish {ambient 0 diffuse 0.7}" )
export_file.puts( "         pigment {" )
export_file.puts( "            " + $cc_type + " turbulence " + $cc_turbulence + "*1.00 translate 200*y" )
export_file.puts( "            color_map {" )
export_file.puts( "               [0.5, rgb " + $cc_color + "*" + $cc_brightness + " transmit 1.0]" )
export_file.puts( "               [1.0, rgb " + $cc_color + "*" + $cc_brightness + " transmit " + transmit_1.to_s + "]" )
export_file.puts( "            }" )
export_file.puts( "            scale " + scale_alt.to_s )
export_file.puts( "         }" )
export_file.puts( "         scale 1500" )
export_file.puts( "      }" )
export_file.puts( "   }" )
export_file.puts( "   plane {y, (2.0*" + plane_alt.to_s + ")}")
export_file.puts( "   plane {y, (2.2*" + plane_alt.to_s + ")}")
export_file.puts( "   plane {y, (2.4*" + plane_alt.to_s + ")}")
# second layer
export_file.puts( "      texture {" )
export_file.puts( "         finish {ambient 0 diffuse 0.7}" )
export_file.puts( "         pigment {" )
export_file.puts( "            " + $cc_type + " turbulence " + $cc_turbulence + "*0.35 translate 300*y" )
export_file.puts( "            color_map {" )
export_file.puts( "               [0.5, rgb " + $cc_color + "*" + $cc_brightness + " transmit 1.0]" )
export_file.puts( "               [0.8, rgb " + $cc_color + "*" + $cc_brightness + " transmit " + transmit_2.to_s + "]" )
export_file.puts( "            }" )
export_file.puts( "         scale 10 warp {turbulence 0.7} scale 1/10" )
export_file.puts( "            scale " + scale_alt.to_s )
export_file.puts( "      }" )
export_file.puts( "      scale 2500" )
export_file.puts( "   }" )
# third layer
export_file.puts( "      texture {" )
export_file.puts( "         finish {ambient 0 diffuse 0.5}" )
export_file.puts( "         pigment {" )
export_file.puts( "            " + $cc_type + " turbulence " + $cc_turbulence + "*0.35 translate 400*y" )
export_file.puts( "            color_map {" )
export_file.puts( "               [0.5, rgb " + $cc_color + "*" + $cc_brightness + " transmit 1.0]" )
export_file.puts( "               [0.8, rgb " + $cc_color + "*" + $cc_brightness + " transmit " + transmit_3.to_s  + "]" )
export_file.puts( "            }" )
export_file.puts( "         scale 10 warp {turbulence 0.7} scale 1/10" )
export_file.puts( "            scale " + scale_alt.to_s )
export_file.puts( "      }" )
export_file.puts( "      scale 2500" )
export_file.puts( "   }" )
export_file.puts( "   hollow double_illuminate" )
if( $cc_shadows == "No" )
  export_file.puts( "   no_shadow" )
end
export_file.puts( "   scale 1/200" )
export_file.puts( "}" )
# object call
export_file.puts( "object {Custom_clouds scale 30000}" )
# Force to use SU background color
sky_color = (get_rendering_option "BackgroundColor")
s_red = sky_color.red / 255.0
s_green = sky_color.green / 255.0
s_blue = sky_color.blue / 255.0
s_col = "color rgb <" + s_red.to_s + "," + s_green.to_s + "," + s_blue.to_s + ">"
export_file.puts( "background {" )
export_file.puts( "     " + s_col )
export_file.puts( "" )
export_file.puts( "}" )
end

# ----------------------------------- Outputs the custom clouds parameters as a macro call: macro clouds(dc,lc,h,cl,np,pd,ptr,pturb,pocta,plamb,pomeg,td,s,df,b,fs,fe)
def write_custom_clouds( export_file )
# default values if not set by the user
$cc_dark_color = "SkyBlue" if not $cc_dark_color
$cc_light_color = "Gray20"  if not $cc_light_color
$cc_height = "5000" if not $cc_height
$cc_coverage = "0.5" if not $cc_coverage
$cc_nplanes = "60" if not $cc_nplanes
$cc_dplanes = "300" if not $cc_dplanes
$cc_pigment = "bozo" if not $cc_pigment
$cc_tdistance = "30" if not $cc_tdistance
$cc_scale = "1" if not $cc_scale
$cc_dfactor = "0.9" if not $cc_dfactor
$cc_transition = "0.1" if not $cc_transition
$cc_sfilter = "0.7" if not $cc_sfilter
$cc_efilter = "0.8" if not $cc_efilter

# Some conversions to cm
$cc_dplanes = ($cc_dplanes.to_f / 10.0).to_s
$cc_scale = ($cc_scale.to_f * 10000).to_s
# Output clouds
export_file.puts( "object {clouds(" + $cc_dark_color + "," + $cc_light_color + "," + $cc_height + "," + $cc_coverage + "," + $cc_nplanes + "," + $cc_dplanes + ",pigment{" + $cc_pigment + "},0.9,9,3.5,0.5," + $cc_tdistance + "," + $cc_scale + "," + $cc_dfactor + "," + $cc_transition + "," + $cc_sfilter + "," + $cc_efilter + ")")
export_file.puts( "    hollow" )
export_file.puts( "    no_shadow" )
export_file.puts( "rotate 5*y" )
export_file.puts( "}" )
# Output sky

export_file.puts( "// Sky container" )
export_file.puts( "sky_sphere{ pigment{ " + $cc_dark_color + "*" + $cc_light_color + "}}")
end

# ----------------------------------- Outputs fog as set by the user
def write_fog( export_file )
model=Sketchup.active_model
export_file.puts( "// Fog" )
#default values if not set by the user
$fog_type = "1" if not $fog_type
$fog_color = "White" if not $fog_color
$fog_distance = model.bounds.diagonal if not $fog_distance
$fog_alt = model.bounds.center.z if not $fog_alt
# fog type set by the user:
if( $fog_type == "Constant" )
  $fog_type = "1"
end
if( $fog_type == "Ground_fog" )
  $fog_type = "2"
end
# fog color:
col = Sketchup::Color.new $fog_color
red = (col.red / 255.0)
green = (col.green / 255.0)
blue = (col.blue / 255.0)
export_file.puts( "fog {" )
export_file.puts( "    distance " + $fog_distance.to_f.to_s )
export_file.puts( "    color rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + ">" )
export_file.puts( "    fog_type " + $fog_type.to_s )
export_file.puts( "    fog_alt " + $fog_alt.to_f.to_s )
export_file.puts( "}" )
export_file.puts( "" )

# Switch back to options values
if( $fog_type == "1" )
  $fog_type = "Constant"
  else
  $fog_type = "Ground_fog"
end
end

# ----------------------------------- Outputs fog as set by the user
def write_su_fog( export_file )
model=Sketchup.active_model
(get_fog_settings)

# fog type from SU to POV values:  0 dense, 1 haze, 2 depth cueing
if( $fog_type == 0 ) or ( $fog_type == 2 )
  fog_typ = "1"
  else
  fog_typ = "2"
end
# Density [0..1]
fog_dens = $fog_density * 100.0
fog_dist = $fog_distance / fog_dens

# fog color:
col = $fog_color.to_a
red = col[0] / 255.0
green = col[1] / 255.0
blue = col[2] / 255.0
export_file.puts( "// Fog" )
export_file.puts( "fog {" )
export_file.puts( "    fog_type " + fog_typ.to_s )
export_file.puts( "    distance " + fog_dist.to_s )
export_file.puts( "    color rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + ">" )
case $fog_type
  when 1
    export_file.puts( "    fog_alt " + $fog_alt.to_f.to_s )
    export_file.puts( "    turbulence <1.0, 0.0, 1.0>" )
end
# End fog
export_file.puts( "}" )
export_file.puts( "" )

end

# ----------------------------------- Outputs infinite ground, if currently displayed in SU
def write_su_ground(export_file, report_file)
if (get_rendering_option "DrawGround")
  # get the color of ground
  ground_color = (get_rendering_option "GroundColor")
  red = ground_color.red / 255.0
  green = ground_color.green / 255.0
  blue = ground_color.blue / 255.0
  col = "rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + ">"
  # write infinite plane with color
  export_file.puts( "// Infinite ground plane" )
  export_file.puts( "plane {" )
  export_file.puts( "    <0, 1, 0>, 0" )
  export_file.puts( "    pigment { color rgb <" + red.to_s + "," + green.to_s + "," + blue.to_s + "> }" )
  export_file.puts( "}" )
  export_file.puts( "" )
  # Report Ground
  report_file.puts( "SU ground exported.")
  else
  # Report No SU Ground
  report_file.puts( "SU ground not exported.")
end
end

# ----------------------------------- Outputs gradient sky, if currently displayed in SU
def write_su_sky(export_file, report_file)
if (get_rendering_option "DrawHorizon")
  # get the color of sky
  ground_color = (get_rendering_option "SkyColor")
  s_red = ground_color.red / 255.0
  s_green = ground_color.green / 255.0
  s_blue = ground_color.blue / 255.0
  s_col = "rgb <" + s_red.to_s + "," + s_green.to_s + "," + s_blue.to_s + ">"
  # get the color of horizon
  ground_color = (get_rendering_option "HorizonColor")
  h_red = ground_color.red / 255.0
  h_green = ground_color.green / 255.0
  h_blue = ground_color.blue / 255.0
  h_col = "rgb <" + h_red.to_s + "," + h_green.to_s + "," + h_blue.to_s + ">"
  export_file.puts( "sky_sphere {" )
  export_file.puts( "  pigment {" )
  export_file.puts( "    gradient y" )
  export_file.puts( "    color_map {" )
  export_file.puts( "      [0 color rgb <" + s_red.to_s + "," + s_green.to_s + "," + s_blue.to_s + ">]" )
  export_file.puts( "      [1 color rgb 1.0]" )
  export_file.puts( "    }" )
  export_file.puts( "    scale -1.5" )
  export_file.puts( "    translate -1" )
  export_file.puts( "  }" )
  export_file.puts( "}" )  
  # Report Sky
  report_file.puts( "SU sky exported.")
end
end

# ----------------------------------- Sets the user fog settings (old)
def set_fog_settings
model=Sketchup.active_model
#default values
$fog_type = "Constant"
$fog_color_1 = "-"
$fog_color_2 = "-"
$fog_color_3 = "White"
$fog_distance = model.bounds.diagonal * 2
$fog_alt = model.bounds.center.z * 1.5

UI.messagebox("Fog parameters will be used if you select \"Add_Fog\" in the \"Weather settings\" dialog box.")
fog_list1 = %w[Constant Ground_fog].join("|")
(create_colors_dropdowns)
colors_list_1 = [$colors_ag.join("|")]
colors_list_2 = [$colors_ho.join("|")]
colors_list_3 = [$colors_pz.join("|")]
dropdowns = [fog_list1, colors_list_1, colors_list_2, colors_list_3]

prompts=["Fog type  ", "Fog color [A-G]  ", "Fog color [H-O]  ", "Fog color [P-Z]  ", " Fog distance  ", "Fog altitude  "]
values=[$fog_type, $fog_color_1, $fog_color_2, $fog_color_3, $fog_distance, $fog_alt]
$fog_type, $fog_color_1, $fog_color_2, $fog_color_3, $fog_distance, $fog_alt  = inputbox prompts, values, dropdowns, "Fog parameters"
$fog_color = (select_color $fog_color_1, $fog_color_2, $fog_color_3 )
end

# ----------------------------------- Gets the user fog settings from SU renderingoptions class
def get_fog_settings
model=Sketchup.active_model
# current values
$fog_use = (get_rendering_option "DisplayFog")
# 0: dense, 1: haze, 2: depth cueing
$fog_type = (get_rendering_option "FogType")
if( (get_rendering_option "FogUseBkColor") == true )
  $fog_color = (get_rendering_option "BackgroundColor")
  else
  $fog_color = (get_rendering_option "FogColor")
end
$fog_density = (get_rendering_option "FogDensity")
$fog_start = (get_rendering_option "FogStart")
$fog_end = (get_rendering_option "FogEnd")
$fog_distance = (model.bounds.diagonal).to_cm
$fog_alt = (model.bounds.center.z).to_cm
end

# ------------------------------------------------------------------------
# ------------------------------------ MISC FUNCTIONS
# ------------------------------------------------------------------------

def setstatusbar(percent) 
  1.upto(percent) {|i|
  Sketchup.set_status_text("Exporting faces: " + percent.to_s + "%")
  }
end
# ----------------------------------- make the report and pov file more readable
def underline( export_file, report_file )
export_file.puts( "// -------------------------------------------" )
export_file.puts( "" )
report_file.puts( "----------------------------------------------" )
report_file.puts( "" )
end

# ----------------------------------- outputs a whitepoint reference and a color system
def write_whitepoint( s_type, export_file, report_file )
export_file.puts( "//Color system for lights")
# Color system default
export_file.puts( "CIE_ColorSystem(CIE_ColSys)" )
# White point
export_file.puts( "//White point reference for lights")
case s_type
  when "Outdoor_Day"
    export_file.puts( "CIE_ColorSystemWhitepoint(CIE_ColSys, Daylight2Whitepoint(6503.62))")
    export_file.puts( "CIE_GamutMapping(off)" )
    export_file.puts( "#declare Lightsys_Brightness = 1.0;" )
    export_file.puts( "#declare Lightsys_Filter = <1,1,1>;" )
  when "Indoor_Day"
    export_file.puts( "CIE_ColorSystemWhitepoint(CIE_ColSys, Daylight2Whitepoint(7504.17))")
    export_file.puts( "CIE_GamutMapping(off)" )
    export_file.puts( "#CIE_ChromaticAdaption(off)" )
    export_file.puts( "#declare Lightsys_Brightness = 0.95;" )
    export_file.puts( "#declare Lightsys_Filter = <1,1,1>;" )
  when "Outdoor_Night"
    export_file.puts( "CIE_ColorSystemWhitepoint(CIE_ColSys, Blackbody2Whitepoint(Kt_Moonlight))")
    export_file.puts( "CIE_GamutMapping(off)" )
    export_file.puts( "#CIE_ChromaticAdaption(off)" )
    export_file.puts( "#declare Lightsys_Brightness = 1.0;" )
    export_file.puts( "#declare Lightsys_Filter = <1,1,1>;" )
  when "Indoor_Night"
    export_file.puts( "CIE_ColorSystemWhitepoint(CIE_ColSys, Blackbody2Whitepoint(Kt_Moonlight))")
    export_file.puts( "CIE_GamutMapping(off)" )
    export_file.puts( "#CIE_ChromaticAdaption(off)" )
    export_file.puts( "#declare Lightsys_Brightness = 0.9;" )
    export_file.puts( "#declare Lightsys_Filter = <1,1,1>;" )
end
export_file.puts( "" )
end

# ----------------------------------- creates the 3 arrays for colors dropdowns
def create_colors_dropdowns
all_colors = Sketchup::Color.names
$colors_ag =[]
$colors_ag[0] = "-"
$colors_ho =[]
$colors_ho[0] = "-"
$colors_pz =[]
$colors_pz[0] = "-"
i = 0
0.upto(52) do |c|
  $colors_ag[i+1] = all_colors[i]
  i = i + 1
end
i = 0
53.upto(101) do |c|
  $colors_ho[i+1] = all_colors[c]
  i = i + 1
end
i = 0
102.upto(139) do |c|
  $colors_pz[i+1] = all_colors[c]
  i = i + 1
end
end

# ----------------------------------- selects the correct color in the 3 parameters coming from the dialog box dropdowns
def select_color( c1, c2, c3 )
if( c1 != "-")
  return_col = c1
end
if( c2 != "-")
  return_col = c2
end
if( c3 != "-")
  return_col = c3
end
return return_col
end

# ----------------------------------- sets the dict_key_color's with the color parameter
def set_dict_key_colors( c )
if( $colors_ag.include?(c) )
  dict_key_color_1 = c
  dict_key_color_2 = "-"
  dict_key_color_3 = "-"
end
if( $colors_ho.include?(c) )
  dict_key_color_1 = "-"
  dict_key_color_2 = c
  dict_key_color_3 = "-"
end
if( $colors_pz.include?(c) )
  dict_key_color_1 = "-"
  dict_key_color_2 = "-"
  dict_key_color_3 = c
end
return [dict_key_color_1, dict_key_color_2, dict_key_color_3]
end

# ----------------------------------- Get a SU current rendering option, return its value
def get_rendering_option( option )
model=Sketchup.active_model
$rop = model.rendering_options.to_a
ret = ""
$rop.each do |pair|
  if( pair[0] == option )
    ret = pair[1]
  end
end
return ret
end
# ----------------------------------- explodes groups and components, except lights, medialights, focalblurpoint and such.
def explode_groups
model=Sketchup.active_model
entities=model.entities
components_names = ["su2pov_sun", "su2pov_pointlight", "su2pov_realpointlight", "su2pov_spotlight", "su2pov_realspotlight", "su2pov_boxmedialight", "su2pov_conemedialight", "su2pov_focalblurpoint", "su2pov_arealight", "su2pov_lensflare", "su2pov_tree" ]
array_to_explode = []
# First pass for groups and components

entities.each do |e|
  if( e.typename == "Group" )
    array_to_explode[array_to_explode.length] = e
  end
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if not components_names.include?( name )
      array_to_explode[array_to_explode.length] = e
    end
  end
end
# Explode higher level components and groups
i = 0
0.upto(array_to_explode.length-1) do |cp|
  Sketchup.set_status_text("Exploding groups and components, highest level: " + i.to_s)
  array_to_explode[i].explode
  i = i + 1
end

# Intermediate model save
su_current_model = model.path           # get the current model file name and path
model_filename = su_current_model.split("\\")
model_filename = model_filename[model_filename.length-1].split(".")[0]
#model.save( model_filename + ".skp" )

# Second pass for nested groups and components

array_to_explode = []
entities=model.entities
entities.each do |e|
  if( e.typename == "Group" )
    array_to_explode[array_to_explode.length] = e
  end
  if( e.typename == "ComponentInstance" )
    name = e.definition.name
    if not components_names.include?( name )
      array_to_explode[array_to_explode.length] = e
    end
  end
end
# Explode lower level components and groups
i = 0
0.upto(array_to_explode.length-1) do |cp|
  Sketchup.set_status_text("Exploding groups and components, lowest level: " + i.to_s)
  array_to_explode[i].explode
  i = i + 1
end
end

# ----------------------------------- Sets the option pov-ray/report file after export
def set_launch_option
# sets the default value for dialog box
$launch_option = "Display_report_file" if not $launch_option
# Sets the options list
list = %w[Display_report_file Launch_POV-Ray].join("|")
dropdown = [list]

# Dialog box:
prompts = ["What to do after export  "]
values = [$launch_option]
result = inputbox prompts,values, dropdown, "After export "
$launch_option = result[0]
end

# ----------------------------------- Outputs radiosity or not
def write_render_type (r_type,export_file)
# Ray-tracing needs nothing here
case r_type
  when "Radiosity_Fast"
    export_file.puts( "radiosity { Rad_Settings(2, off, off) }" )
  when "Radiosity_Normal"
    export_file.puts( "radiosity { Rad_Settings(3, off, off) }" )
  when "Radiosity_Final"
    export_file.puts( "radiosity { Rad_Settings(5, off, off) }" )
  when "Radiosity_Outdoor_Low"
    export_file.puts( "radiosity { Rad_Settings(6, off, off) }" )
  when "Radiosity_Outdoor_High"
    export_file.puts( "radiosity { Rad_Settings(7, off, off) }" )
  when "Radiosity_Indoor_Low"
    export_file.puts( "radiosity { Rad_Settings(9, off, off) }" )
  when "Radiosity_Indoor_High"
    export_file.puts( "radiosity { Rad_Settings(10, off, off) }" )
end
export_file.puts( "}" )
export_file.puts( "" )
end

# ----------------------------------- Camera parameters as in SU current view
def write_camera( camera, export_file, name, commented )
model = Sketchup.active_model
entities=model.entities
view = model.active_view

# default values if not set by the user
$use_special_camera = "No" if not $use_special_camera
# Camera type
cam_type = camera.perspective?
# Get camera viewpoint and target
eye = camera.eye
vx = eye.x.to_f.to_cm
vy = eye.y.to_f.to_cm
vz = eye.z.to_f.to_cm
target = camera.target
tx = target.x.to_f.to_cm
ty = target.y.to_f.to_cm
tz = target.z.to_f.to_cm
# Focal angle:
vfovdeg = camera.fov
vfovrad = vfovdeg*Math::PI/180.0
h = Float(view.vpheight)
w = Float(view.vpwidth)
f = (h/2)/Math.tan(vfovrad/2.0)
hfovrad = 2.0*Math.atan2((w/2),f)
hfovdeg = hfovrad*180.0/Math::PI

# Search for a focal blur point
$focalblur = false
entities.each do |e|
  if( e.typename == "ComponentInstance" )
    if( e.definition.name == "su2pov_focalblurpoint")
      #location
      trans = e.transformation.to_a
      $fbp_x = trans[12].to_cm
      $fbp_y = trans[13].to_cm
      $fbp_z = trans[14].to_cm
      $focalblur = true
      #params
      dict_name="su2pov"
      dicts=e.attribute_dictionaries
      $dict_key_aperture = e.get_attribute(dict_name, "aperture")
      $dict_key_confidence = e.get_attribute(dict_name, "confidence")
      $dict_key_variance = e.get_attribute(dict_name, "variance")
    end
  end
end

# default values for fbp if found but not set by the user
if( $focalblur == true ) and not $dict_key_aperture
  $dict_key_aperture = "10"
  $dict_key_confidence = "0.95"
  $dict_key_variance = "1/200"
end

# Camera header comment
if( commented != true )
  # This is the current page's camera
  export_file.puts( "// SketchUp current camera: " + name)
  else
  # This is another page's camera
  export_file.puts( "// SketchUp camera: " + name)
end

# Camera params  
export_file.puts( "#if (camera_active=" + $n_camera.to_s + ")" )
export_file.puts( "camera {")
if( camera.perspective? == false )
  export_file.puts( "     orthographic" )
  else
    case $use_special_camera
      when "No"
        export_file.puts( "     perspective" )
      when "Cylindrical"
        export_file.puts( "     cylinder 1" )
        export_file.puts( "     angle 180" )
      when "Fisheye"
        export_file.puts( "     fisheye" )
        export_file.puts( "     angle 180" )
      when "Omnimax"
        export_file.puts( "omnimax" )
      when "Panoramic"
        export_file.puts( "panoramic" )
        export_file.puts( "     angle 360" )
      when "Ultra_wide_angle"
        export_file.puts( "ultra_wide_angle" )
      when "Spherical"
        export_file.puts( "spherical" )
        export_file.puts( "     angle 360, 180" )
    end
end
# location and target for all camera types
export_file.puts( "     location <" + vx.to_s + "," + vz.to_s + "," + vy.to_s + ">")
export_file.puts( "     look_at <" + tx.to_s + "," + tz.to_s + "," + ty.to_s + ">")
# angle, depends on camera type
case $use_special_camera
  when "No"
    export_file.puts( "     angle " + hfovdeg.to_s )
  when "ultra_wide_angle"
    export_file.puts( "     angle " + hfovdeg.to_s )
end

# Focal blur only for active (or uncommented) camera
if( $focalblur == true ) and ( commented != true )
  export_file.puts( "     aperture " + $dict_key_aperture )
  export_file.puts( "     blur_samples 10" )
  export_file.puts( "     focal_point <" + $fbp_x.to_s + "," + $fbp_z.to_s + "," + $fbp_y.to_s + ">" )
  export_file.puts( "     confidence " + $dict_key_confidence )
  export_file.puts( "     variance " + $dict_key_variance)
end
export_file.puts( "     }" )
export_file.puts( "" )
$n_camera = $n_camera + 1
export_file.puts( "#end" )

# lens_flare effect for current camera, camera_location and camera_look_at globals for glow effects
if( commented == false )

  $camera_location = "<" + vx.to_s + "," + vz.to_s + "," + vy.to_s + ">"
  $camera_look_at = "<" + tx.to_s + "," + tz.to_s + "," + ty.to_s + ">"
  # Search for a lens flare component
  $found = false
  entities.each do |e|
    if( e.typename == "ComponentInstance" )
      if( e.definition.name == "su2pov_lensflare")
        $found = true
        #location
        trans = e.transformation.to_a
        $lfe_x = trans[12].to_cm
        $lfe_y = trans[13].to_cm
        $lfe_z = trans[14].to_cm
        #params
        dict_name="su2pov"
        dicts=e.attribute_dictionaries
        $dict_key_type = e.get_attribute(dict_name, "type")
        $dict_key_scale = e.get_attribute(dict_name, "scale")
        $dict_key_brightness = e.get_attribute(dict_name, "brightness")
        $dict_key_intensity = e.get_attribute(dict_name, "intensity")
      end
    end
  end
  #default values if component found but not set by the user
  $dict_key_type = "Sun" if not $dict_key_type
  $dict_key_scale = "1" if not $dict_key_scale
  $dict_key_brightness = "1" if not $dict_key_brightness
  $dict_key_intensity = "1" if not $dict_key_intensity
  if( $found == true )
  export_file.puts( "// Lens effect for current camera")
  export_file.puts( "#declare camera_location = <" + vx.to_s + "," + vz.to_s + "," + vy.to_s + "> ;")
  export_file.puts( "#declare camera_look_at = <" + tx.to_s + "," + tz.to_s + "," + ty.to_s + "> ;")
  export_file.puts( "#declare effect_location = <" + $lfe_x.to_s + "," + $lfe_z.to_s + "," + $lfe_y.to_s + "> ;")
  export_file.puts( "#declare effect_type = \"" + $dict_key_type + "\"" + " ;")
  export_file.puts( "#declare effect_scale = " + $dict_key_scale.to_s + " ;" )
  export_file.puts( "#declare effect_brightness = " + $dict_key_brightness.to_s + " ;" )
  export_file.puts( "#declare effect_intensity = " + $dict_key_intensity.to_s + " ;" )
  #if used with focalblur
  if( $focalblur == true )
  export_file.puts( "#declare effect_always_on_top = true" )
  end
  export_file.puts( "#include \"Lens.inc\"")
  export_file.puts( "" )
  end
end
end

# ----------------------------------- Camera parameters as in SU current view
def write_cameras( export_file )
Sketchup.set_status_text("Exporting cameras...")
model = Sketchup.active_model
entities = model.entities
view = model.active_view
cam_view = view.camera
pages = model.pages
$n_cameras = pages.size
ind_active_camera = 1
# Detects indice of current camera in model.pages, set indice to 1 if no pages
ind = 1
pages.each do |page|
  if( page == pages.selected_page )
    ind_active_camera = ind
  end
    ind = ind + 1
end
export_file.puts( "// Cameras control" )
export_file.puts( "#declare camera_active=" + ind_active_camera.to_s+ ";" )
$n_camera = 1

if( $n_cameras > 0 )
  pages.each do |page|
    if( page == pages.selected_page )
      # Get active camera of currently selected page
      camera = page.camera
      (write_camera camera, export_file, page.name, false )
      else
      # Get camera of page
      camera = page.camera
      (write_camera camera, export_file, page.name, true)
    end
  end
  else
  (write_camera cam_view, export_file, "no name" , false)
  $n_cameras = 1  
end
end

# ----------------------------------- Set the use of lens flare and focal blur for current camera
def set_camera_params
#perspective | orthographic | fisheye (angle=180, angle=360) | ultra_wide_angle (no angle needed) |
#omnimax (no angle needed) | cylinder CylinderType | spherical (angle)

# default values if not set by the user
$use_focal_blur = "No" if not $use_focal_blur
$use_lens_flare = "No" if not $use_lens_flare
$use_special_camera = "No" if not $use_special_camera
use_special_camera_list = %w[No Cylindrical Fisheye Omnimax Panoramic Ultra_wide_angle Spherical].join("|")
use_focalblur_list = %w[Yes No].join("|")
use_lensflare_list = %w[Yes No].join("|")
dropdowns = [use_special_camera_list,use_focalblur_list, use_lensflare_list]
# Dialog box:
prompts = ["Use special camera type  ", "Use lens flare effect  ", "Use focal blur effect  "]
values = [$use_special_camera, $use_focal_blur, $use_lens_flare]
$use_special_camera, $use_focal_blur, $use_lens_flare = inputbox prompts,values, dropdowns, "Camera parameters"
end

# ----------------------------------- Set the use of lens flare and focal blur for current camera
def set_lensflare_params
model=Sketchup.active_model
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("lens flare not selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_lensflare" )
#default values
dict_key_type = "Sun" if not dict_key_type
#dict_key_color = "White" if not dict_key_color
dict_key_scale = "1" if not dict_key_scale
dict_key_brightness = "1" if not dict_key_brightness
dict_key_intensity = "1" if not dict_key_intensity

# Retrieve old parameters if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_type = e.get_attribute(dict_name, "type")
  #dict_key_color = e.get_attribute(dict_name, "color")
  dict_key_scale = e.get_attribute(dict_name, "scale")
  dict_key_brightness = e.get_attribute(dict_name, "brightness")
  dict_key_intensity = e.get_attribute(dict_name, "intensity")
  else
  UI.messagebox("No parameters set for the lens flare effect.")
end

#Dialog box:
lens_effects_list = %w[Camera Camera2 Candle Diamond Headlight Headlight2 Rainbow Rainbow2 SoftGlow Sparkle Sparkle2 Spotlight Spotlight2 Star Star2 Sun Sun2 Sun3 Undersea].join("|")
prompts=["Type  ", "Scale  ", "Brightness  ", "Intensity  "]
values=[dict_key_type, dict_key_scale, dict_key_brightness, dict_key_intensity]
dropdown = [lens_effects_list]
dict_key_type, dict_key_scale, dict_key_brightness, dict_key_intensity  = inputbox prompts, values, dropdown, "Lens flare parameters"

dict_name="su2pov"
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"type",dict_key_type)
e.set_attribute(dict_name,"scale",dict_key_scale)
e.set_attribute(dict_name,"brightness",dict_key_brightness)
e.set_attribute(dict_name,"intensity",dict_key_intensity)
else
  UI.messagebox("This is not the lens flare effect.")
end
end

# ----------------------------------- Set the focal blur parameters
def set_focalblur_params
model=Sketchup.active_model
e = model.selection[0]
if e
  type = e.typename
  else
  UI.messagebox("Focal blur point not selected")
  return nil
end
if(type == "ComponentInstance" )
  name = e.definition.name
end

if ( name == "su2pov_focalblurpoint" )
#default values
dict_key_aperture = "10" if not dict_key_aperture
dict_key_confidence = "0.95" if not dict_key_confidence
dict_key_variance = "1/200" if not dict_key_variance

# Retrieve old parameters if any
dict_name="su2pov"
dicts=e.attribute_dictionaries
if( dicts != nil and dicts[dict_name] )
  dict_key_aperture = e.get_attribute(dict_name, "aperture")
  dict_key_confidence = e.get_attribute(dict_name, "confidence")
  dict_key_variance = e.get_attribute(dict_name, "variance")
  else
  UI.messagebox("No parameters set for the focal blur point.")
end

#Dialog box:

prompts=["Aperture [0...N] (larger is blurrier)  ", "Confidence [0...1] (smaller is less accurate)  ", "Variance [0...1] (smaller is more accurate)  "]
values=[dict_key_aperture, dict_key_confidence, dict_key_variance]
dict_key_aperture, dict_key_confidence, dict_key_variance  = inputbox prompts, values, "Focal blur point parameters"

dict_name="su2pov"
e.attribute_dictionary(dict_name, true)
e.set_attribute(dict_name,"aperture",dict_key_aperture)
e.set_attribute(dict_name,"confidence",dict_key_confidence)
e.set_attribute(dict_name,"variance",dict_key_variance)
else
  UI.messagebox("This is not the focal blur point.")
end
end

# ------------------------------------------------------------------------
# ------------------------------------ MATERIALS RELATED FUNCTIONS
# ------------------------------------------------------------------------

# ----------------------------------- Create the global array for POV-Ray material names matching
def create_pov_materials_array
$pov_materials_array = ["Jade", "Red_Marble", "White_Marble", "Blood_Marble", "Blue_Agate", "Sapphire_Agate", "Brown_Agate", "Pink_Granite", "PinkAlabaster", "Cork", "Cherry_Wood", "Pine_Wood", "Dark_Wood", "Tan_Wood", "White_Wood", "Tom_Wood", "DMFWood1", "DMFWood2", "DMFWood3", "DMFWood4", "DMFWood5", "DMFLightOak", "DMFDarkOak", "DMFWood6", "EMBWood1", "Yellow_Pine", "Rosewood", "Sandalwood", "Glass"", Glass2", "Glass3", "Green_Glass", "NBglass", "NBoldglass", "NBwinebottle", "NBbeerbottle", "Ruby_Glass", "Dark_Green_Glass", "Vicks_Bottle_Glass", "Yellow_Glass", "Orange_Glass", "Chrome_Metal", "Brass_Metal", "Bronze_Metal", "Gold_Metal", "Silver_Metal", "Copper_Metal", "Polished_Chrome", "Polished_Brass", "New_Brass", "Spun_Brass", "Brushed_Aluminum", "Silver1", "Silver2", "Silver3", "Brass_Valley", "Rust", "Rusty_Iron", "Soft_Silver", "New_Penny", "Tinny_Brass", "Gold_Nugget", "Aluminum", "Bright_Bronze", "Candy_Cane", "Y_Gradient", "X_Gradient", "Peel", "Water", "Lightning1", "Lightning2", "Starfield", "Blue_Sky", "Bright_Blue_Sky", "Blue_Sky2", "Blue_Sky3", "Blood_Sky", "Apocalypse", "Clouds", "FBM_Clouds", "Shadow_Clouds", "T_Gold_1A", "T_Gold_1B", "T_Gold_1C", "T_Gold_1D", "T_Gold_1E", "T_Gold_2A", "T_Gold_2B", "T_Gold_2C", "T_Gold_2D", "T_Gold_2E", "T_Gold_3A", "T_Gold_3B", "T_Gold_3C", "T_Gold_3D", "T_Gold_3E", "T_Gold_4A", "T_Gold_4B", "T_Gold_4C", "T_Gold_4D", "T_Gold_4E", "T_Gold_5A", "T_Gold_5B", "T_Gold_5C", "T_Gold_5D", "T_Gold_5E"]

end

# ----------------------------------- Change the finishes variables from option strings to include names
def switch_finish (f0,f1)
case f0
  when "Very_soft_and_dull"
    $finish_type_0 = "F_MetalA"
  when "Fairly_soft_and_dull"
    $finish_type_0 = "F_MetalB"
  when "Medium_reflectivity"
    $finish_type_0 = "F_MetalC"
  when "Highly_hard_and_polished"
    $finish_type_0 = "F_MetalD"
  when "Very_highly_polished"
    $finish_type_0 = "F_MetalE"
end
case f1
  when "Very_soft_and_dull"
    $finish_type_1 = "F_MetalA"
  when "Fairly_soft_and_dull"
    $finish_type_1 = "F_MetalB"
  when "Medium_reflectivity"
    $finish_type_1 = "F_MetalC"
  when "Highly_hard_and_polished"
    $finish_type_1 = "F_MetalD"
  when "Very_highly_polished"
    $finish_type_1 = "F_MetalE"
end
end

# ----------------------------------- Change the finishes variables from include names to option names
def switch_back_finish (f0,f1)
case f0
  when "F_MetalA"
    $finish_type_0 = "Very_soft_and_dull"
  when "F_MetalB"
    $finish_type_0 = "Fairly_soft_and_dull"
  when "F_MetalC"
    $finish_type_0 = "Medium_reflectivity"
  when "F_MetalD"
    $finish_type_0 = "Highly_hard_and_polished"
  when "F_MetalE"
    $finish_type_0 = "Very_highly_polished"
end
case f1
  when "F_MetalA"
    $finish_type_0 = "Very_soft_and_dull"
  when "F_MetalB"
    $finish_type_0 = "Fairly_soft_and_dull"
  when "F_MetalC"
    $finish_type_0 = "Medium_reflectivity"
  when "F_MetalD"
    $finish_type_0 = "Highly_hard_and_polished"
  when "F_MetalE"
    $finish_type_0 = "Very_highly_polished"
end
end

# ----------------------------------- Return a material.name without spaces or "-"
def change_material_name(m)
mname = m.sub(/([ -])/, '_')
mname = mname.sub(/([ -])/, '_')
mname = mname.sub(/([ -])/, '_')
return mname
end

def change_material_names
model = Sketchup.active_model
materials = model.materials
materials.each do |mat|
name = mat.display_name
if name
  mname = name.sub(/[ ]/, '_')
  mname = name.sub(/[ ]/, '_')
  mname = name.sub(/[-]/, '_')
end
tex = mat.texture
if tex
  tname = name.sub(/[ ]/, '_')
  tname = name.sub(/[ ]/, '_')
  tname = name.sub(/[-]/, '_')
end
end
end

# ----------------------------------- Checks if this material is a transparent one, returns bool
def tranparent_material( m )
ret = false
ret = true if m.use_alpha?
return ret
end
# ----------------------------------- Ouputs the material of a face
def write_face_material( f,export_file )
  mat = f.material
  bmat = f.back_material
  
  case 
    when ((not mat) and (not bmat))
      output_mat = "Default"
    when ((not mat) and bmat)
      output_mat = "Reversed"
    when (mat and (not bmat))
      output_mat = mat
    when (mat and bmat)
      output_mat = mat
  end
  
  if( output_mat != "Default" ) and ( output_mat != "Reversed" )
    # check if it is a Pov-Ray material
    if not ( $pov_materials_array.include?( output_mat.display_name ) )
      tex = output_mat.texture
      # check if it need a double_iluminate statement
      double_ill = (tranparent_material output_mat)
      if not tex
        # Solid color
        export_file.puts( "texture { " + (change_material_name output_mat.display_name) + " }" )
        else
        # textured material
        export_file.puts( "texture { " + (change_material_name output_mat.display_name) )
        # rotate texture
        #normx = f.plane[0]
        #normy = f.plane[1]
        #normz = f.plane[2]
        #rotx = normx * 90.0
        #roty = normy * 90.0
        #rotz = normz * 90.0
        
        vec_t = Geom::Vector3d.new(0.0, -1.0, 0.0)
        vec_f = f.normal
        vec_proj_xy = Geom::Vector3d.new(vec_f[0], vec_f[1], 0.0)
        vec_proj_xz = Geom::Vector3d.new(vec_f[0], 0.0, vec_f[2])
        vec_proj_yz = Geom::Vector3d.new(0.0, vec_f[1], vec_f[2])
        rx = vec_proj_yz.angle_between(vec_t)
        ry = vec_proj_xz.angle_between(vec_t)
        rz = vec_proj_xy.angle_between(vec_t)
        rotx = (rad2deg rx)
        roty = (rad2deg ry)
        rotz = (rad2deg rz)
        #GOOD_OLD export_file.puts( "rotate<" + rotx.to_s + "," + rotz.to_s + ",0.0>" )
        #export_file.puts( "rotate<" + rotx.to_s + "," + (0.0 - rotz).to_s + "," + roty.to_s + ">" )
        export_file.puts( "rotate<" + rotx.to_s + "," + (0.0 - rotz).to_s + ",0.0>" )
        export_file.puts( "}" )
      end
      # Adds a double_illuminate statement if needed
      if( double_ill == true )
        export_file.puts( "double_illuminate" )
      end
    else
      # this is a Pov-Ray material
      export_file.puts( "texture { " + output_mat.display_name )
      export_file.puts( "    scale 50" )
      export_file.puts( " }" )
    end
  end
  
  if( output_mat == "Default" )
    # No material found for this face, both sides
    export_file.puts( "texture { Default }" )
  end
  
  # This is a reversed face, perhaps with a material, but not visible
  if ( output_mat == "Reversed" )
    export_file.puts( "texture { pigment { color rgb <1,0,0> }}" )
  end
end
# ----------------------------------- Outputs all materials of the SU model, except image materials
def declare_materials( export_file,finish_0, finish_1)
model = Sketchup.active_model
materials = model.materials

# Default material
export_file.puts( "#declare Default = texture {" )
export_file.puts( "     pigment { color rgbft <0.5,0.5,0.5,0.0,0.0> }" )
#if( finish_0 == "Standard")
  export_file.puts( "     finish {" )
  export_file.puts( "             diffuse 0.6" )
  export_file.puts( "             brilliance 1.0" )
  export_file.puts( "             crand 0.0" )
  export_file.puts( "             phong 0.0 phong_size 40" )
  export_file.puts( "             specular 0.005 roughness 0.05" )
  export_file.puts( "             reflection {0.001}" )
  export_file.puts( "             }" )
  #else
  #export_file.puts( "     finish { " + finish_0 + " }" )
#end  
export_file.puts( "}" )
export_file.puts( "" )
# stores it in materials_array for report
$materials_array[$n_materials] = "Default"
$textures_array[$n_materials] = "None"
$n_materials = $n_materials + 1

# Other materials
materials.each do |mat|
  Sketchup.set_status_text("Exporting materials: " + $n_materials.to_s + "/" + materials.length.to_s)
  name = mat.display_name
  # check if it is a Pov-ray material, if not, declare it
  if not ( $pov_materials_array.include?( name ) )
  # change the name if necessary, stores it in materials_array for report
  name = (change_material_name name)
  $materials_array[$n_materials] = name
  
  col = mat.color
  tex = mat.texture
  alpha = mat.alpha
  lin = ""
  # solid colors
  if not tex
    r = col.red / 255.0
    g = col.green / 255.0
    b = col.blue / 255.0
    f = (1.0 - alpha) 
    #t = 0.0
    t = (1.0 - alpha)
    $textures_array[$n_materials] = "None"
    export_file.puts( "#declare " + name + " = texture {" )
    export_file.puts( "    pigment { color rgbft <" + r.to_s + "," + g.to_s + "," + b.to_s + "," + f.to_s + "," + t.to_s + "> }" )
    if( alpha == 1.0 )
      if( finish_0 == "Standard")
        export_file.puts( "    finish {" )
        export_file.puts( "        diffuse 0.6" )
        export_file.puts( "        brilliance 1.0" )
        export_file.puts( "        crand 0.0" )
        export_file.puts( "        phong 0.0 phong_size 40" )
        export_file.puts( "        specular 0.0 roughness 0.05" )
        export_file.puts( "        reflection {0.0}" )
        export_file.puts( "        }" )
        else
        export_file.puts( "     finish { " + finish_0 + " }" )
      end
      export_file.puts( "}" )
      export_file.puts( "" )
      else
      # write the selected glass finish
      case $glass_type
        when "Drab_dark_low_reflexive"
          export_file.puts( "    finish { F_Glass7 }" )
        when "Drab_half_reflexive"
          export_file.puts( "    finish { F_Glass3 }" )
        when "Almost_mirror"
          export_file.puts( "    finish { F_Glass2 }" )
        when "Reflexive"
          export_file.puts( "    finish { F_Glass1 }" )
        when "Very_reflexive"
          export_file.puts( "    finish { F_Glass8 }" )
        when "Mirror_like"
          export_file.puts( "    finish { F_Glass9 }" )
        when "Dark_reflexive"
          export_file.puts( "    finish { F_Glass6 }" )
      end
      export_file.puts( "}" )
      export_file.puts( "" )
    end
  end
  # textured material
  if tex
    full_name = tex.filename.split("\\")
    filename = full_name[full_name.length-1]
    filename = (change_material_name filename)
    file = filename.split(".")[0]
    
    ext = filename.split(".")[1]
    $textures_array[$n_materials] = tex.filename
    if( ext == "JPG" or ext == "jpg")
      ext = "jpeg"
      # texture average color
    end
    # Texture scale
    tw = tex.width.to_cm
    th = tex.height.to_cm
    export_file.puts( "#declare " + name + " = texture {" )
    export_file.puts( "    pigment {" )
    export_file.puts( "        image_map {" )
    export_file.puts( "            " + ext + " \"" + filename + "\"" )
    export_file.puts( "            interpolate 2" )
    export_file.puts( "            map_type 0" )
    # Texture transparency
    if( alpha == 1.0 )
      export_file.puts( "            transmit all 0" )
      export_file.puts( "            filter all 0 " )
      else
      export_file.puts( "            transmit all " + (1.0 - alpha).to_s )
      export_file.puts( "            filter all " + (1.0 - alpha).to_s )
    end
    export_file.puts( "            }" )
    export_file.puts( "            scale <" + tw.to_s + "," + th.to_s + ",100>" )
    export_file.puts( "        }" )
    # glass finish
    if not ( alpha == 1.0 )
      case $glass_type
        when "Drab_dark_low_reflexive"
          export_file.puts( "    finish { F_Glass7 }" )
        when"Drab_half_reflexive"
          export_file.puts( "    finish { F_Glass3 }" )
        when "Almost_mirror"
          export_file.puts( "    finish { F_Glass2 }" )
        when "Reflexive"
          export_file.puts( "    finish { F_Glass1 }" )
        when "Very_reflexive"
          export_file.puts( "    finish { F_Glass8 }" )
        when "Mirror_like"
          export_file.puts( "    finish { F_Glass9 }" )
        when "Dark_reflexive"
          export_file.puts( "    finish { F_Glass6 }" )
      end
      else
      if( finish_1 == "Standard" )
        export_file.puts( "    finish {" )
        export_file.puts( "        diffuse 0.6" )
        export_file.puts( "        brilliance 1.0" )
        export_file.puts( "        crand 0.0" )
        export_file.puts( "        phong 0.0 phong_size 40" )
        export_file.puts( "        specular 0.0 roughness 0.05" )
        export_file.puts( "        reflection {0.0}" )
        export_file.puts( "        }" )
      else
        export_file.puts( "            finish { " + finish_1 + " }" )
      end  
    end
    export_file.puts( "}" )
  end
  $n_materials = $n_materials + 1
  end #of non-Pov-Ray material
end #of materials.each
export_file.puts( "" )
end

# ----------------------------------- Outputs image material
def declare_image_material( im,export_file,finish)
image_path = im.path
names = image_path.split("\\")
basename = names[names.length-1]
# change the name if necessary
basename = (change_material_name basename)
partnames = basename.split(".")
name = partnames[0]
ext = partnames[1]

#stores it in materials_array for report
$materials_array[$n_materials] = name
$textures_array[$n_materials] = image_path
lin = ""
  
# texture
image_model_height = im.height.to_cm 
image_model_width = im.width.to_cm
if( ext == "JPG" or ext == "jpg")
  ext = "jpeg"
end

# outputs only if not processed yet
if( $images_array.include?( basename ) )
  export_file.puts( "#declare " + name + " = texture {" )
  export_file.puts( "    pigment {" )
  export_file.puts( "        image_map {" )
  export_file.puts( "            " + ext + " \"" + basename + "\"" )
  export_file.puts( "            interpolate 2" )
  export_file.puts( "            map_type 0" )
  export_file.puts( "            transmit all 0" )
  export_file.puts( "            filter all 0 " )
  export_file.puts( "            }" )
  export_file.puts( "        }" )
  export_file.puts( "}" )
  export_file.puts( "" )
  $n_materials = $n_materials + 1
  # delete image file name from the array
  $images_array.delete_if {|x| x == basename }
end
end

# ----------------------------------- Ouputs a single face as a POV-polygon object, with or without holes
def write_single_face_to_polygon( f,export_file )
lin = ""
loops = f.loops
plane = f.plane
# default texture plane [0.0, -1.0, 0.0, 22.0330127760323] -> [0.0, 0.0, -1.0]
a = plane[0].to_f
b = plane[1].to_f
c = plane[2].to_f
d = plane[3].to_f
vertices = f.outer_loop.vertices
export_file.puts( "polygon {" )
export_file.puts( (f.vertices.length + f.loops.length).to_s  + "," )
# Loops vertices output
loops.each do |loop|
  lin = ""
  vertices = loop.vertices
  vertices.each do |v|
    pos2 = v.position
    pos = pos2.project_to_plane(a,b,c,d)
    px = sprintf( "%.13f",pos.x.to_f.to_cm )
    py = sprintf( "%.13f",pos.y.to_f.to_cm )
    pz = sprintf( "%.13f",pos.z.to_f.to_cm )
    lin = lin + "<" + px + ", " + pz + ", " + py + ">,"
  end
  # Last vertex to close the loop
  pos2 = vertices[0].position
  pos = pos2.project_to_plane(a,b,c,d)
  p0x = sprintf( "%.13f",pos.x.to_f.to_cm )
  p0y = sprintf( "%.13f",pos.y.to_f.to_cm )
  p0z = sprintf( "%.13f",pos.z.to_f.to_cm )
  lin = lin + "<" + p0x + "," + p0z + "," + p0y + ">"
  export_file.puts( lin )
end

# Face material
(write_face_material f,export_file)

# End of polygon
export_file.puts( "}" )
end

# ----------------------------------- Ouputs a single image as a POV-polygon object with texture on it
def write_single_image_to_polygon( i,export_file)

image_path = i.path
names = image_path.split("\\")
basename = names[names.length-1]
# change the name if necessary
basename = (change_material_name basename)
partnames = basename.split(".")
name = partnames[0]
ext = partnames[1]
if( ext == "jpg" ) or ( ext == "JPG" )
  ext = "jpeg"
end
# Image size
image_width = i.width
image_height = i.height
#Images vectors to extract image vertices
image_axes = i.normal.axes
vec_x = image_axes[0]
vec_y = image_axes[1]
vec_z = image_axes[2]
#Image vertices:
p1 = i.origin
p2 = p1.offset( vec_x, image_width ).to_a
p3 = p1.offset( vec_y, image_height ).to_a
p4 = p2.offset( vec_y, image_height ).to_a

x1 = p1[0].to_f.to_cm
y1 = p1[1].to_f.to_cm
z1 = p1[2].to_f.to_cm
x2 = p2[0].to_f.to_cm
y2 = p2[1].to_f.to_cm
z2 = p2[2].to_f.to_cm
x3 = p3[0].to_f.to_cm
y3 = p3[1].to_f.to_cm
z3 = p3[2].to_f.to_cm
x4 = p4[0].to_f.to_cm
y4 = p4[1].to_f.to_cm
z4 = p4[2].to_f.to_cm

export_file.puts( "mesh {" )
export_file.puts( "    triangle {" )
export_file.puts( "      <"+ x1.to_s + "," + z1.to_s + "," + y1.to_s + ">, <" + x2.to_s + "," + z2.to_s + "," + y2.to_s + ">, <" + x4.to_s + "," + z4.to_s + "," + y4.to_s + ">" )
export_file.puts( "      uv_vectors <0,0>, <1,0>, <1,1>" )
export_file.puts( "    }" )
export_file.puts( "triangle {" )
export_file.puts( "      <"+ x1.to_s + "," + z1.to_s + "," + y1.to_s + ">, <" + x4.to_s + "," + z4.to_s + "," + y4.to_s + ">, <" + x3.to_s + "," + z3.to_s + "," + y3.to_s + ">" )
export_file.puts( "    uv_vectors <0,0>, <1,1>, <0,1>" )
export_file.puts( "}" )
export_file.puts( "texture {" )
export_file.puts( "    uv_mapping pigment {" )
export_file.puts( "    image_map {" )
export_file.puts( "    " + ext + " \"" + name + "\"" )
export_file.puts( "    map_type 0" )
export_file.puts( "    interpolate 0" )
export_file.puts( "    }" )
export_file.puts( "}" )
export_file.puts( "finish { ambient 1 }" )
export_file.puts( "}" )
export_file.puts( "    no_shadow" )
export_file.puts( "}" )
export_file.puts( "" )
end
# -----------------------------------  SU2POV MAIN PROG
def su2pov

model = Sketchup.active_model

# Init's
version = Sketchup.version
model = Sketchup.active_model
entities = model.entities
$n_faces = 0
$n_images = 0
$n_cameras = 0
$n_materials = 0
$n_pointlights = 0
$n_realpointlights = 0
$n_spotlights = 0
$n_realspotlights = 0
$n_arealights = 0
$n_boxmedialights = 0
$n_conemedialights = 0
all_colors = Sketchup::Color.names
$materials_array = []
$textures_array = []
$images_array = []
$fade_distance = 1000
$fade_power = 2
(create_colors_dropdowns)
(create_pov_materials_array)
$n_trees = 0

# Pathes and files
su_install_path = Sketchup.find_support_file("SketchUp.exe").split("SketchUp.exe")[0]
materials_path = su_install_path + "Materials"
su_current_folder = Dir.getwd           # Get the working folder
su_current_model = model.path           # get the current model file name and path
model_filename = su_current_model.split("\\")
model_filename = model_filename[model_filename.length-1].split(".")[0]

# Saves the model if modified
if( model.modified? == true )
  UI.messagebox("Model save required...")
  model.save( model_filename + ".skp" )
end

# Starts the undo command
model.start_operation "Export to POV-Ray"

# sets the default values for dialog boxes, for first export in the session
$model_type = "All" if not $model_type
$scene_type = "Outdoor_Day" if not $scene_type
$light_type = "Sun_for_day_scene" if not $light_type
$finish_type_0 = "Standard" if not $finish_type_0
$finish_type_1 = "Dull" if not $finish_type_1
$glass_type = "Drab_half_reflexive" if not $glass_type
$render_type = "Ray-tracing" if not $render_type
$use_focal_blur = "No" if not $use_focal_blur
$use_lens_flare = "No" if not $use_lens_flare
$lens_flare_type = "Sun" if not $lens_flare_type

# Sets the options lists
list0 = %w[All Visible_layers_only].join("|")
list1 = %w[Outdoor_Day Outdoor_Night Indoor_Day Indoor_Night].join("|")
list1_1 = %w[Sun_for_day_scene Lights_for_night_scene All].join("|")
list4_0 = %w[Standard Very_soft_and_dull Dull Fairly_soft_and_dull Medium_reflectivity Highly_hard_and_polished Very_highly_polished Shiny Glossy Phong_Dull Phong_Shiny Phong_Glossy].join("|")
list4_1 = %w[Standard Very_soft_and_dull Dull Fairly_soft_and_dull Medium_reflectivity Highly_hard_and_polished Very_highly_polished Shiny Glossy Phong_Dull Phong_Shiny Phong_Glossy].join("|")
list5 = %w[Almost_mirror Drab_dark_low_reflexive Drab_half_reflexive Reflexive Very_reflexive Mirror_like Dark_reflexive].join("|")
list6 = %w[Ray-tracing Radiosity_Fast Radiosity_Normal Radiosity_Final Radiosity_Outdoor_Low Radiosity_Outdoor_High Radiosity_Indoor_Low Radiosity_Indoor_High].join("|")
array_of_dropdowns = [list0, list1, list1_1, list4_0, list4_1, list5, list6]

# Output file:
f_name = UI.openpanel( "Select file (with .pov extension) to export model", "", "*.pov" )
#f_split = f_name.split("\\")
#f_split = f_split[f_split.length-1]             #for instance "my_export.pov"
f_split = File.basename(f_name)

# Create the export folder and performs a changedir
#Dir.mkdir(f_name.split(".")[0])
#Dir.chdir(f_name.split(".")[0])
export_file = File.new( f_name, "w" )
report_file = File.new( f_split + ".txt", "w" )
#export_file = File.new( su_current_folder + "\/" + model_filename + "\/" + f_split.to_s, "w" )
#report_file = File.new( su_current_folder + "\/" + model_filename + "\/" + f_split.to_s + ".txt", "w" )
#report_file = File.new( f_name.split(".")[0] + ".txt", "w" )

# Dialog box 1: main params
prompts = ["Model export  ", "Scene type  ", "Lighting type", "Material finish (non-textured) ", "Material finish textured) ", "Glass finish  ", "Render  "]
values = [$model_type, $scene_type, $light_type, $finish_type_0, $finish_type_1, $glass_type, $render_type]
$model_type, $scene_type, $light_type, $finish_type_0, $finish_type_1, $glass_type, $render_type = inputbox prompts,values, array_of_dropdowns, "Scene main parameters"

# Header:
export_file.puts( "// Persistence Of Vision raytracer version 3.5 (or higher) file." )
export_file.puts( "// Exported from SketchUp " + version + " by su2pov 2.0 - by D. Bur." )
# Report header:
report_file.puts( "// Persistence Of Vision raytracer version 3.5 (or higher) REPORT FILE." )
report_file.puts( "// Exported from SketchUp " + version + " by su2pov 2.0 - by D. Bur." )
(underline export_file, report_file )

# Required Include:
export_file.puts( "// Pov-Ray standard includes ")
export_file.puts( "#include \"colors.inc\"" )
export_file.puts( "#include \"glass.inc\"" )
export_file.puts( "#include \"glass_old.inc\"" )
export_file.puts( "#include \"metals.inc\"" )
export_file.puts( "#include \"woods.inc\"" )
export_file.puts( "#include \"stones.inc\"" )
export_file.puts( "#include \"skies.inc\"" )
export_file.puts( "#include \"stars.inc\"" )
export_file.puts( "#include \"finish.inc\"" )
export_file.puts( "#include \"rad_def.inc\"" )
export_file.puts( "#include \"textures.inc\"" )
export_file.puts( "#include \"golds.inc\"" )
export_file.puts( "#include \"sunpos.inc\"" )
export_file.puts( "// Lightsys4 includes" )
export_file.puts( "#include \"CIE.inc\"" )
export_file.puts( "#include \"lightsys.inc\"" )
export_file.puts( "#include \"lightsys_constants.inc\"" )
export_file.puts( "#include \"espd_lightsys.inc\"" )
export_file.puts( "// Custom trees includes " )
export_file.puts( "#include \"leafdefs.inc\"" )
export_file.puts( "#include \"splinetree.inc\"" )

# Report includes:
report_file.puts( "Pov-Ray standard includes: \ncolors.inc, glass.inc, glass_old.inc, metals.inc, woods.inc, stones.inc, skies.inc, stars.inc, finish.inc, rad_def.inc, textures.inc, golds.inc, sunpos.inc" )
report_file.puts( "Lightsys4 standard includes: \nCIE.inc, lightsys.inc, lightsys_constants.inc, espd_lightsys.inc, CIE_Skylight.inc" )
report_file.puts( "Custom trees includes: \nleafdefs.inc, splinetree.inc" )
(underline export_file, report_file )

# Explode groups and components
(explode_groups)

# General purpose declare's
export_file.puts( "// Declare section" )
export_file.puts( "#declare GAMMA=1.0;" )
export_file.puts( "// ColorSystem and Whitepoint used for ALL pigment colors" )
export_file.puts( "CIE_ColorSystem(Beta_ColSys)" )
export_file.puts( "" )
(declare_bulb export_file)
if( $render_type != "Ray-tracing" )
  export_file.puts( "//No ambient finish because of radiosity algoritm selected" )
  export_file.puts( "#default { texture { finish { ambient 0 } } } ")
  export_file.puts( "" )
end

# Standard materials declare's
(switch_finish $finish_type_0,$finish_type_1)
(declare_materials export_file,$finish_type_0,$finish_type_1)
export_file.puts( "" )
(switch_back_finish $finish_type_0,$finish_type_1)

# Images materials declare's
Sketchup.set_status_text("Exporting images materials ...")
# first collect images to avoid duplicates
entities.each do |e|
  next if not e.kind_of? Sketchup::Image
    image_path = e.path
    names = image_path.split("\\")
    basename = names[names.length-1]
    $images_array.push( basename )
end
$images_array.uniq!

entities.each do |e|
  next if not e.kind_of? Sketchup::Image
  case $model_type
    when "All"
      (declare_image_material e,export_file,$finish_type)
    when "Visible_layers_only"
      if e.layer.visible?
        (declare_image_material e,export_file,$finish_type)
      end
  end
end
export_file.puts( "" )

# Report materials and images materials
report_file.puts( $n_materials.to_s + " materials exported, here are their MODIFIED names:" )
report_file.puts( "Please be sure to have all these textures in the same folder than the pov file." )
i = 0
$materials_array.each do |m|
  t = $textures_array[i]
  report_file.puts("Material: " + m + " uses texture: " + t)
  i = i + 1
end
(underline export_file, report_file )

# Global settings:
#-----------------
Sketchup.set_status_text("Exporting global settings...")
export_file.puts( "global_settings {" )
(write_ambient_type $scene_type,export_file, $render_type)
export_file.puts( "assumed_gamma GAMMA" )
# ADC (reflective and transparent surfaces become insignificant)
export_file.puts( "adc_bailout 0.005" )
export_file.puts( "max_intersections 128" )
export_file.puts( "max_trace_level 64" )

# Radiosity macro settings
(write_render_type $render_type,export_file)
# Report Render
report_file.puts( "Render type: " + $render_type)
(underline export_file, report_file )

# CIE   ColorSystem WhitePoint: Select a color/device system but change also the whitepoint to any non default value.
(write_whitepoint $scene_type,export_file,report_file)
(underline export_file, report_file )

# Background:
(write_background export_file, report_file)
(underline export_file, report_file )

# Camera(s) (one per page) output:
(write_cameras export_file)
if( $focalblur == true )
  report_file.puts( "Camera(s): " + $n_cameras.to_s + " (Focal blur active for current camera)")
  else
  report_file.puts( "Camera(s): " + $n_cameras.to_s + " (No focal blur)")
end
(underline export_file, report_file )

# Light sources export, depending of what lights to export:
case $scene_type
  when "Outdoor_Day"
    if( $light_type == "Sun_for_day_scene" ) or ( $light_type == "All" )
      # Sun
      Sketchup.set_status_text("Exporting sun...")
      (get_shadow_info export_file, report_file )
      # Box Media lights
      (write_boxmedialights export_file)
      report_file.puts( "Box Media lights: " + $n_boxmedialights.to_s)
      # Cone Media lights
      (write_conemedialights export_file)
      report_file.puts( "Cone(s) Media lights: " + $n_conemedialights.to_s)
    end
    if( $light_type == "Lights_for_night_scene" )
      Sketchup.set_status_text("Exporting lights...")
      # Point lights
      (write_point_lights export_file)
      report_file.puts( "Point lights: " + $n_pointlights.to_s)
      # Real point lights
      (write_real_point_lights export_file)
      report_file.puts( "Real Point lights: " + $n_realpointlights.to_s)
      # Spotlights
      (write_spotlights export_file)
      report_file.puts( "Spotlights: " + $n_spotlights.to_s)
      # Area lights:
      (write_arealights export_file)
      report_file.puts( "Arealights: " + $n_arealights.to_s)
      # Box Media lights
      (write_boxmedialights export_file)
      report_file.puts( "Box Media lights: " + $n_boxmedialights.to_s)
      # Cone Media lights
      (write_conemedialights export_file)
      report_file.puts( "Cone(s) Media lights: " + $n_conemedialights.to_s)
    end
    if( $light_type == "All" )
      Sketchup.set_status_text("Exporting lights...")
      # Point lights
      (write_point_lights export_file)
      report_file.puts( "Point lights: " + $n_pointlights.to_s)
      # Real point lights
      (write_real_point_lights export_file)
      report_file.puts( "Real Point lights: " + $n_realpointlights.to_s)
      # Spotlights
      (write_spotlights export_file)
      report_file.puts( "Spotlights: " + $n_spotlights.to_s)
      # Area lights:
      (write_arealights export_file)
      report_file.puts( "Arealights: " + $n_arealights.to_s)
    end
  when "Indoor_Day"
    if( $light_type == "Sun_for_day_scene" ) or ( $light_type == "All" )
      # Sun  
      Sketchup.set_status_text("Exporting sun...")
      (get_shadow_info export_file, report_file )
      # Box Media lights
      (write_boxmedialights export_file)
      report_file.puts( "Box Media lights: " + $n_boxmedialights.to_s)
      # Cone Media lights
      (write_conemedialights export_file)
      report_file.puts( "Cone(s) Media lights: " + $n_conemedialights.to_s)
    end
    if( $light_type == "Lights_for_night_scene" )
      Sketchup.set_status_text("Exporting lights...")
      # Point lights
      (write_point_lights export_file)
      report_file.puts( "Point lights: " + $n_pointlights.to_s)
      # Real point lights
      (write_real_point_lights export_file)
      report_file.puts( "Real Point lights: " + $n_realpointlights.to_s)
      # Spotlights
      (write_spotlights export_file)
      report_file.puts( "Spotlights: " + $n_spotlights.to_s)
      # Area lights:
      (write_arealights export_file)
      report_file.puts( "Arealights: " + $n_arealights.to_s)
      # Box Media lights
      (write_boxmedialights export_file)
      report_file.puts( "Box Media lights: " + $n_boxmedialights.to_s)
      # Cone Media lights
      (write_conemedialights export_file)
      report_file.puts( "Cone(s) Media lights: " + $n_conemedialights.to_s)
    end
    if( $light_type == "All" )
      Sketchup.set_status_text("Exporting lights...")
      # Point lights
      (write_point_lights export_file)
      report_file.puts( "Point lights: " + $n_pointlights.to_s)
      # Real point lights
      (write_real_point_lights export_file)
      report_file.puts( "Real Point lights: " + $n_realpointlights.to_s)
      # Spotlights
      (write_spotlights export_file)
      report_file.puts( "Spotlights: " + $n_spotlights.to_s)
      # Area lights:
      (write_arealights export_file)
      report_file.puts( "Arealights: " + $n_arealights.to_s)
    end
  when "Outdoor_Night"
    Sketchup.set_status_text("Exporting lights...")
    # Point lights
    (write_point_lights export_file)
    report_file.puts( "Point lights: " + $n_pointlights.to_s)
    # Real point lights
    (write_real_point_lights export_file)
    report_file.puts( "Real Point lights: " + $n_realpointlights.to_s)
    # Spotlights
    (write_spotlights export_file)
    report_file.puts( "Spotlights: " + $n_spotlights.to_s)
    # Area lights:
    (write_arealights export_file)
    report_file.puts( "Arealights: " + $n_arealights.to_s)
    # Box Media lights
    (write_boxmedialights export_file)
    report_file.puts( "Box Media lights: " + $n_boxmedialights.to_s)
    # Cone Media lights
    (write_conemedialights export_file)
    report_file.puts( "Cone(s) Media lights: " + $n_conemedialights.to_s)
  when "Indoor_Night"
    Sketchup.set_status_text("Exporting lights...")
    # Point lights
    (write_point_lights export_file)
    report_file.puts( "Point lights: " + $n_pointlights.to_s)
    # Real point lights
    (write_real_point_lights export_file)
    report_file.puts( "Real Point lights: " + $n_realpointlights.to_s)
    # Spotlights
    (write_spotlights export_file)
    report_file.puts( "Spotlights: " + $n_spotlights.to_s)
    # Area lights:
    (write_arealights export_file)
    report_file.puts( "Arealights: " + $n_arealights.to_s)
    # Box Media lights
    (write_boxmedialights export_file)
    report_file.puts( "Box Media lights: " + $n_boxmedialights.to_s)
    # Cone Media lights
    (write_conemedialights export_file)
    report_file.puts( "Cone(s) Media lights: " + $n_conemedialights.to_s)
end
export_file.puts( "" )
(underline export_file, report_file )

# Reset the color system to default
export_file.puts( "//Reset color system for geometry")
export_file.puts( "CIE_ColorSystem(Beta_ColSys)" )
export_file.puts( "")

# Ground:
(write_su_ground export_file, report_file)
(underline export_file, report_file )

# Trees:
(write_trees export_file, report_file)
(underline export_file, report_file )

# Faces:
# First count faces...
total_faces = 0
done_faces = 0
entities.each do |e|
  next if not e.kind_of? Sketchup::Face
  total_faces = total_faces + 1
end

entities.each do |e|
  next if not e.kind_of? Sketchup::Face
  case $model_type
    when "All"
      (write_single_face_to_polygon e,export_file)
      $n_faces = $n_faces + 1
    when "Visible_layers_only"
      if e.layer.visible?
        (write_single_face_to_polygon e,export_file)
        $n_faces = $n_faces + 1
      end
  end
  # Status bar progress
  done_faces = done_faces + 1
  percentage_done = (done_faces * 100) / total_faces
  setstatusbar( percentage_done.to_i )

end
# Report faces:
report_file.puts( "Faces total number: " + $n_faces.to_s + "\n")
(underline export_file, report_file )

# Images:
Sketchup.set_status_text("Exporting images...")
entities.each do |e|
  next if not e.kind_of? Sketchup::Image
  case $model_type
    when "All"
      (write_single_image_to_polygon e,export_file)
      $n_images = $n_images + 1
    when "Visible_layers_only"
      if e.layer.visible?
        (write_single_image_to_polygon e,export_file)
        $n_images = $n_images + 1
      end
  end
end
# Report images:
report_file.puts( "Images total number: " + $n_images.to_s + "\n")
(underline export_file, report_file )

# Close the export file:
export_file.close
# Close the report file:
report_file.close
# That's all
Sketchup.set_status_text("Export to POV-Ray -> OK.")
# Open the report file or launch Pov-Ray:
$launch_option = "Display_report_file" if not $launch_option
if( $launch_option == "Display_report_file" )
  UI.openURL("file://" + f_name.split(".")[0] + ".pov.txt")
  else
  UI.openURL("file://" + f_name.split(".")[0] + ".pov")
end

# End of undo
model.commit_operation

end

# ---------------------------------------------------------------------------
# Menu items:
#----------------------------------------------------------------------------
if( not $su2pov_menu_loaded )
    add_separator_to_menu("Plugins")
    utilities_menu = UI.menu("Plugins").add_submenu("POV-Ray converter")
    utilities_menu.add_item("_____________   WEATHER SETTINGS   _____________") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set/Edit weather parameters") { (set_weather_settings) }
    utilities_menu.add_item("Set/Edit clouds parameters") { (set_clouds_settings) }
    utilities_menu.add_item("Set/Edit custom clouds parameters") { (set_fastsky) }
    #utilities_menu.add_item("Set/Edit fog parameters") { (set_fog_settings) }
    utilities_menu.add_item("Set/Edit fog parameters") { Sketchup.send_action 21406}
    utilities_menu.add_item("_____________    LIGHTS SETTINGS    _____________") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set/Edit sunlight parameters") { (set_sunlight_params) }
    utilities_menu.add_item("Set/Edit pointlight parameters") { (set_pointlight_params) }
    utilities_menu.add_item("Set/Edit realpointlight parameters") { (set_realpointlight_params) }
    utilities_menu.add_item("Set/Edit spotlight parameters") { (set_spotlight_params) }
    utilities_menu.add_item("Set/Edit arealight parameters") { (set_arealight_params) }
    utilities_menu.add_item(" ") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set/Edit glow effect to a light") { (set_light_glow_effect) }
    utilities_menu.add_item("Delete glow effect to a light") { (delete_light_glow_effect) }
    utilities_menu.add_item("_____________ MEDIA LIGHTS SETTINGS _____________") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set/Edit medialight parameters") { (set_medialight_params) }
    utilities_menu.add_item("Assign cone medialight to spotlight") { (assign_conemedialight_to_spot) }
    utilities_menu.add_item("_____________  CAMERA SETTINGS  _____________") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set/Edit camera parameters") { (set_camera_params) }
    utilities_menu.add_item("Set/Edit focal blur parameters") { (set_focalblur_params) }
    utilities_menu.add_item("Set/Edit lens flare parameters") { (set_lensflare_params) }
    utilities_menu.add_item("_____________  TREE SETTINGS  _____________") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set/Edit tree parameters") { (set_tree_params) }
    utilities_menu.add_item(" ") { UI.messagebox("No command here !") }
    utilities_menu.add_item("_____________        EXPORT         _____________") { UI.messagebox("No command here !") }
    utilities_menu.add_item("Set action") { (set_launch_option) }
    utilities_menu.add_item("Export model to POV-Ray") { (su2pov) }
    $su2pov_menu_loaded = true
end

file_loaded("su2pov.rb")

