# Copyright 2004, Todd Burch - Burchwood USA   http://www.burchwoodusa.com 

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#-----------------------------------------------------------------------------

require 'sketchup.rb'

#-----------------------------------------------------------------------------

##  VERSION 1.0

def create_cabinet
    # First prompt for the overall box dimensions.  This is done using the inputbox
    # method.  In this case, we will actually use a wrapper for UI.inputbox
    # that is defined in sketchup.rb which does some extra error checking
    
    # The first step is to create some arrays which contain the prompts
    # and default values.
    prompts = ["Cabinet Overall Width", "Finished Height", "Finished Depth", "Material Thickness", "Countertop Thickness",     "Face Frame Thickness"]
    values = [36.inch, 30.inch, 24.inch, 0.75.inch, 0.75.inch, 0.75.inch]

    # A few constants, for now 

    back_thickness = 0.25.inch                            # Assume a 1/4" thick plywood back.  
    bottom_rail_height = 6.inch                           # This is the top of the bottom rail. 
    bottom_shelf_height = bottom_rail_height-0.125.inch   # This allows the bottom rail to be 1/8" higher. 
    
    # Note that for the default values, we specify the values like "6.feet"
    # This says that the value is  Langth, and tells what the units of
    # the Length are.  When you display a Length value in an input box, it
    # will be formatted using your current units settings and it will parse
    # the values that you enter as a Length.
    
    # Now display the inputbox
    results = inputbox prompts, values, "Cabinet Dimensions"
    
    # The values that the user entered are returned in the results value.
    # If the user hit the "Cancel" button, then the function will return
    # nil.  Otherwise it is an array of values.  SKetchUp tries to match
    # the type of the returned values with the types of the default value
    # supplied, so in this case, since the default values were Lengths we
    # will get back an Array of Lengths
    return if not results # This means that the user canceld the operation
    
    width, height, depth, material_thickness, countertop_thickness, ff_thickness = results
    
   
    # The first thing that we will do is bracket all of the entity creation
    # so that this looks like a single operation for undo.  If we didn't do this
    # you would get a whole bunch of separate undo items for each step
    # of the entity creation.
    model = Sketchup.active_model
    model.start_operation "Create Box"
    
    # We will add the new entities to the "active_entities" collection.  If
    # you are not doing a component edit, this will be the main model.
    # if you are doing a component edit, it will be the open component.
    # You could also use model.entities which is the top level collection
    # regardless of whether or not you are doing a component edit.
    entities = model.active_entities

    # If you wanted the box to be created as simple top level entities
    # rather than a Group, you could comment out the following two lines.
    group = entities.add_group
    entities = group.entities
    
    # First we will create a rectangle for the base.  There are a few
    # variations on the add_face method.  This uses the version that
    # takes points and automatically creates the edges needed.
    ls_pts = []
    ls_pts[0] = [0, ff_thickness, 0]
    ls_pts[1] = [material_thickness, ff_thickness, 0]
    ls_pts[2] = [material_thickness, depth-back_thickness, 0] 
    ls_pts[3] = [material_thickness/2, depth-back_thickness, 0] 
    ls_pts[4] = [material_thickness/2, depth, 0]
    ls_pts[5] = [0, depth, 0]
    ls_base = entities.add_face ls_pts
    
    
    rs_pts = []
    rs_pts[0] = [width-material_thickness, ff_thickness, 0]
    rs_pts[1] = [width, ff_thickness, 0]
    rs_pts[2] = [width, depth, 0]
    rs_pts[3] = [width-(material_thickness/2), depth, 0]
    rs_pts[4] = [width-(material_thickness/2), depth-back_thickness, 0] 
    rs_pts[5] = [width-material_thickness, depth-back_thickness, 0]
    rs_base = entities.add_face rs_pts

    bot_pts = [] 
    bot_pts[0] = [material_thickness/2, ff_thickness, bottom_shelf_height ] 
    bot_pts[1] = [material_thickness/2, ff_thickness, bottom_shelf_height-material_thickness ]
    bot_pts[2] = [material_thickness/2, depth, bottom_shelf_height-material_thickness ] 
    bot_pts[3] = [material_thickness/2, depth, bottom_shelf_height-(material_thickness/2) ] 
    bot_pts[4] = [material_thickness/2, depth-back_thickness, bottom_shelf_height-(material_thickness/2) ] 
    bot_pts[5] = [material_thickness/2, depth-back_thickness, bottom_shelf_height ] 

    # You could use a similar technique to crete the other faces of
    # the box.  For this example, we will use the pushpull method instead.
    # When you use pushpull, the direction is determined by the direction
    # of the fromnt of the face.  In order to control the direction and
    # get the pushpull to go in the direction we want, we first check the
    # direction of the face normal.  If it is not in the direction that
    # we want, we will reverse the sign of the distance.
    height = -height if( ls_base.normal.dot(Z_AXIS) < 0 )
    
    # Now we can do the pushpull to create the left and right cabinet sides.
    ls_base.pushpull (height+countertop_thickness)
    rs_base.pushpull (height+countertop_thickness)
    
    height = height.abs    # Reset to a positive value if changed. 

    # Cut a dado in both sides for the bottom shelf.  Do this by drawing on the inside face and then pushing
    # into the side 1/2 way through. 

    ls_dado_pts = [] 
    ls_dado_pts[0] = [material_thickness, ff_thickness, bottom_shelf_height ] 
    ls_dado_pts[1] = [material_thickness, ff_thickness, bottom_shelf_height-material_thickness ] 
    ls_dado_pts[2] = [material_thickness, depth-back_thickness, bottom_shelf_height-material_thickness ] 
    ls_dado_pts[3] = [material_thickness, depth-back_thickness, bottom_shelf_height ] 
    
    rs_dado_pts = [] 
    rs_dado_pts[0] = [width-material_thickness, ff_thickness, bottom_shelf_height ] 
    rs_dado_pts[1] = [width-material_thickness, ff_thickness, bottom_shelf_height-material_thickness ] 
    rs_dado_pts[2] = [width-material_thickness, depth-back_thickness, bottom_shelf_height-material_thickness ] 
    rs_dado_pts[3] = [width-material_thickness, depth-back_thickness, bottom_shelf_height ] 
    
    ls_dadotop = entities.add_line(ls_dado_pts[0],ls_dado_pts[3])     # Draw the left side dado rectangle on the face.
    ls_dadobot = entities.add_line(ls_dado_pts[1],ls_dado_pts[2])     # Draw the dado rectangle on the face.
    
    rs_dadotop = entities.add_line(rs_dado_pts[0],rs_dado_pts[3])     # Draw the right side dado rectangle on the face.
    rs_dadobot = entities.add_line(rs_dado_pts[1],rs_dado_pts[2])     # Draw the dado rectangle on the face.
    

    # Find the common face so that it can be pushed into the side to make a dado. 
    ls_dado_face = ls_dadotop.common_face(ls_dadobot) 
    ls_dado_face.pushpull -material_thickness/2 
    rs_dado_face = rs_dadotop.common_face(rs_dadobot) 
    rs_dado_face.pushpull -material_thickness/2 

    # Now that the dados are "cut", draw the bottom shelf. 
    bottom = entities.add_face bot_pts
    bottom.pushpull width-(material_thickness)  


    # Put a dado on the top of each side.
    lstd_pts = []   # lstd = Left Side Top Dado. 
    lstd_pts[0] = [material_thickness, ff_thickness, height-countertop_thickness] 
    lstd_pts[1] = [material_thickness, ff_thickness, height-countertop_thickness-material_thickness] 
    lstd_pts[2] = [material_thickness, depth-back_thickness, height-countertop_thickness-material_thickness] 
    lstd_pts[3] = [material_thickness, depth-back_thickness, height-countertop_thickness] 

    lstd_top = entities.add_line(lstd_pts[0], lstd_pts[3])
    lstd_bot = entities.add_line(lstd_pts[1], lstd_pts[2])
    
    # Find the face name and push it into the side.
    lstd_face = lstd_top.common_face(lstd_bot) 
    lstd_face.pushpull (-(material_thickness/2) )

    # Right side top dado.  "rstd". 
    rstd_pts = [] 
    rstd_pts[0] = [width-material_thickness, ff_thickness, height-countertop_thickness] 
    rstd_pts[1] = [width-material_thickness, ff_thickness, height-countertop_thickness-material_thickness] 
    rstd_pts[2] = [width-material_thickness, depth-back_thickness, height-countertop_thickness-material_thickness] 
    rstd_pts[3] = [width-material_thickness, depth-back_thickness, height-countertop_thickness] 

    rstd_top = entities.add_line(rstd_pts[0], rstd_pts[3])
    rstd_bot = entities.add_line(rstd_pts[1], rstd_pts[2])
    
    # Find the face name and push it into the side.
    rstd_face = rstd_top.common_face(rstd_bot) 
    rstd_face.pushpull (-(material_thickness/2) )

    # Add a couple cleats to the top, set into the dado.  One at the front and one at the back. 
    fc_p = [] 
    fc_p[0] = [material_thickness/2, ff_thickness, height-countertop_thickness ] 
    fc_p[1] = [material_thickness/2, ff_thickness, height-countertop_thickness-material_thickness ] 
    fc_p[2] = [material_thickness/2, ff_thickness+3.inch, height-countertop_thickness-material_thickness ] 
    fc_p[3] = [material_thickness/2, ff_thickness+3.inch, height-countertop_thickness ] 
    fc_face = entities.add_face fc_p
    fc_face.pushpull (width-material_thickness) 

    rc_p = [] 
    rc_p[0] = [material_thickness/2, depth-back_thickness, height-countertop_thickness ] 
    rc_p[1] = [material_thickness/2, depth-back_thickness, height-countertop_thickness-material_thickness ] 
    rc_p[2] = [material_thickness/2, depth-back_thickness-3.inch, height-countertop_thickness-material_thickness ] 
    rc_p[3] = [material_thickness/2, depth-back_thickness-3.inch, height-countertop_thickness ] 
    rc_face = entities.add_face rc_p
    rc_face.pushpull (width-material_thickness) 

    # Put a back on the cabinet. 

    back_p = [] 
    back_p[0] = [material_thickness/2, depth-back_thickness, height-countertop_thickness ] 
    back_p[1] = [material_thickness/2, depth-back_thickness, bottom_shelf_height-(material_thickness/2) ] 
    back_p[2] = [width-material_thickness/2, depth-back_thickness, bottom_shelf_height-(material_thickness/2) ] 
    back_p[3] = [width-material_thickness/2, depth-back_thickness, height-countertop_thickness ] 
    back_face = entities.add_face back_p 
    back_face.pushpull back_thickness

    ##----------------------------------------------------------------------------------------------- 
    # Zoom to Extents. 
    view = model.active_view 
    view.zoom_extents 

    ##-----------------------------------------------------------------------------------------------
    # Prompt for more information at this point.  Face Frame Parts. 
    prompts = ["Top Rail Width", "Bottom Rail Width", "Stile Width" ]
    values = [3.inch, 2.5.inch, 2.5.inch]

    # Now display the inputbox
    results = inputbox prompts, values, "Face Frame Dimensions"   
    top_rail_width, bottom_rail_width, stile_width = results

    #Draw the Left Stile First. 
    x1 = 0 
    x2 = stile_width  
    y1 = 0
    y2 = ff_thickness  
    z1 = 0 
    z2 = 0 
    stile_p = [] 
    stile_p[0] = [x1, y1, z1 ] 
    stile_p[1] = [x1, y2, z1 ] 
    stile_p[2] = [x2, y2, z2 ] 
    stile_p[3] = [x2, y1, z2 ]
    stile_face = entities.add_face stile_p 
    if stile_face.normal.dot(Z_AXIS) < 0 then   # We want to pull up: positive "z" axis. 
      stile_face.reverse! 
      end 
    stile_face.pushpull height-countertop_thickness 

    #Draw the Right Stile.   Adjust x1 & x2, other values are the same. 

    x1 = width-stile_width 
    x2 = width  
    stile_p = [] 
    stile_p[0] = [x1, y1, z1 ] 
    stile_p[1] = [x1, y2, z1 ] 
    stile_p[2] = [x2, y2, z2 ] 
    stile_p[3] = [x2, y1, z2 ]
    stile_face = entities.add_face stile_p 
    x = stile_face.normal 
    if stile_face.normal.dot(Z_AXIS) < 0 then   # We want to pull up: positive "z" axis. 
      stile_face.reverse! 
      end 
    stile_face.pushpull height-countertop_thickness    

    # Draw the Top Rail. 

    x1 = stile_width 
    x2 = stile_width
    y1 = 0
    y2 = ff_thickness 
    z1 = height-countertop_thickness-top_rail_width
    z2 = height-countertop_thickness
    stile_p = [] 
    stile_p[0] = [x1, y1, z1 ] 
    stile_p[1] = [x1, y2, z1 ] 
    stile_p[2] = [x2, y2, z2 ] 
    stile_p[3] = [x2, y1, z2 ]
    stile_face = entities.add_face stile_p 
    if stile_face.normal.dot(X_AXIS) < 0 then   # We want to pull right: positive "x" axis. 
      stile_face.reverse! 
      end 
    stile_face.pushpull width-(stile_width*2)     

    # Draw the Bottom Rail. 

    x1 = stile_width 
    x2 = stile_width 
    y1 = 0
    y2 = ff_thickness 
    z1 = bottom_rail_height 
    z2 = z1 - bottom_rail_width
    stile_p = [] 
    stile_p[0] = [x1, y1, z1 ] 
    stile_p[1] = [x1, y2, z1 ] 
    stile_p[2] = [x2, y2, z2 ] 
    stile_p[3] = [x2, y1, z2 ]
    stile_face = entities.add_face stile_p 
    if stile_face.normal.dot(X_AXIS) < 0 then   # We want to pull right: positive "x" axis. 
      stile_face.reverse! 
      end 
    stile_face.pushpull width-(stile_width*2)     
        
    
    # Now we are done and we can end the operation
    model.commit_operation
end


