# Copyright 2004, @Last Software, Inc.
# The code related to click 3 points was reused here from an original
# example titled rectangle.rb by @Last Software, Inc. 
# To it I added the house code. cheers! Tim 
# Name:           Housetool01
# Description:    Create simple houses for mass modeling applications
# Usage:          click 3 points, create a simple house
# Date:           2004,10,25 
# Type:           Tool
# Revised:        2004,10,26 Roof thickness option added 
#                 2004,10,27 Grouping & commenting added
#-----------------------------------------------------------------------------
require 'sketchup.rb'

class Housetool01

def initialize
    @ip = Sketchup::InputPoint.new
    @ip1 = Sketchup::InputPoint.new
    reset
    
    # sets the default house settings
    $heights = 350.cm if not $heights
    $rthick = 25.cm if not $rthick
    $slopes = 30 if not $slopes
     
    # Dialog box
    prompts = ["House height", "Roof Thickness" ,"Roof Slope"]
    values = [$heights, $rthick, $slopes]
    results = inputbox prompts, values, "House parameters"

    return if not results
    $heights, $rthick, $slopes = results
end

def reset
    @pts = []
    @state = 0
    @ip1.clear
    @drawn = false
    Sketchup::set_status_text "", SB_VCB_LABEL
    Sketchup::set_status_text "", SB_VCB_VALUE
    Sketchup::set_status_text "Click anywhere to start"
    @shift_down_time = Time.now
end

def activate
    self.reset
end

def deactivate(view)
    view.invalidate if @drawn
end

def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        @pts[4] = @pts[0]
        need_draw = @ip.display? || @drawn
    when 1
        @pts[1] = @ip.position
        @length = @pts[0].distance @pts[1]
        Sketchup::set_status_text @length.to_s, SB_VCB_VALUE
    when 2
        pt1 = @ip.position
        pt2 = pt1.project_to_line @pts
        vec = pt1 - pt2

        @pt1 = @ip.position
        @pt2 = pt1.project_to_line @pts
        @vec = @pt1 - @pt2


        @width = vec.length
        if( @width > 0 )
            # test for a square
            square_point = pt2.offset(vec, @length)
            if( view.pick_helper.test_point(square_point, x, y) )
                @width = @length
                @pts[2] = @pts[1].offset(vec, @width)
                @pts[3] = @pts[0].offset(vec, @width)
                view.tooltip = "Square"
            else
                @pts[2] = @pts[1].offset(vec)
                @pts[3] = @pts[0].offset(vec)
            end
        else
            @pts[2] = @pts[1]
            @pts[3] = @pts[0]
        end
        Sketchup::set_status_text @width.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end

def onMouseMove(flags, x, y, view)
    self.set_current_point(x, y, view)
end

def create_rectangle
#--------------------------------------------------------------------------
    model = Sketchup.active_model
    model.start_operation "Create House"
    entities = model.active_entities

#Do Grouping: 
#Place a # before each of the 2 following lines to ungroup the house
    group = entities.add_group
    entities = group.entities
        
    # creates a new point from p1 in the direction of p2-p3 with length d
    # params are Point3d, 2 vertices, a length, returns a Point3d
    def translate(p1, p2, p3, d)
        v = p3 - p2
        v.length = d
        trans = Geom::Transformation.translation(v)
        return p1.transform(trans)
    end   
    
    #------plan of house
    base = entities.add_face(@pts[0], @pts[1], @pts[2], @pts[3])
    base.pushpull -$heights
    
    #------create new points at the eaves "$heights"
    @pt0=Geom::Point3d.new(@pts[0][0], @pts[0][1], @pts[0][2]+$heights)
    @pt1=Geom::Point3d.new(@pts[1][0], @pts[1][1], @pts[1][2]+$heights)
    @pt2=Geom::Point3d.new(@pts[2][0], @pts[2][1], @pts[2][2]+$heights)
    @pt3=Geom::Point3d.new(@pts[3][0], @pts[3][1], @pts[3][2]+$heights)

    #------create new points for roof thickness "$rthick"
    @pt00=Geom::Point3d.new(@pts[0][0], @pts[0][1], @pts[0][2]+$heights+$rthick)
    @pt11=Geom::Point3d.new(@pts[1][0], @pts[1][1], @pts[1][2]+$heights+$rthick)
    @pt22=Geom::Point3d.new(@pts[2][0], @pts[2][1], @pts[2][2]+$heights+$rthick)
    @pt33=Geom::Point3d.new(@pts[3][0], @pts[3][1], @pts[3][2]+$heights+$rthick)
        
    #------create a new mid-point pt0m btw. @pt0 & @pt3
    @pt0m = translate( @pt0, @pt0, @pt3, @width/2)
    #------create a new mid-point pt1m btw. $pt1 & @pt2
    @pt1m = translate( @pt1, @pt1, @pt2, @width/2)
         
    @roofpitch=@width/2* Math::tan($slopes.degrees)
    #------create a new point pt3hm (Ridge)
    @pt0mh=Geom::Point3d.new(@pt0m[0], @pt0m[1], @pt0m[2]+@roofpitch)
    #------create a new point pt2hm (Ridge)
    @pt1mh=Geom::Point3d.new(@pt1m[0], @pt1m[1], @pt1m[2]+@roofpitch)

    #------create a new point pt3hm (Ridge+rthick)
    @pt0mh_rthick=Geom::Point3d.new(@pt0m[0], @pt0m[1], @pt0m[2]+@roofpitch+$rthick)
    #------create a new point pt2hm (Ridge+rthick)
    @pt1mh_rthick=Geom::Point3d.new(@pt1m[0], @pt1m[1], @pt1m[2]+@roofpitch+$rthick)
    
    #------Do the inner roof planes           
    base = entities.add_face(@pt3, @pt0mh, @pt1mh,@pt2)
    base = entities.add_face(@pt0, @pt0mh, @pt1mh,@pt1)
    #------Do gable ends
    base = entities.add_face(@pt0, @pt3, @pt0mh)
    base = entities.add_face(@pt1, @pt2, @pt1mh)
    #------Do the outer roof planes           
    base = entities.add_face(@pt33, @pt0mh_rthick, @pt1mh_rthick,@pt22)
    base = entities.add_face(@pt00, @pt0mh_rthick, @pt1mh_rthick,@pt11)
    #------Do roof end gables
    base = entities.add_face(@pt0, @pt00, @pt0mh_rthick, @pt33, @pt3, @pt0mh)
    base = entities.add_face(@pt1, @pt11, @pt1mh_rthick, @pt22, @pt2, @pt1mh)
    #------Do Barge Board
    base = entities.add_face(@pt0, @pt00, @pt11, @pt1)
    base = entities.add_face(@pt3, @pt33, @pt22, @pt2)
#-----------------------------------------------------------------------------
    #view = Sketchup.active_model.active_view 
    #view.zoom_extents 
#----------------------------------------------------------------------------- 
    model.commit_operation
    self.reset
end

def increment_state
    @state += 1
    case @state
    when 1
        @ip1.copy! @ip
        Sketchup::set_status_text "Click for second point"
        Sketchup::set_status_text "Length", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 2
        @ip1.clear
        Sketchup::set_status_text "Click for third point"
        Sketchup::set_status_text "Width", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 3
        self.create_rectangle
    end
end

def onLButtonDown(flags, x, y, view)
    self.set_current_point(x, y, view)
    self.increment_state
    view.lock_inference
end

def onCancel(flag, view)
    view.invalidate if @drawn
    self.reset
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
    
    case @state
    when 1
        # update the width
        vec = @pts[1] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[1] = @pts[0].offset(vec)
            view.invalidate
            self.increment_state
        end
    when 2
        # update the height
        vec = @pts[3] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[2] = @pts[1].offset(vec)
            @pts[3] = @pts[0].offset(vec)
            self.increment_state
        end
    end
end

def getExtents
    bb = Geom::BoundingBox.new
    case @state
    when 0
        # We are getting the first point
        if( @ip.valid? && @ip.display? )
            bb.add @ip.position
        end
    when 1
        bb.add @pts[0]
        bb.add @pts[1]
    when 2
        bb.add @pts
    end
    bb
end

def draw(view)
    @drawn = false
    
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end

    # show the rectangle
    if( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
        inference_locked = view.inference_locked?
        view.line_width = 3 if inference_locked
        view.draw(GL_LINE_STRIP, @pts[0], @pts[1])
        view.line_width = 1 if inference_locked
        @drawn = true
    elsif( @state > 1 )
        # draw the curve
        view.drawing_color = "black"
        view.draw(GL_LINE_STRIP, @pts)
        @drawn = true
    end
end

def onKeyDown(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY && rpt == 1 )
        @shift_down_time = Time.now
        
        # if we already have an inference lock, then unlock it
        if( view.inference_locked? )
            view.lock_inference
        elsif( @state == 0 )
            view.lock_inference @ip
        elsif( @state == 1 )
            view.lock_inference @ip, @ip1
        end
    end
end

def onKeyUp(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY &&
        view.inference_locked? &&
        (Time.now - @shift_down_time) > 0.5 )
        view.lock_inference
    end
end
    
end # class Housetool01

#=============================================================================
def Housetool01
    Sketchup.active_model.select_tool Housetool01.new
end

#if( not file_loaded?("Housetool01.rb") )
#    UI.menu("Plugins").add_item("HouseTool") { Sketchup.active_model.select_tool Housetool01.new }
#end
#-----------------------------------------------------------------------------
file_loaded("Housetool01.rb")


