# Copyright 2004, @Last Software, Inc.
# The code related to click 3 points was reused here from an original
# example titled rectangle.rb by @Last Software, Inc. 
# To it I added the rafter code. cheers! Tim 
# Name:           RafterTool
# Description:    Create a string of rafters @ centers and mirrored about a ridge  
# Usage:          click 3 points, create a bunch of rafters contained within a defined area
# Date:           2004,11,11 
# Type:           Tool
# Revised:        2004,11,13 fixed alignment problem
#                 2004,11,15 changed all the FRIGIN' push-pulls to planes  
#                 because push-pull will reverse direction, should the user rectangle  
#                 be picked in an direction opposite to the push-pull direction.
#                 Reverse face problem remains, due to user input of rectangle direction.
#-------------------------------------------------------------------------------------------
require 'sketchup.rb'

class RafterTool

def initialize

    @ip = Sketchup::InputPoint.new
    @ip1 = Sketchup::InputPoint.new
    reset
    
    # sets the default rafter settings
    @radepth = 30.cm if not @radepth
    @rathick = 5.cm if not @rathick
    @rapitch = 6.0 if not @rapitch
    @raoc = 40.cm if not @raoc 
    @raoh = 0.0.cm if not @raoh
     
    # Dialog box
    prompts = ["Depth", "Thickness ", "Pitch", "Spacing", "Overhang "]
    values = [@radepth, @rathick, @rapitch, @raoc, @raoh]
    results = inputbox prompts, values, "Rafter parameters"

    return if not results
    @radepth, @rathick, @rapitch, @raoc, @raoh = results
end

def reset
    @pts = []
    @state = 0
    @ip1.clear
    @drawn = false
    Sketchup::set_status_text "", SB_VCB_LABEL
    Sketchup::set_status_text "", SB_VCB_VALUE
    Sketchup::set_status_text "Click anywhere to start"
    @shift_down_time = Time.now
end

def activate
    self.reset
end

def deactivate(view)
    view.invalidate if @drawn
end

def set_current_point(x, y, view)
    if( !@ip.pick(view, x, y, @ip1) )
        return false
    end
    need_draw = true
    
    # Set the tooltip that will be displayed
    view.tooltip = @ip.tooltip
        
    # Compute points
    case @state
    when 0
        @pts[0] = @ip.position
        @pts[4] = @pts[0]
        need_draw = @ip.display? || @drawn
    when 1
        @pts[1] = @ip.position
        @length = @pts[0].distance @pts[1]
        Sketchup::set_status_text @length.to_s, SB_VCB_VALUE
    when 2
        pt1 = @ip.position
        pt2 = pt1.project_to_line @pts
        vec = pt1 - pt2

        @pt1 = @ip.position
        @pt2 = pt1.project_to_line @pts
        @vec = @pt1 - @pt2


        @width = vec.length
        if( @width > 0 )
            # test for a square
            square_point = pt2.offset(vec, @length)
            if( view.pick_helper.test_point(square_point, x, y) )
                @width = @length
                @pts[2] = @pts[1].offset(vec, @width)
                @pts[3] = @pts[0].offset(vec, @width)
                view.tooltip = "Square"
            else
                @pts[2] = @pts[1].offset(vec)
                @pts[3] = @pts[0].offset(vec)
            end
        else
            @pts[2] = @pts[1]
            @pts[3] = @pts[0]
        end
        Sketchup::set_status_text @width.to_s, SB_VCB_VALUE
    end

    view.invalidate if need_draw
end

def onMouseMove(flags, x, y, view)
    self.set_current_point(x, y, view)
end

def create_rectangle
#--------------------------------------------------------------------------
    model = Sketchup.active_model
    model.start_operation "Create Rafters"
    entities = model.active_entities
    
    # creates a new point from p1 in the direction of p2-p3 with length d
    # params are Point3d, 2 vertices, a length, returns a Point3d
    def translate(p1, p2, p3, d)
        v = p3 - p2
        v.length = d
        trans = Geom::Transformation.translation(v)
        return p1.transform(trans)
    end  
    
    #------plan of rectangle for checking purposes
    #base = entities.add_face(@pts[0], @pts[1], @pts[2], @pts[3])
    
    #------create a new set of points from the original 3 point pick
    @pt0=Geom::Point3d.new(@pts[0][0], @pts[0][1], @pts[0][2])
    @pt1=Geom::Point3d.new(@pts[1][0], @pts[1][1], @pts[1][2])
    @pt2=Geom::Point3d.new(@pts[2][0], @pts[2][1], @pts[2][2])
    @pt3=Geom::Point3d.new(@pts[3][0], @pts[3][1], @pts[3][2])
    
    #------create a point to locate thickness of the rafter 
    @pt0a=translate(@pt0, @pt0, @pt3, @rathick)
    @pt1a=translate(@pt1, @pt1, @pt2, @rathick)
    @pt2a=translate(@pt2, @pt2, @pt1, @rathick)
    @pt3a=translate(@pt3, @pt3, @pt0, @rathick)
    
    #------create a new mid-point (ridge)
    @pt5=translate(@pt0, @pt0, @pt1, @length/2)
    @pt5a=translate(@pt0a, @pt0a, @pt1a, @length/2)
    @pt6=translate(@pt3, @pt3, @pt2, @length/2)
    @pt6a=translate(@pt3a, @pt3a, @pt2a, @length/2)
    
    #------formula defines similar triangles & Pythagoras 
    @raheight=(@rapitch/12)*(@length/2)
    @ralength=Math::sqrt((@length/2)*(@length/2)+@raheight*@raheight)
    
    #------formula defines plump cut of roof by Similar triangles    
    @racut=@radepth*(@ralength/(@length/2))
    
    #------formula defines overhang of roof by Similar triangles
    @raslope=@raoh*(@ralength/(@length/2))
    
    #------formula defines rafter on center spacing    
    @num_rafters=((@width/@raoc)+1).to_i 
    
   #------points related to left & right Rafter planes
    @pt55=Geom::Point3d.new(@pt5[0], @pt5[1], @pt5[2]+@raheight)
    @pt55a=Geom::Point3d.new(@pt5a[0],@pt5a[1],@pt5a[2]+@raheight)
    @pt555=Geom::Point3d.new(@pt55[0], @pt55[1], @pt55[2]+@racut)
    @pt555a=Geom::Point3d.new(@pt55a[0], @pt55a[1], @pt55a[2]+@racut)
    @pt00=Geom::Point3d.new(@pt0[0], @pt0[1], @pt0[2]+@racut)
    @pt00a=Geom::Point3d.new(@pt0a[0], @pt0a[1], @pt0a[2]+@racut)
    @pt11=Geom::Point3d.new(@pt1[0], @pt1[1], @pt1[2]+@racut)
    @pt11a=Geom::Point3d.new(@pt1a[0], @pt1a[1], @pt1a[2]+@racut)
    #------points related to left & right Rafter overhang planes
    @pt7=translate(@pt0, @pt55, @pt0, @raslope)
    @pt7a=translate(@pt0a, @pt55a, @pt0a, @raslope)
    @pt77=translate(@pt00, @pt555, @pt00, @raslope)
    @pt77a=translate(@pt00a, @pt555a, @pt00a, @raslope)
    @pt8=translate(@pt1, @pt55, @pt1, @raslope)
    @pt8a=translate(@pt1a, @pt55a, @pt1a, @raslope)
    @pt88=translate(@pt11, @pt555, @pt11, @raslope)
    @pt88a=translate(@pt11a, @pt555a, @pt11a, @raslope)
    
    definitions=model.definitions
    count=definitions.add
    entities=count.entities
    
    #-----Do Rafter Planes 
    if @raoh == 0 
    #-----Do Rafter Planes without overhang left
    base = entities.add_face(@pt0, @pt00, @pt555, @pt55)
    base.reverse!
    base = entities.add_face(@pt0, @pt0a, @pt55a, @pt55)
    base = entities.add_face(@pt0, @pt0a, @pt00a, @pt00)
    base = entities.add_face(@pt0a, @pt00a, @pt555a, @pt55a)
    base = entities.add_face(@pt00, @pt00a, @pt555a, @pt555)
    base = entities.add_face(@pt55, @pt55a, @pt555a, @pt555)
    #-----Do Rafter Planes without overhang right
    base = entities.add_face(@pt1, @pt11, @pt555, @pt55) 
    base = entities.add_face(@pt1, @pt1a, @pt55a, @pt55)
    base = entities.add_face(@pt1, @pt1a, @pt11a, @pt11)    
    base = entities.add_face(@pt1a, @pt11a, @pt555a, @pt55a)
    base = entities.add_face(@pt11, @pt11a, @pt555a, @pt555)
    #------Do transform of location and copy rafters to their new locations  
    t=Geom::Transformation.translation(Geom::Vector3d.new(0, 0, 0))
    entities=model.active_entities
    entities.add_instance(count, t)
    i = 1
    while i < @num_rafters
       # Transformation
       i = i + 1
       vec = @pt3 - @pt0
       vec.length = @raoc*(i-1)
       t=Geom::Transformation.translation(Geom::Vector3d.new(0,@raoc*(i-1), 0))
       t=Geom::Transformation.translation(vec)
       entities.add_instance(count, t)
    end #if 
    else 
    #------Do Rafter Planes with overhang left
    base = entities.add_face(@pt7, @pt77, @pt555, @pt55)
    base.reverse!
    base = entities.add_face(@pt7a, @pt77a, @pt555a, @pt55a)
    base = entities.add_face(@pt7, @pt77, @pt77a, @pt7a)
    base = entities.add_face(@pt7, @pt7a, @pt55a, @pt55)
    base = entities.add_face(@pt77, @pt77a, @pt555a, @pt555)
    base = entities.add_face(@pt55, @pt55a, @pt555a, @pt555)
    #------Do Rafter Planes with overhang right
    base = entities.add_face(@pt8, @pt88, @pt555, @pt55)
    base = entities.add_face(@pt8a, @pt88a, @pt555a, @pt55a)
    base = entities.add_face(@pt8, @pt88, @pt88a, @pt8a)
    base = entities.add_face(@pt88, @pt88a, @pt555a, @pt555)
    base = entities.add_face(@pt55, @pt55a, @pt8a, @pt8)
    #------Do transform of location and copy rafters to their new locations  
    t=Geom::Transformation.translation(Geom::Vector3d.new(0, 0, 0))
    entities=model.active_entities
    entities.add_instance(count, t)
    i = 1
    while i < @num_rafters
       # Transformation
       i = i + 1
       vec = @pt3 - @pt0
       vec.length = @raoc*(i-1)
       t=Geom::Transformation.translation(Geom::Vector3d.new(0,@raoc*(i-1), 0))
       t=Geom::Transformation.translation(vec)
       entities.add_instance(count, t)
    end #if
    end #end @raoh == 0
    #--------------------------------------------------------------------------
    #------points related to left & right Rafter planes, for Last Rafter planes
    @pt33=Geom::Point3d.new(@pt3[0], @pt3[1], @pt3[2]+@racut)
    @pt33a=Geom::Point3d.new(@pt3a[0], @pt3a[1], @pt3a[2]+@racut)
    @pt22=Geom::Point3d.new(@pt2[0], @pt2[1], @pt2[2]+@racut)
    @pt22a=Geom::Point3d.new(@pt2a[0], @pt2a[1], @pt2a[2]+@racut)
    @pt66=Geom::Point3d.new(@pt6[0],@pt6[1],@pt6[2]+@raheight)
    @pt66a=Geom::Point3d.new(@pt6a[0],@pt6a[1],@pt6a[2]+@raheight)
    @pt666=Geom::Point3d.new(@pt66[0], @pt66[1], @pt66[2]+@racut)
    @pt666a=Geom::Point3d.new(@pt66a[0], @pt66a[1], @pt66a[2]+@racut)
    #------points related to left & right Rafter overhang planes, for Last Rafter planes
    @pt9=translate(@pt3, @pt66, @pt3, @raslope)
    @pt9a=translate(@pt3a, @pt66a, @pt3a, @raslope)
    @pt99=translate(@pt33, @pt666, @pt33, @raslope)
    @pt99a=translate(@pt33a, @pt666a, @pt33a, @raslope)
    @pt10=translate(@pt2, @pt66, @pt2, @raslope)
    @pt10a=translate(@pt2a, @pt66a, @pt2a, @raslope)
    @pt1010=translate(@pt22, @pt666, @pt22, @raslope)
    @pt1010a=translate(@pt22a, @pt666a, @pt22a, @raslope)
    
    group=entities.add_group
    entities=group.entities
    
    #-----Do Last Rafter planes
    if @raoh == 0 
    #-----Do Rafter planes without overhang left
    base=entities.add_face(@pt3a, @pt33a, @pt666a, @pt66a)
    base.reverse!
    base=entities.add_face(@pt3, @pt33, @pt666, @pt66)
    base=entities.add_face(@pt3, @pt3a, @pt33a, @pt33)
    base=entities.add_face(@pt33, @pt33a, @pt666a, @pt666)
    base=entities.add_face(@pt3, @pt3a, @pt66a, @pt66)
    base=entities.add_face(@pt66, @pt666, @pt666a, @pt66a)
    #-----Do Rafter planes without overhang right
    base=entities.add_face(@pt2a, @pt22a, @pt666a, @pt66a)
    base=entities.add_face(@pt2, @pt22, @pt666, @pt66)
    base=entities.add_face(@pt2, @pt2a, @pt22a, @pt22)
    base=entities.add_face(@pt22, @pt22a, @pt666a, @pt666)
    base=entities.add_face(@pt2, @pt2a, @pt66a, @pt66)
    base=entities.add_face(@pt66, @pt666, @pt666a, @pt66a)
    else
    #-----Do Rafter Planes with overhang left
    base=entities.add_face(@pt9a, @pt99a, @pt666a, @pt66a)
    base.reverse!
    base=entities.add_face(@pt9, @pt99, @pt666, @pt66)
    base=entities.add_face(@pt9, @pt9a, @pt99a, @pt99)
    base=entities.add_face(@pt99, @pt99a, @pt666a, @pt666)
    base=entities.add_face(@pt9, @pt9a, @pt66a, @pt66)
    base=entities.add_face(@pt66, @pt666, @pt666a, @pt66a)
    #-----Do Rafter Planes with overhang right
    base=entities.add_face(@pt10a, @pt1010a, @pt666a, @pt66a)
    base=entities.add_face(@pt10, @pt1010, @pt666, @pt66)
    base=entities.add_face(@pt10, @pt10a, @pt1010a, @pt1010)
    base=entities.add_face(@pt1010, @pt1010a, @pt666a, @pt666)
    base=entities.add_face(@pt10, @pt10a, @pt66a, @pt66)
    base=entities.add_face(@pt66, @pt666, @pt666a, @pt66a)
    end #end @raoh == 0
#----------------------------------------------------------------------------
    #view = Sketchup.active_model.active_view 
    #view.zoom_extents 
#----------------------------------------------------------------------------
    model.commit_operation
    self.reset
end

def increment_state
    @state += 1
    case @state
    when 1
        @ip1.copy! @ip
        Sketchup::set_status_text "Click for second point"
        Sketchup::set_status_text "Length", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 2
        @ip1.clear
        Sketchup::set_status_text "Click for third point"
        Sketchup::set_status_text "Width", SB_VCB_LABEL
        Sketchup::set_status_text "", SB_VCB_VALUE
    when 3
        self.create_rectangle
    end
end

def onLButtonDown(flags, x, y, view)
    self.set_current_point(x, y, view)
    self.increment_state
    view.lock_inference
end

def onCancel(flag, view)
    view.invalidate if @drawn
    self.reset
end

# This is called when the user types a value into the VCB
def onUserText(text, view)
    # The user may type in something that we can't parse as a length
    # so we set up some exception handling to trap that
    begin
        value = text.to_l
    rescue
        # Error parsing the text
        UI.beep
        value = nil
        Sketchup::set_status_text "", SB_VCB_VALUE
    end
    return if !value
    
    case @state
    when 1
        # update the width
        vec = @pts[1] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[1] = @pts[0].offset(vec)
            view.invalidate
            self.increment_state
        end
    when 2
        # update the height
        vec = @pts[3] - @pts[0]
        if( vec.length > 0.0 )
            vec.length = value
            @pts[2] = @pts[1].offset(vec)
            @pts[3] = @pts[0].offset(vec)
            self.increment_state
        end
    end
end

def getExtents
    bb = Geom::BoundingBox.new
    case @state
    when 0
        # We are getting the first point
        if( @ip.valid? && @ip.display? )
            bb.add @ip.position
        end
    when 1
        bb.add @pts[0]
        bb.add @pts[1]
    when 2
        bb.add @pts
    end
    bb
end

def draw(view)
    @drawn = false
    
    # Show the current input point
    if( @ip.valid? && @ip.display? )
        @ip.draw(view)
        @drawn = true
    end

    # show the rectangle
    if( @state == 1 )
        # just draw a line from the start to the end point
        view.set_color_from_line(@ip1, @ip)
        inference_locked = view.inference_locked?
        view.line_width = 3 if inference_locked
        view.draw(GL_LINE_STRIP, @pts[0], @pts[1])
        view.line_width = 1 if inference_locked
        @drawn = true
    elsif( @state > 1 )
        # draw the curve
        view.drawing_color = "black"
        view.draw(GL_LINE_STRIP, @pts)
        @drawn = true
    end
end

def onKeyDown(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY && rpt == 1 )
        @shift_down_time = Time.now
        
        # if we already have an inference lock, then unlock it
        if( view.inference_locked? )
            view.lock_inference
        elsif( @state == 0 )
            view.lock_inference @ip
        elsif( @state == 1 )
            view.lock_inference @ip, @ip1
        end
    end
end

def onKeyUp(key, rpt, flags, view)
    if( key == CONSTRAIN_MODIFIER_KEY &&
        view.inference_locked? &&
        (Time.now - @shift_down_time) > 0.5 )
        view.lock_inference
    end
end
    
end # class RafterTool

#=============================================================================
def RafterTool
    Sketchup.active_model.select_tool RoofTool.new
end

#if( not file_loaded?("RafterTool.rb") )
#    UI.menu("Plugins").add_item("RafterTool") { Sketchup.active_model.select_tool RafterTool.new }
#end
#-----------------------------------------------------------------------------
file_loaded("RafterTool.rb")


