# Name :          Screw 2.0
# Description :   screw a set of edges around the z-axis (if "turns" is a negative number, rotation is clockwise)
# Author :        Frank Wiesner
# Usage :         1. select edges (and Components)
#                 2. run script
# Date :          30.Aug.2oo4
# Type :          script
# History:        
#                 2.1 (31.Aug.2004) - remembers turn/step values during SU session
#                                     expressions in dialog (i.e. use turns = 483/360.0 for a 483 degrees turn)
#                 2.0 (30.Aug.2004) - support for ConstructionPoints
#                                     instant drawing (although Sketchup does not update during execution)
#                                     status info
#                                     new algorithm for offset vector
#                                     fractional value support
#                 1.4  (5.Aug.2oo4) - cleaned up ugly transformation code, script now faster
#                                     error checking for "add_face': Points are not planar" errors
#                 1.3  (4.Aug.2oo4) - support for components
#                 1.2  (4.Aug.2oo4) - edges are smoothed
#                 1.1  (3.Aug.2oo4) - if "turns" is a negative number, rotation is clockwise, minor fixes
#                 1.o  (3.Aug.2oo4) - first version

require 'sketchup.rb'

class Screw

def initialize
    @turns = "2.25"
    @steps = "10.0"
end

def screw_selection
	mo=Sketchup.active_model()
	ss = mo.selection()

	if ss.empty? 
		UI.messagebox("Nothing to screw")
	else
		prompts = ["Steps", "Turns"]
		values = [@steps, @turns]
		results = inputbox prompts, values, "Screw"
		temp1, temp2 = results
		if (temp1 != false)
			@steps = temp1
			@turns = temp2

			steps = eval(@steps)
			turns = eval(@turns)

			
			# neg. steps not allowed!
			if (steps < 0) 
				steps = -1*steps
			end
			
			rotDir = 1
			if (turns < 0) 
				turns = -1*turns
				rotDir = -1
			end
			
			angularOffset = 360.degrees/(steps.floor)
			angularOffsetComponent = 360.degrees/steps
			targetAngle = 360.degrees*turns

			# copy all relevant points in an array...
			
			verticesArray = Array.new()
			edgeArray = Array.new()
			compArray = Array.new()
			constrPtsArray = Array.new()
			
			ss.each {|e| 
				if (e.kind_of? Sketchup::Edge)
					verticesArray.push(e.vertices) 
					edgeArray.push(e) 
				end
				compArray.push(e) if e.kind_of? Sketchup::ComponentInstance
				constrPtsArray(e.position) if e.kind_of? Sketchup::ConstructionPoint
			}
			if (edgeArray.empty? && compArray.empty?)
				UI.messagebox("At least one edge or component must be selected!")
			else
				verticesArray.flatten!
				ptArray = Array.new()
				verticesArray.each do |v|
					ptArray.push(Geom::Point3d.new(v.position))
				end
				
				# determin vertical and radial Offset
				
				# first we find edge with the closest Vertex to z-axis...
				
				closestEdge = edgeArray[0]
				closestVertex = edgeArray[0].start

				edgeArray.each do |e|
					e.vertices.each do |v|
						if ( (v.position.x*v.position.x+v.position.y*v.position.y) < (closestVertex.position.x*closestVertex.position.x+closestVertex.position.y*closestVertex.position.y) )
							closestEdge = e
							closestVertex = v
						end 
					end
				end
				
				# now we look for start and endpoints of the polyline...
				
				vertex1 = closestEdge.start
				vertex2 = closestEdge.end
				edge1 = closestEdge
				edge2 = closestEdge
				cyclic = false

				currentEdge = get_next_edge(edge1, vertex1)
				while ((currentEdge != nil) && (cyclic == false))
					vertex1 = currentEdge.other_vertex(vertex1)
					cyclic = true if (vertex1 == vertex2)
					edge1 = currentEdge
					currentEdge = get_next_edge(edge1, vertex1)
				end

				currentEdge = get_next_edge(edge2, vertex2)
				while ((currentEdge != nil) && (cyclic == false))
					vertex2 = currentEdge.other_vertex(vertex2)
					edge2 = currentEdge
					currentEdge = get_next_edge(edge2, vertex2)
				end

				# now we determine the offset vector...
				
				fullOffsetVector = Geom::Vector3d.new(0, 0, 0) 

				if (!cyclic)
					# if any direction is present, use it...
					if ((vertex1 == edge1.start) && (vertex2 == edge2.end))
						fullOffsetVector = vertex2.position-vertex1.position
					elsif ((vertex1 == edge1.end) && (vertex2 == edge2.start))
						fullOffsetVector = vertex1.position-vertex2.position
					else
						# if no direction is indicated by polyline, default is up the blue axis...
						if (vertex1.position.z < vertex2.position.z)
							fullOffsetVector = vertex2.position-vertex1.position
						elsif (vertex1.position.z > vertex2.position.z)
							fullOffsetVector = vertex1.position-vertex2.position
						else
							# if both endpoints even have the same height, direction is away from the blue axis (e.g. outward)...
							if (vertex1.position.x*vertex1.position.x+vertex1.position.y*vertex1.position.y < vertex2.position.x*vertex2.position.x+vertex2.position.y*vertex2.position.y)
								fullOffsetVector = vertex2.position-vertex1.position
							else
								fullOffsetVector = vertex1.position-vertex2.position
							end
						end
					end
				end

				faceOrientation = 1
				faceOrientation = -1 if (fullOffsetVector.z<0)
				xOff = fullOffsetVector.x/steps
				yOff = fullOffsetVector.y/steps
				zOff = fullOffsetVector.z/steps

				totalNumberOfOperations = steps*turns*(ptArray.length/2)+steps*turns
				doneNumberOfOperations = 0

				# now we can do the transformations...
				# we build a 2D Array for all points
				
				mo.start_operation "Screw"
				
				# screwing Edges and Construction Points...

				newPtsArray = Array.new()
				oldPtsArray = ptArray;
				newConPtsArray = Array.new()
				oldConPtsArray = constrPtsArray;

				targetAngle = 360.degrees if !fullOffsetVector.valid? 

				i = 1
				while (i*angularOffset < targetAngle)
					
					# defining transformations

					rot=Geom::Transformation.rotation(ORIGIN, Z_AXIS, i*angularOffset*rotDir)
					trans=Geom::Transformation.translation([i*xOff, i*yOff, i*zOff])

					# transform Edge Points

					newPtsArray = Array.new()
					for k in 0..(ptArray.length-1) 
						newpt=Geom::Point3d.new([ptArray[k].x ,ptArray[k].y ,ptArray[k].z])
						newpt.transform!(trans)
						newpt.transform!(rot) 
						newPtsArray[k] = newpt
						if ((k%2)==1)
							make_edges_and_faces(newPtsArray[k-1], newPtsArray[k], oldPtsArray[k-1], oldPtsArray[k], angularOffset*rotDir, faceOrientation)
						end
					end
					oldPtsArray = newPtsArray

					# now the Construction Points

					newConPtsArray = Array.new()
					for k in 0..(constrPtsArray.length-1) 
						newpt=Geom::Point3d.new([constrPtsArray[k].x ,constrPtsArray[k].y ,constrPtsArray[k].z])
						newpt.transform!(trans)
						newpt.transform!(rot) 
						newConPtsArray[k] = newpt
						mo.entities.add_edges(oldConPtsArray[k], newConPtsArray[k])
					end
					oldConPtsArray = newConPtsArray
					
					doneNumberOfOperations = doneNumberOfOperations + (ptArray.length/2)
					Sketchup.set_status_text("screw: screwing edges ("+String((100*Float(doneNumberOfOperations)/Float(totalNumberOfOperations)).floor)+"%)") 
				
					i = i+1
				end

				# doing ramaining edges (fractional part)
				
				# defining transformations

				rot=Geom::Transformation.rotation(ORIGIN, Z_AXIS, i*angularOffset*rotDir)
				trans=Geom::Transformation.translation([i*xOff, i*yOff, i*zOff])
				rotAux=Geom::Transformation.rotation(ORIGIN, Z_AXIS, targetAngle)

				# transform Edge Points

				newPtsArray = Array.new()
				for k in 0..(ptArray.length-1) 
					newpt=Geom::Point3d.new([ptArray[k].x ,ptArray[k].y ,ptArray[k].z])
					newpt.transform!(trans)
					newpt.transform!(rot) 
					aux=Geom::Point3d.new([ptArray[k].x ,ptArray[k].y ,ptArray[k].z])
					aux.transform!(rotAux) 
					slicePlane = [aux, Z_AXIS.cross(ORIGIN-aux)]
					sliceLine = [newpt, oldPtsArray[k]] 
					newPtsArray[k] = Geom.intersect_line_plane(sliceLine,slicePlane)
					if ((k%2)==1)
						make_edges_and_faces(newPtsArray[k-1], newPtsArray[k], oldPtsArray[k-1], oldPtsArray[k], angularOffset*rotDir, faceOrientation)
					end
				end
				oldPtsArray = newPtsArray

				
				# transform Construction Points

				newConPtsArray = Array.new()
				for k in 0..(constrPtsArray.length-1) 
					newpt=Geom::Point3d.new([constrPtsArray[k].x ,constrPtsArray[k].y ,constrPtsArray[k].z])
					newpt.transform!(trans)
					newpt.transform!(rot) 
					aux=Geom::Point3d.new([ptArray[k].x ,ptArray[k].y ,ptArray[k].z])
					aux.transform!(rotAux) 
					slicePlane = [aux, Z_AXIS.cross(ORIGIN-aux)]
					sliceLine = [newpt, oldPtsArray[k]] 
					newConPtsArray[k] = Geom.intersect_line_plane(sliceLine,slicePlane)
					mo.entities.add_edges(oldConPtsArray[k], newConPtsArray[k])
				end
				oldConPtsArray = newConPtsArray

				# screwing Components...

				targetAngle = 360.degrees-0.001 if !fullOffsetVector.valid? 

				i = 1
				while (i*angularOffsetComponent < targetAngle)
					
					# defining transformations

					rot=Geom::Transformation.rotation(ORIGIN, Z_AXIS, i*angularOffsetComponent*rotDir)
					trans=Geom::Transformation.translation([i*xOff, i*yOff, i*zOff])

					# insert Components and transform...

					compArray.each do |c|
						nc = mo.entities.add_instance(c.definition, c.transformation) 
						nc.transform!(trans)
						nc.transform!(rot)
					end

					doneNumberOfOperations = doneNumberOfOperations + 1
					Sketchup.set_status_text("screw: screwing components ("+String((100*Float(doneNumberOfOperations)/Float(totalNumberOfOperations)).floor)+"%)") 

					i = i+1
				end

				mo.commit_operation
				Sketchup.set_status_text("screw: ready.") 
			end 
		end 
	end
end

def get_next_edge(e,v)
	conEdges = v.edges
	return nil if (conEdges.length < 2) 
	return conEdges[1] if (e == conEdges[0]) 
	return conEdges[0]
end

def make_edges_and_faces(p1, p2, p3, p4, ao, fo)
	mo=Sketchup.active_model()
	lines = Array.new()
	lines.push(mo.entities.add_edges(p1, p2).first)
	lines.push(mo.entities.add_edges(p4, p1).first)
	lines.push(mo.entities.add_edges(p3, p1).first)
	lines.push(mo.entities.add_edges(p4, p2).first)
	# test for colinear to avoid "add_face': Points are not planar" errors
	notColinear = true;
	vec1 = p3 - p4
	vec2 = p3 - p1
	notColinear = vec1.cross(vec2).valid?
	if (notColinear)
		if (ao*fo < 0) 
			mo.entities.add_face(p3, p4, p1)
			mo.entities.add_face(p4, p1, p2)
		else
			mo.entities.add_face(p3, p1, p4)
			mo.entities.add_face(p4, p2, p1)
		end
	end
	#smoothing edges...
	lines.each do |e|
		e.smooth=true
		e.soft=true
	end
end

end #class

screwTool = Screw.new

#if( not file_loaded?("screw_21.rb") )
#	UI.menu("Plugins").add_item("Screw 2.1") { screwTool.screw_selection }
#end
file_loaded("screw_21.rb")
