# Copyright 2004, Rick Wilson 

# Permission to use, copy, modify, and distribute this software for 
# any purpose and without fee is hereby granted, provided that the above
# copyright notice appear in all copies.

# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

# Name :          Windowizer 2.1
# Description :   makes storefront windows from selected quadrilaterals
# Author :        RW
# Usage :         select 4-edged faces, set preferences,
#                 and POOF! - windows!
# Date :          16.Jul.2004
# Type :		tool
# History:
#			2.1 (17.Aug.2004) - fixed floating point bug in dialog box
#						- fixed context menu validation bug
#						- fixed selection validation bug
#			2.0 (12.Aug.2004) - added features: corner selections, non-bounded face support,
#						  multiple face support
#			1.1e(26.Jul.2004) - metric support
#			1.1d(23.Jul.2004) - new selection filters; added flush corner & floor windows
#			1.1c(23.Jul.2004) - revised the revision to correct 0 col/row glitch
#			1.1 (23.Jul.2004) - revised to correct vector bugs resulting in
#						  strange windows outside of frames
#			1.0 (16.Jul.2004) - first version

require 'sketchup.rb'

class Windowizer

def initialize
	@model = Sketchup.active_model
	@entities = @model.entities
	@materials=@model.materials
	if ($vPanes == nil)
		$vPanes = 2
		$hPanes = 2
		$framewidth = 50.mm
		$frameheight = 50.mm
		$mullwidth = 50.mm
		$mullheight = 50.mm
		$frameinset = 50.mm
		$glassinset = 49.mm
		$framecolor = "DimGray"
		$glasscolor = "Blue Glass"
	end
	$allSame=nil
	add_paint()							#ADD DEFAULT PAINT COLORS IF NECESSARY
	activate
end

#	FIND THE INSET FACE

def find_face(face1,face2,discard)
	$face=nil
	face1edges=face1.edges
	face2edges=face2.edges
	0.upto(3) do |a|
		0.upto(3) do |b|
			if ((face1edges[a] != face2edges[b]) and (face1edges[a].common_face(face2edges[b]) != discard ))
				if ((face1edges[a].common_face(face2edges[b]) != face1) and (face1edges[a].common_face(face2edges[b]) != face2))
					if (face1edges[a].common_face(face2edges[b]))
						return face1edges[a].common_face(face2edges[b])
						#face = face1edges[a].common_face(face2edges[b])
					end
				end
			end
		end
	end
end

#	FIND THE LOWEST 2 POINTS ON THE SELCTED FACE

def find_lowPoints(face)
	fVerts=face.vertices
	low1=fVerts[0]
	low2=low1
	0.upto(3) do |a|
		fVertsZ1=fVerts[a].position[2]
		if (fVertsZ1 <= low2.position[2])
			low2=fVerts[a]
		end
		if (fVertsZ1 <= low1.position[2])
			low2=low1
			low1=fVerts[a]
		end
	end
	0.upto(3) do |a|
		edge=face.edges[a]
		if ((edge.used_by?(low1)) && (edge.used_by?(low2)))
			$lowE=a
		end
	end
end

#	PAINT THE INSET FACES

def insetPaint(face)
	$faceEdges = face.edges					#Select edges so we know where we've been
	if $glassinset != 0
		face.pushpull (- $glassinset)			#Inset the glass
		discard=$faceEdges[0].faces[0]		#These lines find the new inset face
		face1=$faceEdges[0].faces[1]			# |
		face2=$faceEdges[1].faces[1]			# |
		face=find_face(face1,face2,discard)		#_V____
	end
	face.material=$glasscolor				#Paint the new inset face
end

#	BEGIN THE MAIN ROUTINE

def activate
	s = @model.selection
	$ss=[]
	$unb=[]
	$bnd=[]

#		CHECK THE SELECTIONS!

	0.upto(s.length-1) do |x|
		$ss.push(s[x]) if ((s[x].class == Sketchup::Face) && (s[x].edges.length == 4))
	end

	s.clear
	return UI.messagebox("ѡһ4棡") if $ss.empty?
	@model.start_operation "Windowizer"

	if $ss.length>1
		same = UI.messagebox("涼ͬ", MB_YESNO)
	else
		same=6
	end

	if same==6
		$allSame=true
		return nil if windowizer_settings().to_s=="end"
	end

#		CHECK FOR FACE-IN-A-FACE

	while $ss.length > 0
		if (($ss.first.edges[0].faces != $ss.first.edges[1].faces) && ($ss.first.edges[1].faces != $ss.first.edges[2].faces) && ($ss.first.edges[2].faces != $ss.first.edges[3].faces) && ($ss.first.edges[3].faces != $ss.first.edges[0].faces))
			$unb.push($ss.first)				#NOT A FACE-IN-FACE
		else
			$bnd.push($ss.first)				#IS A FACE-IN-FACE
		end
		$ss.shift
	end

	$bnd.each {|bb| bounded(bb)}
	$unb.each {|bb| unbounded(bb)}
	$bnd=$unb=$ss=$allSame=nil
	@model.selection.clear
	@model.commit_operation
end

#	DEFAULT SETTINGS - PROMPT THE USER FOR THE WINDOWIZER SETTINGS
#	(Note to self to add separate mullion width & height settings)

def windowizer_settings
	if $allSame == true
		prompts = ["", "", "Frame Width", "Frame Height", "Mullion Width", "Mullion Height", "Frame Inset", "Glass Inset", "Frame Color", "Glass Color"]
		values = [$vPanes, $hPanes, $framewidth, $frameheight, $mullwidth, $mullheight, $frameinset, $glassinset, $framecolor, $glasscolor]
		results = inputbox prompts, values, "Windowizer Settings"
	else
		prompts = ["", "", "Frame Width", "Frame Height", "Mullion Width", "Mullion Height", "Glass Inset", "Glass Color"]
		values = [$vPanes, $hPanes, $framewidth, $frameheight, $mullwidth, $mullheight, $glassinset, $glasscolor]
		results = inputbox prompts, values, "Windowizer Settings"
	end
	if not results
		return "end"
	end
	if $allSame == true
		$vPanes = results[0]
		$hPanes = results[1]
		$framewidth = results[2]
		$frameheight = results[3]
		$mullwidth = results[4]
		$mullheight = results[5]
		$frameinset = results[6]
		$glassinset = results[7]
		$framecolor = results[8]
		$glasscolor = results[9]
	else
		$vPanes = results[0]
		$hPanes = results[1]
		$framewidth = results[2]
		$frameheight = results[3]
		$mullwidth = results[4]
		$mullheight = results[5]
		$glassinset = results[6]
		$glasscolor = results[7]
	end

	$vPanes = 1 if $vPanes < 1
	$hPanes = 1 if $hPanes < 1
	$framewidth = 0.25.inch if $framewidth <= 0.inch
	$frameheight = 0.25.inch if $frameheight <= 0.inch
	$mullwidth = 0.25.inch if $mullwidth <= 0.inch
	$mullheight = 0.25.inch if $mullheight <= 0.inch
#	$frameinset = 0.5.inch if $frameinset == 0.inch
#	$glassinset = 0.5.inch if $glassinset == 0.inch
end

#	ADD THE DEFAULT PAINTS

def add_paint()

#		ADD THE GLASS PAINT

	if ((@materials["Blue Glass"] == nil) && (@materials["<Blue Glass>"] == nil))
		glass=@materials.add("Blue Glass")
		glass.color=[100,149,237]
		glass.alpha=0.8
	end

#		ADD THE FRAME PAINT

	if (@materials["DimGray"] == nil)
		frame=@materials.add("DimGray")
		frame.color="DimGray"
	end
end

# START BOUNDED FACE OPERATIONS

def bounded(face)
	if not $allSame
		Sketchup.active_model.selection.clear
		Sketchup.active_model.selection.add(face)
		prompts=["Frame Inset", "Frame Color"]
		values =[$frameinset, $framecolor]
		results=inputbox prompts, values, "Windowizer Settings"
		return nil if not results
		$frameinset = results[0]
		$framecolor = results[1]
	end
 	if $frameinset != 0
		_faceEdges = face.edges					#Select edges so we know where we've been
		face.pushpull (-$frameinset)	#Inset the frame
		faceEdges = []
		0.upto(3) do |a|
			faceEdges[faceEdges.length]=_faceEdges[a] if (_faceEdges[a].valid?)	#Pass valid edges to new handler
		end

		face=nil
		discard=faceEdges[0].faces[0]				#These lines find the new inset face
		face1=faceEdges[0].faces[1]				# |
		face2=faceEdges[1].faces[1]				# |
		face = nil							# |
		face=find_face(face1,face2,discard)			#_V____
	end
	$unb.push(face)
end

# START UNBOUNDED FACE OPERATIONS

def unbounded(face)
	if not $allSame
		@model.selection.clear
		@model.selection.add(face)
		return nil if windowizer_settings() == "end"
	end
	return if not face
	face.material=$framecolor				#PAINT THE NEW INSET FACE
	discard = nil
	discard = face						#THE INSET FACE IS NOW THE DISCARD FOR THE WINDOW PANES

	find_lowPoints(face)

	$edge1=face.edges[$lowE]				#PUT THE FACE EDGES IN A USABLE ORDER
	$edge2=face.edges[$lowE-3]
	$edge3=face.edges[$lowE-2]
	$edge4=face.edges[$lowE-1]

#	$hLength1=(($edge1.length-$framewidth)/$hPanes)-$framewidth		#Glass length, bottom edge
#	$hLength2=(($edge3.length-$framewidth)/$hPanes)-$framewidth		#Glass length, top edge
#	$vLength1=(($edge4.length-$frameheight)/$vPanes)-$frameheight	#Glass length, left edge
#	$vLength2=(($edge2.length-$frameheight)/$vPanes)-$frameheight	#Glass length, right edge

	$hLength1=($edge1.length-((2*$framewidth)+($mullwidth * ($hPanes - 1))))/$hPanes	#Glass length, bottom edge
	$hLength2=($edge3.length-((2*$framewidth)+($mullwidth * ($hPanes - 1))))/$hPanes	#Glass length, top edge
	$vLength1=($edge4.length-((2*$frameheight)+($mullheight * ($vPanes - 1))))/$vPanes	#Glass length, left edge
	$vLength2=($edge2.length-((2*$frameheight)+($mullheight * ($vPanes - 1))))/$vPanes	#Glass length, right edge

	if ($edge4.used_by?($edge1.start))
		$hpt1=Geom::Point3d.new($edge1.start.position)
	else
		$hpt1=Geom::Point3d.new($edge1.end.position)
	end

	0.upto(3) do |a|
		if (face.vertices[a].position == $hpt1)
			$hpt2=Geom::Point3d.new(face.vertices[a-3].position)
			$hpt3=Geom::Point3d.new(face.vertices[a-1].position)
			$hpt4=Geom::Point3d.new(face.vertices[a-2].position)
		end		
	end
	
	if ($edge1.start.position == $hpt1)
		$hVec1=$edge1.line[1]
	else
		$hVec1=$edge1.line[1].reverse
	end
	if ($edge3.start.position == $hpt4)	
		$hVec2=$edge3.line[1].reverse
	else
		$hVec2=$edge3.line[1]
	end
	if ($edge4.start.position == $hpt3)
		$vVec1=$edge4.line[1].reverse
	else
		$vVec1=$edge4.line[1]
	end
	if ($edge2.start.position == $hpt2)
		$vVec2=$edge2.line[1]
	else
		$vVec2=$edge2.line[1].reverse
	end

	1.upto($vPanes) do |n|
		1.upto($hPanes) do |m|
			$hpt1x=$hpt1.offset($hVec1, ($framewidth+($mullwidth*(m-1))+($hLength1 * (m-1))))
			$hpt2x=$hpt1.offset($hVec1, ($framewidth+($mullwidth*(m-1))+($hLength1 * m)))
			$hpt3x=$hpt3.offset($hVec2, ($framewidth+($mullwidth*(m-1))+($hLength2 * (m-1))))
			$hpt4x=$hpt3.offset($hVec2, ($framewidth+($mullwidth*(m-1))+($hLength2 * m)))

			$vpt1x=$hpt1.offset($vVec1, ($frameheight+($mullheight*(n-1))+($vLength1 * (n-1))))
			$vpt2x=$hpt1.offset($vVec1, ($frameheight+($mullheight*(n-1))+($vLength1 * n)))
			$vpt3x=$hpt2.offset($vVec2, ($frameheight+($mullheight*(n-1))+($vLength2 * (n-1))))
			$vpt4x=$hpt2.offset($vVec2, ($frameheight+($mullheight*(n-1))+($vLength2 * n)))

			$line1=[Geom::Point3d.new($hpt1x), Geom::Point3d.new($hpt3x)]
			$line2=[Geom::Point3d.new($hpt2x), Geom::Point3d.new($hpt4x)]
			$line3=[Geom::Point3d.new($vpt1x), Geom::Point3d.new($vpt3x)]
			$line4=[Geom::Point3d.new($vpt2x), Geom::Point3d.new($vpt4x)]

			$pt0=Geom.intersect_line_line($line1, $line3)
			$pt1=Geom.intersect_line_line($line2, $line3)
			$pt2=Geom.intersect_line_line($line2, $line4)
			$pt3=Geom.intersect_line_line($line1, $line4)

			pts = []
			pts[0] = $pt0
			pts[1] = $pt1
			pts[2] = $pt2
			pts[3] = $pt3

			face = @entities.add_face pts
			insetPaint(face)
		end
	end
end

def Windowizer::validate_selection
	Sketchup.active_model.selection.each {|e| return true if e.class == Sketchup::Face}
	return nil
end

end	#END CLASS WINDOWIZER


if not file_loaded?("windowizer.rb")
	UI.add_context_menu_handler do |menu|
		menu.add_separator if (Windowizer.validate_selection)
 		menu.add_item("濪  ") { Windowizer.new } if (Windowizer.validate_selection)
	end
end

#-----------------------------------------------------------------------------
file_loaded("windowizer.rb")
