// ------------------------------------------------------------------------------------------------
//
//  File: Delegates.inl
//
//      Description		- Implements the Delegate##DELEGATE_PARAM_COUNT class which allow for generic and
//						  flexible function pointers, if you need to add more parameters, please
//						  add them via the #define/#include method used in "Delegates.h"
//
//      Author			- Grant Peters
//
//  Modifications:
//  --------------
//      Sept 8, 2009	- Grant: Copied from my private engine.
//
// ------------------------------------------------------------------------------------------------

#define DELEGATE_DECLARE_NAME			PP_CONCAT(Delegate, DELEGATE_PARAM_COUNT)
#define DELEGATE_DECLARE_FUNC_PARAMS	PP_REPEAT_TEMPLATE_PARAM(DELEGATE_PARAM_COUNT)
#define DELEGATE_DECLARE_FUNC_VALUES	PP_REPEAT_TEMPLATE_VALUE(DELEGATE_PARAM_COUNT)
#define DELEGATE_DECLARE_FUNC_TYPES		PP_REPEAT_TEMPLATE_TYPE(DELEGATE_PARAM_COUNT)

#define EVENT_DECLARE_NAME			PP_CONCAT(Event, DELEGATE_PARAM_COUNT)

namespace Mortar
{
	template<class ReturnType PP_COMMA_IF(DELEGATE_PARAM_COUNT) PP_REPEAT_TEMPLATE_DECL(DELEGATE_PARAM_COUNT)>
	class DELEGATE_DECLARE_NAME
	{
		struct BaseDelegate;
		typedef StackAllocatedPointer<BaseDelegate, 32> DelegatePointer;

		struct BaseDelegate
		{
			friend class DELEGATE_DECLARE_NAME/*<ReturnType PP_COMMA_IF(DELEGATE_PARAM_COUNT)  DELEGATE_DECLARE_FUNC_TYPES>*/;
			friend class StackAllocatedPointer<BaseDelegate, 32>;
			virtual ~BaseDelegate() {}

			inline bool operator==(const BaseDelegate &rhs) const
			{
				if(GetTypeID() == rhs.GetTypeID())
					return Compare(rhs);
				return false;
			}

		private:
			virtual void Clone(DelegatePointer &ptr) const = 0;
			virtual ReturnType Call(DELEGATE_DECLARE_FUNC_PARAMS) = 0;
			virtual TypeID GetTypeID() const = 0;
			virtual bool Compare(const BaseDelegate &rhs) const = 0;
		};

		DelegatePointer m_delegate;
	public:

		DELEGATE_DECLARE_NAME() {}

		ReturnType operator() (DELEGATE_DECLARE_FUNC_PARAMS) const { return Call(DELEGATE_DECLARE_FUNC_VALUES);	}

		ReturnType Call(DELEGATE_DECLARE_FUNC_PARAMS) const
		{
			BaseDelegate *func = m_delegate.Resolve();
			if(func != NULL)
				return func->Call(DELEGATE_DECLARE_FUNC_VALUES);

			// If the program fails to compile at this line, use 'CallNoNullCheck()' instead
			// you should also wrap the code in 'if(Delegate.IsNull())' when doing this so as
			// to handle NULL pointers and AVOID A CRASH!!!
			return ReturnType();	
		}

		// You should also wrap the code in 'if(Delegate.IsNull())' when using this function
		// so as to handle NULL pointers and AVOID A CRASH!!!
		ReturnType CallNoNullCheck (DELEGATE_DECLARE_FUNC_PARAMS) const
		{
			BaseDelegate *func = m_delegate.Resolve();
			ASSERT_MSG(func != NULL, "'Delegate.CallNoNullCheck()' called when the delegate was NULL,\nplease wrap code in an 'if(Delegate.IsNull())' to protect against this scenario!");
			return func->Call(DELEGATE_DECLARE_FUNC_VALUES);
		}

		bool IsNull()
		{
			return m_delegate.Resolve() == NULL;
		}

		inline bool operator==(const BaseDelegate &rhs) const
		{
			BaseDelegate *myPtr = m_delegate.Resolve();;
			if(myPtr == NULL)
				return false;
			return  (*myPtr) == rhs;
		}

		inline bool operator==(const DELEGATE_DECLARE_NAME &rhs) const
		{
			BaseDelegate *myPtr = m_delegate.Resolve();;
			BaseDelegate *rhsPtr = rhs.m_delegate.Resolve();
			if(rhsPtr == myPtr) // Pointers match, this shouldn't happen unless we are both NULL
			{
				ASSERT_MSG(myPtr == NULL, "Internal error, each Delegate class should have its own copy of the 'BaseDelegate'");
				return true;
			}
			if(rhsPtr == NULL)
				return false; // Other is NULL, we aren't
			return  (*myPtr) == (*rhsPtr);
		}

#pragma region(BaseDelegate assignments)
		void operator=(const BaseDelegate &func)
		{
			func.Clone(m_delegate);
		}
		DELEGATE_DECLARE_NAME(const BaseDelegate &func)
		{
			func.Clone(m_delegate);
		}
#pragma endregion

#pragma region(Static/Global function delegate)
		class Global : public BaseDelegate
		{
		public:
			typedef ReturnType (*FuncPtr)(DELEGATE_DECLARE_FUNC_PARAMS);
			Global(FuncPtr function)
				: m_function(function)
			{
			}

			Global(const Global &copy)
				: m_function(copy.m_function)
			{
			}

		private:
			FuncPtr m_function;

			void Clone(DelegatePointer &ptr) const
			{
				ptr.CopyConstruct(*this);
			}

			ReturnType Call(DELEGATE_DECLARE_FUNC_PARAMS)
			{
				return m_function(DELEGATE_DECLARE_FUNC_VALUES);
			}

			TypeID GetTypeID() const { return GenerateTypeID<Global>(); }
			bool Compare(const BaseDelegate &rhs) const { return m_function == reinterpret_cast<const Global &>(rhs).m_function; }
		};
#pragma region(Assignment helper)
		DELEGATE_DECLARE_NAME(typename Global::FuncPtr globalFunc)
		{
			if(globalFunc != NULL)
				*this = Global(globalFunc);
			else
				m_delegate.Delete();
		}
		void operator=(typename Global::FuncPtr globalFunc)
		{
			if(globalFunc != NULL)
				*this = Global(globalFunc);
			else
				m_delegate.Delete();
		}
#pragma endregion
#pragma endregion

#pragma region(Callee function delegate)
		template<typename CalleeType>
		class Callee : public BaseDelegate
		{
		public:
			typedef ReturnType (CalleeType::*FuncPtr)(DELEGATE_DECLARE_FUNC_PARAMS);

			Callee(CalleeType *callee, FuncPtr function)
				: m_callee(callee), m_function(function)
			{
			}
			Callee(CalleeType &callee, FuncPtr function)
				: m_callee(&callee), m_function(function)
			{
			}

		private:
			CalleeType *m_callee;
			FuncPtr m_function;

			void Clone(DelegatePointer &ptr) const
			{
				ptr.CopyConstruct(*this);
			}

			ReturnType Call(DELEGATE_DECLARE_FUNC_PARAMS)
			{
				return (m_callee->*m_function)(DELEGATE_DECLARE_FUNC_VALUES);
			}

			TypeID GetTypeID() const { return GenerateTypeID<Callee>(); }
			bool Compare(const BaseDelegate &rhs) const { return m_function == reinterpret_cast<const Callee &>(rhs).m_function && m_callee == reinterpret_cast<const Callee &>(rhs).m_callee; }
		};
#pragma region(Assignment helper)
		template <typename CalleeType> static Callee<CalleeType> QCallee(CalleeType &callee, ReturnType (CalleeType::*func)(DELEGATE_DECLARE_FUNC_PARAMS))	{ return Callee<CalleeType>(callee, func); }
		template <typename CalleeType> static Callee<CalleeType> QCallee(CalleeType *callee, ReturnType (CalleeType::*func)(DELEGATE_DECLARE_FUNC_PARAMS))	{ return Callee<CalleeType>(callee, func); }
#pragma endregion
#pragma endregion

#pragma region(CopyCallee function delegate)
		template<typename CalleeType>
		class CopyCallee : public BaseDelegate
		{
		public:
			typedef ReturnType (CalleeType::*FuncPtr)(DELEGATE_DECLARE_FUNC_PARAMS);

			CopyCallee(CalleeType *callee, FuncPtr function)
				: m_callee(*callee), m_function(function)
			{
			}
			CopyCallee(CalleeType &callee, FuncPtr function)
				: m_callee(callee), m_function(function)
			{
			}

		private:
			CalleeType m_callee;
			FuncPtr m_function;

			void Clone(DelegatePointer &ptr) const
			{
				ptr.CopyConstruct(*this);
			}

			ReturnType Call(DELEGATE_DECLARE_FUNC_PARAMS)
			{
				return (m_callee.*m_function)(DELEGATE_DECLARE_FUNC_VALUES);
			}

			TypeID GetTypeID() const { return GenerateTypeID<CopyCallee>(); }
			bool Compare(const BaseDelegate &rhs) const { return m_function == reinterpret_cast<const CopyCallee &>(rhs).m_function && m_callee == reinterpret_cast<const CopyCallee &>(rhs).m_callee; }
		};
#pragma region(Assignment helper)
		template <typename CalleeType> static CopyCallee<CalleeType> QCopyCallee(CalleeType &callee, ReturnType (CalleeType::*func)(DELEGATE_DECLARE_FUNC_PARAMS))	{ return CopyCallee<CalleeType>(callee, func); }
		template <typename CalleeType> static CopyCallee<CalleeType> QCopyCallee(CalleeType *callee, ReturnType (CalleeType::*func)(DELEGATE_DECLARE_FUNC_PARAMS))	{ return CopyCallee<CalleeType>(callee, func); }
#pragma endregion
#pragma endregion
	};

#if DELEGATE_PARAM_COUNT > 0
#define TYPENAME_IF_PARAM_COUNT typename
#else
#define TYPENAME_IF_PARAM_COUNT
#endif

	// Basically an array of delegates called in the order they were assigned
	// due to thier being multiple, a return value doesn't make sense as we've got no
	// way of knowing how to combine them
#if DELEGATE_PARAM_COUNT > 0
	template<PP_REPEAT_TEMPLATE_DECL(DELEGATE_PARAM_COUNT)>
#endif
	class EVENT_DECLARE_NAME
	{
	public:
		typedef DELEGATE_DECLARE_NAME<void PP_COMMA_IF(DELEGATE_PARAM_COUNT) PP_REPEAT_TEMPLATE_TYPE(DELEGATE_PARAM_COUNT)> DelegateType;

	private:
		typedef std::list<DelegateType> EventList;
		EventList m_events;

	public:
		inline void Register(const DelegateType &callback) { m_events.push_back(callback); }
		inline void UnRegister(const DelegateType &callback) { m_events.remove(callback); }

		inline void Trigger(DELEGATE_DECLARE_FUNC_PARAMS) const
		{
			const TYPENAME_IF_PARAM_COUNT EventList::const_iterator endIter = m_events.end();
			for(TYPENAME_IF_PARAM_COUNT EventList::const_iterator iter = m_events.begin(); iter != endIter; ++iter)
			{
				iter->Call(DELEGATE_DECLARE_FUNC_VALUES);
			}
		}

		inline void operator +=(const DelegateType &toTrigger) { Register(toTrigger); }
		inline void operator -=(const DelegateType &toTrigger) { UnRegister(toTrigger); }
		inline void operator ()(DELEGATE_DECLARE_FUNC_PARAMS) { Trigger(DELEGATE_DECLARE_FUNC_VALUES); }
	};

}
#undef EVENT_DECLARE_NAME
#undef TYPENAME_IF_PARAM_COUNT

#undef DELEGATE_DECLARE_NAME
#undef DELEGATE_DECLARE_FUNC_PARAMS
#undef DELEGATE_DECLARE_FUNC_VALUES
#undef DELEGATE_PARAM_COUNT