// ------------------------------------------------------------------------------------------------
//
//  List.inl
//
// ------------------------------------------------------------------------------------------------

#include "../Startup/Macros.h"
#include "../Memory/MortarMemory.h"

template<typename T>
FreeList* List<T>::CreateNewMemoryAlloc(StackHeap *heap, u32 size)
{
	FreeList *free_list = new FreeList(heap, size);
	return free_list;
}

template<typename T>
void* List<T>::AllocMemory(FreeList *free_list)
{
	return free_list->Allocate();
}

template<typename T>
void List<T>::ReleaseMemory(FreeList *free_list, void *p)
{
	free_list->Release(p);
}

// Constructor
//
template<typename T>
List<T>::List()
{
	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
	private_mem_alloc = 0;

	memory_allocator = NULL;
}

template<typename T>
List<T>::List(StackHeap *heap)
{
	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
	private_mem_alloc = 1;

	if(heap)
		memory_allocator = CreateNewMemoryAlloc(heap, sizeof(Node));
	else
		memory_allocator = NULL;
}

template<typename T>
List<T>::List(FreeList *list)
{
	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
	private_mem_alloc = 0;

	memory_allocator = list;
}

template<typename T>
void List<T>::Init()
{
	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
	private_mem_alloc = 0;

	memory_allocator = NULL;
}

template<typename T>
void List<T>::Init(StackHeap *heap)
{
	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
	private_mem_alloc = 1;

	if(heap)
		memory_allocator = CreateNewMemoryAlloc(heap, sizeof(Node));
	else
		memory_allocator = NULL;
}

template<typename T>
void List<T>::Init(FreeList *list)
{
	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
	private_mem_alloc = 0;

	memory_allocator = list;
}

// Destructor
//
template<typename T>
List<T>::~List()
{
	Destroy();
}


// Destroy
//
template<typename T>
void List<T>::Destroy()
{
	Clear();
	if(private_mem_alloc)
		SAFE_DELETE(memory_allocator);
}

// Method: Clear
//
//      The list is completely cleared by this call and is then free to be reused with either method.
//
template<typename T>
void List<T>::Clear()
{
	// Only delete nodes if we created them.
	if (method == METHOD_CREATE)
	{
		Node* curr = head;
		Node* next;

		if(!memory_allocator)
		{
			while (curr)
			{
				next = curr->next;

				delete curr;
				

				curr = next;
			}
		}
		else
		{
			
			while (curr)
			{
				next = curr->next;

				ReleaseMemory(memory_allocator, curr);				

				curr = next;
			}
			
		}

	}

	count  = 0;        
	head   = 0;
	tail   = 0;
	method = METHOD_NONE;
//	if(memory_allocator)
}


template<typename T>
void List<T>::ClearFreeList()
{
	ASSERT(private_mem_alloc);
	if(memory_allocator)memory_allocator->Clear();
}




// Adds a node at the head of the linked list.
//
template<typename T>
void List<T>::AddHead(Node* node)
{
	ASSERT(node);        

	if (method == METHOD_NONE)
		method =  METHOD_ADD;

	ASSERT(method == METHOD_ADD);

	AddNodeToHead(node);
}



// Adds a node at the tail of the linked list.
//
template<typename T>
void List<T>::AddTail(Node* node)
{
	ASSERT(node);        

	if (method == METHOD_NONE)
		method =  METHOD_ADD;

	ASSERT(method == METHOD_ADD);
	
	AddNodeToTail(node);
}



// Method: AddHead
//
//      Adds an item at the head of the linked list.
//
template<typename T>
void List<T>::AddHead(T item)
{
	if (method == METHOD_NONE)
		method =  METHOD_CREATE;

	ASSERT(method == METHOD_CREATE);
	
	Node* node = NULL;
	if(!memory_allocator)
	{
		node = new Node(item);
	}
	else
	{
		node = (Node*)AllocMemory(memory_allocator);
	}
	ASSERT(node);
	
	node->item = item;
	node->prev = 0;
	node->next = 0;

	
	
	AddNodeToHead(node);
}



// Method: AddTail
//
//      Adds a node at the tail of the linked list.
//
template<typename T>
void List<T>::AddTail(T item)
{
	if (method == METHOD_NONE)
		method =  METHOD_CREATE;

	ASSERT(method == METHOD_CREATE);
	
	Node* node = NULL;
	if(!memory_allocator)
	{
		node = new Node(item);
	}
	else
	{
		node = (Node*)AllocMemory(memory_allocator);
	}
	
	ASSERT(node);

	node->item = item;
	node->prev = 0;
	node->next = 0;

	 
	
	AddNodeToTail(node); 
}



// Finds an items node if it is held within the list.
//
template<typename T>
void List<T>::Remove(const T item)
{
	Node* node = Find(item);

	if ( node)
	{
        Remove(node);
	}
}



// Finds an items node if it is held within the list.
//
template<typename T>
void List<T>::Remove(Node* node)
{
	if (node)
	{
		// Keep a pointer to the current node, so we can delete it.
		Node* temp = node;


		// Remove the head node?
		if (node == head)
		{
			// Are there any nodes left?
			if (head->next)
			{
				// Remove old head node.
				head  = head->next;
				
				head->prev = 0;
			}
			else
			{
				head = 0;
				tail = 0;
			}
		}

		// Remove the tail node?
		else if (node == tail)
		{
			// Are there any nodes left?
			if (tail->prev)
			{
				// Remove old tail node.
				tail = tail->prev;
				
				tail->next = 0;

			}
			else
			{
				head = 0;
				tail = 0;
			}
		}

		// Remove inbetween node.
		else
		{
			node->prev->next = node->next;
			node->next->prev = node->prev;
		}

	
		// Now we can call delete on the node that has been unlinked.
		if (method == METHOD_CREATE)
		{   
			if(!memory_allocator)
			{
				delete temp;
			}
			else
			{
				ReleaseMemory(memory_allocator, temp);
			}
		}
	
	
		// Reduce count.
		if (--count == 0)
		{
			method = METHOD_NONE;
		}
	}
}



// Inserts an item into the list.
//
template<typename T>
void List<T>::Insert(Node* curr, const T item, const InsertPos insert)
{
    if (method == METHOD_NONE)
		method =  METHOD_CREATE;

	ASSERT(method == METHOD_CREATE);

	Node* node = NULL;
	if(!memory_allocator)
	{
		node = new Node(item);
	}
	else
	{
		node = (Node*)AllocMemory(memory_allocator);
	}
	node->item = item;
	node->prev = 0;
	node->next = 0;

	ASSERT(node);
	ASSERT(curr);

    InsertNode(curr, node, insert);
}



// Inserts a node into the list.
//
template<typename T>
void List<T>::Insert(Node* curr, Node* node, const InsertPos insert)
{
	if (method == METHOD_NONE)
		method =  METHOD_ADD;

	ASSERT(method == METHOD_ADD);
	ASSERT(node);
	ASSERT(curr);

    InsertNode(curr, node, insert);
}


//      Inserts an item into the list.
//
template<typename T>
void List<T>::Insert(Iterator it, const T item, const InsertPos insert)
{
    if (method == METHOD_NONE)
		method =  METHOD_CREATE;

	ASSERT(method == METHOD_CREATE);

	Node* node = NULL;
	if(!memory_allocator)
	{
		node = new Node(item);
	}
	else
	{
		node = (Node*)AllocMemory(memory_allocator);
	}
	node->item = item;
	node->prev = 0;
	node->next = 0;

	ASSERT(node);
	ASSERT(it.curr);

    InsertNode(it.curr, node, insert);
}


//      Inserts a node into the list.
template<typename T>
void List<T>::Insert(Iterator it, Node* node, const InsertPos insert)
{
	if (method == METHOD_NONE)
		method =  METHOD_ADD;

	ASSERT(method == METHOD_ADD);
	ASSERT(node);
	ASSERT(it.curr);

    InsertNode(it.curr, node, insert);
}


// Removes an current item from the list.
template<typename T>
void List<T>::RemoveAt(int index)
{
    ASSERT(index < count);
    ASSERT(head);

    Node* node = head;
    int i = 0;
    while(node)
    {
        if (i++ == index)
        {
            Remove(node);
            return;
        }

        node = node->next;
    }
}


// Removes all the items at the specified indices
//
template<typename T>
void List<T>::RemoveRange(int start, int end)
{
    ASSERT(start < count);
    ASSERT(end   < count);
    ASSERT(start > -1);
    ASSERT(end   > -1);

    if (start > end)
    {
        SWAP(start, end);
    }

    int count = (end - start);

    for (int i=0; i<count; i++)
    {
        RemoveAt(start); // Removes same index *count* times as the links shift down
    }
}


// ------------------------------------------------------------------------------------------------
//
//  Private methods.
//
// ------------------------------------------------------------------------------------------------

// Adds a new node at the lists beginning.
//
template<typename T>
void List<T>::AddNodeToHead(Node* node)
{
	if (0 == head)
	{
		head = node;
		tail = node;

		node->prev = 0;
		node->next = 0;
	}
	else
	{
		node->prev = 0;         // As this is the new head node, there will be no previous.
		node->next = head;      // New head nodes next will be old head node.
		head->prev = node;      // Old head nodes previous must now point to the new head.
		head       = node;      // Set new head node.
	}

	count++;
}


// Adds a new node at the lists end.
//
template<typename T>
void List<T>::AddNodeToTail(Node* node)
{
	if (0 == tail)
	{
		head = node;
		tail = node;

		node->prev = 0;
		node->next = 0;
	}
	else
	{
		tail->next = node;      // Old tails next must be set to point at the new node.
		node->prev = tail;      // New nodes previous must be the current tail.
		node->next = 0;         // New nodes next is 0.
		tail       = node;      // Now set tail node to point to the new node.
	}

	count++;
}


// Inserts a node into the list.
//
template<typename T>
void List<T>::InsertNode(Node* curr, Node* node, const InsertPos insert)
{
    if (insert == AFTER)
    {
        node->next = curr->next;

        node->prev = curr;

        if (curr->next)
        {
            curr->next->prev = node;
        }

        curr->next = node;


        // Is inserted node the new tail?
        if (curr == tail)
        {
             tail = node;
        }
    }
    else
    {
        node->prev = curr->prev;

        node->next = curr;

        if (curr->prev)
        {
            curr->prev->next = node;
        }

        curr->prev = node;


        // Is inserted node the new head?
        if (curr == head)
        {
             head = node;
        }
    }

    count++;
}
