// ------------------------------------------------------------------------------------------------
//
//  File: Vector2.inl
//
//      Description     - Implementation of the inline Vector2 functions
//      Author          - Aidan Millott
//
//  Modifications:
//  --------------
//      July 29, 2009   - Initial creation.
//		July 30, 2009	- Woody Updated to templates
//		July 30, 2009	- Paul McNab converted for 2D use.
//
// ------------------------------------------------------------------------------------------------


// Operator +
//
template <class T> const _Vector2<T> _Vector2<T>::One  = _Vector2<T>(1.0f, 1.0f);
template <class T> const _Vector2<T> _Vector2<T>::Zero = _Vector2<T>(0.0f, 0.0f); 
template <class T> const _Vector2<T> _Vector2<T>::UnitY = _Vector2<T>(0.0f, 1.0f); 
template <class T> const _Vector2<T> _Vector2<T>::UnitX = _Vector2<T>(1.0f, 0.0f); 



template <class T> _Vector2<T> _Vector2<T>::operator + ( const _Vector2<T>& rhs ) const
{
	return _Vector2<T>( x + rhs.x, y + rhs.y );
}

template <class T> _Vector2<T> _Vector2<T>::operator + ( T rhs ) const
{
	return _Vector2<T>( x + rhs, y + rhs );
}


// Operator -
//
template <class T> _Vector2<T> _Vector2<T>::operator - ( const _Vector2<T>& rhs ) const
{
	return _Vector2<T>( x - rhs.x, y - rhs.y );
}

template <class T> _Vector2<T> _Vector2<T>::operator - ( T rhs ) const
{
	return _Vector2<T>( x - rhs, y - rhs );
}


// Operator *
//
template <class T> _Vector2<T> _Vector2<T>::operator * ( const T& rhs ) const
{
	return	_Vector2<T>
		(
		x * rhs,
		y * rhs
		);
}

template <class T> _Vector2<T> _Vector2<T>::operator * (const _Vector2<T>& val) const
{
	return _Vector2<T>
		(
		x * val.x,
		y * val.y
		);
}


// Operator >>/<<
//
template <class T> _Vector2<T> _Vector2<T>::operator >> ( const u32 val ) const
{
	return	_Vector2<T>
		(
		x>>val,
		y>>val
		);
}


template <class T> _Vector2<T> _Vector2<T>::operator << ( const u32 val ) const
{
	return	_Vector2<T>
		(
		x<<val,
		y<<val
		);
}


// Operator /
//
template <class T> _Vector2<T> _Vector2<T>::operator / ( const T& rhs ) const
{
	return	_Vector2<T>
		(
		x / rhs,
		y / rhs
		);
}



// Operator +=
//
template <class T> _Vector2<T>& _Vector2<T>::operator += (const _Vector2<T>& rhs )
{
	x += rhs.x;
	y += rhs.y;

	return *this;
}


// Operator >>=/<<=
//
template <class T> _Vector2<T>& _Vector2<T>::operator >>= (const u32 val )
{
	x >>= val;
	y >>= val;

	return *this;
}

template <class T> _Vector2<T>& _Vector2<T>::operator <<= (const u32 val )
{
	x <<= val;
	y <<= val;

	return *this;
}


// Operator -=
//
template <class T> _Vector2<T>& _Vector2<T>::operator -= ( const _Vector2<T>& rhs )
{
	x -= rhs.x;
	y -= rhs.y;

	return *this;
}


// Operator *=
//
template <class T> _Vector2<T>& _Vector2<T>::operator *= ( const T val )
{
	x *= val;
	y *= val;

	return *this;
}

template <class T> _Vector2<T> _Vector2<T>::operator *= (const _Vector2<T>& val)
{
	x *= val.x;
	y *= val.y;

	return *this;
}


// Operator /=
//
template <class T> _Vector2<T>& _Vector2<T>::operator /= ( const T val )
{
	x /= val;
	y /= val;

	return *this;
}


// Operator -
//
template <class T> _Vector2<T> _Vector2<T>::operator - () const
{
	return _Vector2<T>(-x, -y);
}


// Operator ==
//
template <class T> bool _Vector2<T>::operator == (const _Vector2<T> &v) const
{
	return (x == v.x && y == v.y);
}


// Operator !=
//
template <class T> bool _Vector2<T>::operator != (const _Vector2<T> &v) const
{
	return (x != v.x || y != v.y );
}

template <class T> bool _Vector2<T>::Equals(const _Vector2 &v, float tolerance) const
{
	return
		BETWEEN(x-v.x, -tolerance, tolerance) &&
		BETWEEN(y-v.y, -tolerance, tolerance);
}


// Magnitude
//
template <class T> T _Vector2<T>::Magnitude() const
{
	return Math::Sqrt(MagnitudeSqr());
}


// MagnitudeSqr
//
template <class T> T _Vector2<T>::MagnitudeSqr() const
{
	return Dot(*this);
}


// Normalise
//
template <class T> float _Vector2<T>::Normalise()
{
	if(x == 0 && y == 0)
		return 0;

	T denom = Magnitude();
	if(denom != 0)
	{
		*this /= denom;
	}
	else
	{
		//wasn't 0, but was very small, multiply by a large number
		*this *= 1000000;
		Normalise();
	}

	return denom;
}


// Cross
//
/*
template <class T> T _Vector2<T>::Cross(const _Vector2<T>& rhs) const
{
	_Vector2<T> res;

	res.x = (y * rhs.z) - (rhs.y * z);
	res.y = (z * rhs.x) - (rhs.z * x);

	return res;
}
*/


// Dot
//
template <class T> T _Vector2<T>::Dot(const _Vector2<T>& rhs) const
{
	return (x * rhs.x) + (y * rhs.y);
}


// Lerp
//
template <class T> _Vector2<T> _Vector2<T>::Lerp(const _Vector2<T>& rhs, const T time) const
{
	_Vector2<T> tmp = (rhs - *this) * time;
	return *this + tmp;
}


// Magnitude
//
template <class T> T _Vector2<T>::Magnitude(_Vector2<T> v)
{
	return v.Magnitude();
}


// MagnitudeSqr
//
template <class T> T _Vector2<T>::MagnitudeSqr(_Vector2<T> v)
{
	return v.MagnitudeSqr();
}


// Normalise
//
template <class T> _Vector2<T> _Vector2<T>::Normalise(_Vector2<T> v)
{
	v.Normalise();
	return v;
}


// Cross
//
template <class T> _Vector2<T> _Vector2<T>::Cross(_Vector2<T> lhs, _Vector2<T> rhs)
{
	return lhs.Cross(rhs);
}


// Dot
//
template <class T> T _Vector2<T>::Dot(_Vector2<T> lhs, _Vector2<T> rhs)
{
	return lhs.Dot(rhs);
}


// Lerp
//
template <class T> _Vector2<T> _Vector2<T>::Lerp(_Vector2<T> lhs, _Vector2<T> rhs, T time)
{
	return lhs.Lerp(rhs, time);
}


template <class T> inline _Vector2<T> operator + ( const T& lhs, const _Vector2<T>& rhs) { return _Vector2<T>(lhs + rhs.x, lhs + rhs.y); }
template <class T> inline _Vector2<T> operator - ( const T& lhs, const _Vector2<T>& rhs) { return _Vector2<T>(lhs - rhs.x, lhs - rhs.y); }
template <class T> inline _Vector2<T> operator * ( const T& lhs, const _Vector2<T>& rhs) { return _Vector2<T>(lhs * rhs.x, lhs * rhs.y); }
template <class T> inline _Vector2<T> operator / ( const T& lhs, const _Vector2<T>& rhs) { return _Vector2<T>(lhs / rhs.x, lhs / rhs.y); }

// Returns the points data as a string.
//
template <class T> const char *_Vector2<T>::ToString() const
{
	static char buffer[256];
#if defined(PLATFORM_PC) || defined(PLATFORM_360)
	sprintf_s(buffer, "%i, %i", (s32)x, (s32)y);
#elif defined(PLATFORM_BADA)
    snprintf(buffer, sizeof(buffer), "%i, %i", (s32)x, (s32)y);
#else
    sprintf(buffer, "%i, %i", (s32)x, (s32)y);
#endif
	return buffer;
}
