﻿// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.define('ntp', function() {
  'use strict';

  // We can't pass the currently dragging tile via dataTransfer because of
  // http://crbug.com/31037
  var currentlyDraggingTile = null;
  function getCurrentlyDraggingTile() {
    return currentlyDraggingTile;
  }

  function addPinnedUrl(item, index) {
    console.log(" ---------- chrome.send: addPinnedURL");
    chrome.send('addPinnedURL', [item.url, item.title, item.faviconUrl || '', item.thumbnailUrl || '', String(index)]);
  }

  function saveMostVisitedPageList() {
    var idx = 0,
    data = {},
    keys = [],
    grids = document.querySelectorAll('.most-visited-page .tile-grid a:not([class~="drag-representation"])');
    //grids = document.querySelectorAll('.most-visited-page a.real');
    if (grids.length % 4 != 0 && saveMostVisitedPageListTimes++ < 5) {
      setTimeout(function() {
        saveMostVisitedPageList()
      },200);
      return;
    }

    saveMostVisitedPageListTimes = 0;

    var thumbData = [];
    var thumbDataArray = [];
    for (var i in grids) {
      var grid = grids[i];
      if (!grid.nodeType) continue;

      if (grid.data_) {
        //thumbData.push(grid.data_);
        thumbDataArray.push(grid.data_.url);
        thumbDataArray.push(grid.data_.pinned ? true: false);
      } else {
        // 空格
        thumbDataArray.push('');
        thumbDataArray.push(false);
      }
    }
    try {
      console.log(" ---------- chrome.send: saveMostVisitedPageList([" + thumbDataArray + "])");
      chrome.send('saveMostVisitedPageList', thumbDataArray);
    } catch(e) {}
  }

  /**
   * Creates a new Tile object. Tiles wrap content on a TilePage, providing
   * some styling and drag functionality.
   * @constructor
   * @extends {HTMLDivElement}
   */
  function Tile(contents) {
    var tile = cr.doc.createElement('div');
    tile.__proto__ = Tile.prototype;
    tile.initialize(contents);

    return tile;
  }

  Tile.prototype = {
    __proto__: HTMLDivElement.prototype,

    initialize: function(contents) {
      // 'real' as opposed to doppleganger.
      this.className = 'tile real';
      this.appendChild(contents);
      contents.tile = this;

      this.addEventListener('dragstart', this.onDragStart_);
      this.addEventListener('drag', this.onDragMove_);
      this.addEventListener('dragend', this.onDragEnd_);

      this.firstChild.addEventListener('webkitAnimationEnd', this.onContentsAnimationEnd_.bind(this));

      this.eventTracker = new EventTracker();
    },

    get index() {
      return Array.prototype.indexOf.call(this.parentNode.children, this);
    },

    get tilePage() {
      return findAncestorByClass(this, 'tile-page');
    },

    /**
     * Position the tile at |x, y|, and store this as the grid location, i.e.
     * where the tile 'belongs' when it's not being dragged.
     * @param {number} x The x coordinate, in pixels.
     * @param {number} y The y coordinate, in pixels.
     */
    setGridPosition: function(x, y) {
      this.gridX = x;
      this.gridY = y;
      this.moveTo(x, y);
    },

    /**
     * Position the tile at |x, y|.
     * @param {number} x The x coordinate, in pixels.
     * @param {number} y The y coordinate, in pixels.
     */
    moveTo: function(x, y) {
      // left overrides right in LTR, and right takes precedence in RTL.
      this.style.left = x + 'px';
      this.style.right = x + 'px';
      this.style.top = y + 'px';
    },

    checkDrag_: function(){
         return true;
    },
    /**
     * The handler for dragstart events fired on |this|.
     * @param {Event} e The event for the drag.
     * @private
     */
    onDragStart_: function(e) {
      if(!this.checkDrag_()) return;
      // The user may start dragging again during a previous drag's finishing
      // animation.
      if (this.classList.contains('dragging')) this.finalizeDrag_();

      currentlyDraggingTile = this;

      //e.dataTransfer.effectAllowed = 'copyMove';
      // TODO(estade): fill this in.
      //e.dataTransfer.setData('text/plain', 'foo');
      // The drag clone is the node we use as a representation during the drag.
      // It's attached to the top level document element so that it floats above
      // image masks.
      this.dragClone = this.cloneNode(true);
      this.dragClone.classList.add('drag-representation');
      this.ownerDocument.documentElement.appendChild(this.dragClone);
      this.eventTracker.add(this.dragClone, 'webkitTransitionEnd', this.onDragCloneTransitionEnd_.bind(this));

      this.classList.add('dragging');
      // offsetLeft is mirrored in RTL. Un-mirror it.
      var offsetLeft = ntp.isRTL() ? this.parentNode.clientWidth - this.offsetLeft: this.offsetLeft;
      this.dragOffsetX = e.x - offsetLeft - this.parentNode.offsetLeft;
      this.dragOffsetY = e.y - this.offsetTop -
      // Unlike offsetTop, this value takes scroll position into account.
      this.parentNode.getBoundingClientRect().top;

      this.onDragMove_(e);

      this.ori_index = this.index;
    },

    /**
     * The handler for drag events fired on |this|.
     * @param {Event} e The event for the drag.
     * @private
     */
    onDragMove_: function(e) {
    if(!this.checkDrag_()) return;
      if (e.view != window || (e.x == 0 && e.y == 0)) {
        // attribute hidden seems to be overridden by display.
        this.dragClone.classList.add('hidden');
        return;
      }
      var right_btn = $("page-switcher-end");
      if(right_btn && right_btn.offsetLeft){
        if(e.x > right_btn.offsetLeft)
         return;
      }
      this.dragClone.classList.remove('hidden');
      this.dragClone.style.left = (e.x - this.dragOffsetX) + 'px';
      this.dragClone.style.top = (e.y - this.dragOffsetY) + 'px';
    },

    /**
     * The handler for dragend events fired on |this|.
     * @param {Event} e The event for the drag.
     * @private
     */
    onDragEnd_: function(e) {
    if(!this.checkDrag_()) return;
      // The drag clone can still be hidden from the last drag move event.
      this.dragClone.classList.remove('hidden');
      this.dragClone.classList.add('placing');
      currentlyDraggingTile = null;
      this.tilePage.positionTile_(this.index);

      // The tile's contents may have moved following the respositioning; adjust
      // for that.
      var contentDiffX = this.dragClone.firstChild.offsetLeft - this.firstChild.offsetLeft;
      var contentDiffY = this.dragClone.firstChild.offsetTop - this.firstChild.offsetTop;
      this.dragClone.style.left = (this.gridX + this.parentNode.offsetLeft - contentDiffX) + 'px';
      this.dragClone.style.top = (this.gridY + this.parentNode.getBoundingClientRect().top - contentDiffY) + 'px';

      /* add by 360 to save most-visited grid-index and pinned */
      setTimeout(function() {saveMostVisitedPageList();},360);
      /* add over */

      /* add by 360 */
      // dragend to pinned
      var mostVisited = this.firstChild;
      if (mostVisited.togglePinned_ && this.index != this.ori_index) {
        mostVisited.data_.pinned = true;
        mostVisited.updatePinnedState_();
        if ($("add-dot-container").style.display != "block") {
            setTimeout(function () {
                chrome.send("cacheNTP");
            },200);
        }
      }
    },

    /**
     * Creates a clone of this node offset by the coordinates. Used for the
     * dragging effect where a tile appears to float off one side of the grid
     * and re-appear on the other.
     * @param {number} x x-axis offset, in pixels.
     * @param {number} y y-axis offset, in pixels.
     */
    showDoppleganger: function(x, y) {
    if(!this.checkDrag_()) return;
      // We always have to clear the previous doppleganger to make sure we get
      // style updates for the contents of this tile.
      this.clearDoppleganger();

      var clone = this.cloneNode(true);
      clone.classList.remove('real');
      clone.classList.add('doppleganger');
      var clonelets = clone.querySelectorAll('.real');
      for (var i = 0; i < clonelets.length; i++) {
        clonelets[i].classList.remove('real');
      }

      this.appendChild(clone);
      this.doppleganger_ = clone;

      if (ntp.isRTL())
        x *= -1;

      this.doppleganger_.style.WebkitTransform = 'translate(' + x + 'px, ' + y + 'px)';
    },

    /**
     * Destroys the current doppleganger.
     */
    clearDoppleganger: function() {
      if (this.doppleganger_) {
        this.removeChild(this.doppleganger_);
        this.doppleganger_ = null;
      }
    },

    /**
     * Returns status of doppleganger.
     * @return {boolean} True if there is a doppleganger showing for |this|.
     */
    hasDoppleganger: function() {
      return !!this.doppleganger_;
    },

    /**
     * Cleans up after the drag is over. This is either called when the
     * drag representation finishes animating to the final position, or when
     * the next drag starts (if the user starts a 2nd drag very quickly).
     * @private
     */
    finalizeDrag_: function() {
      assert(this.classList.contains('dragging'));

      var clone = this.dragClone;
      this.dragClone = null;
      clone.parentNode.removeChild(clone);
      this.eventTracker.remove(clone, 'webkitTransitionEnd');
      this.classList.remove('dragging');
      if (this.firstChild.finalizeDrag) this.firstChild.finalizeDrag();
    },

    /**
     * Called when the drag representation node is done migrating to its final
     * resting spot.
     * @param {Event} e The transition end event.
     */
    onDragCloneTransitionEnd_: function(e) {
      if (this.classList.contains('dragging') && (e.propertyName == 'left' || e.propertyName == 'top' || e.propertyName == '-webkit-transform')) {
        this.finalizeDrag_();
      }
    },

    /**
     * Called when an app is removed from Chrome. Animates its disappearance.
     */
    doRemove: function() {
      this.firstChild.classList.add('removing-tile-contents');
    },

    /**
     * Callback for the webkitAnimationEnd event on the tile's contents.
     * @param {Event} e The event object.
     */
    onContentsAnimationEnd_: function(e) {
      if (this.firstChild.classList.contains('new-tile-contents')) this.firstChild.classList.remove('new-tile-contents');
      if (this.firstChild.classList.contains('removing-tile-contents')) {
		var tilePage = this.tilePage;
        tilePage.removeTile(this);
        
		var event = document.createEvent('Event');
        event.initEvent('removed', true, true);
		tilePage.dispatchEvent(event);
	  }
    },
  };

  /**
   * Gives the proportion of the row width that is devoted to a single icon.
   * @param {number} rowTileCount The number of tiles in a row.
   * @return {number} The ratio between icon width and row width.
   */
  function tileWidthFraction(rowTileCount) {
    return rowTileCount + (rowTileCount - 1) * TILE_SPACING_FRACTION;
  }

  /**
   * Calculates an assortment of tile-related values for a grid with the
   * given dimensions.
   * @param {number} width The pixel width of the grid.
   * @param {number} numRowTiles The number of tiles in a row.
   * @return {Object} A mapping of pixel values.
   */
  function tileValuesForGrid(width, numRowTiles) {
    var tileWidth = width / tileWidthFraction(numRowTiles);
    var offsetX = tileWidth;//* (1 + TILE_SPACING_FRACTION);
    //var interTileSpacing = offsetX - tileWidth;
    var interTileSpacing = 0;//offsetX - tileWidth - 15;
    return {
      tileWidth: tileWidth,
      offsetX: offsetX,
      interTileSpacing: interTileSpacing,
    };
  }

  // The proportion of the tile width which will be used as spacing between
  // tiles.
  var TILE_SPACING_FRACTION = 1 / 8;

  // The smallest amount of horizontal blank space to display on the sides when
  // displaying a wide arrangement.
  var MIN_WIDE_MARGIN = 110;

  /**
   * Creates a new TilePage object. This object contains tiles and controls
   * their layout.
   * @param {string} name The display name for the page.
   * @param {Object} gridValues Pixel values that define the size and layout
   *     of the tile grid.
   * @constructor
   * @extends {HTMLDivElement}
   */
  function TilePage(name, gridValues) {
    var el = cr.doc.createElement('div');
    el.pageName = name;
    el.gridValues_ = gridValues;
    el.__proto__ = TilePage.prototype;
    el.initialize();

    return el;
  }

  /**
   * Takes a collection of grid layout pixel values and updates them with
   * additional tiling values that are calculated from TilePage constants.
   * @param {Object} grid The grid layout pixel values to update.
   */
  TilePage.initGridValues = function(grid) {
    // The amount of space we need to display a narrow grid (all narrow grids
    // are this size).
    grid.narrowWidth = grid.minTileWidth * tileWidthFraction(grid.minColCount);
    // The minimum amount of space we need to display a wide grid.
    grid.minWideWidth = grid.minTileWidth * tileWidthFraction(grid.maxColCount);
    // The largest we will ever display a wide grid.
    grid.maxWideWidth = grid.maxTileWidth * tileWidthFraction(grid.maxColCount);
    // Tile-related pixel values for the narrow display.
    grid.narrowTileValues = tileValuesForGrid(grid.narrowWidth, grid.minColCount);
    // Tile-related pixel values for the minimum narrow display.
    grid.wideTileValues = tileValuesForGrid(grid.minWideWidth, grid.maxColCount);
  };

  TilePage.prototype = {
    __proto__: HTMLDivElement.prototype,

    initialize: function() {
      this.className = 'tile-page';

      // 合并16版的滚动条
      // Div that acts as a custom scrollbar. The scrollbar has to live
      // outside the content div so it doesn't flicker when scrolling (due to
      // repainting after the scroll, then repainting again when moved in the
      // onScroll handler). |scrollbar_| is only aesthetic, and it only
      // represents the thumb. Actual events are still handled by the invisible
      // native scrollbars. This div gives us more flexibility with the visuals.
      this.scrollbar_ = this.ownerDocument.createElement('div');
      this.scrollbar_.className = 'tile-page-scrollbar';
      this.scrollbar_.hidden = true;
      this.appendChild(this.scrollbar_);

      // This contains everything but the scrollbar.
      this.content_ = this.ownerDocument.createElement('div');
      this.content_.className = 'tile-page-content';
      this.appendChild(this.content_);

      // Div that holds the tiles.
      this.tileGrid_ = this.ownerDocument.createElement('div');
      this.tileGrid_.className = 'tile-grid';
      if (this.gridValues_)
        this.tileGrid_.style.minWidth = this.gridValues_.narrowWidth + 'px';
      this.content_.appendChild(this.tileGrid_);

      // Ordered list of our tiles.
      this.tileElements_ = this.tileGrid_.getElementsByClassName('tile real');

      this.eventTracker = new EventTracker();
      this.eventTracker.add(window, 'resize', this.onResize_.bind(this));

      this.content_.addEventListener('scroll', this.onScroll_.bind(this));

      this.tileGrid_.addEventListener('dragenter', this.onDragEnter_.bind(this));
      this.tileGrid_.addEventListener('dragover', this.onDragOver_.bind(this));
      this.tileGrid_.addEventListener('drop', this.onDrop_.bind(this));
      this.tileGrid_.addEventListener('dragleave', this.onDragLeave_.bind(this));

      // add by huangximing
      $('thumbnailMarginRange').addEventListener('change', function () {
        $("thumbnailMarginbar").style.width = (this.value / 16) * (this.clientWidth - 17) + "px";
        $("thumbnailMarginRangeText").innerText = this.value;
        localStorage.thumbMarginValue = this.value;
      });
      $("thumbnailMarginRange").addEventListener("mouseup", function () {
        localStorage.thumbMarginValue = this.value;
      });
      this.eventTracker.add($('thumbnailMarginRange'), 'change', this.repositionTile.bind(this));
      // add by huangximing end
    },

    get tiles() {
      return this.tileElements_;
    },

    get tileCount() {
      return this.tileElements_.length;
    },

    get selected() {
      return Array.prototype.indexOf.call(this.parentNode.children, this) ==
        ntp.getCardSlider().currentCard;
    },

    /**
     * The size of the margin (unused space) on the sides of the tile grid, in
     * pixels.
     * @type {number}
     */
    get sideMargin() {
      return this.layoutValues_ ? this.layoutValues_.leftMargin: 0;
    },

    /**
     * Returns the width of the scrollbar, in pixels, if it is active, or 0
     * otherwise.
     * @type {number}
     */
    get scrollbarWidth() {
      return this.scrollbar_.hidden ? 0: 17;
    },

    /**
     * Returns any extra padding to insert to the bottom of a tile page.  By
     * default there is none, but subclasses can override.
     * @type {number}
     */
    get extraBottomPadding() {
      return 0;
    },

    /**
     * Cleans up resources that are no longer needed after this TilePage
     * instance is removed from the DOM.
     */
    tearDown: function() {
      this.eventTracker.removeAll();
    },
    /* add by 360 */
    emptyTile: function() {
        this.tileGrid_.innerHTML = '';
        $("search_test").style.top = '0px';
      this.tileElements_ = this.tileGrid_.getElementsByClassName('tile real');
    },
    setTileContent: function(Element) {
      this.tileGrid_.appendChild(Element);
    },
    /**
     * Appends a tile to the end of the tile grid.
     * @param {HTMLElement} tileElement The contents of the tile.
     * @param {?boolean} animate If true, the append will be animated.
     * @protected
     */
    appendTile: function(tileElement, animate) {
      this.addTileAt(tileElement, this.tileElements_.length, animate);
    },

    /**
     * Adds the given element to the tile grid.
     * @param {Node} tileElement The tile object/node to insert.
     * @param {number} index The location in the tile grid to insert it at.
     * @param {?boolean} animate If true, the tile in question will be animated
     *     (other tiles, if they must reposition, do not animate).
     * @protected
     */
    addTileAt: function(tileElement, index, animate) {
      this.classList.remove('animating-tile-page');
      if (animate) tileElement.classList.add('new-tile-contents');
      var wrapperDiv = new Tile(tileElement);
      if (index == this.tileElements_.length) {
        this.tileGrid_.appendChild(wrapperDiv);
      } else {
        this.tileGrid_.insertBefore(wrapperDiv, this.tileElements_[index]);
      }
      this.calculateLayoutValues_();
      //this.heightChanged_();

      this.positionTile_(index);
    },

    /**
     * Removes the given tile and animates the respositioning of the other
     * tiles.
     * @param {HTMLElement} tile The tile to remove from |tileGrid_|.
     * @param {?boolean} animate If true, tiles will animate.
     */
    removeTile: function(tile, animate) {
      if (animate) this.classList.add('animating-tile-page');
      var index = tile.index;
      tile.parentNode.removeChild(tile);
      this.calculateLayoutValues_();
      for (var i = index; i < this.tileElements_.length; i++) {
        this.positionTile_(i);
      }
    },

    /**
     * Removes all tiles from the page.
     */
    removeAllTiles: function() {
      this.tileGrid_.innerHTML = '';
    },

    /**
     * Called when the page is selected (in the card selector).
     * @param {Event} e A custom cardselected event.
     * @private
     */
    handleCardSelection_: function(e) {
      this.tabIndex = 1;

      // When we are selected, we re-calculate the layout values. (See comment
      // in doDrop.)
      this.calculateLayoutValues_();
    },

    /**
     * Called when the page loses selection (in the card selector).
     * @param {Event} e A custom carddeselected event.
     * @private
     */
    handleCardDeselection_: function(e) {
      this.tabIndex = -1;
      if (this.currentFocusElement_)
        this.currentFocusElement_.tabIndex = -1;
    },

    /**
     * When we get focus, pass it on to the focus element.
     * @param {Event} e The focus event.
     * @private
     */
    handleFocus_: function(e) {
      if (this.focusableElements_.length == 0) return;

      this.updateFocusElement_();
    },

    /**
     * Since we are doing custom focus handling, we have to manually
     * set focusability on click (as well as keyboard nav above).
     * @param {Event} e The focus event.
     * @private
     */
    handleMouseDown_: function(e) {
      var focusable = findAncestorByClass(e.target, 'focusable');
      if (focusable) {
        this.focusElementIndex_ = Array.prototype.indexOf.call(this.focusableElements_, focusable);
        this.updateFocusElement_();
      } else {
        // This prevents the tile page from getting focus when the user clicks
        // inside the grid but outside of any tile.
        e.preventDefault();
      }
    },

    /**
     * Handle arrow key focus nav.
     * @param {Event} e The focus event.
     * @private
     */
    handleKeyDown_: function(e) {
      // We only handle up, down, left, right without control keys.
      if (e.metaKey || e.shiftKey || e.altKey || e.ctrlKey) return;

      // Wrap the given index to |this.focusableElements_|.
      var wrap = function(idx) {
        return (idx + this.focusableElements_.length) % this.focusableElements_.length;
      }.bind(this);

      switch (e.keyIdentifier) {
      case 'Right':
      case 'Left':
        var direction = e.keyIdentifier == 'Right' ? 1: -1;
        this.focusElementIndex_ = wrap(this.focusElementIndex_ + direction);
        break;
      case 'Up':
      case 'Down':
        // Look through all focusable elements. Find the first one that is
        // in the same column.
        var direction = e.keyIdentifier == 'Up' ? -1: 1;
        var currentIndex =
          Array.prototype.indexOf.call(this.focusableElements_,
            this.currentFocusElement_);
        var newFocusIdx = wrap(currentIndex + direction)
        var tile = this.currentFocusElement_.parentNode;
        for (;; newFocusIdx = wrap(newFocusIdx + direction)) {
          var newTile = this.focusableElements_[newFocusIdx].parentNode;
          var rowTiles = this.layoutValues_.numRowTiles;
          if ((newTile.index - tile.index) % rowTiles == 0) break;
        }

        this.focusElementIndex_ = newFocusIdx;
        break;

      default:
        return;
      }

      this.updateFocusElement_();

      e.preventDefault();
      e.stopPropagation();
    },

    /**
     * Focuses the element for |this.focusElementIndex_|. Makes the current
     * focus element, if any, no longer eligible for focus.
     * @private
     */
    updateFocusElement_: function() {
      this.focusElementIndex_ = Math.min(this.focusableElements_.length - 1, this.focusElementIndex_);
      this.focusElementIndex_ = Math.max(0, this.focusElementIndex_);

      var newFocusElement = this.focusableElements_[this.focusElementIndex_];
      var lastFocusElement = this.currentFocusElement_;
      if (lastFocusElement && lastFocusElement != newFocusElement) lastFocusElement.tabIndex = -1;

      newFocusElement.tabIndex = 1;
      newFocusElement.focus();
      this.tabIndex = -1;
    },

    /**
     * The current focus element is that element which is eligible for focus.
     * @type {HTMLElement} The node.
     * @private
     */
    get currentFocusElement_() {
      return this.querySelector('.focusable[tabindex="1"]');
    },

    /**
     * Makes some calculations for tile layout. These change depending on
     * height, width, and the number of tiles.
     * TODO(estade): optimize calls to this function. Do nothing if the page is
     * hidden, but call before being shown.
     * @private
     */
    calculateLayoutValues_: function() {
      var grid = this.gridValues_;
      if (!grid) {
        return;
      }
      var tileGrid_width = this.tileGrid_.parentNode.parentNode.clientWidth - 16;
      var availableSpace = tileGrid_width - 2 * MIN_WIDE_MARGIN;

      var wide = availableSpace >= grid.minWideWidth;
      var numRowTiles = wide ? grid.maxColCount: grid.minColCount;

      var effectiveGridWidth = wide ? Math.min(Math.max(availableSpace, grid.minWideWidth), grid.maxWideWidth) : grid.narrowWidth;

      if (grid.type == 'apps') {
          var bow = document.documentElement.clientWidth,
          boh = document.documentElement.clientHeight;
          var whp = bow / boh;
          if (whp > 1.6 && (boh < 780 || bow < 950)) {
              effectiveGridWidth -= 100 * whp;
              if (whp > 2.0 && whp < 2.2 && boh < 645 && bow < 1426) {
                  effectiveGridWidth += 26 * whp;
              } else if (whp > 1.8 && whp < 1.9 && boh < 780 && bow < 1440) {
                  effectiveGridWidth += 50 * whp;
              }
          }
      } else {
          wide = false;
      }

      var realTileValues = tileValuesForGrid(effectiveGridWidth, numRowTiles);

        // leftMargin centers the grid within the avaiable space.
      var minMargin = wide ? MIN_WIDE_MARGIN : 0;
      var leftMargin = Math.max(minMargin, (tileGrid_width - effectiveGridWidth) / 2);

      var rowHeight = this.heightForWidth(realTileValues.tileWidth) + realTileValues.interTileSpacing;

      if (grid.type == 'mosts') {
          var areaHeight = window.innerHeight - 140 - 55;
          var cachedMarginValue = parseInt(localStorage.thumbMarginValue) || 0;
          var gridH = (rowHeight + cachedMarginValue) * 3;
          if (gridH > areaHeight) {
              rowHeight = Math.max(115, areaHeight / 3 - cachedMarginValue);
              var rowWidth = rowHeight * 278 / 162;
              effectiveGridWidth = (rowWidth + cachedMarginValue) * numRowTiles + cachedMarginValue + 35;
              leftMargin = Math.max(minMargin, (tileGrid_width - effectiveGridWidth) / 2);
              realTileValues = {
                  offsetX: rowWidth,
                  tileWidth: rowWidth
              };
          }
      }

      this.layoutValues_ = {
        numRowTiles: numRowTiles,
        leftMargin: leftMargin,
        colWidth: realTileValues.offsetX,
        rowHeight: rowHeight,
        tileWidth: realTileValues.tileWidth,
        wide: wide,
      };
//      if(this.classList.contains("most-visited-page")){
//        document.querySelectorAll(".channel-wrapper").foreach(function (e,i,a) {
//            var scaleRate = Math.max(realTileValues.tileWidth/grid.maxTileWidth ,grid.minTileWidth/grid.maxTileWidth);
//            //e.style.webkitTransform = "scale("+scaleRate+")";
//            //e.style.webkitTransformationOrigin = "top left";//This method can not make the wrapper div in center
//            e.style.zoom = scaleRate;
//        });
//      }

      this.updateTopMargin_();

      this.firePageLayoutEvent_();
    },

    /**
     * Dispatches the custom pagelayout event.
     * @private
     */
    firePageLayoutEvent_: function() {
      cr.dispatchSimpleEvent(this, 'pagelayout', true, true);
    },
    repositionTile : function(e) {
      for (var i = 0; i < this.tileElements_.length; i++) {
        this.positionTile_(i);
      }
      //ntp.updateGridShadow();
    },
    /**
     * Calculates the x/y coordinates for an element and moves it there.
     * @param {number} index The index of the element to be positioned.
     * @param {number} indexOffset If provided, this is added to |index| when
     *     positioning the tile. The effect is that the tile will be positioned
     *     in a non-default location.
     * @private
     */
    positionTile_: function(index, indexOffset) {
      var grid = this.gridValues_;
      var layout = this.layoutValues_;
      indexOffset = typeof indexOffset != 'undefined' ? indexOffset: 0;
      // Add the offset _after_ the modulus division. We might want to show the
      // tile off the side of the grid.
      var col = index % layout.numRowTiles + indexOffset;
      var row = Math.floor(index / layout.numRowTiles);
      if(!this.classList.contains("apps-page")){
        if(col>=4) { col = 0;row++;};
        if(col<0) { col = 3;row--;};
      }
      var tile = this.tileElements_[index];
      var cachedMarginValue = parseInt(localStorage.thumbMarginValue) || 0;
      // Calculate the final on-screen position for the tile.
      var realX = col * (layout.colWidth+cachedMarginValue) + layout.leftMargin;
      var realY = row * (layout.rowHeight+cachedMarginValue);
      var most_visited_page = $('page-list').querySelector('.most-visited-page').querySelector('.tile-grid');
      if(most_visited_page) most_visited_page.style.top = cachedMarginValue + 'px';
      var marginLeft_ = 0;
      if(ntp.getCardSlider().cards_.length<=1
       || loadTimeData.getInteger('shown_page_type') == loadTimeData.getInteger('apps_page_id')
       ) marginLeft_ = 52;
      most_visited_page.style.marginLeft = marginLeft_ + 'px';
      var switchBtn = $('page-switcher-end');
      // Calculate the portion of the tile's position that should be animated.
      var animatedTileValues = layout.wide ? grid.wideTileValues: grid.narrowTileValues;
      // Animate the difference between three-wide and six-wide.
      var animatedLeftMargin = layout.wide ? 0: (grid.minWideWidth - MIN_WIDE_MARGIN - grid.narrowWidth) / 2;
      var animatedX = col * animatedTileValues.offsetX + animatedLeftMargin ;
      var animatedY = row * (this.heightForWidth(animatedTileValues.tileWidth) + animatedTileValues.interTileSpacing);

      tile.setGridPosition(animatedX + switchBtn.clientWidth/2 - cachedMarginValue, animatedY - cachedMarginValue);
      tile.firstChild.setBounds(layout.tileWidth, realX - animatedX, realY - animatedY);

       var targetElements = tile.querySelectorAll(".most-visited,.thumbnail,.edit-bar,.thumbnail-title");
       var len = targetElements.length;
       if(cachedMarginValue == 0){
            for(var i=0;i<len;i++){
                targetElements[i].classList.add("zero");
            }
        }else{
            for(var i=0;i<len;i++){
                targetElements[i].classList.remove("zero");
            }
        }
      // This code calculates whether the tile needs to show a clone of itself
      // wrapped around the other side of the tile grid.
      var offTheRight = col == layout.numRowTiles || (col == layout.numRowTiles - 1 && tile.hasDoppleganger());
      var offTheLeft = col == -1 || (col == 0 && tile.hasDoppleganger());
      if (this.isCurrentDragTarget_ && (offTheRight || offTheLeft)) {
        var sign = offTheRight ? 1: -1;
        tile.showDoppleganger(-layout.numRowTiles * layout.colWidth * sign, layout.rowHeight * sign);
      } else {
        tile.clearDoppleganger();
      }

      //
      if (index == this.tileElements_.length - 1) {
        this.tileGrid_.style.height = (realY + layout.rowHeight) + 'px';

        this.queueUpdateScrollbars_();
      }
    },

    /**
     * Gets the index of the tile that should occupy coordinate (x, y). Note
     * that this function doesn't care where the tiles actually are, and will
     * return an index even for the space between two tiles. This function is
     * effectively the inverse of |positionTile_|.
     * @param {number} x The x coordinate, in pixels, relative to the top left
     *     of tileGrid_.
     * @param {number} y The y coordinate.
     * @private
     */
    getWouldBeIndexForPoint_: function(x, y) {
      var grid = this.gridValues_;
      var layout = this.layoutValues_;

      var gridClientRect = this.tileGrid_.getBoundingClientRect();
      var col = Math.floor((x - gridClientRect.left - layout.leftMargin) / layout.colWidth);
      if (col < 0 || col >= layout.numRowTiles) return -1;

      if (ntp.isRTL()) col = layout.numRowTiles - 1 - col;

      var row = Math.floor((y - gridClientRect.top) / layout.rowHeight);
      return row * layout.numRowTiles + col;
    },

    /**
     * Window resize event handler. Window resizes may trigger re-layouts.
     * @param {Object} e The resize event.
     */
    onResize_: function(e) {
      if (this.lastWidth_ == this.clientWidth && this.lastHeight_ == this.clientHeight) {
        return;
      }

      this.calculateLayoutValues_();

      this.lastWidth_ = this.clientWidth;
      this.lastHeight_ = this.clientHeight;
      this.classList.add('animating-tile-page');
      for (var i = 0; i < this.tileElements_.length; i++) {
        this.positionTile_(i);
      }

      // redraw scrollbars
      this.queueUpdateScrollbars_();
    },

    updateTopMargin_: function(page) {
      var currentSlider= page?page:document.querySelector(".selected-card");
      if(!currentSlider) return;
      var gridHeight = currentSlider.querySelector(".tile-grid").clientHeight;

//      if (currentSlider.classList.contains("most-visited-page")) {
//        var firstMostVisited = document.querySelector('.most-visited');
//        gridHeight += firstMostVisited&&firstMostVisited.clientHeight>0 ? firstMostVisited.clientHeight : 160;
//      }
      var arailWidth = document.documentElement.clientWidth,
          arailHeight = document.documentElement.clientHeight,
          footerEle = document.querySelector(".footer"),
          footerEleBottom = 65,
          minSearchHeight = 140,
          newMargin = (arailWidth > 1024) ? minSearchHeight : 130;
      ;
      var whp = arailWidth/arailHeight;
      if((whp>=2 && arailHeight < 645) || (whp<2 && ((arailHeight<680 && arailWidth>1000) || (arailHeight<710 && arailWidth>1280)))){
        footerEleBottom = 10;
        if(whp<2 && arailHeight<680 && arailWidth>1000) footerEleBottom += 30;
      }

      var footerHeight = footerEleBottom + footerEle.clientHeight;
      var content_Margin = Math.max((arailHeight - footerHeight - gridHeight) / 2, (loadTimeData.getBoolean("showSearchEngineBar") ? newMargin : 10));
      if (!currentSlider.classList.contains("nav-page") && !currentSlider.classList.contains("apps-page"))
        currentSlider.querySelector(".tile-page-content").style.paddingTop = content_Margin + 'px';
      newMargin = Math.max(minSearchHeight, content_Margin);
      $("search-form").style.marginTop = (newMargin - $("search-form").clientHeight) / 2 + "px";
      var search_test_style = $("search_test").style;
      if(page) search_test_style.top = -currentSlider.querySelector(".tile-page-content").scrollTop + "px";
      search_test_style.height = newMargin + "px";
      search_test_style.display = loadTimeData.getBoolean("showSearchEngineBar") && !currentSlider.classList.contains("apps-page") && !currentSlider.classList.contains("nav-page")  ? "block" : "none";
      footerEle.style.bottom = footerEleBottom + "px";
//      cacheNTP();
//      console.log("CacheNTP:topmargin")
    },

    /**
     * Handles final setup that can only happen after |this| is inserted into
     * the page.
     * @private
     */
    onNodeInsertedIntoDocument_: function(e) {
      this.calculateLayoutValues_();
      //this.heightChanged_();
    },

    /**
     * Called when the height of |this| has changed: update the size of
     * tileGrid.
     * @private
     */
    heightChanged_: function() {
      // The tile grid will expand to the bottom footer, or enough to hold all
      // the tiles, whichever is greater. It would be nicer if tilePage were
      // a flex box, and the tile grid could be box-flex: 1, but this exposes a
      // bug where repositioning tiles will cause the scroll position to reset.
      this.tileGrid_.style.minHeight = (this.clientHeight - this.tileGrid_.offsetTop - this.content_.offsetTop - this.extraBottomPadding) + 'px';
    },

    /**
     * Scrolls the page in response to a mousewheel event.
     * @param {Event} e The mousewheel event.
     */
    handleMouseWheel: function(e) {
      this.content_.scrollTop -= e.wheelDeltaY / 3;
      if ($("search-hotwords").classList.contains("triangle-click"))
        $("search-hotwords").classList.remove("triangle-click");
      if (!$("hot-items").classList.contains("hidden"))
        $("hot-items").classList.add("hidden");
       $("search-sg").style.display = "none";
    },

    /**
     * Handles mouse wheels on |this|. We handle this explicitly because we want
     * a consistent experience whether the user scrolls on the page or on the
     * page switcher (handleMouseWheel provides a common conversion factor
     * between wheel delta and scroll delta).
     * @param {Event} e The mousewheel event.
     * @private
     */
    onMouseWheel_: function(e) {
      if (e.wheelDeltaY == 0) return;

      this.handleMouseWheel(e);
      e.preventDefault();
      e.stopPropagation();
    },

    /**
     * Handler for the 'scroll' event on |content_|.
     * @param {Event} e The scroll event.
     * @private
     */
    onScroll_: function(e) {
      this.queueUpdateScrollbars_();
      chrome.send("clearCacheNTP",["scroll"]);
    },

    /**
     * ID of scrollbar update timer. If 0, there's no scrollbar re-calc queued.
     * @private
     */
    scrollbarUpdate_: 0,

    /**
     * Queues an update on the custom scrollbar. Used for two reasons: first,
     * coalescing of multiple updates, and second, because action like
     * repositioning a tile can require a delay before they affect values
     * like clientHeight.
     * @private
     */
    queueUpdateScrollbars_: function() {
      if (this.scrollbarUpdate_) return;

      this.scrollbarUpdate_ = window.setTimeout(
      this.doUpdateScrollbars_.bind(this), 0);
    },

    /**
     * Does the work of calculating the visibility, height and position of the
     * scrollbar thumb (there is no track or buttons).
     * @private
     */
    doUpdateScrollbars_: function() {
      this.scrollbarUpdate_ = 0;

      var content = this.content_;

      // Adjust scroll-height to account for possible header-bar.
      var adjustedScrollHeight = content.scrollHeight - content.offsetTop;

      if (adjustedScrollHeight <= content.clientHeight) {
        this.scrollbar_.hidden = true;
        return;
      } else {
        this.scrollbar_.hidden = false;
      }

      var thumbTop = content.offsetTop + content.scrollTop / adjustedScrollHeight * content.clientHeight;
      var thumbHeight = content.clientHeight / adjustedScrollHeight * this.clientHeight;
      $("search_test").style.top = - content.scrollTop + "px";
      if(!$("hot-items").classList.contains("hidden")){
        var hotItems = $('hot-items');
        var searchInput = document.querySelector("#search-kw");
        var offset = searchInput.getBoundingClientRect();
        if (!hotItems.classList.contains("hidden")) {
            hotItems.style.left = offset.left + "px";
            hotItems.style.top = offset.top + searchInput.clientHeight + 5 + "px";
        }
      }
      this.scrollbar_.style.top = thumbTop + 'px';
      this.scrollbar_.style.height = thumbHeight + 'px';
      this.firePageLayoutEvent_();
    },

    /**
     * Get the height for a tile of a certain width. Override this function to
     * get non-square tiles.
     * @param {number} width The pixel width of a tile.
     * @return {number} The height for |width|.
     */
    heightForWidth: function(width) {
      return width;
    },

    /** Dragging **/

    /**
     * The number of un-paired dragenter events that have fired on |this|. This
     * is incremented by |onDragEnter_| and decremented by |onDragLeave_|. This
     * is necessary because dragging over child widgets will fire additional
     * enter and leave events on |this|. A non-zero value does not necessarily
     * indicate that |isCurrentDragTarget_| is true.
     * @type {number}
     * @private
     */
    dragEnters_: 0,

    /**
     * Whether the tile page is currently being dragged over with data it can
     * accept.
     * @type {boolean}
     * @private
     */
    isCurrentDragTarget_: false,

    /**
     * Handler for dragenter events fired on |tileGrid_|.
     * @param {Event} e A MouseEvent for the drag.
     * @private
     */
    onDragEnter_: function(e) {
      if (++this.dragEnters_ > 1) return;
      this.doDragEnter_(e);
    },

    /**
     * Thunk for dragover events fired on |tileGrid_|.
     * @param {Event} e A MouseEvent for the drag.
     * @private
     */
    onDragOver_: function(e) {
      if (!this.isCurrentDragTarget_) return;
      this.doDragOver_(e);
    },

    /**
     * Thunk for drop events fired on |tileGrid_|.
     * @param {Event} e A MouseEvent for the drag.
     * @private
     */
    onDrop_: function(e) {
      this.dragEnters_ = 0;
      if (!this.isCurrentDragTarget_) return;
      this.doDrop_(e);
    },

    /**
     * Thunk for dragleave events fired on |tileGrid_|.
     * @param {Event} e A MouseEvent for the drag.
     * @private
     */
    onDragLeave_: function(e) {
      if (--this.dragEnters_ > 0) return;

      this.isCurrentDragTarget_ = false;
      this.cleanUpDrag_();
    },

    /**
     * Performs all actions necessary when a drag enters the tile page.
     * @param {Event} e A mouseover event for the drag enter.
     * @private
     */
    doDragEnter_: function(e) {
      if (!this.shouldAcceptDrag(e.dataTransfer)) {
        return;
      }

      this.isCurrentDragTarget_ = true;

      this.classList.add('animating-tile-page');
      this.withinPageDrag_ = this.contains(currentlyDraggingTile);
      this.dragItemIndex_ = this.withinPageDrag_ ? currentlyDraggingTile.index: this.tileElements_.length;
      this.currentDropIndex_ = this.dragItemIndex_;
    },

    /**
     * Performs all actions necessary when the user moves the cursor during
     * a drag over the tile page.
     * @param {Event} e A mouseover event for the drag over.
     * @private
     */
    doDragOver_: function(e) {
      e.preventDefault();

      if (currentlyDraggingTile) e.dataTransfer.dropEffect = 'move';
      else e.dataTransfer.dropEffect = 'copy';

      var newDragIndex = this.getWouldBeIndexForPoint_(e.pageX, e.pageY);
      if (newDragIndex < 0 || newDragIndex >= this.tileElements_.length) newDragIndex = this.dragItemIndex_;
      this.updateDropIndicator_(newDragIndex);
    },

    /**
     * Performs all actions necessary when the user completes a drop.
     * @param {Event} e A mouseover event for the drag drop.
     * @private
     */
    doDrop_: function(e) {
      e.stopPropagation();
      this.isCurrentDragTarget_ = false;

      var index = this.currentDropIndex_;
      // Only change data if this was not a 'null drag'.
      if (! ((index == this.dragItemIndex_) && this.withinPageDrag_)) {

        var adjustedIndex = this.currentDropIndex_ + (index > this.dragItemIndex_ ? 1: 0);
        if (currentlyDraggingTile) {
          /* add by 360 to save most-visited grid-index and pinned */
          //addPinnedUrl(thumbData[currentlyDraggingTile.index], index);
          /* add over */
          this.tileGrid_.insertBefore(currentlyDraggingTile, this.tileElements_[adjustedIndex]);
          this.tileMoved(currentlyDraggingTile);
        } else {
          this.addOutsideData(e.dataTransfer, adjustedIndex);
        }
      }

      this.classList.remove('animating-tile-page');
      this.cleanUpDrag_();
    },

    /**
     * Makes sure all the tiles are in the right place after a drag is over.
     * @private
     */
    cleanUpDrag_: function() {
      for (var i = 0; i < this.tileElements_.length; i++) {
        // The current drag tile will be positioned in its dragend handler.
        if (this.tileElements_[i] == currentlyDraggingTile) {
          //currentlyDraggingTile.classList.remove('dragging');
          continue;
        }
        this.positionTile_(i);
      }
    },

    /**
     * Updates the visual indicator for the drop location for the active drag.
     * @param {Event} e A MouseEvent for the drag.
     * @private
     */
    updateDropIndicator_: function(newDragIndex) {
      var oldDragIndex = this.currentDropIndex_;
      if (newDragIndex == oldDragIndex) return;

      /**
    this.positionTile_(newDragIndex, this.dragItemIndex_ - newDragIndex);
    this.positionTile_(oldDragIndex);
      this.currentDropIndex_ = newDragIndex;

    return false;
/**/
      var repositionStart = Math.min(newDragIndex, oldDragIndex);
      var repositionEnd = Math.max(newDragIndex, oldDragIndex);

      for (var i = repositionStart; i <= repositionEnd; i++) {
        if (i == this.dragItemIndex_) continue;
        else if (i > this.dragItemIndex_) var adjustment = i <= newDragIndex ? -1: 0;
        else var adjustment = i >= newDragIndex ? 1: 0;

        this.positionTile_(i, adjustment);
      }
      this.currentDropIndex_ = newDragIndex;
    },

    /**
     * Checks if a page can accept a drag with the given data.
     * @param {Object} dataTransfer The dataTransfer object, if the drag object
     *     is not a tile (e.g. it is a link).
     * @return {boolean} True if this page can handle the drag.
     */
    shouldAcceptDrag: function(dataTransfer) {
      return false;
    },

    /**
     * Called to accept a drag drop.
     * @param {Object} dataTransfer The data transfer object that holds the drop
     *     data.
     * @param {number} index The tile index at which the drop occurred.
     */
    addOutsideData: function(dataTransfer, index) {
      // This should not get called unless there is a non-default
      // implementation.
      assert(false);
    },

    /**
     * Called when a tile has been moved (via dragging). Override this to make
     * backend updates.
     * @param {Node} draggedTile The tile that was dropped.
     */
    tileMoved: function(draggedTile) {},
  };

  return {
    getCurrentlyDraggingTile: getCurrentlyDraggingTile,
    TilePage: TilePage
  };
});
