// Copyright (c) 2011 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

cr.define('options', function () {
    var OptionsPage = options.OptionsPage;

    /**
    * ImportDataOverlay class
    * Encapsulated handling of the 'Import Data' overlay page.
    * @class
    */
    function ImportDataOverlay() {
        OptionsPage.call(this,
                     'importData',
                     templateData.importDataOverlayTabTitle,
                     'import-data-overlay');
    }

    cr.addSingletonGetter(ImportDataOverlay);

    ImportDataOverlay.prototype = {
        // Inherit from OptionsPage.
        __proto__: OptionsPage.prototype,

        /**
        * Initialize the page.
        */
        initializePage: function () {
            // Call base class implementation to start preference initialization.
            OptionsPage.prototype.initializePage.call(this);
            var self = this;
            $('import-browsers').onchange = function () {
                self.updateCheckboxes_();
                self.validateCommitButton_();
            };

            $('import-data-commit').onclick = function () {
                var importOptions = ['history', 'passwords', 'search',
                'google-extension', 'google-app', 'google-opentabs'];
                var param = [];
                param[0] = String($('import-browsers').value);
                for (var j = 0; j < importOptions.length; j++) {
                    var checkbox = $('import-' + importOptions[j]);
                    if (checkbox != null) param[j + 1] = String(checkbox.checked);
                    else param[j + 1] = String(false);
                }
                chrome.send('importData', param);
            };

            $('import-data-cancel').onclick = function () {
                ImportDataOverlay.dismiss();
            };

            $('import-data-show-bookmarks-bar').onchange = function () {
                // Note: The callback 'toggleShowBookmarksBar' is handled within the
                // browser options handler -- rather than the import data handler --
                // as the implementation is shared by several clients.
                chrome.send('toggleShowBookmarksBar');
            };

            $('import-data-confirm').onclick = function () {
                ImportDataOverlay.dismiss();
            };

            // Form controls are disabled until the profile list has been loaded.
            //self.setControlsSensitive_(false);
        },

        /**
        * Set enabled and checked state of the commit button.
        * @private
        */
        validateCommitButton_: function () {
            var somethingToImport = false;
            var checkboxes = document.querySelectorAll('#import-checkboxes input[type=checkbox]');
            for (var i = 0; i < checkboxes.length; i++) {
                somethingToImport = (somethingToImport || checkboxes[i].checked);
            }
            $('import-data-commit').disabled = !somethingToImport;
        },

        /**
        * Sets the sensitivity of all the checkboxes and the commit button.
        * @private
        */
        setControlsSensitive_: function (sensitive) {
            var checkboxes =
            document.querySelectorAll('#import-checkboxes input[type=checkbox]');
            for (var i = 0; i < checkboxes.length; i++) {
                this.setUpCheckboxState_(checkboxes[i], sensitive);
            }
            $('import-data-commit').disabled = !sensitive;
        },

        /**
        * Set enabled and checked states a checkbox element.
        * @param {Object} checkbox A checkbox element.
        * @param {boolean} enabled The enabled state of the chekbox.
        * @private
        */
        setUpCheckboxState_: function (checkbox, enabled) {
            //checkbox.setDisabled("noProfileData", !enabled);
            checkbox.disabled = !enabled;
        },

        /**
        * Update the enabled and checked states of all checkboxes.
        * @private
        */
        updateCheckboxes_: function () {
            var index = $('import-browsers').selectedIndex;
            var browserProfile;
            if (this.browserProfiles.length > index)
                browserProfile = this.browserProfiles[index];
            else return;
            var import_tds = ['import-table-01',
                              'import-table-02',
                              'import-table-11',
                              'import-table-12',
                              'import-table-21',
                              'import-table-22'];
            for (var i = 0; i < import_tds.length; i++) {
                $(import_tds[i]).innerHTML = "&nbsp;";
            }
            var importOptions = ['history', 'passwords', 'search',
                'google-extension', 'google-app', 'google-opentabs'];
            var pos = 0;
            for (var j = 0; j < importOptions.length; j++) {
                var node_text = importOptions[j];
                if (browserProfile[node_text]) {
                    $(import_tds[pos++]).innerHTML = this.checkbox_list_text[node_text];
                }
            }
            for (var k = 0; k < importOptions.length; k++) {
                var node_text = importOptions[k];
                var checkbox = document.getElementById('import-' + node_text);
                if (checkbox) {
                    checkbox.onchange = function () {
                        self.validateCommitButton_();
                    }
                    checkbox.checked = true;
                    this.setUpCheckboxState_(checkbox, true);
                }
            }
        },

        /**
        * Update the supported browsers popup with given entries.
        * @param {array} browsers List of supported browsers name.
        * @private
        */
        updateSupportedBrowsers_: function (browsers) {
            this.checkbox_list_text = new Object();
            var importOptions = ['history', 'passwords', 'search',
                'google-extension', 'google-app', 'google-opentabs'];
            for (var j = 0; j < importOptions.length; j++) {
                var option_div = $('import-' + importOptions[j] + '-div');
                if (option_div != null) {
                    this.checkbox_list_text[importOptions[j]] = option_div.innerHTML;
                }
            }
            this.browserProfiles = browsers;
            var browserSelect = $('import-browsers');
            var selectedIndex = browserSelect.selectedIndex;
            var selectedText = "";
            var setSelect = false;
            if (selectedIndex >= 0) {
                selectedText = browserSelect.options[selectedIndex].text;
                setSelect = true;
            }
            browserSelect.remove(0);  // Remove the 'Loading...' option.
            browserSelect.textContent = '';
            var browserCount = browsers.length;

            if (browserCount == 0) {
                var option = new Option(templateData.noProfileFound, 0);
                browserSelect.appendChild(option);
                this.setControlsSensitive_(false);
            } else {
                for (var i = 0; i < browserCount; i++) {
                    var browser = browsers[i];
                    var option = new Option(browser['name'], browser['index']);
                    browserSelect.appendChild(option);
                    if (setSelect == true) {
                        if (selectedText == browser['name'])
                            option.selected = true;
                    }
                }
                this.updateCheckboxes_();
                this.validateCommitButton_();
                this.setControlsSensitive_(true);
            }
        },

        /**
        * Clear import prefs set when user checks/unchecks a checkbox so that each
        * checkbox goes back to the default "checked" state (or alternatively, to
        * the state set by a recommended policy).
        * @private
        */
        clearUserPrefs_: function () {
            var importPrefs = [
                                'import_history',
                                'import_saved_passwords',
                                'import_search_engine',
                                'import-google-extension',
                                'import-google-app',
                                'import-google-opentabs'
                               ];
            for (var i = 0; i < importPrefs.length; i++)
                Preferences.clearPref(importPrefs[i], undefined);
        }
    };

    ImportDataOverlay.clearUserPrefs = function () {
        ImportDataOverlay.getInstance().clearUserPrefs_();
    };

    /**
    * Update the supported browsers popup with given entries.
    * @param {array} list of supported browsers name.
    */
    ImportDataOverlay.updateSupportedBrowsers = function (browsers) {
        ImportDataOverlay.getInstance().updateSupportedBrowsers_(browsers);
    };

    /**
    * Update the UI to reflect whether an import operation is in progress.
    * @param {boolean} state True if an import operation is in progress.
    */
    ImportDataOverlay.setImportingState = function (state) {
        var checkboxes =
        document.querySelectorAll('#import-checkboxes input[type=checkbox]');
        for (var i = 0; i < checkboxes.length; i++) {
            checkboxes[i].disabled = state;
            //checkboxes[i].setDisabled("Importing", state);
        }
        //        if (!state)
        //            ImportDataOverlay.getInstance().updateCheckboxes_();
        $('import-browsers').disabled = state;
        $('import-throbber').style.visibility = state ? "visible" : "hidden";
        ImportDataOverlay.getInstance().validateCommitButton_();
    };

    /**
    * Remove the import overlay from display.
    */
    ImportDataOverlay.dismiss = function () {
        ImportDataOverlay.clearUserPrefs();
        OptionsPage.closeOverlay();
    };

    /**
    * Show a message confirming the success of the import operation.
    */
    ImportDataOverlay.confirmSuccess = function () {
        //        var showBookmarksMessage = $('import-favorites').checked;
        ImportDataOverlay.setImportingState(false);
        $('import-data-configure').hidden = true;
        $('import-data-success').hidden = false;
        $('import-find-your-bookmarks').hidden = true; //!showBookmarksMessage;
    };

    // Export
    return {
        ImportDataOverlay: ImportDataOverlay
    };
});
