/*
 * Decompiled with CFR 0.152.
 */
package git4idea.config;

import com.intellij.execution.ExecutionException;
import com.intellij.execution.configurations.GeneralCommandLine;
import com.intellij.execution.process.CapturingProcessHandler;
import com.intellij.execution.process.ProcessOutput;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.vfs.CharsetToolkit;
import git4idea.config.GitVersion;
import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class GitExecutableDetector {
    private static final Logger LOG = Logger.getInstance(GitExecutableDetector.class);
    private static final String[] UNIX_PATHS = new String[]{"/usr/local/bin", "/usr/bin", "/opt/local/bin", "/opt/bin", "/usr/local/git/bin"};
    private static final String UNIX_EXECUTABLE = "git";
    private static final File WIN_ROOT = new File("C:");
    private static final String GIT_CMD = "git.cmd";
    private static final String GIT_EXE = "git.exe";
    public static final String DEFAULT_WIN_GIT = "git.exe";
    public static final String PATH_ENV = "PATH";

    @NotNull
    public String detect() {
        if (SystemInfo.isWindows) {
            String string = this.detectForWindows();
            if (string == null) {
                throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detect"));
            }
            return string;
        }
        String string = GitExecutableDetector.detectForUnix();
        if (string == null) {
            throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detect"));
        }
        return string;
    }

    @NotNull
    private static String detectForUnix() {
        for (String p : UNIX_PATHS) {
            File f = new File(p, UNIX_EXECUTABLE);
            if (!f.exists()) continue;
            String string = f.getPath();
            if (string == null) {
                throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detectForUnix"));
            }
            return string;
        }
        if (UNIX_EXECUTABLE == null) {
            throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detectForUnix"));
        }
        return UNIX_EXECUTABLE;
    }

    @NotNull
    private String detectForWindows() {
        String exec = this.checkInPath();
        if (exec != null) {
            String string = exec;
            if (string == null) {
                throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detectForWindows"));
            }
            return string;
        }
        exec = GitExecutableDetector.checkProgramFiles();
        if (exec != null) {
            String string = exec;
            if (string == null) {
                throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detectForWindows"));
            }
            return string;
        }
        exec = GitExecutableDetector.checkCygwin();
        if (exec != null) {
            String string = exec;
            if (string == null) {
                throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detectForWindows"));
            }
            return string;
        }
        String string = this.checkSoleExecutable();
        if (string == null) {
            throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "detectForWindows"));
        }
        return string;
    }

    @Nullable
    private String checkInPath() {
        String PATH = this.getPath();
        if (PATH == null) {
            return null;
        }
        List pathEntries = StringUtil.split((String)PATH, (String)";");
        for (String pathEntry : pathEntries) {
            if (!GitExecutableDetector.looksLikeGit(pathEntry)) continue;
            return GitExecutableDetector.checkBinDir(new File(pathEntry));
        }
        return null;
    }

    private static boolean looksLikeGit(@NotNull String path) {
        if (path == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "path", "git4idea/config/GitExecutableDetector", "looksLikeGit"));
        }
        List dirs = FileUtil.splitPath((String)path);
        for (String dir : dirs) {
            if (!dir.toLowerCase().startsWith(UNIX_EXECUTABLE)) continue;
            return true;
        }
        return false;
    }

    @Nullable
    private static String checkProgramFiles() {
        String[] PROGRAM_FILES = new String[]{"Program Files", "Program Files (x86)"};
        ArrayList<File> distrs = new ArrayList<File>();
        for (String programFiles : PROGRAM_FILES) {
            File pf = new File(WIN_ROOT, programFiles);
            File[] children = pf.listFiles(new FileFilter(){

                @Override
                public boolean accept(File pathname) {
                    return pathname.isDirectory() && pathname.getName().toLowerCase().startsWith(GitExecutableDetector.UNIX_EXECUTABLE);
                }
            });
            if (!pf.exists() || children == null) continue;
            distrs.addAll(Arrays.asList(children));
        }
        Collections.sort(distrs, Collections.reverseOrder(new VersionDirsComparator()));
        for (File distr : distrs) {
            String exec = GitExecutableDetector.checkDistributive(distr);
            if (exec == null) continue;
            return exec;
        }
        return null;
    }

    @Nullable
    private static String checkCygwin() {
        String[] OTHER_WINDOWS_PATHS;
        for (String otherPath : OTHER_WINDOWS_PATHS = new String[]{FileUtil.toSystemDependentName((String)"cygwin/bin/git.exe")}) {
            File file = new File(WIN_ROOT, otherPath);
            if (!file.exists()) continue;
            return file.getPath();
        }
        return null;
    }

    @NotNull
    private String checkSoleExecutable() {
        if (this.runs(GIT_CMD)) {
            if (GIT_CMD == null) {
                throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "checkSoleExecutable"));
            }
            return GIT_CMD;
        }
        if ("git.exe" == null) {
            throw new IllegalStateException(String.format("@NotNull method %s.%s must not return null", "git4idea/config/GitExecutableDetector", "checkSoleExecutable"));
        }
        return "git.exe";
    }

    @Nullable
    private static String checkDistributive(@Nullable File gitDir) {
        String[] binDirs;
        if (gitDir == null || !gitDir.exists()) {
            return null;
        }
        for (String binDir : binDirs = new String[]{"cmd", "bin"}) {
            String exec = GitExecutableDetector.checkBinDir(new File(gitDir, binDir));
            if (exec == null) continue;
            return exec;
        }
        return null;
    }

    @Nullable
    private static String checkBinDir(@NotNull File binDir) {
        if (binDir == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "binDir", "git4idea/config/GitExecutableDetector", "checkBinDir"));
        }
        if (!binDir.exists()) {
            return null;
        }
        for (String exec : new String[]{GIT_CMD, "git.exe"}) {
            File fe = new File(binDir, exec);
            if (!fe.exists()) continue;
            return fe.getPath();
        }
        return null;
    }

    protected boolean runs(@NotNull String exec) {
        if (exec == null) {
            throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "exec", "git4idea/config/GitExecutableDetector", "runs"));
        }
        GeneralCommandLine commandLine = new GeneralCommandLine();
        commandLine.setExePath(exec);
        commandLine.setCharset(CharsetToolkit.getDefaultSystemCharset());
        try {
            CapturingProcessHandler handler = new CapturingProcessHandler(commandLine);
            ProcessOutput result = handler.runProcess((int)TimeUnit.SECONDS.toMillis(5L));
            return !result.isTimeout();
        }
        catch (ExecutionException e) {
            return false;
        }
    }

    @Nullable
    protected String getPath() {
        return System.getenv(PATH_ENV);
    }

    private static class VersionDirsComparator
    implements Comparator<File> {
        private VersionDirsComparator() {
        }

        @Override
        public int compare(File f1, File f2) {
            String name1 = f1.getName().toLowerCase();
            String name2 = f2.getName().toLowerCase();
            if (name1.equals(GitExecutableDetector.UNIX_EXECUTABLE)) {
                return name2.equals(GitExecutableDetector.UNIX_EXECUTABLE) ? VersionDirsComparator.fallback(f1, f2) : 1;
            }
            if (name2.equals(GitExecutableDetector.UNIX_EXECUTABLE)) {
                return -1;
            }
            Pattern GIT_WITH_VERSION = Pattern.compile("^git[ _]*([\\d\\.]*).*$");
            Matcher m1 = GIT_WITH_VERSION.matcher(name1);
            Matcher m2 = GIT_WITH_VERSION.matcher(name2);
            if (m1.matches() && m2.matches()) {
                GitVersion v1 = VersionDirsComparator.parseGitVersion(m1.group(1));
                GitVersion v2 = VersionDirsComparator.parseGitVersion(m2.group(1));
                if (v1 == null || v2 == null) {
                    return VersionDirsComparator.fallback(f1, f2);
                }
                int compareVersions = v1.compareTo(v2);
                return compareVersions == 0 ? VersionDirsComparator.fallback(f1, f2) : compareVersions;
            }
            return VersionDirsComparator.fallback(f1, f2);
        }

        private static int fallback(@NotNull File f1, @NotNull File f2) {
            if (f1 == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "f1", "git4idea/config/GitExecutableDetector$VersionDirsComparator", "fallback"));
            }
            if (f2 == null) {
                throw new IllegalArgumentException(String.format("Argument for @NotNull parameter '%s' of %s.%s must not be null", "f2", "git4idea/config/GitExecutableDetector$VersionDirsComparator", "fallback"));
            }
            int compareParents = f1.getParentFile().getName().compareTo(f2.getParentFile().getName());
            if (compareParents != 0) {
                return -compareParents;
            }
            String name1 = f1.getName().toLowerCase();
            String name2 = f2.getName().toLowerCase();
            return name1.compareTo(name2);
        }

        @Nullable
        private static GitVersion parseGitVersion(@Nullable String name) {
            if (name == null) {
                return null;
            }
            Pattern VERSION = Pattern.compile("(\\d+)(?:\\.(\\d+))?(?:\\.(\\d+))?(?:\\.(\\d+))?.*");
            Matcher m = VERSION.matcher(name);
            if (!m.matches()) {
                return null;
            }
            try {
                int major = Integer.parseInt(m.group(1));
                return new GitVersion(major, VersionDirsComparator.parseOrNull(m.group(2)), VersionDirsComparator.parseOrNull(m.group(3)), VersionDirsComparator.parseOrNull(m.group(4)));
            }
            catch (NumberFormatException e) {
                LOG.info("Unexpected NFE when parsing [" + name + "]", (Throwable)e);
                return null;
            }
        }

        private static int parseOrNull(String group) {
            return group == null ? 0 : Integer.parseInt(group);
        }
    }
}

