'use strict';
/* global ApplicationSource */
/* global Bookmark */
/* global BookmarkSource */
/* global Collection */
/* global CollectionSource */
/* global dispatchEvent */
/* global Divider */
/* global configurator */

(function(exports) {

  const DB_VERSION = 1;

  const DB_NAME = 'verticalhome';

  const DB_ITEM_STORE = 'items';
  const DB_SV_APP_STORE_NAME = 'svAppsInstalled';

  var db;

  function sort(entries, order) {

    if (!order || !order.length) {
      return entries;
    }

    var newEntries = [];
    function isEqual(lookFor, compareWith) {
      if (!lookFor || !lookFor.manifestURL ||
          !compareWith.detail || !compareWith.detail.manifestURL) {
        return false;
      }
      if (compareWith.detail.entryPoint) {
        return lookFor.manifestURL === compareWith.detail.manifestURL &&
               lookFor.entry_point === compareWith.detail.entryPoint;
      } else {
        return lookFor.manifestURL === compareWith.detail.manifestURL;
      }
    }

    for (var i = 0, iLen = order.length; i < iLen; i++) {
      // Add all entries of current section
      for (var j = 0, jLen = order[i].length; j < jLen; j++) {
        var ind = entries.findIndex(
                           isEqual.bind(null, order[i][j]));
        if (ind >= 0) {
          newEntries.push(entries.splice(ind,1)[0]);
        }
      }
      // If we have more sections add a divider
      if (i < iLen - 1) {
        newEntries.push(new Divider());
      }
    }
    // If entries is not empty yet add orderless entries
    if (entries.length > 0) {
        newEntries.push(new Divider());
        newEntries = newEntries.concat(entries);
    }
    for (i = 0, iLen = newEntries.length; i < iLen; i++) {
      if (newEntries[i].detail) {
        newEntries[i].detail.index = i;
      }
    }
   return newEntries;
  }

  function loadTable(table, indexName, iterator, aNext) {
    newTxn(table, 'readonly', function(txn, store) {
      var index = store.index(indexName);
      index.openCursor().onsuccess = function onsuccess(event) {
        var cursor = event.target.result;
        if (!cursor) {
          return;
        }
        iterator(cursor.value);
        cursor.continue();
      };
    }, aNext);
  }

  function newTxn(storeName, txnType, withTxnAndStore, successCb) {
    var txn = db.transaction([storeName], txnType);
    var store = txn.objectStore(storeName);

    txn.oncomplete = function(event) {
      if (successCb) {
        successCb(event);
      }
    };

    txn.onerror = function(event) {
      console.warn('Error during transaction.');
    };

    withTxnAndStore(txn, store);
  }

  function ItemStore() {
    var self = this;
    this.applicationSource = new ApplicationSource(this);
    this.bookmarkSource = new BookmarkSource(this);
    this.collectionSource = new CollectionSource(this);

    this.sources = [this.applicationSource, this.bookmarkSource,
                    this.collectionSource];

    this.ready = false;

    var isEmpty = false;
    self.gridOrder = null;

    var request = window.indexedDB.open(DB_NAME, DB_VERSION);

    request.onupgradeneeded = function _onupgradeneeded(event) {
      var db = event.target.result;

      var oldVersion = event.oldVersion || 0;
      switch (oldVersion) {
        case 0:
          // Create the item store
          var objectStore = db.createObjectStore(DB_ITEM_STORE,
            { keyPath: 'index'});

          objectStore.createIndex('index', 'index', { unique: true });
          isEmpty = true;
          self.gridOrder = configurator.getGrid();
          var objectSV = db.createObjectStore(DB_SV_APP_STORE_NAME,
            { keyPath: 'manifestURL' });
          objectSV.createIndex('indexSV', 'indexSV', { unique: true });
      }
    };

    request.onsuccess = function _onsuccess() {
      db = request.result;

      if (isEmpty) {
        self.populate(
          self.fetch.bind(self, self.synchronize.bind(self)));
      } else {
        self.initSources(
          self.fetch.bind(self, self.synchronize.bind(self)));
      }
    };
  }

  ItemStore.prototype = {

    /**
     * A list of all items. These are item objects (App, Bookmark, Divider)
     */
    _allItems: [],

    /**
     * Maintains the current index of the last grid item.
     */
    nextPosition: 0,

    /**
     * Fetches a list of all items in the store.
     */
    all: function(success) {
      if (!this.ready) {
        window.addEventListener('databaseready', this.all.bind(this, success));
        return;
      }

      success(this._allItems);
    },

    saveTable: function(table, objArr, column, checkPersist, aNext) {
      newTxn(table, 'readwrite', function(txn, store) {
        store.clear();
        for (var i = 0, iLen = objArr.length; i < iLen; i++) {
          if (!checkPersist || (checkPersist && objArr[i].persistToDB)) {
            store.put(column?objArr[i][column]:objArr[i]);
          }
        }
        if (typeof aNext === 'function') {
          aNext();
        }
      });
    },

    /**
     * Saves all icons to the database.
     */
    save: function(entries, aNext) {
      entries = sort(entries, this.gridOrder);
      this.gridOrder = null;
      // The initial config is simply the list of apps
      this.saveTable(DB_ITEM_STORE, entries, 'detail', true, aNext);
    },

    /**
     * Save reference to SingleVariant app previously installed
     */
    savePrevInstalledSvApp: function(svApps, aNext) {
      this.saveTable(DB_SV_APP_STORE_NAME, svApps, null, false, aNext);
    },

    /**
     * Fetches items from the database.
     * @param {Function} callback A function to call after fetching all items.
     */
    fetch: function(callback) {
      var collected = [];

      function iterator(value) {
        collected.push(value);
      }

      function iteratorSV(value) {
        /* jshint validthis: true */
        this.applicationSource.addPreviouslyInstalledSvApp(value.manifestURL);
      }

      loadTable(DB_SV_APP_STORE_NAME, 'indexSV', iteratorSV.bind(this));
      loadTable(DB_ITEM_STORE, 'index', iterator, finish.bind(this));

      function finish() {
        /* jshint validthis: true */
        // Transforms DB results into item classes
        for (var i = 0, iLen = collected.length; i < iLen; i++) {
          var thisItem = collected[i];
          if (thisItem.type === 'app') {
            var itemObj = this.applicationSource.mapToApp(thisItem);
            this._allItems.push(itemObj);
          } else if (thisItem.type === 'divider') {
            var divider = new Divider(thisItem);
            this._allItems.push(divider);
          } else if (thisItem.type === 'bookmark') {
            var bookmark = new Bookmark(thisItem);
            this._allItems.push(bookmark);
          } else if (thisItem.type === 'collection') {
            var collection = new Collection(thisItem);
            this._allItems.push(collection);
          }
        }

        this.notifyReady();

        if (callback && typeof callback === 'function') {
          callback();
        }
      }
    },

    /**
     * We have fetched data from our local database and displayed it,
     * but data inside of our application or bookmark store may be outdated.
     * We need to synchronize each source and delete/add records.
     */
    synchronize: function() {
      this.sources.forEach(function eachSource(source) {
        source.synchronize();
      });
    },

    /**
     * Initializes all sources.
     * @param {Function} callback The callback to fire after all sources init.
     */
    initSources: function(callback) {
      var pending = this.sources.length;

      var allEntries = [];
      var self = this;

      var current = 0;
      function handleSource() {
        var source = self.sources[current];
        current++;
        source.populate(next);
      }
      handleSource();

      function next(entries) {
        allEntries = allEntries.concat(entries);
        if (!(--pending)) {
          callback(allEntries);
        } else {
          handleSource();
        }
      }
    },

    /**
     * Populates the database with the initial data.
     * @param {Function} callback Callback after database is populated.
     */
    populate: function(callback) {
      this.initSources(function(entries) {
        this.save(entries, callback);
      }.bind(this));
    },

    /**
     * Notifies consumers that the database is ready for queries to be makde.
     */
    notifyReady: function() {
      this.ready = true;
      dispatchEvent(new CustomEvent('databaseready'));
    },

    /**
     * Gets the next available position in the grid
     */
    getNextPosition: function() {
      var nextPosition = this.nextPosition;
      this.nextPosition++;
      return nextPosition;
    }

  };

  exports.ItemStore = ItemStore;

}(window));
