//
//  ECStandard.h
//  ECCdmaMgr
//
//  Created by gdcn on 11-10-21.
//  Copyright 2011年 CDC. All rights reserved.
//
/**
 * @file    ECStandard.h
 * @brief   cdma模块接口协议定义
 * @author  liw
 * @date    2011-11-15
 * @version  v1.0
 */

#import <Foundation/Foundation.h>
#import "ECCdmaDef.h"


/**
    设备操作接口协议定义
 */
@protocol ECStandardAT <NSObject>


/**
 判断是否已加载设备
 @returns 有加载返回true，否则返回false
 */
- (bool)IsLoaded;

/**
 判断是否拥有独立的AT指令端口
 @returns 有独立AT指令端口返回true，否则返回false
 */
- (bool)IsHavePCUIDevice;


/**
 尝试卸载设备使用
 @returns 成功返回0，失败返回其他值
 */
- (int)tryUnloadDevice;


/**
 卸载设备使用，不管有没有加载设备
 @returns 成功返回0，失败返回其他值
 */
- (int)unloadDevice;


/**
	获取当前PIN状态
	@param pinStateInfo PIN信息结构体，请参考CT_PIN_INFO定义
    @warning eModePin:CT_MODE_PIN_LOCK表示已有PIN锁定，具体PIN状态参考eStatePin，CT_MODE_PIN_UNLOCK已解锁状态，则表示用户已进行了PIN码难，或没有PIN码
	@returns 0表示成功，其他表示失败
 */
- (int)getPinStateInfo:(CT_PIN_INFO*)pinStateInfo;



/**
 验证PIN码
 @param pinCode 当前PIN码
 @param pPinTimes PIN码重试次数
 @returns 成功返回0，失败返回其他值
 */
- (int)verifyPinWithCode:(NSString*)pinCode TryTimes:(int*)pPinTimes;


/**
 启用或关闭PIN码保护
 @param pinCode 当前PIN码
 @param bEanble 启用或关闭标志，true表示启用，false表示关闭
 @param pPinTimes 剩余PIN码重试次数
 @returns 成功返回0，失败返回其他值
 */
- (int)enablePinWithCode:(NSString*)pinCode State:(bool)bEanble TryTimes:(int*)pPinTimes;


/**
 修改PIN码
 @param oldCode 当前PIN码
 @param newCode 新PIN码
 @param pPinTimes 返回剩余PIN码重试次数
 @returns 成功返回0，失败返回其他值
 */
- (int)moidfyPinWithOldCode:(NSString*)oldCode NewCode:(NSString*)newCode TryTimes:(int*)pPinTimes;


/**
 使用PUK码解锁PIN码
 @param pukCode UIM卡的PUK码
 @param newPinCode 新的PIN码
 @param pPukTimes 剩余PUK码重试次数（失败有效，成功可不理此参数）
 @returns 成功返回0，失败返回非0值
 */
- (int)unlockWithPukCode:(NSString*)pukCode NewPin:(NSString*)newPinCode PukTimes:(int*)pPukTimes;


/**
 获取1X信号量
 @returns 返回1X当前信号量
 */
- (int)get1XSignal;


/**
 获取3G信号量
 @returns 返回3G当前信号量
 */
- (int)getHdrSignal;


/**
 获取当前上网卡模式，现在主要有1X和3G模式
 @returns 返回当前模式
 */
- (CT_SYSTEM_MODE)getPreMode;


/*!
 * @brief SetPreMode
 *
 *           设置模式，现在支持只支持2：1xＣＤＭＡ模式，4：ＨＤＲ3Ｇ模式，8混合模式
 * @params mode: 模式值，请参考_ECCdmaMode定义
 * @return successed return 0, other return other
 */
- (int)setPreMode:(CT_SYSTEM_MODE)mode;

/**
 设备拨号设备帐号和密码，此接口只对华为上网卡有用
 @param account 帐号名
 @param password 用户密码
 @returns 成功返回0，失败返回其他值
 */
- (int)setDialAccount:(NSString*)account Password:(NSString*)password;


/**
 获取Modem设备名
 @returns Modem设备名
 */
- (NSString*)getModemName;

/**
	获取设备型号
	@returns 设备型号
 */
- (NSString*)getDeviceModel;



/**
 使用上网卡设备发送一条短信
 @param toNumber 目标号码
 @param content 短信内容
 @returns 成功返回0，失败返回其他值
 */
- (int)sendMessage:(NSString*)toNumber content:(NSString*)content;


/**
 从指定设备中读取一条短信
 @param memoryType 设备介质类型,请参考_ECTYPE_MEMORY定义 
 @param msgID 短信ID或索引号
 @param msg 返回的短信相关信息
 @warning 返回值为临时创建，需要用户手动释放内存
 @returns 失败返回nil,成功返回短信字典，key包括：
         MsgID:短信ID(NSNumber)
         PhoneNum:发送号码(NSString)
         Content:短信内容(NSString)
         RecvTime:接收时间(NSString)
         Format:短信编码格式(NSNumber)
 */
- (NSDictionary*)readMessage:(CT_TYPE_MEMORY)memoryType MsgID:(int)msgID;

/**
 删除指定设备上的指定短信
 @param memoryType 设备介质类型,请参考_ECTYPE_MEMORY定义 
 @param msgID 短信索引号或短信ID
 @returns 成功返回0，失败返回其他值
 */
- (int)delMessage:(CT_TYPE_MEMORY)memoryType MsgID:(int)msgID;

/**
	获取短信存储空间状态
	@param status 短信存储空间状态结构体，请参考EC_STATUS_SPACE定义
	@returns 成功返回0，失败返回其他值
 */
- (int)getMessageSpaceStatus:(CT_STATUS_SPACE*)status;

/**
	设置拨号时AT指令延时时间，拨号时禁止下发AT指令调用
	@param time 具体的延时时间，为-1时一直等待
	@returns 成功返回0，失败返回其他值
 */
- (int)setCmdDelayTime:(int)time;

/**
 获取IM卡的IMSI号
 @returns 成功返回IMSI,失败返回nil
 */
- (NSString*)getIMSI;

/**
 获取上网卡设备信息
 @param info 设备信息结构体
 @returns 成功返回0，失败返回其他值
 */
- (int)getDeviceInfo:(CT_DEVICE_INFO*)info;
@end