/*
 *    @(#)TextLineNumber.java   10/03/06
 *
 *    Copyright (c) 2010 Jan Weber
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

package org.jw.menage.ui.components;

//~--- JDK imports ------------------------------------------------------------

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import java.util.HashMap;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.border.Border;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.MatteBorder;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.Element;
import javax.swing.text.JTextComponent;
import javax.swing.text.StyleConstants;
import javax.swing.text.Utilities;

/**
 * Class offering a Panel with line numbers for a JTextComponent as presented at http://tips4java.wordpress.com/2009/05/23/text-component-line-number/
 *
 *
 * @version        1.0, 10/03/06
 * @author         Jan Weber
 */
public class TextLineNumber extends JPanel implements CaretListener, DocumentListener, PropertyChangeListener
{

    public final static float CENTER = 0.5f;
    public final static float LEFT = 0.0f;
    public final static float RIGHT = 1.0f;
    private final static Border OUTER = new MatteBorder (0, 0, 0, 2, Color.GRAY);
    private final static int HEIGHT = Integer.MAX_VALUE - 1000000;
    private int borderGap;
    private JTextComponent component;
    private Color currentLineForeground;
    private float digitAlignment;
    private HashMap<String, FontMetrics> fonts;
    private int lastDigits;
    private int lastHeight;
    private int lastLine;
    private int minimumDisplayDigits;
    private boolean updateFont;

    /**
     *  Create a line number component for a text component. This minimum
     *  display width will be based on 3 digits.
     *
     *  @param component  the related text component
     */
    public TextLineNumber (JTextComponent component)
    {
        this (component, 3);
    }

    /**
     *  Create a line number component for a text component.
     *
     *  @param component  the related text component
     *  @param minimumDisplayDigits  the number of digits used to calculate
     *                               the minimum width of the component
     */
    public TextLineNumber (JTextComponent component, int minimumDisplayDigits)
    {
        this.component = component;
        setFont (component.getFont ());
        setBorderGap (5);
        setCurrentLineForeground (Color.RED);
        setDigitAlignment (RIGHT);
        setMinimumDisplayDigits (minimumDisplayDigits);
        component.getDocument ().addDocumentListener (this);
        component.addCaretListener (this);
        component.addPropertyChangeListener ("font", this);
    }

    /**
     * Method to redraw line numbers when carret changes occur
     *
     *
     * @param e
     */
    @Override
    public void caretUpdate (CaretEvent e)
    {

        int     caretPosition = component.getCaretPosition ();
        Element root          = component.getDocument ().getDefaultRootElement ();
        int     currentLine   = root.getElementIndex (caretPosition);

        if (lastLine != currentLine)
        {
            repaint ();
            lastLine = currentLine;
        }
    }

    /**
     * Method to act on changes and draw new line numbers
     *
     *
     * @param e
     */
    @Override
    public void changedUpdate (DocumentEvent e)
    {
        documentChanged ();
    }

    /**
     *  Gets the border gap
     *
     *  @return the border gap in pixels
     */
    public int getBorderGap ()
    {
        return borderGap;
    }

    /**
     *  Gets the current line rendering Color
     *
     *  @return the Color used to render the current line number
     */
    public Color getCurrentLineForeground ()
    {
        return (currentLineForeground == null)
               ? getForeground ()
               : currentLineForeground;
    }

    /**
     *  Gets the digit alignment
     *
     *  @return the alignment of the painted digits
     */
    public float getDigitAlignment ()
    {
        return digitAlignment;
    }

    /**
     *  Gets the minimum display digits
     *
     *  @return the minimum display digits
     */
    public int getMinimumDisplayDigits ()
    {
        return minimumDisplayDigits;
    }

    /**
     *  Gets the update font property
     *
     *  @return the update font property
     */
    public boolean getUpdateFont ()
    {
        return updateFont;
    }

    /**
     * Method to act on document changes and draw new line numbers
     *
     *
     * @param e
     */
    @Override
    public void insertUpdate (DocumentEvent e)
    {
        documentChanged ();
    }

    /**
     *  Draw the line numbers
     *
     * @param g
     */
    @Override
    public void paintComponent (Graphics g)
    {
        super.paintComponent (g);

        FontMetrics fontMetrics    = component.getFontMetrics (component.getFont ());
        Insets      insets         = getInsets ();
        int         availableWidth = getSize ().width - insets.left - insets.right;

        Rectangle clip           = g.getClipBounds ();
        int       rowStartOffset = component.viewToModel (new Point (0, clip.y));
        int       endOffset      = component.viewToModel (new Point (0, clip.y + clip.height));

        while (rowStartOffset <= endOffset)
        {
            try
            {
                if (isCurrentLine (rowStartOffset))
                {
                    g.setColor (getCurrentLineForeground ());
                }
                else
                {
                    g.setColor (getForeground ());
                }

                String lineNumber  = getTextLineNumber (rowStartOffset);
                int    stringWidth = fontMetrics.stringWidth (lineNumber);
                int    x           = getOffsetX (availableWidth, stringWidth) + insets.left;
                int    y           = getOffsetY (rowStartOffset, fontMetrics);

                g.drawString (lineNumber, x, y);

                rowStartOffset = Utilities.getRowEnd (component, rowStartOffset) + 1;
            }
            catch (Exception e) {}
        }
    }

    /**
     * Method to act on font changes
     *
     *
     * @param evt
     */
    @Override
    public void propertyChange (PropertyChangeEvent evt)
    {
        if (evt.getNewValue () instanceof Font)
        {
            if (updateFont)
            {
                Font newFont = (Font) evt.getNewValue ();

                setFont (newFont);
                lastDigits = 0;
                setPreferredWidth ();
            }
            else
            {
                repaint ();
            }
        }
    }

    @Override
    public void removeUpdate (DocumentEvent e)
    {
        documentChanged ();
    }

    /**
     *  The border gap is used in calculating the left and right insets of the
     *  border. Default value is 5.
     *
     *  @param borderGap  the gap in pixels
     */
    public void setBorderGap (int borderGap)
    {
        this.borderGap = borderGap;

        Border inner = new EmptyBorder (0, borderGap, 0, borderGap);

        setBorder (new CompoundBorder (OUTER, inner));
        lastDigits = 0;
        setPreferredWidth ();
    }

    /**
     *  The Color used to render the current line digits. Default is Coolor.RED.
     *
     *  @param currentLineForeground  the Color used to render the current line
     */
    public void setCurrentLineForeground (Color currentLineForeground)
    {
        this.currentLineForeground = currentLineForeground;
    }

    /**
     *  Specify the horizontal alignment of the digits within the component.
     *  Common values would be:
     *  <ul>
     *  <li>TextLineNumber.LEFT
     *  <li>TextLineNumber.CENTER
     *  <li>TextLineNumber.RIGHT (default)
     *  </ul>
     *
     * @param digitAlignment
     */
    public void setDigitAlignment (float digitAlignment)
    {
        this.digitAlignment = (digitAlignment > 1.0f)
                              ? 1.0f
                              : (digitAlignment < 0.0f)
                                ? -1.0f
                                : digitAlignment;
    }

    /**
     *  Specify the mimimum number of digits used to calculate the preferred
     *  width of the component. Default is 3.
     *
     *  @param minimumDisplayDigits  the number digits used in the preferred
     *                               width calculation
     */
    public void setMinimumDisplayDigits (int minimumDisplayDigits)
    {
        this.minimumDisplayDigits = minimumDisplayDigits;
        setPreferredWidth ();
    }

    /**
     *  Set the update font property. Indicates whether this Font should be
     *  updated automatically when the Font of the related text component
     *  is changed.
     *
     *  @param updateFont  when true update the Font and repaint the line
     *                     numbers, otherwise just repaint the line numbers.
     */
    public void setUpdateFont (boolean updateFont)
    {
        this.updateFont = updateFont;
    }


    /**
     * Get the line number to be drawn. The empty string will be returned
     *  when a line of text has wrapped.
     *
     * @param rowStartOffset
     *
     * @return
     */
    protected String getTextLineNumber (int rowStartOffset)
    {
        Element root  = component.getDocument ().getDefaultRootElement ();
        int     index = root.getElementIndex (rowStartOffset);
        Element line  = root.getElement (index);

        if (line.getStartOffset () == rowStartOffset)
        {
            return String.valueOf (index + 1);
        }
        else
        {
            return "";
        }
    }

    private void documentChanged ()
    {

        SwingUtilities.invokeLater (new Runnable ()
        {
            public void run ()
            {
                int preferredHeight = component.getPreferredSize ().height;

                if (lastHeight != preferredHeight)
                {
                    setPreferredWidth ();
                    repaint ();
                    lastHeight = preferredHeight;
                }
            }
        });
    }

    /**
     * Determine the X offset to properly align the line number when drawn
     *
     *
     * @param availableWidth
     * @param stringWidth
     *
     * @return
     */
    private int getOffsetX (int availableWidth, int stringWidth)
    {
        return (int) ((availableWidth - stringWidth) * digitAlignment);
    }

    /**
     * Determine the Y offset for the current row
     *
     *
     * @param rowStartOffset
     * @param fontMetrics
     *
     * @return
     *
     * @throws BadLocationException
     */
    private int getOffsetY (int rowStartOffset, FontMetrics fontMetrics) throws BadLocationException
    {

        Rectangle r          = component.modelToView (rowStartOffset);
        int       lineHeight = fontMetrics.getHeight ();
        int       y          = r.y + r.height;
        int       descent    = 0;


        if (r.height == lineHeight)    
        {
            descent = fontMetrics.getDescent ();
        }
        else                           
        {
            if (fonts == null)
            {
                fonts = new HashMap<String, FontMetrics> ();
            }

            Element root  = component.getDocument ().getDefaultRootElement ();
            int     index = root.getElementIndex (rowStartOffset);
            Element line  = root.getElement (index);

            for (int i = 0; i < line.getElementCount (); i++)
            {
                Element      child      = line.getElement (i);
                AttributeSet as         = child.getAttributes ();
                String       fontFamily = (String) as.getAttribute (StyleConstants.FontFamily);
                Integer      fontSize   = (Integer) as.getAttribute (StyleConstants.FontSize);
                String       key        = fontFamily + fontSize;
                FontMetrics  fm         = fonts.get (key);

                if (fm == null)
                {
                    Font font = new Font (fontFamily, Font.PLAIN, fontSize);

                    fm = component.getFontMetrics (font);
                    fonts.put (key, fm);
                }

                descent = Math.max (descent, fm.getDescent ());
            }
        }

        return y - descent;
    }

    /**
     *  We need to know if the caret is currently positioned on the line we
     *  are about to paint so the line number can be highlighted.
     *
     * @param rowStartOffset
     *
     * @return
     */
    private boolean isCurrentLine (int rowStartOffset)
    {
        int     caretPosition = component.getCaretPosition ();
        Element root          = component.getDocument ().getDefaultRootElement ();

        if (root.getElementIndex (rowStartOffset) == root.getElementIndex (caretPosition))
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     *  Calculate the width needed to display the maximum line number
     */
    private void setPreferredWidth ()
    {
        Element root   = component.getDocument ().getDefaultRootElement ();
        int     lines  = root.getElementCount ();
        int     digits = Math.max (String.valueOf (lines).length (), minimumDisplayDigits);

        if (lastDigits != digits)
        {
            lastDigits = digits;

            FontMetrics fontMetrics    = getFontMetrics (getFont ());
            int         width          = fontMetrics.charWidth ('0') * digits;
            Insets      insets         = getInsets ();
            int         preferredWidth = insets.left + insets.right + width;
            Dimension   d              = getPreferredSize ();

            d.setSize (preferredWidth, HEIGHT);
            setPreferredSize (d);
            setSize (d);
        }
    }
}
