///**********************************************************************************************************************************
///  DKBSPObjectStorage.h
///  DrawKit ©2005-2008 Apptree.net
///
///  Created by graham on 03/01/2009.
///
///	 This software is released subject to licensing conditions as detailed in DRAWKIT-LICENSING.TXT, which must accompany this source file. 
///
///**********************************************************************************************************************************

#import <Cocoa/Cocoa.h>
#import "DKLinearObjectStorage.h"

@class DKBSPIndexTree;

/// node types

typedef enum
{
	kNodeHorizontal,
	kNodeVertical,
	kNodeLeaf
}
DKLeafType;

/// tree operations

typedef enum
{
	kDKOperationInsert,
	kDKOperationDelete,
	kDKOperationAccumulate
}
DKBSPOperation;

/// The actual storage object. This inherits the linear array which actually stores the objects, but maintains a BSP tree in parallel, which
/// stores indexes that refer to this array. Thus the objects' Z-order is strictly maintained by the array as for the linear case, but objects can
/// be extracted very rapidly when performing a spatial query.

@interface DKBSPObjectStorage : DKLinearObjectStorage
{
@private
	DKBSPIndexTree*	mTree;
	NSUInteger		mTreeDepth;
	NSUInteger		mLastItemCount;
}

- (void)			setTreeDepth:(NSUInteger) aDepth;
- (id)				tree;

@end

/// node object - used internally with DKBSPIndexTree. Note members are public as this is frequently accessed
/// internally via a pointer for high performance

@interface DKBSPNode : NSObject
{
@public
	DKLeafType		mType;
	CGFloat			mOffset;
	id				mLeafStorage;
}

@property (nonatomic, assign) DKLeafType	type;			// node type
@property (nonatomic, assign) CGFloat		offset;			// split point in the given dimension
@property (nonatomic, retain) id			storage;		// for a leaf node, the storage object

@end

#pragma mark -

/// tree object
/// this stores indexes in mutable index sets. The indexes refer to the index of the object within the linear array. Given a rect query, this returns an index set which
/// is the indexes of all objects that intersect the rect. Using -objectsAtIndexes: on the linear array then returns the relevant objects sorted by Z-order. The tree only
/// stores the indexes of visible objects, thus it doesn't need to test for visibility - the storage will manage adding and removing indexes as object visibility changes.

/// note that this is equivalent to a binary search in 2 dimensions. The purpose is to weed out as many irrelevant objects as possible in advance of returning them to the
/// client for drawing. Internally it is tuned for speed but it relies heavily on the performance of Cocoa's NSIndexSet class, and -addIndexes: in particular. If these turn
/// out to be slow, this may be detrimental to drawing performance.

/// tree nodes and leaves are all stored in the same array (mNodes) and the actual storage at the leaf is a property of
/// the leaf node. While all nodes have this property, it is nil for non-leaf nodes, and even for leaf nodes it is
/// allocated lazily on object insertion, so may be nil if no objects have been inserted at that position.

@interface DKBSPIndexTree : NSObject
{
@protected
	NSMutableArray*		mNodes;
	NSSize				mCanvasSize;
	NSBezierPath*		mDebugPath;
	NSUInteger			mLeafCount;
	NSUInteger			mFirstLeafIndex;
}

@property (nonatomic, readonly) NSArray* nodes;

+ (Class)			leafClass;

- (id)				initWithCanvasSize:(NSSize) size depth:(NSUInteger) depth;
- (NSSize)			canvasSize;

- (void)			setDepth:(NSUInteger) depth;
- (NSUInteger)		countOfLeaves;

- (void)			insertItemIndex:(NSUInteger) idx withRect:(NSRect) rect;
- (void)			removeItemIndex:(NSUInteger) idx withRect:(NSRect) rect;

- (NSIndexSet*)		itemsIntersectingRects:(const NSRect*) rects count:(NSUInteger) count;
- (NSIndexSet*)		itemsIntersectingRect:(NSRect) rect;
- (NSIndexSet*)		itemsIntersectingPoint:(NSPoint) point;

- (void)			shiftIndexesStartingAtIndex:(NSUInteger) startIndex by:(NSInteger) delta;

- (NSBezierPath*)	debugStorageDivisions;

@end

@interface DKBSPIndexTree (InternalOperations)

// these are implemented by DKBSPIndexTree as private methods, but prototyped here so
// we can make use of them in a subclass

- (void)			recursivelySearchWithRect:(NSRect) rect index:(NSUInteger) indx withObject:(id) obj info:(void*) info;
- (void)			recursivelySearchWithPoint:(NSPoint) pt index:(NSUInteger) indx withObject:(id) obj info:(void*) info;
- (void)			operateOnLeaf:(DKBSPNode*) leaf withObject:(id) obj info:(void*) info;
- (void)			removeObject:(id<DKStorableObject>) obj;

@end

/// info structure used to specify the operation through the recursive searches

typedef struct
{
	DKBSPOperation		operation;
	NSUInteger			opIndex;
}
DKBSPIndexInfo;


#define	kDKBSPSlack				48
#define kDKMinimumDepth			8U
#define kDKMaximumDepth			10U		// set 0 for no limit
