//
//  DKBezierPath.h
//  GCDrawKit
//
//  Created by Graham on 24/11/2013.
//  Copyright (c) 2013 Apptree. All rights reserved.
//

#import <Foundation/Foundation.h>

// each element of the path is described using this struct, which is kept in a simple C array within the object.
// Note that order of the points is not the same as NSBezierPath

typedef struct
{
	CGPathElementType		fType;				// element type
	CGPoint					fPoints[3];			// associated points for type
	CGFloat					fLen;				// calculated length of this element, or zero
}
DKBezierElement, *DKBezierElementPtr;

// for iterating the elements using a block, this type is declared. The return value indicates whether iteration should
// continue

typedef BOOL (^DKBezierElementBlock)( DKBezierElementPtr element, CGPoint* previousPoint );

// this object is used to describe points and slopes at particular positions along a path.

@interface DKBezierSlopePoint : NSObject
{
@private
	CGPoint		mPoint;
	CGFloat		mSlope;
}

@property (nonatomic, assign) CGPoint		point;	// the point on the path
@property (nonatomic, assign) CGFloat		slope;	// the slope at that point, in radians
@property (nonatomic, readonly) CGFloat		normal;	// the normal to the slope, radians + pi/2 (90°)

@end

// for iterating constant length intervals along a path, this block type is declared. The return value indicates whether
// iteration should continue. Using -enumeratePathAtFixedInterval:usingBlock:info: is designed to be very efficient by avoiding
// repeated calculation of points on the path and by caching aggressively.

typedef BOOL (^DKBezierPathIntervalBlock)( DKBezierSlopePoint* sp, CGFloat accumLength );

// for getting flattened points from the recursive length function, we use this block type

typedef void (^DKFlatPointBlock)(CGPoint p);

#pragma mark -

// the path class. This is very similar to NSBezierPath, except it uses CG basic types throughout, and does a lot more caching
// so that calculating lengths and sublengths are much cheaper if done repeatedly.

@interface DKBezierPath : NSObject <NSCoding, NSCopying>
{
@private
	CGRect					mBounds;
	CGRect					mControlPointBounds;
	CGPoint					mSubpathStartingPoint;
	NSUInteger				mElementCount;
	NSUInteger				mElementsArrayLength;
	NSUInteger				mSubpathsCount;
	NSUInteger				mDashCount;
	BOOL					mSubpathOpen;
	BOOL					mIterateFlag;
	CGPathRef				mCGPath;
	CGFloat					mLength;
	CGFloat					mLineWidth;
	CGFloat					mFlatness;
	CGFloat					mMiterLimit;
	CGFloat					mDashPhase;
	CGLineCap				mLineCapStyle;
	CGLineJoin				mLineJoinStyle;
	NSWindingRule			mWindingRule;
	CGFloat*				mLineDash;
	DKBezierElementPtr		mElements;
	NSMutableDictionary*	mFixedIntervalsCache;
}

@property (nonatomic, readonly) NSUInteger		elementCount;				// number of elements in the path, 0 if empty
@property (nonatomic, readonly) CGRect			bounds;						// bounding rect of the path only, excluding control points and stroking parameters
@property (nonatomic, readonly) CGRect			controlPointBounds;			// bounding rect of the path's control points
@property (nonatomic, readonly) CGPathRef		CGPath;						// a core graphics path equivalent to the path
@property (nonatomic, readonly) BOOL			isEmpty;					// returns YES if the path contains no elements
@property (nonatomic, readonly) CGFloat			length;						// returns the overall length of the path in points
@property (nonatomic, assign) CGFloat			lineWidth;					// stroke line width
@property (nonatomic, assign) CGFloat			flatness;					// flatness of the path
@property (nonatomic, assign) CGFloat			miterLimit;					// stroke miter limit
@property (nonatomic, assign) CGLineJoin		lineJoinStyle;				// stroke line join style
@property (nonatomic, assign) CGLineCap			lineCapStyle;				// stroke line cap style
@property (nonatomic, assign) NSWindingRule		windingRule;				// winding rule for fills & clips

+ (DKBezierPath*)	bezierPath;
+ (DKBezierPath*)	bezierPathWithRect:(CGRect) rect;
+ (DKBezierPath*)	bezierPathWithOvalInRect:(CGRect) rect;
+ (DKBezierPath*)	bezierPathWithRoundedRect:(CGRect) rect xRadius:(CGFloat) rx yRadius:(CGFloat) ry;

- (instancetype)	initWithCGPath:(CGPathRef) path;
- (instancetype)	initWithRect:(CGRect) rect;

- (void)			moveToPoint:(CGPoint) point;
- (void)			lineToPoint:(CGPoint) point;
- (void)			curveToPoint:(CGPoint) endPoint controlPoint1:(CGPoint) controlPoint1 controlPoint2:(CGPoint) controlPoint2;
- (void)			closePath;

- (void)			relativeMoveToPoint:(CGPoint) point;
- (void)			relativeLineToPoint:(CGPoint) point;
- (void)			relativeCurveToPoint:(CGPoint) endPoint controlPoint1:(CGPoint) controlPoint1 controlPoint2:(CGPoint) controlPoint2;

- (void)			appendBezierPath:(DKBezierPath*) path;
- (void)			addBezierPath:(DKBezierPath*) path;

- (CGPoint)			currentPoint;
- (void)			removeAllPoints;

- (void)			setLineDash:(CGFloat*) pattern count:(NSUInteger) count phase:(CGFloat) phase;
- (void)			getLineDash:(CGFloat*) pattern count:(NSUInteger*) count phase:(CGFloat*) phase;

- (CGPathElementType) elementAtIndex:(NSUInteger) index;
- (CGPathElementType) elementAtIndex:(NSUInteger) index associatedPoints:(CGPoint*) points;
- (void)			setAssociatedPoints:(CGPoint*) points atIndex:(NSUInteger) index;

- (DKBezierElementPtr)	bezierElementAtIndex:(NSUInteger) index;
- (void)				setBezierElement:(DKBezierElement) element atIndex:(NSUInteger) index;

- (void)			enumerateElementsUsingBlock:(DKBezierElementBlock) block;
- (void)			enumeratePathAtFixedInterval:(CGFloat) interval usingBlock:(DKBezierPathIntervalBlock) block;
- (void)			enumeratePathAtIntervals:(CGFloat*) intervals count:(NSUInteger) count usingBlock:(DKBezierPathIntervalBlock) block;

- (void)			stroke;
- (void)			fill;
- (void)			addClip;

@end

#pragma mark -

@interface DKBezierPath (DKBezierPathUtilities)

- (DKBezierSlopePoint*) pointAndSlopeAtPathStart;
- (DKBezierSlopePoint*) pointAndSlopeAtPathEnd;
- (DKBezierSlopePoint*)	pointAndSlopeAtLength:(CGFloat) length;

- (DKBezierPath*)	bezierPathToLength:(CGFloat) length;
- (DKBezierPath*)	bezierPathFromLength:(CGFloat) length;
- (DKBezierPath*)	bezierPathFromLength:(CGFloat) length1 toLength:(CGFloat) length2;

- (NSUInteger)		countOfSubpaths;
- (NSArray*)		subpaths;

- (DKBezierPath*)	bezierPathByFlatteningPath;
- (DKBezierPath*)	bezierPathByReversingPath;
- (DKBezierPath*)	bezierPathByApplyingTransform:(CGAffineTransform) tfm;
- (DKBezierPath*)	bezierPathByStrokingPath;

+ (DKBezierPath*)	bezierPathWithNSBezierPath:(NSBezierPath*) nspath;

@end

#pragma mark -


#define DK_PTR_FREE( p )	if(p){free(p);(p)= NULL;}

// array indexes for control points in DKBezierElement. NOTE: different from NSBezierPath

#define DK_END_POINT			0
#define DK_CONTROL_POINT_1		1
#define DK_CONTROL_POINT_2		2

// default flatness

#define DK_DEFAULT_FLATNESS		0.6

/*

This is intended to become a replacement for NSBezierPath in DK, where it is a) portable and b) more performant,
 due to more aggressive caching of internal values such as bounds, etc. It calls Core Graphics functions to
 do drawing.


*/

