//
//  DKQuadTreeStorage.h
//  GCDrawKit
//
//  Created by Graham Cox on 4/07/12.
//  Copyright (c) 2012 Apptree. All rights reserved.
//

#import "DKLinearObjectStorage.h"


@class DKQuadTreeNode;



@interface DKQuadTreeStorage : DKLinearObjectStorage
{
@private
	DKQuadTreeNode*		mRoot;
	NSSize				mCanvasSize;
}

@end

#pragma mark -


typedef enum
{
	kDKQuadTreeNodeNW		= 0,
	kDKQuadTreeNodeNE		= 1,
	kDKQuadTreeNodeSW		= 2,
	kDKQuadTreeNodeSE		= 3
}
DKQuadTreeNodeIndex;




@interface DKQuadTreeNode : NSObject
{
@private
	NSMutableSet*		mObjects;			// stored objects at this node level, if any (may be nil or empty)
	DKQuadTreeNode*		mParent;			// the node's parent, weak ref
	DKQuadTreeNode*		mChildren[4];		// the four child nodes, may be nil. node owns its children
	DKQuadTreeNodeIndex	mIndexInParent;		// whether this node is NE, NW, SE or SW in parent node
	CGFloat				mWidth;				// the width of the node
	CGFloat				mHeight;			// the height of the node
	CGFloat				mHSplit;			// absolute horizontal split position
	CGFloat				mVSplit;			// absolute vertical split position
	NSUInteger			mLevel;				// node depth in the tree
}

@property (assign, nonatomic) DKQuadTreeNode*		parent;
@property (assign, nonatomic) DKQuadTreeNodeIndex	indexInParent;
@property (readonly) NSSet*							children;
@property (retain, nonatomic) NSSet*				objects;
@property (readonly) CGFloat						horizontalSplit;
@property (readonly) CGFloat						verticalSplit;
@property (readonly) NSUInteger						nodeIdentifier;
@property (assign, nonatomic) CGFloat				width;
@property (assign, nonatomic) CGFloat				height;
@property (assign, nonatomic) NSUInteger			level;

+ (DKQuadTreeNode*)		rootNodeForSize:(NSSize) size;

// spatial queries invoked on root only

- (NSSet*)				objectsInRects:(const NSRect[]) rects count:(NSUInteger) count;
- (NSSet*)				objectsInRect:(NSRect) rect;
- (NSSet*)				objectsContainingPoint:(NSPoint) point;

// object insertion and deletion invoked on root only

- (void)				insertObject:(id<DKStorableObject>) object;
- (void)				removeObject:(id<DKStorableObject>) object;
- (void)				removeObjects:(NSSet*) objects;
- (void)				object:(id<DKStorableObject>) object didChangeBoundsFrom:(NSRect) oldBounds;
- (void)				bulkLoadObjects:(NSArray*) objs;

// all nodes:

- (NSUInteger)			count;
- (BOOL)				isEmpty;
- (NSUInteger)			countOfSubnodes;
- (void)				removeFromSupernode;
- (void)				prune;

// designated initializer:

- (id)					initWithParent:(DKQuadTreeNode*) parent index:(DKQuadTreeNodeIndex) index;

// debugging:

- (NSBezierPath*)		debugPath;

@end



/*

Implements a quad-tree storage data structure conforming to DKObjectStorage.
 
 Each node in the tree divides the space into four parts. If an object straddles the split in either dimension, it lives in the node, otherwise in the appropriate child node.

Unlike the current BSP implementation, this tree structure is grown dynamically and is only as deep and wide as needed for the objects inserted. An object only exists as a member
 of exactly one node, unlike BSP, so memory management is simpler.

This should perform nearly as well as BSP but is a much simpler implementation.
 
 One drawback of this storage is that if objects happen to straddle a split at a high level, even if they are very small, they'll be included in a search even though they
 may be a long way outside the query rectangle. This adds some overhead to draw the objects compared with BSP.



*/