///**********************************************************************************************************************************
///  DKRastGroup.h
///  DrawKit ©2005-2008 Apptree.net
///
///  Created by graham on 17/03/2007.
///
///	 This software is released subject to licensing conditions as detailed in DRAWKIT-LICENSING.TXT, which must accompany this source file. 
///
///**********************************************************************************************************************************

#import "DKRasterizer.h"


@class DKRastGroup;



@interface DKRastGroup : DKRasterizer <NSCoding, NSCopying>
{
@private
	NSMutableArray*		m_renderList;
}

- (void)				setRenderList:(NSArray*) list;
- (NSArray*)			renderList;

- (DKRastGroup*)		root;

- (void)				observableWasAdded:(GCObservableObject*) observable;
- (void)				observableWillBeRemoved:(GCObservableObject*) observable;
	
- (void)				addRenderer:(DKRasterizer*) renderer;
- (void)				removeRenderer:(DKRasterizer*) renderer;
- (void)				moveRendererAtIndex:(NSUInteger) src toIndex:(NSUInteger) dest;
- (void)				insertRenderer:(DKRasterizer*) renderer atIndex:(NSUInteger) index;
- (void)				removeRendererAtIndex:(NSUInteger) index;
- (NSUInteger)			indexOfRenderer:(DKRasterizer*) renderer;

- (DKRasterizer*)		rendererAtIndex:(NSUInteger) index;
- (DKRasterizer*)		rendererWithName:(NSString*) name;

- (NSUInteger)			countOfRenderList;
- (BOOL)				containsRendererOfClass:(Class) cl;
- (NSArray*)			renderersOfClass:(Class) cl;
- (NSArray*)			nonSubstitutableRenderers;
	
- (BOOL)				isValid;

- (void)				removeAllRenderers;
- (void)				removeRenderersOfClass:(Class) cl inSubgroups:(BOOL) subs;

// KVO compliant variants of the render list management methods, key = "renderList"
// all other list management methods eventually call these.

- (id)					objectInRenderListAtIndex:(NSUInteger) indx;
- (void)				insertObject:(id) obj inRenderListAtIndex:(NSUInteger) index;
- (void)				removeObjectFromRenderListAtIndex:(NSUInteger) indx;

- (NSUndoManager*)		undoManager;

// for special use only - add a renderer without any notification, even KVO:

- (void)				addRendererWithoutNotifying:(DKRasterizer*) rast;
- (void)				removeRendererWithoutNotifying:(DKRasterizer*) rast;

@end


/*

A rendergroup is a single renderer which contains a list of other renderers. Each renderer is applied to the object
in list order.

Because the group is itself a renderer, it can be added to other groups, etc to form complex trees of rendering
behaviour.

A group saves and restores the graphics state around all of its calls, so can also be used to "bracket" sets of
rendering operations together.

The rendering group is the basis for the more application-useful drawing style object.

Because DKRasterizer inherits from GCObservableObject, the group object supports a KVO-based approach for observing its
components. Whenever a component is added or removed from a group, the root object (typically a style) is informed through 
the observableWasAdded: observableWillBeRemoved: methods. If the root object is indeed interested in observing the object,
it should call its setUpKVOForObserver and tearDownKVOForObserver methods. Groups propagate these messages down the tree
as well, so the root object is given the opportunity to observe any component anywhere in the tree. Additionally, groups
themselves are observed for changes to their lists, so the root object is able to track changes to the group structure
as well.
 
 Update 1.9:
 
 The KVO mechanism described above does not scale well when a model has hundreds or more styles. The set up/tear down takes
 significant time especially during dearchiving. So for INTERNAL notifications between the rasterizer and the style that owns it,
 a direct means is used, the DKRasterizerObserver protocol. Each property setter now invokes the willChange/didChange methods on
 the observer directly, requiring no setup or teardown other than setting the observer pointer. The speed up on large files
 is dramatic. KVO is still used for EXTERNAL observations, such as for a UI to change the parameters of a rasterizer.
 
 In addition, a rastGroup now manages the undo of changes to its renderList directly, rather than relying on the owning style
 to do it. This improves undo reliability. As yet individual rasterizers don't do the same however. The -undoManager method
 just calls the same method on the root object, so it requires that the style has an undoManager assigned.


*/
