///**********************************************************************************************************************************
///  DKRasterizer.h
///  DrawKit �2005-2008 Apptree.net
///
///  Created by graham on 23/11/2006.
///
///	 This software is released subject to licensing conditions as detailed in DRAWKIT-LICENSING.TXT, which must accompany this source file. 
///
///**********************************************************************************************************************************

#import "DKRasterizerProtocol.h"
#import "GCObservableObject.h"


@class DKRasterizer, DKRastGroup;

// a rasterizer observer can get change notifications directly if it implements this protocol. This is used to replace KVO
// for observing changes within a style, because setting up and tearing down the KVO is a performance problem for files.
// However KVO is still required for editing a style using e.g. style inspector.

@protocol DKRasterizerObserver <NSObject>

- (void)	rasterizerWillChange:(DKRasterizer*) rast keyPath:(NSString*) kp;
- (void)	rasterizerDidChange:(DKRasterizer*) rast keyPath:(NSString*) kp;

@end


// clipping values:


typedef enum
{
	kDKClippingNone			= 0,
	kDKClipOutsidePath		= 1,
	kDKClipInsidePath		= 2
}
DKClippingOption;

/// transformations are moving to DKRepeatTransformGroup. Set this to 0 to remove all old code and ivars
/// related to the previous support for transformations. Note that this does not remove compatibility code that
/// migrates older rasterizers to the group approach.

#define DK_HAS_OLD_STYLE_TRANSFORMATIONS		0


@interface DKRasterizer : GCObservableObject <DKRasterizer, NSCoding, NSCopying>
{
@private
	DKRastGroup*		mContainerRef;		// group that contains this
	id<DKRasterizerObserver> mRastObserver;	// an observer of property changes (typically the owning style)
	NSString*			mDescription;		// description
	BOOL				m_enabled;			// YES if actually drawn
	DKClippingOption	mClipping;			// set path clipping to this
	CGBlendMode			mBlendMode;			// blend mode
	CGFloat				mOpacity;			// opacity
	BOOL				mOpacityEnabled;	// YES to apply opacity/blend setting

#if DK_HAS_OLD_STYLE_TRANSFORMATIONS
	NSUInteger			mRepeats;			// number of repeats of effect
	CGFloat				mDX, mDY;			// linear offset per repeat
	CGFloat				mSX, mSY;			// linear scale per repeat
	CGFloat				mRepRotation;		// rotation per repeat
	BOOL				mRepAngleRelative;	// YES if rep angle is relative to object
	BOOL				mRepeatRandom;		// randomise offset, scale and angle per repeat
	BOOL				mRepeatReverse;		// if YES, repeats are executed in reverse order
	NSColor*			mRepeatBlendColour;	// if non-nil, transform effect can blend to this colour
#endif
@protected
	NSString*			m_name;				// optional name
}

@property(nonatomic, copy) NSString*					componentDescription;
@property(nonatomic, assign) id<DKRasterizerObserver>	rastObserver;

+ (DKRasterizer*)	rasterizerFromPasteboard:(NSPasteboard*) pb;
+ (BOOL)			allowSubstitutionForHitTesting;

- (DKRastGroup*)	container;
- (void)			setContainer:(DKRastGroup*) container;

- (void)			setName:(NSString*) name;
- (NSString*)		name;

- (BOOL)			isValid;
- (BOOL)			isVisibleFill;

- (void)			setEnabled:(BOOL) enable;
- (BOOL)			enabled;

- (void)			setClipping:(DKClippingOption) clipping;
- (void)			setClippingWithoutNotifying:(DKClippingOption) clipping;
- (DKClippingOption) clipping;

- (void)			setBlendMode:(CGBlendMode) bmode;
- (CGBlendMode)		blendMode;

- (void)			setOpacity:(CGFloat) opacity;
- (CGFloat)			opacity;

- (void)			setOpacityEnabled:(BOOL) eo;
- (BOOL)			opacityEnabled;

- (NSBezierPath*)	pathForObject:(id<DKRenderable>) object;

- (void)			setTemporaryColour:(NSColor*) colour;

- (BOOL)			copyToPasteboard:(NSPasteboard*) pb;

@end

// deprecated:

#define DK_RASTERIZER_DEPRECATED		1

#if DK_RASTERIZER_DEPRECATED

@interface DKRasterizer (Deprecated)

- (NSString*)		label;

#if DK_HAS_OLD_STYLE_TRANSFORMATIONS
- (NSBezierPath*)	renderingPathForObject:(id<DKRenderable>) object repeatNumber:(NSUInteger) rep;

- (void)			setRepeats:(NSUInteger) repeats;
- (NSUInteger)		repeats;

- (void)			setRepeatRandom:(BOOL) repRandom;
- (BOOL)			repeatRandom;

- (void)			setRepeatReverse:(BOOL) rev;
- (BOOL)			repeatReverse;

- (void)			setRepeatBlendColour:(NSColor*) repBlend;
- (NSColor*)		repeatBlendColour;
- (NSColor*)		blendedColourFromColour:(NSColor*) inColour repeatNumber:(NSUInteger) n;

- (void)			setRepeatXOffset:(CGFloat) repeatX;
- (CGFloat)			repeatXOffset;
- (void)			setRepeatYOffset:(CGFloat) repeatY;
- (CGFloat)			repeatYOffset;

- (void)			setRepeatXScale:(CGFloat) repXScale;
- (CGFloat)			repeatXScale;
- (void)			setRepeatYScale:(CGFloat) repYScale;
- (CGFloat)			repeatYScale;

- (void)			setRepeatRotation:(CGFloat) angleInRadians;
- (CGFloat)			repeatRotation;
- (void)			setRepeatRotationInDegrees:(CGFloat) degrees;
- (CGFloat)			repeatRotationInDegrees;

- (void)			setRepeatRotationIsRelative:(BOOL) rel;
- (BOOL)			repeatRotationIsRelative;
#endif

@end


#endif


extern NSString*	kDKRasterizerPasteboardType;

extern NSString*	kDKRasterizerPropertyWillChange;
extern NSString*	kDKRasterizerPropertyDidChange;
extern NSString*	kDKRasterizerChangedPropertyKey;


/*
 DKRasterizer is an abstract base class that implements the DKRasterizer protocol. Concrete subclasses
 include DKStroke, DKFill, DKHatching, DKFillPattern, DKGradient, etc.
 
 A renderer is given an object and renders it according to its behaviour to the current context. It can
 do whatever it wants. Normally it will act upon the object's path so as a convenience the renderPath method
 is called by default. Subclasses can override at the object or the path level, as they wish.
 
 Renderers are obliged to accurately return the extra space they need to perform their rendering, over and
 above the bounds of the path. For example a standard stroke is aligned on the path, so the extra space should
 be half of the stroke width in both width and height. This additional space is used to compute the correct bounds
 of a shape when a set of rendering operations is applied to it.

*/


@interface NSObject (DKRendererDelegate)

- (NSBezierPath*)	renderer:(DKRasterizer*) aRenderer willRenderPath:(NSBezierPath*) aPath;

@end


/*
 Renderers can now have a delegate attached which is able to modify behaviours such as changing the path rendered, etc.

*/
