///**********************************************************************************************************************************
///  DKStyle.h
///  DrawKit �2005-2008 Apptree.net
///
///  Created by graham on 13/08/2006.
///
///	 This software is released subject to licensing conditions as detailed in DRAWKIT-LICENSING.TXT, which must accompany this source file. 
///
///**********************************************************************************************************************************


#import "DKRastGroup.h"
#import "DKStyleNameProvider.h"

@class	DKDrawableObject, DKUndoManager, DKStyle;


// notifications are going away in favour of direct client notification. A client implements this protocol and adds itself
// as a client of the style using -addClient: and removes itself with -removeClient:. Note that the style keeps these
// objects as weak references. This should yield huge performance improvements with very large drawings, as no KVO or notification
// set-up and tear-down is required.

@protocol DKStyleClient

@property (nonatomic, assign) NSUInteger	clientIndex;

- (void)	styleWillChange:(DKStyle*) style keyPath:(NSString*) kp;
- (void)	styleDidChange:(DKStyle*) style keyPath:(NSString*) kp;

@end

// swatch types that can be passed to -styleSwatchWithSize:type:

typedef enum
{
	kDKStyleSwatchAutomatic			= -1,
	kDKStyleSwatchRectanglePath		= 0,
	kDKStyleSwatchCurvePath			= 1
}
DKStyleSwatchType;


// options that can be passed to -derivedStyleWithPasteboard:withOptions:


typedef enum
{
	kDKDerivedStyleDefault			= 0,
	kDKDerivedStyleForPathHint		= 1,
	kDKDerivedStyleForShapeHint		= 2
}
DKDerivedStyleOptions;



#define STYLE_SWATCH_SIZE		NSMakeSize( 128.0, 128.0 )

// n.b. for style registry API, see DKStyleRegistry.h



@interface DKStyle : DKRastGroup <DKRasterizerObserver, NSCoding, NSCopying, NSMutableCopying>
{
@private
	NSDictionary*			m_textAttributes;		// supports text additions
	NSUndoManager*			m_undoManagerRef;		// style's undo manager
	BOOL					m_shared;				// YES if the style is shared
	BOOL					m_locked;				// YES if style can't be edited
	BOOL					m_quiet;				// YES to suppress notifications to clients, etc
	id						m_renderClientRef;		// valid only while actually drawing
	NSString*				m_uniqueKey;			// unique key, set once for all time
	BOOL					m_mergeFlag;			// set to YES when a style is read in from a file and was saved in a registered state.
	NSTimeInterval			m_lastModTime;			// timestamp to determine when styles have been updated
	NSString*				mUserDescription;		// user description, if any
	NSPointerArray*			mClients;				// weak references to clients
}

// basic standard styles:

+ (DKStyle*)			defaultStyle;			// very boring, black stroke and light gray fill
+ (DKStyle*)			defaultTrackStyle;		// grey stroke over wider black stroke, no fill

// easy construction of other simple styles:

+ (DKStyle*)			styleWithFillColour:(NSColor*) fc strokeColour:(NSColor*) sc;
+ (DKStyle*)			styleWithFillColour:(NSColor*) fc strokeColour:(NSColor*) sc strokeWidth:(CGFloat) sw;
+ (DKStyle*)			styleFromPasteboard:(NSPasteboard*) pb;

+ (NSArray*)			stylePasteboardTypes;
+ (BOOL)				canInitWithPasteboard:(NSPasteboard*) pb;

// pasted styles - separate non-persistent registry

+ (DKStyle*)			styleWithPasteboardName:(NSString*) name;
+ (void)				registerStyle:(DKStyle*) style withPasteboardName:(NSString*) pbname;

// whether sharing enabled and default sharing flag

+ (void)				setStyleSharingEnabled:(BOOL) enable;
+ (BOOL)				styleSharingEnabled;

+ (void)				setStylesAreSharableByDefault:(BOOL) share;
+ (BOOL)				stylesAreSharableByDefault;

// shadows:

+ (NSShadow*)			defaultShadow;
+ (BOOL)				setWillDrawShadows:(BOOL) drawShadows;
+ (BOOL)				willDrawShadows;

// performance options:

+ (void)				setShouldAntialias:(BOOL) aa;
+ (BOOL)				shouldAntialias;
+ (void)				setShouldSubstitutePlaceholderStyle:(BOOL) substitute;
+ (BOOL)				shouldSubstitutePlaceholderStyle;

// setting a name provider:

+ (void)				setNameProvider:(id<DKStyleNameProvider>) nameProvider;
+ (id<DKStyleNameProvider>) nameProvider;
+ (NSString*)			providedNameForCopyOfStyle:(DKStyle*) style;
+ (void)				disableNameProvider;
+ (void)				enableNameProvider;
+ (BOOL)				isNameProviderEnabled;

// updating & notifying clients:

- (void)				addClient:(id<DKStyleClient>) client;
- (void)				removeClient:(id<DKStyleClient>) client;

- (void)				notifyClientsBeforeChange;
- (void)				notifyClientsAfterChange;
- (void)				styleWasAttached:(DKDrawableObject*) toObject;
- (void)				styleWillBeRemoved:(DKDrawableObject*) fromObject;
- (void)				setDoNotPostClientNotifications:(BOOL) donotpost;
- (BOOL)				doNotPostClientNotifications;

// (text) attributes - basic support

- (void)				setTextAttributes:(NSDictionary*) attrs;
- (void)				setTextAttributes:(NSDictionary*) attrs notifying:(BOOL) notify;
- (NSDictionary*)		textAttributes;
- (BOOL)				hasTextAttributes;
- (void)				removeTextAttributes;
- (BOOL)				isPureTextStyle;

// shared and locked status:

- (void)				setStyleSharable:(BOOL) share;
- (BOOL)				isStyleSharable;
- (void)				setLocked:(BOOL) lock;
- (BOOL)				locked;

// registry info:

- (BOOL)				isStyleRegistered;
- (NSArray*)			registryKeys;
- (NSString*)			uniqueKey;
- (void)				assignUniqueKey;
- (BOOL)				requiresRemerge;
- (void)				clearRemergeFlag;
- (NSTimeInterval)		lastModificationTimestamp;

- (BOOL)				isEqualToStyle:(DKStyle*) aStyle;

- (void)				setUserDescription:(NSString*) userDesc;
- (NSString*)			userDescription;

// undo:

- (void)				setUndoManager:(NSUndoManager*) undomanager;
- (NSUndoManager*)		undoManager;
- (void)				changeKeyPath:(NSString*) keypath ofObject:(id) object toValue:(id) value;

// stroke utilities:

- (void)				scaleStrokeWidthsBy:(CGFloat) scale withoutInformingClients:(BOOL) quiet;
- (CGFloat)				maxStrokeWidth;
- (CGFloat)				maxStrokeWidthDifference;
- (void)				applyStrokeAttributesToPath:(NSBezierPath*) path;
- (NSUInteger)			countOfStrokes;

// clipboard:

- (BOOL)				copyToPasteboard:(NSPasteboard*) pb;
- (DKStyle*)			derivedStyleWithPasteboard:(NSPasteboard*) pb;
- (DKStyle*)			derivedStyleWithPasteboard:(NSPasteboard*) pb withOptions:(DKDerivedStyleOptions) options;
- (DKStyle*)			derivedStyleWithScaledStrokes:(CGFloat) scaleFactor;

// query methods:

- (BOOL)				hasStroke;
- (BOOL)				hasFill;
- (BOOL)				hasVisibleFill;
- (BOOL)				hasHatch;
- (BOOL)				hasTextAdornment;
- (BOOL)				isEmpty;

// swatch images:

- (NSImage*)			styleSwatchWithSize:(NSSize) size type:(DKStyleSwatchType) type;
- (NSImage*)			standardStyleSwatch;
- (NSImage*)			image;
- (NSImage*)			imageToFitSize:(NSSize) aSize;

// currently rendering client (may be queried by renderers)

- (id)					currentRenderClient;

// making derivative styles:

- (DKStyle*)			styleByMergingFromStyle:(DKStyle*) otherStyle;
- (DKStyle*)			styleByRemovingRenderersOfClass:(Class) aClass;
- (id)					clone;

@end



// deprecated methods


@interface DKStyle (Deprecated)

- (NSUInteger)			countOfClients;

@end

// swatch caching:
// this is now centralised to permit taking advantage of NSCache on 10.6 

#define DKSTYLE_SWATCH_COUNT_LIMIT		200



@interface DKStyle (SwatchCaching)

+ (NSImage*)			swatchForStyle:(DKStyle*) style;
+ (void)				setSwatch:(NSImage*) swatch forStyle:(DKStyle*) style;
+ (id)					swatchCache;

@end

// pasteboard types:

extern NSString*		kDKStylePasteboardType;
extern NSString*		kDKStyleKeyPasteboardType;

// notifications:

extern NSString*		kDKStyleWillChangeNotification;
extern NSString*		kDKStyleDidChangeNotification;
extern NSString*		kDKStyleTextAttributesDidChangeNotification; 
extern NSString*		kDKStyleWasAttachedNotification;
extern NSString*		kDKStyleWillBeDetachedNotification;
extern NSString*		kDKStyleLockStateChangedNotification;
extern NSString*		kDKStyleSharableFlagChangedNotification;
extern NSString*		kDKStyleNameChangedNotification;
extern NSString*		kDKStyleWasCopiedNotification;
extern NSString* const	kDKStyleSharingEnabledChangeNotification;

// preferences keys

extern NSString*		kDKStyleDisplayPerformance_no_anti_aliasing;
extern NSString*		kDKStyleDisplayPerformance_no_shadows;
extern NSString*		kDKStyleDisplayPerformance_substitute_styles;
extern NSString* const	kDKStyleSharableDefaultsKey;
extern NSString* const	kDKStyleSharingDisabledDefaultsKey;
