(function() {
  describe("Autoflow package", function() {
    var autoflow, editor, editorElement, _ref;
    _ref = [], autoflow = _ref[0], editor = _ref[1], editorElement = _ref[2];
    describe("autoflow:reflow-selection", function() {
      beforeEach(function() {
        var activationPromise;
        activationPromise = null;
        waitsForPromise(function() {
          return atom.workspace.open();
        });
        runs(function() {
          editor = atom.workspace.getActiveTextEditor();
          editorElement = atom.views.getView(editor);
          atom.config.set('editor.preferredLineLength', 30);
          activationPromise = atom.packages.activatePackage('autoflow');
          return atom.commands.dispatch(editorElement, 'autoflow:reflow-selection');
        });
        return waitsForPromise(function() {
          return activationPromise;
        });
      });
      it("uses the preferred line length based on the editor's scope", function() {
        atom.config.set('editor.preferredLineLength', 4, {
          scopeSelector: '.text.plain.null-grammar'
        });
        editor.setText("foo bar");
        editor.selectAll();
        atom.commands.dispatch(editorElement, 'autoflow:reflow-selection');
        return expect(editor.getText()).toBe("foo\nbar");
      });
      it("rearranges line breaks in the current selection to ensure lines are shorter than config.editor.preferredLineLength", function() {
        editor.setText("This is the first paragraph and it is longer than the preferred line length so it should be reflowed.\n\nThis is a short paragraph.\n\nAnother long paragraph, it should also be reflowed with the use of this single command.");
        editor.selectAll();
        atom.commands.dispatch(editorElement, 'autoflow:reflow-selection');
        return expect(editor.getText()).toBe("This is the first paragraph\nand it is longer than the\npreferred line length so it\nshould be reflowed.\n\nThis is a short paragraph.\n\nAnother long paragraph, it\nshould also be reflowed with\nthe use of this single\ncommand.");
      });
      it("reflows the current paragraph if nothing is selected", function() {
        editor.setText("This is a preceding paragraph, which shouldn't be modified by a reflow of the following paragraph.\n\nThe quick brown fox jumps over the lazy\ndog. The preceding sentence contains every letter\nin the entire English alphabet, which has absolutely no relevance\nto this test.\n\nThis is a following paragraph, which shouldn't be modified by a reflow of the preciding paragraph.\n");
        editor.setCursorBufferPosition([3, 5]);
        atom.commands.dispatch(editorElement, 'autoflow:reflow-selection');
        return expect(editor.getText()).toBe("This is a preceding paragraph, which shouldn't be modified by a reflow of the following paragraph.\n\nThe quick brown fox jumps over\nthe lazy dog. The preceding\nsentence contains every letter\nin the entire English\nalphabet, which has absolutely\nno relevance to this test.\n\nThis is a following paragraph, which shouldn't be modified by a reflow of the preciding paragraph.\n");
      });
      return it("allows for single words that exceed the preferred wrap column length", function() {
        editor.setText("this-is-a-super-long-word-that-shouldn't-break-autoflow and these are some smaller words");
        editor.selectAll();
        atom.commands.dispatch(editorElement, 'autoflow:reflow-selection');
        return expect(editor.getText()).toBe("this-is-a-super-long-word-that-shouldn't-break-autoflow\nand these are some smaller\nwords");
      });
    });
    return describe("reflowing text", function() {
      beforeEach(function() {
        return autoflow = require("../lib/autoflow");
      });
      it('respects current paragraphs', function() {
        var res, text;
        text = 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh id magna ullamcorper sagittis. Maecenas\net enim eu orci tincidunt adipiscing\naliquam ligula.\n\nLorem ipsum dolor sit amet, consectetur adipiscing elit.\nPhasellus gravida\nnibh id magna ullamcorper\ntincidunt adipiscing lacinia a dui. Etiam quis erat dolor.\nrutrum nisl fermentum rhoncus. Duis blandit ligula facilisis fermentum.';
        res = 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh\nid magna ullamcorper sagittis. Maecenas et enim eu orci tincidunt adipiscing\naliquam ligula.\n\nLorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh\nid magna ullamcorper tincidunt adipiscing lacinia a dui. Etiam quis erat dolor.\nrutrum nisl fermentum rhoncus. Duis blandit ligula facilisis fermentum.';
        return expect(autoflow.reflow(text, {
          wrapColumn: 80
        })).toEqual(res);
      });
      it('respects indentation', function() {
        var res, text;
        text = 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh id magna ullamcorper sagittis. Maecenas\net enim eu orci tincidunt adipiscing\naliquam ligula.\n\n    Lorem ipsum dolor sit amet, consectetur adipiscing elit.\n    Phasellus gravida\n    nibh id magna ullamcorper\n    tincidunt adipiscing lacinia a dui. Etiam quis erat dolor.\n    rutrum nisl fermentum  rhoncus. Duis blandit ligula facilisis fermentum';
        res = 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh\nid magna ullamcorper sagittis. Maecenas et enim eu orci tincidunt adipiscing\naliquam ligula.\n\n    Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida\n    nibh id magna ullamcorper tincidunt adipiscing lacinia a dui. Etiam quis\n    erat dolor. rutrum nisl fermentum  rhoncus. Duis blandit ligula facilisis\n    fermentum';
        return expect(autoflow.reflow(text, {
          wrapColumn: 80
        })).toEqual(res);
      });
      it('respects prefixed text (comments!)', function() {
        var res, text;
        text = 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh id magna ullamcorper sagittis. Maecenas\net enim eu orci tincidunt adipiscing\naliquam ligula.\n\n  #  Lorem ipsum dolor sit amet, consectetur adipiscing elit.\n  #  Phasellus gravida\n  #  nibh id magna ullamcorper\n  #  tincidunt adipiscing lacinia a dui. Etiam quis erat dolor.\n  #  rutrum nisl fermentum  rhoncus. Duis blandit ligula facilisis fermentum';
        res = 'Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida nibh\nid magna ullamcorper sagittis. Maecenas et enim eu orci tincidunt adipiscing\naliquam ligula.\n\n  #  Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida\n  #  nibh id magna ullamcorper tincidunt adipiscing lacinia a dui. Etiam quis\n  #  erat dolor. rutrum nisl fermentum  rhoncus. Duis blandit ligula facilisis\n  #  fermentum';
        return expect(autoflow.reflow(text, {
          wrapColumn: 80
        })).toEqual(res);
      });
      it('respects multiple prefixes (js/c comments)', function() {
        var res, text;
        text = '// Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida\net enim eu orci tincidunt adipiscing\naliquam ligula.';
        res = '// Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida et\n// enim eu orci tincidunt adipiscing aliquam ligula.';
        return expect(autoflow.reflow(text, {
          wrapColumn: 80
        })).toEqual(res);
      });
      return it('properly handles * prefix', function() {
        var res, text;
        text = '* Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida\net enim eu orci tincidunt adipiscing\naliquam ligula.\n\n  * soidjfiojsoidj foi';
        res = '* Lorem ipsum dolor sit amet, consectetur adipiscing elit. Phasellus gravida et\n* enim eu orci tincidunt adipiscing aliquam ligula.\n\n  * soidjfiojsoidj foi';
        return expect(autoflow.reflow(text, {
          wrapColumn: 80
        })).toEqual(res);
      });
    });
  });

}).call(this);
