(function() {
  var TagFinder, tagFinder;

  TagFinder = require('../lib/tag-finder');

  tagFinder = new TagFinder();

  describe('closeTag', function() {
    describe('TagFinder::parseFragment', function() {
      var fragment;
      fragment = "";
      beforeEach(function() {
        return fragment = "<html><head><body></body>";
      });
      it('returns the last not closed elem in fragment, matching a given pattern', function() {
        var stack;
        stack = tagFinder.parseFragment(fragment, [], /<(\w+)|<\/(\w*)/, function() {
          return true;
        });
        return expect(stack[stack.length - 1]).toBe("head");
      });
      it('stops when cond become true', function() {
        var stack;
        stack = tagFinder.parseFragment(fragment, [], /<(\w+)|<\/(\w*)/, function() {
          return false;
        });
        return expect(stack.length).toBe(0);
      });
      return it('uses the given match expression to match tags', function() {
        var stack;
        stack = tagFinder.parseFragment(fragment, [], /<(body)|(notag)/, function() {
          return true;
        });
        return expect(stack[stack.length - 1]).toBe("body");
      });
    });
    describe('TagFinder::tagsNotClosedInFragment', function() {
      it('returns the outermost tag not closed in an HTML fragment', function() {
        var fragment, tags;
        fragment = "<html><head></head><body><h1><p></p>";
        tags = tagFinder.tagsNotClosedInFragment(fragment);
        return expect(tags).toEqual(['html', 'body', 'h1']);
      });
      return it('is not confused by tag attributes', function() {
        var fragment, tags;
        fragment = '<html><head></head><body class="c"><h1 class="p"><p></p>';
        tags = tagFinder.tagsNotClosedInFragment(fragment);
        return expect(tags).toEqual(['html', 'body', 'h1']);
      });
    });
    describe('TagFinder::tagDoesNotCloseInFragment', function() {
      it('returns true if the given tag is not closed in the given fragment', function() {
        var fragment;
        fragment = "</other1></other2></html>";
        return expect(tagFinder.tagDoesNotCloseInFragment("body", fragment)).toBe(true);
      });
      it('returns false if the given tag is closed in the given fragment', function() {
        var fragment;
        fragment = "</other1></body></html>";
        return expect(tagFinder.tagDoesNotCloseInFragment(["body"], fragment)).toBe(false);
      });
      it('returns true even if the given tag is re-opened and re-closed', function() {
        var fragment;
        fragment = "<other> </other><body></body><html>";
        return expect(tagFinder.tagDoesNotCloseInFragment(["body"], fragment)).toBe(true);
      });
      return it('returns false even if the given tag is re-opened and re-closed before closing', function() {
        var fragment;
        fragment = "<other> </other><body></body></body><html>";
        return expect(tagFinder.tagDoesNotCloseInFragment(["body"], fragment)).toBe(false);
      });
    });
    return describe('TagFinder::closingTagForFragments', function() {
      it('returns the last opened in preFragment tag that is not closed in postFragment', function() {
        var postFragment, preFragment;
        preFragment = "<html><head></head><body><h1></h1><p>";
        postFragment = "</body></html>";
        return expect(tagFinder.closingTagForFragments(preFragment, postFragment)).toBe("p");
      });
      it('correctly handles empty postFragment', function() {
        var postFragment, preFragment;
        preFragment = "<html><head></head><body><h1></h1><p>";
        postFragment = "";
        return expect(tagFinder.closingTagForFragments(preFragment, postFragment)).toBe("p");
      });
      it("correctly handles malformed tags", function() {
        var postFragment, preFragment;
        preFragment = "<html><head></head></htm";
        postFragment = "";
        return expect(tagFinder.closingTagForFragments(preFragment, postFragment)).toBe("html");
      });
      it('returns null if there is no open tag to be closed', function() {
        var postFragment, preFragment;
        preFragment = "<html><head></head><body><h1></h1><p>";
        postFragment = "</p></body></html>";
        return expect(tagFinder.closingTagForFragments(preFragment, postFragment)).toBe(null);
      });
      return it("correctly closes tags containing hyphens", function() {
        var postFragment, preFragment;
        preFragment = "<html><head></head><body><h1></h1><my-element>";
        postFragment = "</body></html>";
        return expect(tagFinder.closingTagForFragments(preFragment, postFragment)).toBe("my-element");
      });
    });
  });

}).call(this);
