(function() {
  var $, path, _;

  _ = require('underscore-plus');

  $ = require('atom-space-pen-views').$;

  path = require('path');

  describe('FindView', function() {
    var activationPromise, editor, editorView, findView, getFindAtomPanel, getResultDecorations, workspaceElement, _ref;
    _ref = [], workspaceElement = _ref[0], editorView = _ref[1], editor = _ref[2], findView = _ref[3], activationPromise = _ref[4];
    getFindAtomPanel = function() {
      return workspaceElement.querySelector('.find-and-replace').parentNode;
    };
    getResultDecorations = function(editor, clazz) {
      var decoration, decorations, markerId, markerIdForDecorations, resultDecorations, _i, _len;
      markerIdForDecorations = editor.decorationsForScreenRowRange(0, editor.getLineCount());
      resultDecorations = [];
      for (markerId in markerIdForDecorations) {
        decorations = markerIdForDecorations[markerId];
        for (_i = 0, _len = decorations.length; _i < _len; _i++) {
          decoration = decorations[_i];
          if (decoration.getProperties()["class"] === clazz) {
            resultDecorations.push(decoration);
          }
        }
      }
      return resultDecorations;
    };
    beforeEach(function() {
      spyOn(atom, 'beep');
      workspaceElement = atom.views.getView(atom.workspace);
      atom.project.setPaths([path.join(__dirname, 'fixtures')]);
      waitsForPromise(function() {
        return atom.workspace.open('sample.js');
      });
      return runs(function() {
        jasmine.attachToDOM(workspaceElement);
        editor = atom.workspace.getActiveTextEditor();
        editorView = atom.views.getView(editor);
        return activationPromise = atom.packages.activatePackage("find-and-replace").then(function(_arg) {
          var mainModule;
          mainModule = _arg.mainModule;
          mainModule.createViews();
          return findView = mainModule.findView, mainModule;
        });
      });
    });
    describe("when find-and-replace:show is triggered", function() {
      it("attaches FindView to the root view", function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          return expect(workspaceElement.querySelector('.find-and-replace')).toBeDefined();
        });
      });
      it("populates the findEditor with selection when there is a selection", function() {
        editor.setSelectedBufferRange([[2, 8], [2, 13]]);
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getFindAtomPanel()).toBeVisible();
          expect(findView.findEditor.getText()).toBe('items');
          findView.findEditor.setText('');
          editor.setSelectedBufferRange([[2, 14], [2, 20]]);
          atom.commands.dispatch(editorView, 'find-and-replace:show');
          expect(getFindAtomPanel()).toBeVisible();
          return expect(findView.findEditor.getText()).toBe('length');
        });
      });
      it("does not change the findEditor text when there is no selection", function() {
        editor.setSelectedBufferRange([[2, 8], [2, 8]]);
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          findView.findEditor.setText('kitten');
          atom.commands.dispatch(editorView, 'find-and-replace:show');
          return expect(findView.findEditor.getText()).toBe('kitten');
        });
      });
      it("does not change the findEditor text when there is a multiline selection", function() {
        editor.setSelectedBufferRange([[2, 8], [3, 12]]);
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getFindAtomPanel()).toBeVisible();
          return expect(findView.findEditor.getText()).toBe('');
        });
      });
      return it("honors config settings for find options", function() {
        atom.config.set('find-and-replace.useRegex', true);
        atom.config.set('find-and-replace.caseSensitive', true);
        atom.config.set('find-and-replace.inCurrentSelection', true);
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(findView.caseOptionButton).toHaveClass('selected');
          expect(findView.regexOptionButton).toHaveClass('selected');
          return expect(findView.selectionOptionButton).toHaveClass('selected');
        });
      });
    });
    describe("when find-and-replace:toggle is triggered", function() {
      return it("toggles the visibility of the FindView", function() {
        atom.commands.dispatch(workspaceElement, 'find-and-replace:toggle');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getFindAtomPanel()).toBeVisible();
          atom.commands.dispatch(workspaceElement, 'find-and-replace:toggle');
          return expect(getFindAtomPanel()).not.toBeVisible();
        });
      });
    });
    describe("when FindView's replace editor is visible", function() {
      return it("keeps the replace editor visible when find-and-replace:show is triggered", function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show-replace');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          atom.commands.dispatch(editorView, 'find-and-replace:show');
          return expect(findView.replaceEditor).toBeVisible();
        });
      });
    });
    describe("core:cancel", function() {
      beforeEach(function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          findView.findEditor.setText('items');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          return findView.focus();
        });
      });
      describe("when core:cancel is triggered on the find view", function() {
        it("detaches from the workspace view", function() {
          atom.commands.dispatch(document.activeElement, 'core:cancel');
          return expect(getFindAtomPanel()).not.toBeVisible();
        });
        return it("removes highlighted matches", function() {
          expect(workspaceElement).toHaveClass('find-visible');
          atom.commands.dispatch(document.activeElement, 'core:cancel');
          return expect(workspaceElement).not.toHaveClass('find-visible');
        });
      });
      describe("when core:cancel is triggered on an empty pane", function() {
        return it("hides the find panel", function() {
          var paneElement;
          paneElement = atom.views.getView(atom.workspace.getActivePane());
          paneElement.focus();
          atom.commands.dispatch(paneElement, 'core:cancel');
          return expect(getFindAtomPanel()).not.toBeVisible();
        });
      });
      describe("when core:cancel is triggered on an editor", function() {
        return it("detaches from the workspace view", function() {
          waitsForPromise(function() {
            return atom.workspace.open();
          });
          return runs(function() {
            atom.commands.dispatch(editorView, 'core:cancel');
            return expect(getFindAtomPanel()).not.toBeVisible();
          });
        });
      });
      return describe("when core:cancel is triggered on a mini editor", function() {
        return it("leaves the find view attached", function() {
          var miniEditor;
          miniEditor = document.createElement('atom-text-editor');
          miniEditor.setAttribute('mini', '');
          atom.workspace.addTopPanel({
            item: miniEditor
          });
          miniEditor.focus();
          atom.commands.dispatch(miniEditor, 'core:cancel');
          return expect(getFindAtomPanel()).toBeVisible();
        });
      });
    });
    describe("serialization", function() {
      it("serializes find and replace history", function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        runs(function() {
          findView.findEditor.setText("items");
          findView.replaceEditor.setText("cat");
          findView.replaceAll();
          findView.findEditor.setText("sort");
          findView.replaceEditor.setText("dog");
          findView.replaceNext();
          findView.findEditor.setText("shift");
          findView.replaceEditor.setText("ok");
          findView.findNext(false);
          atom.packages.deactivatePackage("find-and-replace");
          activationPromise = atom.packages.activatePackage("find-and-replace").then(function(_arg) {
            var mainModule;
            mainModule = _arg.mainModule;
            mainModule.createViews();
            return findView = mainModule.findView, mainModule;
          });
          return atom.commands.dispatch(editorView, 'find-and-replace:show');
        });
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toBe('shift');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toBe('sort');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toBe('items');
          atom.commands.dispatch(findView.replaceEditor.element, 'core:move-up');
          expect(findView.replaceEditor.getText()).toBe('dog');
          atom.commands.dispatch(findView.replaceEditor.element, 'core:move-up');
          return expect(findView.replaceEditor.getText()).toBe('cat');
        });
      });
      return it("serializes find options ", function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        runs(function() {
          expect(findView.caseOptionButton).not.toHaveClass('selected');
          expect(findView.regexOptionButton).not.toHaveClass('selected');
          expect(findView.selectionOptionButton).not.toHaveClass('selected');
          expect(findView.wholeWordOptionButton).not.toHaveClass('selected');
          findView.caseOptionButton.click();
          findView.regexOptionButton.click();
          findView.selectionOptionButton.click();
          findView.wholeWordOptionButton.click();
          expect(findView.caseOptionButton).toHaveClass('selected');
          expect(findView.regexOptionButton).toHaveClass('selected');
          expect(findView.selectionOptionButton).toHaveClass('selected');
          expect(findView.wholeWordOptionButton).toHaveClass('selected');
          atom.packages.deactivatePackage("find-and-replace");
          activationPromise = atom.packages.activatePackage("find-and-replace").then(function(_arg) {
            var mainModule;
            mainModule = _arg.mainModule;
            mainModule.createViews();
            return findView = mainModule.findView, mainModule;
          });
          return atom.commands.dispatch(editorView, 'find-and-replace:show');
        });
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(findView.caseOptionButton).toHaveClass('selected');
          expect(findView.regexOptionButton).toHaveClass('selected');
          expect(findView.selectionOptionButton).toHaveClass('selected');
          return expect(findView.wholeWordOptionButton).toHaveClass('selected');
        });
      });
    });
    describe("finding", function() {
      beforeEach(function() {
        atom.config.set('find-and-replace.focusEditorAfterSearch', false);
        editor.setCursorBufferPosition([2, 0]);
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          findView.findEditor.setText('items');
          return atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
        });
      });
      describe("when find-and-replace:confirm is triggered", function() {
        return it("runs a search", function() {
          findView.findEditor.setText('notinthefile');
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:confirm');
          expect(getResultDecorations(editor, 'find-result')).toHaveLength(0);
          findView.findEditor.setText('items');
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:confirm');
          return expect(getResultDecorations(editor, 'find-result')).toHaveLength(5);
        });
      });
      describe("when the find string contains an escaped char", function() {
        beforeEach(function() {
          editor.setText("\t\n\\t\\\\");
          return editor.setCursorBufferPosition([0, 0]);
        });
        describe("when regex seach is enabled", function() {
          beforeEach(function() {
            return atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
          });
          it("finds a backslash", function() {
            findView.findEditor.setText('\\\\');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[1, 0], [1, 1]]);
          });
          it("finds a newline", function() {
            findView.findEditor.setText('\\n');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[0, 1], [1, 0]]);
          });
          return it("finds a tab character", function() {
            findView.findEditor.setText('\\t');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[0, 0], [0, 1]]);
          });
        });
        return describe("when regex seach is disabled", function() {
          it("finds the literal backslash t", function() {
            findView.findEditor.setText('\\t');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[1, 0], [1, 2]]);
          });
          it("finds a backslash", function() {
            findView.findEditor.setText('\\');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[1, 0], [1, 1]]);
          });
          it("finds two backslashes", function() {
            findView.findEditor.setText('\\\\');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[1, 2], [1, 4]]);
          });
          return it("doesn't find when escaped", function() {
            findView.findEditor.setText('\\\\t');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(editor.getSelectedBufferRange()).toEqual([[0, 0], [0, 0]]);
          });
        });
      });
      describe("when focusEditorAfterSearch is set", function() {
        beforeEach(function() {
          atom.config.set('find-and-replace.focusEditorAfterSearch', true);
          return atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
        });
        return it("selects the first match following the cursor and correctly focuses the editor", function() {
          expect(findView.resultCounter.text()).toEqual('3 of 6');
          expect(editor.getSelectedBufferRange()).toEqual([[2, 34], [2, 39]]);
          return expect(editorView).toHaveFocus();
        });
      });
      describe("when whole-word search is enabled", function() {
        beforeEach(function() {
          editor.setText("-----\nswhole-wordy\nwhole-word\nword\nwhole-swords");
          editor.setCursorBufferPosition([0, 0]);
          return atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-whole-word-option');
        });
        it("finds the whole words", function() {
          findView.findEditor.setText('word');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          return expect(editor.getSelectedBufferRange()).toEqual([[2, 6], [2, 10]]);
        });
        return it("doesn't highlights the search inside words", function() {
          findView.findEditor.setText('word');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(getResultDecorations(editor, 'find-result')).toHaveLength(1);
          return expect(getResultDecorations(editor, 'current-result')).toHaveLength(1);
        });
      });
      it("doesn't change the selection, beeps if there are no matches and keeps focus on the find view", function() {
        editor.setCursorBufferPosition([2, 0]);
        findView.findEditor.setText('notinthefilebro');
        findView.findEditor.focus();
        atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
        expect(editor.getCursorBufferPosition()).toEqual([2, 0]);
        expect(atom.beep).toHaveBeenCalled();
        expect(findView).toHaveFocus();
        return expect(findView.descriptionLabel.text()).toEqual("No results found for 'notinthefilebro'");
      });
      describe("updating the descriptionLabel", function() {
        it("properly updates the info message", function() {
          findView.findEditor.setText('item');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(findView.descriptionLabel.text()).toEqual("6 results found for 'item'");
          findView.findEditor.setText('notinthefilenope');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(findView.descriptionLabel.text()).toEqual("No results found for 'notinthefilenope'");
          findView.findEditor.setText('item');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(findView.descriptionLabel.text()).toEqual("6 results found for 'item'");
          findView.findEditor.setText('');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          return expect(findView.descriptionLabel.text()).toContain("Find in Current Buffer");
        });
        return describe("when there is a find-error", function() {
          beforeEach(function() {
            editor.setCursorBufferPosition([2, 0]);
            return atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
          });
          it("displays the error", function() {
            findView.findEditor.setText('i[t');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            expect(findView.descriptionLabel).toHaveClass('text-error');
            return expect(findView.descriptionLabel.text()).toContain('Invalid regular expression');
          });
          return it("will be reset when there is no longer an error", function() {
            findView.findEditor.setText('i[t');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            expect(findView.descriptionLabel).toHaveClass('text-error');
            findView.findEditor.setText('');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            expect(findView.descriptionLabel).not.toHaveClass('text-error');
            expect(findView.descriptionLabel.text()).toContain("Find in Current Buffer");
            findView.findEditor.setText('item');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            expect(findView.descriptionLabel).not.toHaveClass('text-error');
            return expect(findView.descriptionLabel.text()).toContain("6 results");
          });
        });
      });
      it("selects the first match following the cursor", function() {
        expect(findView.resultCounter.text()).toEqual('2 of 6');
        expect(editor.getSelectedBufferRange()).toEqual([[2, 8], [2, 13]]);
        atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
        expect(findView.resultCounter.text()).toEqual('3 of 6');
        expect(editor.getSelectedBufferRange()).toEqual([[2, 34], [2, 39]]);
        return expect(findView.findEditor).toHaveFocus();
      });
      it("selects the next match when the next match button is pressed", function() {
        findView.nextButton.click();
        expect(findView.resultCounter.text()).toEqual('3 of 6');
        return expect(editor.getSelectedBufferRange()).toEqual([[2, 34], [2, 39]]);
      });
      it("selects the next match when the 'find-and-replace:find-next' event is triggered and correctly focuses the editor", function() {
        expect(findView).toHaveFocus();
        atom.commands.dispatch(editorView, 'find-and-replace:find-next');
        expect(findView.resultCounter.text()).toEqual('3 of 6');
        expect(editor.getSelectedBufferRange()).toEqual([[2, 34], [2, 39]]);
        return expect(editorView).toHaveFocus();
      });
      it("selects the previous match before the cursor when the 'find-and-replace:show-previous' event is triggered", function() {
        expect(findView.resultCounter.text()).toEqual('2 of 6');
        expect(editor.getSelectedBufferRange()).toEqual([[2, 8], [2, 13]]);
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:show-previous');
        expect(findView.resultCounter.text()).toEqual('1 of 6');
        expect(editor.getSelectedBufferRange()).toEqual([[1, 22], [1, 27]]);
        return expect(findView.findEditor).toHaveFocus();
      });
      it("will re-run search if 'find-and-replace:find-next' is triggered after changing the findEditor's text", function() {
        findView.findEditor.setText('sort');
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-next');
        expect(findView.resultCounter.text()).toEqual('3 of 5');
        return expect(editor.getSelectedBufferRange()).toEqual([[8, 11], [8, 15]]);
      });
      it("'find-and-replace:find-next' adds to the findEditor's history", function() {
        findView.findEditor.setText('sort');
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-next');
        expect(findView.resultCounter.text()).toEqual('3 of 5');
        findView.findEditor.setText('nope');
        atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
        return expect(findView.findEditor.getText()).toEqual('sort');
      });
      it("selects the previous match when the 'find-and-replace:find-previous' event is triggered and correctly focuses the editor", function() {
        expect(findView).toHaveFocus();
        atom.commands.dispatch(editorView, 'find-and-replace:find-previous');
        expect(findView.resultCounter.text()).toEqual('1 of 6');
        expect(editor.getSelectedBufferRange()).toEqual([[1, 27], [1, 22]]);
        return expect(editorView).toHaveFocus();
      });
      it("will re-run search if 'find-and-replace:find-previous' is triggered after changing the findEditor's text", function() {
        findView.findEditor.setText('sort');
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-previous');
        expect(findView.resultCounter.text()).toEqual('2 of 5');
        return expect(editor.getSelectedBufferRange()).toEqual([[1, 6], [1, 10]]);
      });
      it("selects all matches when 'find-and-replace:find-all' is triggered and correctly focuses the editor", function() {
        expect(findView).toHaveFocus();
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-all');
        expect(editor.getSelectedBufferRanges()).toEqual([[[1, 27], [1, 22]], [[2, 8], [2, 13]], [[2, 34], [2, 39]], [[3, 16], [3, 21]], [[4, 10], [4, 15]], [[5, 16], [5, 21]]]);
        return expect(editorView).toHaveFocus();
      });
      it("will re-run search if 'find-and-replace:find-all' is triggered after changing the findEditor's text", function() {
        findView.findEditor.setText('sort');
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-all');
        return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 9], [0, 13]], [[1, 6], [1, 10]], [[8, 11], [8, 15]], [[8, 43], [8, 47]], [[11, 9], [11, 13]]]);
      });
      it("replaces results counter with number of results found when user moves the cursor", function() {
        editor.moveDown();
        return expect(findView.resultCounter.text()).toBe('6 found');
      });
      it("replaces results counter x of y text when user selects a marked range", function() {
        editor.moveDown();
        editor.setSelectedBufferRange([[2, 34], [2, 39]]);
        return expect(findView.resultCounter.text()).toEqual('3 of 6');
      });
      it("places the selected text into the find editor when find-and-replace:set-find-pattern is triggered", function() {
        editor.setSelectedBufferRange([[1, 6], [1, 10]]);
        atom.commands.dispatch(workspaceElement, 'find-and-replace:use-selection-as-find-pattern');
        expect(findView.findEditor.getText()).toBe('sort');
        expect(editor.getSelectedBufferRange()).toEqual([[1, 6], [1, 10]]);
        atom.commands.dispatch(workspaceElement, 'find-and-replace:find-next');
        return expect(editor.getSelectedBufferRange()).toEqual([[8, 11], [8, 15]]);
      });
      it("does not highlight the found text when the find view is hidden", function() {
        atom.commands.dispatch(findView.findEditor.element, 'core:cancel');
        return atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-next');
      });
      describe("when the active pane item changes", function() {
        beforeEach(function() {
          return editor.setSelectedBufferRange([[0, 0], [0, 0]]);
        });
        describe("when a new edit session is activated", function() {
          it("reruns the search on the new edit session", function() {
            waitsForPromise(function() {
              return atom.workspace.open('sample.coffee');
            });
            return runs(function() {
              editor = atom.workspace.getActivePaneItem();
              expect(findView.resultCounter.text()).toEqual('7 found');
              return expect(editor.getSelectedBufferRange()).toEqual([[0, 0], [0, 0]]);
            });
          });
          it("initially highlights the found text in the new edit session", function() {
            expect(getResultDecorations(editor, 'find-result')).toHaveLength(6);
            waitsForPromise(function() {
              return atom.workspace.open('sample.coffee');
            });
            return runs(function() {
              var newEditor;
              expect(getResultDecorations(editor, 'find-result')).toHaveLength(0);
              newEditor = atom.workspace.getActiveTextEditor();
              return expect(getResultDecorations(newEditor, 'find-result')).toHaveLength(7);
            });
          });
          return it("highlights the found text in the new edit session when find next is triggered", function() {
            waitsForPromise(function() {
              return atom.workspace.open('sample.coffee');
            });
            return runs(function() {
              var newEditor;
              atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-next');
              newEditor = atom.workspace.getActiveTextEditor();
              expect(getResultDecorations(newEditor, 'find-result')).toHaveLength(6);
              return expect(getResultDecorations(newEditor, 'current-result')).toHaveLength(1);
            });
          });
        });
        describe("when all active pane items are closed", function() {
          return it("updates the result count", function() {
            atom.commands.dispatch(editorView, 'core:close');
            return expect(findView.resultCounter.text()).toEqual('no results');
          });
        });
        describe("when the active pane item is not an edit session", function() {
          var anotherOpener, openerDisposable, _ref1;
          _ref1 = [], anotherOpener = _ref1[0], openerDisposable = _ref1[1];
          beforeEach(function() {
            anotherOpener = function(pathToOpen, options) {
              return document.createElement('div');
            };
            return openerDisposable = atom.workspace.addOpener(anotherOpener);
          });
          afterEach(function() {
            return openerDisposable.dispose();
          });
          return it("updates the result view", function() {
            waitsForPromise(function() {
              return atom.workspace.open("another");
            });
            return runs(function() {
              return expect(findView.resultCounter.text()).toEqual('no results');
            });
          });
        });
        return describe("when a new edit session is activated on a different pane", function() {
          it("initially highlights all the sample.js results", function() {
            return expect(getResultDecorations(editor, 'find-result')).toHaveLength(6);
          });
          it("reruns the search on the new editor", function() {
            var newEditor;
            newEditor = null;
            waitsForPromise(function() {
              return atom.project.open('sample.coffee').then(function(o) {
                return newEditor = o;
              });
            });
            return runs(function() {
              newEditor = atom.workspace.paneForItem(editor).splitRight({
                items: [newEditor]
              }).getActiveItem();
              expect(getResultDecorations(newEditor, 'find-result')).toHaveLength(7);
              expect(findView.resultCounter.text()).toEqual('7 found');
              expect(newEditor.getSelectedBufferRange()).toEqual([[0, 0], [0, 0]]);
              atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-next');
              expect(findView.resultCounter.text()).toEqual('1 of 7');
              return expect(newEditor.getSelectedBufferRange()).toEqual([[1, 9], [1, 14]]);
            });
          });
          it("highlights the found text in the new edit session (and removes the highlights from the other)", function() {
            var newEditor, newEditorView, _ref1;
            _ref1 = [], newEditor = _ref1[0], newEditorView = _ref1[1];
            waitsForPromise(function() {
              return atom.workspace.open('sample.coffee').then(function(o) {
                return newEditor = o;
              });
            });
            return runs(function() {
              expect(getResultDecorations(editor, 'find-result')).toHaveLength(0);
              return expect(getResultDecorations(newEditor, 'find-result')).toHaveLength(7);
            });
          });
          return it("will still highlight results after the split pane has been destroyed", function() {
            var newEditor, newEditorView, _ref1;
            _ref1 = [], newEditor = _ref1[0], newEditorView = _ref1[1];
            waitsForPromise(function() {
              return atom.workspace.open('sample.coffee').then(function(o) {
                return newEditor = o;
              });
            });
            runs(function() {
              var originalPane, splitPane;
              originalPane = atom.workspace.paneForItem(editor);
              splitPane = atom.workspace.paneForItem(editor).splitRight();
              originalPane.moveItemToPane(newEditor, splitPane, 0);
              expect(getResultDecorations(newEditor, 'find-result')).toHaveLength(7);
              newEditorView = atom.views.getView(editor);
              atom.commands.dispatch(newEditorView, 'core:close');
              editorView.focus();
              return expect(atom.workspace.getActiveTextEditor()).toBe(editor);
            });
            return runs(function() {
              return expect(getResultDecorations(editor, 'find-result')).toHaveLength(6);
            });
          });
        });
      });
      describe("when the buffer contents change", function() {
        it("re-runs the search", function() {
          editor.setSelectedBufferRange([[1, 26], [1, 27]]);
          editor.insertText("");
          window.advanceClock(1000);
          expect(findView.resultCounter.text()).toEqual('5 found');
          editor.insertText("s");
          window.advanceClock(1000);
          return expect(findView.resultCounter.text()).toEqual('6 found');
        });
        return it("does not beep if no matches were found", function() {
          editor.setCursorBufferPosition([2, 0]);
          findView.findEditor.setText('notinthefilebro');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          atom.beep.reset();
          editor.insertText("blah blah");
          return expect(atom.beep).not.toHaveBeenCalled();
        });
      });
      describe("when finding within a selection", function() {
        beforeEach(function() {
          return editor.setSelectedBufferRange([[2, 0], [4, 0]]);
        });
        it("toggles find within a selction via and event and only finds matches within the selection", function() {
          findView.findEditor.setText('items');
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-selection-option');
          expect(editor.getSelectedBufferRange()).toEqual([[2, 8], [2, 13]]);
          return expect(findView.resultCounter.text()).toEqual('1 of 3');
        });
        return it("toggles find within a selction via and button and only finds matches within the selection", function() {
          findView.findEditor.setText('items');
          findView.selectionOptionButton.click();
          expect(editor.getSelectedBufferRange()).toEqual([[2, 8], [2, 13]]);
          return expect(findView.resultCounter.text()).toEqual('1 of 3');
        });
      });
      describe("when regex is toggled", function() {
        it("toggles regex via an event and finds text matching the pattern", function() {
          editor.setCursorBufferPosition([2, 0]);
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
          findView.findEditor.setText('i[t]em+s');
          return expect(editor.getSelectedBufferRange()).toEqual([[2, 8], [2, 13]]);
        });
        it("toggles regex via a button and finds text matching the pattern", function() {
          editor.setCursorBufferPosition([2, 0]);
          findView.regexOptionButton.click();
          findView.findEditor.setText('i[t]em+s');
          return expect(editor.getSelectedBufferRange()).toEqual([[2, 8], [2, 13]]);
        });
        it("re-runs the search using the new find text when toggled", function() {
          editor.setCursorBufferPosition([1, 0]);
          findView.findEditor.setText('s(o)rt');
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
          return expect(editor.getSelectedBufferRange()).toEqual([[1, 6], [1, 10]]);
        });
        return describe("when an invalid regex is entered", function() {
          return it("displays an error", function() {
            editor.setCursorBufferPosition([2, 0]);
            atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
            findView.findEditor.setText('i[t');
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            return expect(findView.descriptionLabel).toHaveClass('text-error');
          });
        });
      });
      describe("when whole-word is toggled", function() {
        it("toggles whole-word via an event and finds text matching the pattern", function() {
          editor.setCursorBufferPosition([0, 0]);
          findView.findEditor.setText('sort');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(editor.getSelectedBufferRange()).toEqual([[0, 9], [0, 13]]);
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-whole-word-option');
          return expect(editor.getSelectedBufferRange()).toEqual([[1, 6], [1, 10]]);
        });
        it("toggles whole-word via a button and finds text matching the pattern", function() {
          editor.setCursorBufferPosition([0, 0]);
          findView.findEditor.setText('sort');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(editor.getSelectedBufferRange()).toEqual([[0, 9], [0, 13]]);
          findView.wholeWordOptionButton.click();
          return expect(editor.getSelectedBufferRange()).toEqual([[1, 6], [1, 10]]);
        });
        return it("re-runs the search using the new find text when toggled", function() {
          editor.setCursorBufferPosition([8, 0]);
          findView.findEditor.setText('apply');
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-whole-word-option');
          return expect(editor.getSelectedBufferRange()).toEqual([[11, 20], [11, 25]]);
        });
      });
      describe("when case sensitivity is toggled", function() {
        beforeEach(function() {
          editor.setText("-----\nwords\nWORDs\n");
          return editor.setCursorBufferPosition([0, 0]);
        });
        it("toggles case sensitivity via an event and finds text matching the pattern", function() {
          findView.findEditor.setText('WORDs');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(editor.getSelectedBufferRange()).toEqual([[1, 0], [1, 5]]);
          editor.setCursorBufferPosition([0, 0]);
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-case-option');
          return expect(editor.getSelectedBufferRange()).toEqual([[2, 0], [2, 5]]);
        });
        return it("toggles case sensitivity via a button and finds text matching the pattern", function() {
          findView.findEditor.setText('WORDs');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          expect(editor.getSelectedBufferRange()).toEqual([[1, 0], [1, 5]]);
          editor.setCursorBufferPosition([0, 0]);
          findView.caseOptionButton.click();
          return expect(editor.getSelectedBufferRange()).toEqual([[2, 0], [2, 5]]);
        });
      });
      describe("highlighting search results", function() {
        var getResultDecorationMarker;
        getResultDecorationMarker = function(clazz) {
          var _ref1;
          return (_ref1 = getResultDecorations(editor, clazz)[0]) != null ? _ref1.getMarker() : void 0;
        };
        it("only highlights matches", function() {
          expect(getResultDecorations(editor, 'find-result')).toHaveLength(5);
          findView.findEditor.setText('notinthefilebro');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          return runs(function() {
            return expect(getResultDecorations(editor, 'find-result')).toHaveLength(0);
          });
        });
        it("adds a class to the current match indicating it is the current match", function() {
          var firstResultMarker, nextResultMarker, originalResultMarker;
          firstResultMarker = getResultDecorationMarker('current-result');
          expect(getResultDecorations(editor, 'find-result')).toHaveLength(5);
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          nextResultMarker = getResultDecorationMarker('current-result');
          expect(nextResultMarker).not.toEqual(firstResultMarker);
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-previous');
          atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:find-previous');
          originalResultMarker = getResultDecorationMarker('current-result');
          return expect(originalResultMarker).toEqual(firstResultMarker);
        });
        return it("adds a class to the result when the current selection equals the result's range", function() {
          var newResultMarker, originalResultMarker;
          originalResultMarker = getResultDecorationMarker('current-result');
          expect(originalResultMarker).toBeDefined();
          editor.setSelectedBufferRange([[5, 16], [5, 20]]);
          expect(getResultDecorationMarker('current-result')).toBeUndefined();
          editor.setSelectedBufferRange([[5, 16], [5, 21]]);
          newResultMarker = getResultDecorationMarker('current-result');
          expect(newResultMarker).toBeDefined();
          return expect(newResultMarker).not.toBe(originalResultMarker);
        });
      });
      describe("when user types in the find editor", function() {
        var advance;
        advance = function() {
          return advanceClock(findView.findEditor.getModel().getBuffer().stoppedChangingDelay + 1);
        };
        beforeEach(function() {
          return findView.findEditor.focus();
        });
        return it("updates the search results", function() {
          expect(findView.descriptionLabel.text()).toContain("6 results");
          findView.findEditor.setText('why do I need these 2 lines? The editor does not trigger contents-modified without them');
          advance();
          findView.findEditor.setText('');
          advance();
          expect(findView.descriptionLabel.text()).toContain("Find in Current Buffer");
          expect(findView).toHaveFocus();
          findView.findEditor.setText('sort');
          advance();
          expect(findView.descriptionLabel.text()).toContain("5 results");
          expect(findView).toHaveFocus();
          findView.findEditor.setText('items');
          advance();
          expect(findView.descriptionLabel.text()).toContain("6 results");
          return expect(findView).toHaveFocus();
        });
      });
      return describe("when another find is called", function() {
        var previousMarkers;
        previousMarkers = null;
        beforeEach(function() {
          return previousMarkers = _.clone(editor.getMarkers());
        });
        it("clears existing markers for another search", function() {
          findView.findEditor.setText('notinthefile');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          return expect(editor.getMarkers().length).toEqual(1);
        });
        return it("clears existing markers for an empty search", function() {
          findView.findEditor.setText('');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          return expect(editor.getMarkers().length).toEqual(1);
        });
      });
    });
    describe("replacing", function() {
      beforeEach(function() {
        editor.setCursorBufferPosition([2, 0]);
        atom.commands.dispatch(editorView, 'find-and-replace:show-replace');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          findView.findEditor.setText('items');
          return findView.replaceEditor.setText('cats');
        });
      });
      describe("when the replacement string contains an escaped char", function() {
        describe("when the regex option is chosen", function() {
          beforeEach(function() {
            return atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
          });
          it("inserts tabs and newlines", function() {
            findView.replaceEditor.setText('\\t\\n');
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            return expect(editor.getText()).toMatch(/\t\n/);
          });
          return it("doesn't insert a escaped char if there are multiple backslashs in front of the char", function() {
            findView.replaceEditor.setText('\\\\t\\\t');
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            return expect(editor.getText()).toMatch(/\\t\\\t/);
          });
        });
        return describe("when in normal mode", function() {
          it("inserts backslach n and t", function() {
            findView.replaceEditor.setText('\\t\\n');
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            return expect(editor.getText()).toMatch(/\\t\\n/);
          });
          return it("inserts carriage returns", function() {
            var textWithCarriageReturns;
            textWithCarriageReturns = editor.getText().replace(/\n/g, "\r");
            editor.setText(textWithCarriageReturns);
            findView.replaceEditor.setText('\\t\\r');
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            return expect(editor.getText()).toMatch(/\\t\\r/);
          });
        });
      });
      describe("replace next", function() {
        describe("when core:confirm is triggered", function() {
          it("replaces the match after the cursor and selects the next match", function() {
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            expect(findView.resultCounter.text()).toEqual('2 of 5');
            expect(editor.lineTextForBufferRow(2)).toBe("    if (cats.length <= 1) return items;");
            return expect(editor.getSelectedBufferRange()).toEqual([[2, 33], [2, 38]]);
          });
          it("replaceEditor maintains focus after core:confirm is run", function() {
            findView.replaceEditor.focus();
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            return expect(findView.replaceEditor).toHaveFocus();
          });
          return it("replaces the _current_ match and selects the next match", function() {
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            editor.setSelectedBufferRange([[2, 8], [2, 13]]);
            expect(findView.resultCounter.text()).toEqual('2 of 6');
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            expect(findView.resultCounter.text()).toEqual('2 of 5');
            expect(editor.lineTextForBufferRow(2)).toBe("    if (cats.length <= 1) return items;");
            expect(editor.getSelectedBufferRange()).toEqual([[2, 33], [2, 38]]);
            atom.commands.dispatch(findView.replaceEditor.element, 'core:confirm');
            expect(findView.resultCounter.text()).toEqual('2 of 4');
            expect(editor.lineTextForBufferRow(2)).toBe("    if (cats.length <= 1) return cats;");
            return expect(editor.getSelectedBufferRange()).toEqual([[3, 16], [3, 21]]);
          });
        });
        describe("when the replace next button is pressed", function() {
          return it("replaces the match after the cursor and selects the next match", function() {
            $('.find-and-replace .btn-next').click();
            expect(findView.resultCounter.text()).toEqual('2 of 5');
            expect(editor.lineTextForBufferRow(2)).toBe("    if (cats.length <= 1) return items;");
            expect(editor.getSelectedBufferRange()).toEqual([[2, 33], [2, 38]]);
            return expect(editorView).toHaveFocus();
          });
        });
        return describe("when the 'find-and-replace:replace-next' event is triggered", function() {
          return it("replaces the match after the cursor and selects the next match", function() {
            atom.commands.dispatch(editorView, 'find-and-replace:replace-next');
            expect(findView.resultCounter.text()).toEqual('2 of 5');
            expect(editor.lineTextForBufferRow(2)).toBe("    if (cats.length <= 1) return items;");
            return expect(editor.getSelectedBufferRange()).toEqual([[2, 33], [2, 38]]);
          });
        });
      });
      describe("replace previous", function() {
        return describe("when command is triggered", function() {
          return it("replaces the match after the cursor and selects the previous match", function() {
            atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
            atom.commands.dispatch(findView.element, 'find-and-replace:replace-previous');
            expect(findView.resultCounter.text()).toEqual('1 of 5');
            expect(editor.lineTextForBufferRow(2)).toBe("    if (cats.length <= 1) return items;");
            return expect(editor.getSelectedBufferRange()).toEqual([[1, 22], [1, 27]]);
          });
        });
      });
      describe("replace all", function() {
        describe("when the replace all button is pressed", function() {
          it("replaces all matched text", function() {
            $('.find-and-replace .btn-all').click();
            expect(findView.resultCounter.text()).toEqual('no results');
            expect(editor.getText()).not.toMatch(/items/);
            expect(editor.getText().match(/\bcats\b/g)).toHaveLength(6);
            return expect(editor.getSelectedBufferRange()).toEqual([[2, 0], [2, 0]]);
          });
          return it("all changes are undoable in one transaction", function() {
            $('.find-and-replace .btn-all').click();
            editor.undo();
            return expect(editor.getText()).not.toMatch(/\bcats\b/g);
          });
        });
        return describe("when the 'find-and-replace:replace-all' event is triggered", function() {
          return it("replaces all matched text", function() {
            atom.commands.dispatch(editorView, 'find-and-replace:replace-all');
            expect(findView.resultCounter.text()).toEqual('no results');
            expect(editor.getText()).not.toMatch(/items/);
            expect(editor.getText().match(/\bcats\b/g)).toHaveLength(6);
            return expect(editor.getSelectedBufferRange()).toEqual([[2, 0], [2, 0]]);
          });
        });
      });
      return describe("replacement patterns", function() {
        describe("when the regex option is true", function() {
          return it("replaces $1, $2, etc... with substring matches", function() {
            atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:toggle-regex-option');
            findView.findEditor.setText('(items)([\\.;])');
            findView.replaceEditor.setText('$2$1');
            atom.commands.dispatch(editorView, 'find-and-replace:replace-all');
            expect(editor.getText()).toMatch(/;items/);
            return expect(editor.getText()).toMatch(/\.items/);
          });
        });
        return describe("when the regex option is false", function() {
          return it("replaces the matches with without any regex subsitions", function() {
            findView.findEditor.setText('items');
            findView.replaceEditor.setText('$&cats');
            atom.commands.dispatch(editorView, 'find-and-replace:replace-all');
            expect(editor.getText()).not.toMatch(/items/);
            return expect(editor.getText().match(/\$&cats\b/g)).toHaveLength(6);
          });
        });
      });
    });
    describe("history", function() {
      beforeEach(function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        return waitsForPromise(function() {
          return activationPromise;
        });
      });
      describe("when there is no history", function() {
        return it("retains unsearched text", function() {
          var text;
          text = 'something I want to search for but havent yet';
          findView.findEditor.setText(text);
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          return expect(findView.findEditor.getText()).toEqual(text);
        });
      });
      return describe("when there is history", function() {
        var oneRange, threeRange, twoRange, _ref1;
        _ref1 = [], oneRange = _ref1[0], twoRange = _ref1[1], threeRange = _ref1[2];
        beforeEach(function() {
          atom.commands.dispatch(editorView, 'find-and-replace:show');
          editor.setText("zero\none\ntwo\nthree\n");
          findView.findEditor.setText('one');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          findView.findEditor.setText('two');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          findView.findEditor.setText('three');
          return atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
        });
        it("can navigate the entire history stack", function() {
          expect(findView.findEditor.getText()).toEqual('three');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          expect(findView.findEditor.getText()).toEqual('');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          expect(findView.findEditor.getText()).toEqual('');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('three');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('two');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('one');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('one');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          return expect(findView.findEditor.getText()).toEqual('two');
        });
        it("retains the current unsearched text", function() {
          var text;
          text = 'something I want to search for but havent yet';
          findView.findEditor.setText(text);
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('three');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          expect(findView.findEditor.getText()).toEqual(text);
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('three');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-down');
          return expect(findView.findEditor.getText()).toEqual('');
        });
        it("adds confirmed patterns to the history", function() {
          findView.findEditor.setText("cool stuff");
          atom.commands.dispatch(findView.findEditor.element, 'core:confirm');
          findView.findEditor.setText("cooler stuff");
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          expect(findView.findEditor.getText()).toEqual('cool stuff');
          atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
          return expect(findView.findEditor.getText()).toEqual('three');
        });
        return describe("when user types in the find editor", function() {
          var advance;
          advance = function() {
            return advanceClock(findView.findEditor.getModel().getBuffer().stoppedChangingDelay + 1);
          };
          beforeEach(function() {
            return findView.findEditor.focus();
          });
          return it("does not add live searches to the history", function() {
            expect(findView.descriptionLabel.text()).toContain("1 result");
            findView.findEditor.setText('FIXME: necessary search for some reason??');
            advance();
            findView.findEditor.setText('nope');
            advance();
            expect(findView.descriptionLabel.text()).toContain('nope');
            findView.findEditor.setText('zero');
            advance();
            expect(findView.descriptionLabel.text()).toContain("zero");
            atom.commands.dispatch(findView.findEditor.element, 'core:move-up');
            return expect(findView.findEditor.getText()).toEqual('three');
          });
        });
      });
    });
    return describe("panel focus", function() {
      beforeEach(function() {
        atom.commands.dispatch(editorView, 'find-and-replace:show');
        return waitsForPromise(function() {
          return activationPromise;
        });
      });
      it("focuses the find editor when the panel gets focus", function() {
        findView.replaceEditor.focus();
        expect(findView.replaceEditor).toHaveFocus();
        findView.focus();
        return expect(findView.findEditor).toHaveFocus();
      });
      return it("moves focus between editors with find-and-replace:focus-next", function() {
        findView.findEditor.focus();
        expect(findView.findEditor).toHaveClass('is-focused');
        expect(findView.replaceEditor).not.toHaveClass('is-focused');
        atom.commands.dispatch(findView.findEditor.element, 'find-and-replace:focus-next');
        expect(findView.findEditor).not.toHaveClass('is-focused');
        expect(findView.replaceEditor).toHaveClass('is-focused');
        atom.commands.dispatch(findView.replaceEditor.element, 'find-and-replace:focus-next');
        expect(findView.findEditor).toHaveClass('is-focused');
        return expect(findView.replaceEditor).not.toHaveClass('is-focused');
      });
    });
  });

}).call(this);
