(function() {
  var $, Q, ResultsPaneView, View, fs, os, path, temp, waitsForPromise, _, _ref,
    __hasProp = {}.hasOwnProperty,
    __extends = function(child, parent) { for (var key in parent) { if (__hasProp.call(parent, key)) child[key] = parent[key]; } function ctor() { this.constructor = child; } ctor.prototype = parent.prototype; child.prototype = new ctor(); child.__super__ = parent.prototype; return child; };

  os = require('os');

  path = require('path');

  temp = require('temp');

  _ = require('underscore-plus');

  _ref = require('atom-space-pen-views'), $ = _ref.$, View = _ref.View;

  fs = require('fs-plus');

  Q = require('q');

  ResultsPaneView = require('../lib/project/results-pane');

  waitsForPromise = function(fn) {
    return window.waitsForPromise({
      timeout: 30000
    }, fn);
  };

  describe('ProjectFindView', function() {
    var activationPromise, editor, editorView, getAtomPanel, getExistingResultsPane, projectFindView, resultsPane, searchPromise, workspaceElement, _ref1;
    _ref1 = [], activationPromise = _ref1[0], editor = _ref1[1], editorView = _ref1[2], projectFindView = _ref1[3], searchPromise = _ref1[4], resultsPane = _ref1[5], workspaceElement = _ref1[6];
    getAtomPanel = function() {
      return workspaceElement.querySelector('.project-find').parentNode;
    };
    getExistingResultsPane = function() {
      var pane;
      pane = atom.workspace.paneForURI(ResultsPaneView.URI);
      if (pane != null) {
        return pane.itemForURI(ResultsPaneView.URI);
      }
      return null;
    };
    beforeEach(function() {
      workspaceElement = atom.views.getView(atom.workspace);
      atom.project.setPaths([path.join(__dirname, 'fixtures')]);
      jasmine.attachToDOM(workspaceElement);
      atom.config.set('find-and-replace.openProjectFindResultsInRightPane', false);
      return activationPromise = atom.packages.activatePackage("find-and-replace").then(function(_arg) {
        var mainModule, spy;
        mainModule = _arg.mainModule;
        mainModule.createViews();
        projectFindView = mainModule.projectFindView;
        return spy = spyOn(projectFindView, 'confirm').andCallFake(function() {
          searchPromise = spy.originalValue.call(projectFindView);
          resultsPane = $(workspaceElement).find('.preview-pane').view();
          return searchPromise;
        });
      });
    });
    describe("when project-find:show is triggered", function() {
      it("attaches ProjectFindView to the root view", function() {
        atom.commands.dispatch(workspaceElement, 'project-find:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          projectFindView.findEditor.setText('items');
          expect(getAtomPanel()).toBeVisible();
          expect(projectFindView.find('.preview-block')).not.toBeVisible();
          expect(projectFindView.find('.loading')).not.toBeVisible();
          return expect(projectFindView.findEditor.getModel().getSelectedBufferRange()).toEqual([[0, 0], [0, 5]]);
        });
      });
      describe("with an open buffer", function() {
        editor = null;
        beforeEach(function() {
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          waitsForPromise(function() {
            return activationPromise;
          });
          runs(function() {
            return projectFindView.findEditor.setText('');
          });
          return waitsForPromise(function() {
            return atom.workspace.open('sample.js').then(function(o) {
              return editor = o;
            });
          });
        });
        it("populates the findEditor with selection when there is a selection", function() {
          editor.setSelectedBufferRange([[2, 8], [2, 13]]);
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          expect(getAtomPanel()).toBeVisible();
          expect(projectFindView.findEditor.getText()).toBe('items');
          editor.setSelectedBufferRange([[2, 14], [2, 20]]);
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          expect(getAtomPanel()).toBeVisible();
          return expect(projectFindView.findEditor.getText()).toBe('length');
        });
        return it("populates the findEditor with the previous selection when there is no selection", function() {
          editor.setSelectedBufferRange([[2, 14], [2, 20]]);
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          expect(getAtomPanel()).toBeVisible();
          expect(projectFindView.findEditor.getText()).toBe('length');
          editor.setSelectedBufferRange([[2, 30], [2, 30]]);
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          expect(getAtomPanel()).toBeVisible();
          return expect(projectFindView.findEditor.getText()).toBe('length');
        });
      });
      describe("when the ProjectFindView is already attached", function() {
        beforeEach(function() {
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          waitsForPromise(function() {
            return activationPromise;
          });
          return runs(function() {
            projectFindView.findEditor.setText('items');
            return projectFindView.findEditor.getModel().setSelectedBufferRange([[0, 0], [0, 0]]);
          });
        });
        return it("focuses the find editor and selects all the text", function() {
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          expect(projectFindView.findEditor).toHaveFocus();
          return expect(projectFindView.findEditor.getModel().getSelectedText()).toBe("items");
        });
      });
      return it("honors config settings for find options", function() {
        atom.config.set('find-and-replace.useRegex', true);
        atom.config.set('find-and-replace.caseSensitive', true);
        atom.commands.dispatch(workspaceElement, 'project-find:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(projectFindView.caseOptionButton).toHaveClass('selected');
          return expect(projectFindView.regexOptionButton).toHaveClass('selected');
        });
      });
    });
    describe("when project-find:toggle is triggered", function() {
      return it("toggles the visibility of the ProjectFindView", function() {
        atom.commands.dispatch(workspaceElement, 'project-find:toggle');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getAtomPanel()).toBeVisible();
          atom.commands.dispatch(workspaceElement, 'project-find:toggle');
          return expect(getAtomPanel()).not.toBeVisible();
        });
      });
    });
    describe("when project-find:show-in-current-directory is triggered", function() {
      var DirElement, FileElement, nested, projectPath, tree, _ref2;
      _ref2 = [], nested = _ref2[0], tree = _ref2[1], projectPath = _ref2[2];
      DirElement = (function(_super) {
        __extends(DirElement, _super);

        function DirElement() {
          return DirElement.__super__.constructor.apply(this, arguments);
        }

        DirElement.content = function(path) {
          return this.div({
            "class": 'directory'
          }, (function(_this) {
            return function() {
              return _this.div({
                "class": 'nested-thing'
              }, function() {
                _this.span({
                  outlet: 'name',
                  "class": 'name',
                  'data-path': path
                }, path);
                return _this.ul({
                  outlet: 'files',
                  "class": 'files'
                });
              });
            };
          })(this));
        };

        DirElement.prototype.initialize = function(path) {
          this.path = path;
        };

        DirElement.prototype.createFiles = function(names) {
          var name, _i, _len, _results;
          _results = [];
          for (_i = 0, _len = names.length; _i < _len; _i++) {
            name = names[_i];
            _results.push(this.files.append(new FileElement(path.join(this.path, name), name)));
          }
          return _results;
        };

        return DirElement;

      })(View);
      FileElement = (function(_super) {
        __extends(FileElement, _super);

        function FileElement() {
          return FileElement.__super__.constructor.apply(this, arguments);
        }

        FileElement.content = function(path, name) {
          return this.li({
            "class": 'file',
            'data-path': path
          }, (function(_this) {
            return function() {
              return _this.span({
                outlet: 'name',
                "class": 'name'
              }, name);
            };
          })(this));
        };

        FileElement.prototype.initialize = function(path) {
          return fs.writeFileSync(path, '');
        };

        return FileElement;

      })(View);
      beforeEach(function() {
        var p;
        projectPath = temp.mkdirSync("atom");
        atom.project.setPaths([projectPath]);
        p = atom.project.getPaths()[0];
        tree = new DirElement(p);
        tree.createFiles(['one.js', 'two.js']);
        nested = new DirElement(path.join(p, 'nested'));
        nested.createFiles(['another.js']);
        tree.files.append(nested);
        return workspaceElement.appendChild(tree[0]);
      });
      it("populates the pathsEditor when triggered with a directory", function() {
        atom.commands.dispatch(nested.name[0], 'project-find:show-in-current-directory');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getAtomPanel()).toBeVisible();
          expect(projectFindView.pathsEditor.getText()).toBe('nested');
          expect(projectFindView.findEditor).toHaveFocus();
          atom.commands.dispatch(tree.name[0], 'project-find:show-in-current-directory');
          return expect(projectFindView.pathsEditor.getText()).toBe('');
        });
      });
      it("populates the pathsEditor when triggered on a directory's name", function() {
        atom.commands.dispatch(nested[0], 'project-find:show-in-current-directory');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getAtomPanel()).toBeVisible();
          expect(projectFindView.pathsEditor.getText()).toBe('nested');
          expect(projectFindView.findEditor).toHaveFocus();
          atom.commands.dispatch(tree.name[0], 'project-find:show-in-current-directory');
          return expect(projectFindView.pathsEditor.getText()).toBe('');
        });
      });
      it("populates the pathsEditor when triggered on a file", function() {
        atom.commands.dispatch(nested.files.find('> .file:eq(0)').view().name[0], 'project-find:show-in-current-directory');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          expect(getAtomPanel()).toBeVisible();
          expect(projectFindView.pathsEditor.getText()).toBe('nested');
          expect(projectFindView.findEditor).toHaveFocus();
          atom.commands.dispatch(tree.files.find('> .file:eq(0)').view().name[0], 'project-find:show-in-current-directory');
          return expect(projectFindView.pathsEditor.getText()).toBe('');
        });
      });
      return describe("when there are multiple root directories", function() {
        beforeEach(function() {
          return atom.project.addPath(temp.mkdirSync("another-path-"));
        });
        return it("includes the basename of the containing root directory in the paths-editor", function() {
          atom.commands.dispatch(nested.files.find('> .file:eq(0)').view().name[0], 'project-find:show-in-current-directory');
          waitsForPromise(function() {
            return activationPromise;
          });
          return runs(function() {
            expect(getAtomPanel()).toBeVisible();
            return expect(projectFindView.pathsEditor.getText()).toBe(path.join(path.basename(projectPath), 'nested'));
          });
        });
      });
    });
    describe("finding", function() {
      beforeEach(function() {
        waitsForPromise(function() {
          return atom.workspace.open('sample.js');
        });
        runs(function() {
          editor = atom.workspace.getActiveTextEditor();
          editorView = atom.views.getView(editor);
          return atom.commands.dispatch(workspaceElement, 'project-find:show');
        });
        return waitsForPromise(function() {
          return activationPromise;
        });
      });
      describe("when the find string contains an escaped char", function() {
        beforeEach(function() {
          var projectPath;
          projectPath = temp.mkdirSync("atom");
          fs.writeFileSync(path.join(projectPath, "tabs.txt"), "\t\n\\\t\n\\\\t");
          atom.project.setPaths([projectPath]);
          return atom.commands.dispatch(workspaceElement, 'project-find:show');
        });
        describe("when regex seach is enabled", function() {
          return it("finds a literal tab character", function() {
            atom.commands.dispatch(projectFindView[0], 'project-find:toggle-regex-option');
            projectFindView.findEditor.setText('\\t');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              expect(resultsView).toBeVisible();
              return expect(resultsView.find("li > ul > li")).toHaveLength(2);
            });
          });
        });
        return describe("when regex seach is disabled", function() {
          it("finds the escape char", function() {
            projectFindView.findEditor.setText('\\t');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              expect(resultsView).toBeVisible();
              return expect(resultsView.find("li > ul > li")).toHaveLength(1);
            });
          });
          it("finds a backslash", function() {
            projectFindView.findEditor.setText('\\');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              expect(resultsView).toBeVisible();
              return expect(resultsView.find("li > ul > li")).toHaveLength(3);
            });
          });
          return it("doesn't insert a escaped char if there are multiple backslashs in front of the char", function() {
            projectFindView.findEditor.setText('\\\\t');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              expect(resultsView).toBeVisible();
              return expect(resultsView.find("li > ul > li")).toHaveLength(1);
            });
          });
        });
      });
      describe("when core:cancel is triggered", function() {
        beforeEach(function() {
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          return projectFindView.focus();
        });
        return it("detaches from the root view", function() {
          atom.commands.dispatch(document.activeElement, 'core:cancel');
          return expect(getAtomPanel()).not.toBeVisible();
        });
      });
      describe("splitting into a second pane", function() {
        beforeEach(function() {
          workspaceElement.style.height = '1000px';
          return atom.commands.dispatch(editorView, 'project-find:show');
        });
        it("splits when option is true", function() {
          var initialPane;
          initialPane = atom.workspace.getActivePane();
          atom.config.set('find-and-replace.openProjectFindResultsInRightPane', true);
          projectFindView.findEditor.setText('items');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            var pane1;
            pane1 = atom.workspace.getActivePane();
            return expect(pane1).not.toBe(initialPane);
          });
        });
        it("does not split when option is false", function() {
          var initialPane;
          initialPane = atom.workspace.getActivePane();
          projectFindView.findEditor.setText('items');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            var pane1;
            pane1 = atom.workspace.getActivePane();
            return expect(pane1).toBe(initialPane);
          });
        });
        return it("can be duplicated", function() {
          atom.config.set('find-and-replace.openProjectFindResultsInRightPane', true);
          projectFindView.findEditor.setText('items');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            var length, pane1, pane2, resultsPaneView1, resultsPaneView2;
            resultsPaneView1 = atom.views.getView(getExistingResultsPane());
            pane1 = atom.workspace.getActivePane();
            pane1.splitRight({
              copyActiveItem: true
            });
            pane2 = atom.workspace.getActivePane();
            resultsPaneView2 = atom.views.getView(pane2.itemForURI(ResultsPaneView.URI));
            expect(pane1).not.toBe(pane2);
            expect(resultsPaneView1).not.toBe(resultsPaneView2);
            length = resultsPaneView1.querySelectorAll('li > ul > li').length;
            expect(length).toBeGreaterThan(0);
            expect(resultsPaneView2.querySelectorAll('li > ul > li')).toHaveLength(length);
            return expect(resultsPaneView2.querySelector('.preview-count').innerHTML).toEqual(resultsPaneView1.querySelector('.preview-count').innerHTML);
          });
        });
      });
      describe("serialization", function() {
        return it("serializes if the case and regex options", function() {
          atom.commands.dispatch(editorView, 'project-find:show');
          expect(projectFindView.caseOptionButton).not.toHaveClass('selected');
          projectFindView.caseOptionButton.click();
          expect(projectFindView.caseOptionButton).toHaveClass('selected');
          expect(projectFindView.regexOptionButton).not.toHaveClass('selected');
          projectFindView.regexOptionButton.click();
          expect(projectFindView.regexOptionButton).toHaveClass('selected');
          atom.packages.deactivatePackage("find-and-replace");
          activationPromise = atom.packages.activatePackage("find-and-replace").then(function(_arg) {
            var mainModule;
            mainModule = _arg.mainModule;
            mainModule.createViews();
            return projectFindView = mainModule.projectFindView, mainModule;
          });
          atom.commands.dispatch(editorView, 'project-find:show');
          waitsForPromise(function() {
            return activationPromise;
          });
          return runs(function() {
            expect(projectFindView.caseOptionButton).toHaveClass('selected');
            return expect(projectFindView.regexOptionButton).toHaveClass('selected');
          });
        });
      });
      describe("description label", function() {
        beforeEach(function() {
          atom.commands.dispatch(editorView, 'project-find:show');
          atom.commands.dispatch(projectFindView[0], 'project-find:toggle-regex-option');
          return spyOn(atom.workspace, 'scan').andCallFake(function() {
            return Q();
          });
        });
        return it("shows an error when the pattern is invalid and clears when no error", function() {
          projectFindView.findEditor.setText('[');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          runs(function() {
            expect(projectFindView.descriptionLabel).toHaveClass('text-error');
            expect(projectFindView.descriptionLabel.text()).toContain('Invalid regular expression');
            projectFindView.findEditor.setText('');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            expect(projectFindView.descriptionLabel).not.toHaveClass('text-error');
            expect(projectFindView.descriptionLabel.text()).toContain('Find in Project');
            projectFindView.findEditor.setText('items');
            return atom.commands.dispatch(projectFindView[0], 'core:confirm');
          });
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            expect(projectFindView.descriptionLabel).not.toHaveClass('text-error');
            return expect(projectFindView.descriptionLabel.text()).toContain('items');
          });
        });
      });
      describe("regex", function() {
        beforeEach(function() {
          atom.commands.dispatch(editorView, 'project-find:show');
          projectFindView.findEditor.setText('i(\\w)ems+');
          return spyOn(atom.workspace, 'scan').andCallFake(function() {
            return Q();
          });
        });
        it("escapes regex patterns by default", function() {
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            return expect(atom.workspace.scan.argsForCall[0][0]).toEqual(/i\(\\w\)ems\+/gi);
          });
        });
        it("shows an error when the regex pattern is invalid", function() {
          atom.commands.dispatch(projectFindView[0], 'project-find:toggle-regex-option');
          projectFindView.findEditor.setText('[');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            return expect(projectFindView.descriptionLabel).toHaveClass('text-error');
          });
        });
        describe("when search has not been run yet", function() {
          return it("toggles regex option via an event but does not run the search", function() {
            expect(projectFindView.regexOptionButton).not.toHaveClass('selected');
            atom.commands.dispatch(projectFindView[0], 'project-find:toggle-regex-option');
            expect(projectFindView.regexOptionButton).toHaveClass('selected');
            return expect(atom.workspace.scan).not.toHaveBeenCalled();
          });
        });
        return describe("when search has been run", function() {
          beforeEach(function() {
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            return waitsForPromise(function() {
              return searchPromise;
            });
          });
          it("toggles regex option via an event and finds files matching the pattern", function() {
            expect(projectFindView.regexOptionButton).not.toHaveClass('selected');
            atom.commands.dispatch(projectFindView[0], 'project-find:toggle-regex-option');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              expect(projectFindView.regexOptionButton).toHaveClass('selected');
              return expect(atom.workspace.scan.mostRecentCall.args[0]).toEqual(/i(\w)ems+/gi);
            });
          });
          return it("toggles regex option via a button and finds files matching the pattern", function() {
            expect(projectFindView.regexOptionButton).not.toHaveClass('selected');
            projectFindView.regexOptionButton.click();
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              expect(projectFindView.regexOptionButton).toHaveClass('selected');
              return expect(atom.workspace.scan.mostRecentCall.args[0]).toEqual(/i(\w)ems+/gi);
            });
          });
        });
      });
      describe("case sensitivity", function() {
        beforeEach(function() {
          atom.commands.dispatch(editorView, 'project-find:show');
          spyOn(atom.workspace, 'scan').andCallFake(function() {
            return Q();
          });
          projectFindView.findEditor.setText('ITEMS');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          return waitsForPromise(function() {
            return searchPromise;
          });
        });
        it("runs a case insensitive search by default", function() {
          return expect(atom.workspace.scan.argsForCall[0][0]).toEqual(/ITEMS/gi);
        });
        it("toggles case sensitive option via an event and finds files matching the pattern", function() {
          expect(projectFindView.caseOptionButton).not.toHaveClass('selected');
          atom.commands.dispatch(projectFindView[0], 'project-find:toggle-case-option');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            expect(projectFindView.caseOptionButton).toHaveClass('selected');
            return expect(atom.workspace.scan.mostRecentCall.args[0]).toEqual(/ITEMS/g);
          });
        });
        return it("toggles case sensitive option via a button and finds files matching the pattern", function() {
          expect(projectFindView.caseOptionButton).not.toHaveClass('selected');
          projectFindView.caseOptionButton.click();
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            expect(projectFindView.caseOptionButton).toHaveClass('selected');
            return expect(atom.workspace.scan.mostRecentCall.args[0]).toEqual(/ITEMS/g);
          });
        });
      });
      describe("when project-find:confirm is triggered", function() {
        return it("displays the results and no errors", function() {
          projectFindView.findEditor.setText('items');
          atom.commands.dispatch(projectFindView[0], 'project-find:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            var resultsPaneView, resultsView;
            resultsPaneView = getExistingResultsPane();
            resultsView = resultsPaneView.resultsView;
            expect(resultsView).toBeVisible();
            resultsView.scrollToBottom();
            return expect(resultsView.find("li > ul > li")).toHaveLength(13);
          });
        });
      });
      describe("when core:confirm is triggered", function() {
        beforeEach(function() {
          return atom.commands.dispatch(workspaceElement, 'project-find:show');
        });
        describe("when the there search field is empty", function() {
          return it("does not run the seach but clears the model", function() {
            spyOn(atom.workspace, 'scan');
            spyOn(projectFindView.model, 'clear');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            expect(atom.workspace.scan).not.toHaveBeenCalled();
            return expect(projectFindView.model.clear).toHaveBeenCalled();
          });
        });
        it("reruns the search when confirmed again after focusing the window", function() {
          projectFindView.findEditor.setText('thisdoesnotmatch');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          runs(function() {
            spyOn(atom.workspace, 'scan');
            return atom.commands.dispatch(projectFindView[0], 'core:confirm');
          });
          waitsForPromise(function() {
            return searchPromise;
          });
          runs(function() {
            expect(atom.workspace.scan).not.toHaveBeenCalled();
            atom.workspace.scan.reset();
            $(window).triggerHandler('focus');
            return atom.commands.dispatch(projectFindView[0], 'core:confirm');
          });
          waitsForPromise(function() {
            return searchPromise;
          });
          runs(function() {
            expect(atom.workspace.scan).toHaveBeenCalled();
            atom.workspace.scan.reset();
            return atom.commands.dispatch(projectFindView[0], 'core:confirm');
          });
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            return expect(atom.workspace.scan).not.toHaveBeenCalled();
          });
        });
        describe("when results exist", function() {
          beforeEach(function() {
            return projectFindView.findEditor.setText('items');
          });
          it("displays the results and no errors", function() {
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              expect(resultsView).toBeVisible();
              resultsView.scrollToBottom();
              expect(resultsView.find("li > ul > li")).toHaveLength(13);
              expect(resultsPaneView.previewCount.text()).toBe("13 results found in 2 files for items");
              return expect(projectFindView.errorMessages).not.toBeVisible();
            });
          });
          it("only searches paths matching text in the path filter", function() {
            spyOn(atom.workspace, 'scan').andCallFake(function() {
              return Q();
            });
            projectFindView.pathsEditor.setText('*.js');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              return expect(atom.workspace.scan.argsForCall[0][1]['paths']).toEqual(['*.js']);
            });
          });
          return it("updates the results list when a buffer changes", function() {
            var buffer;
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            buffer = atom.project.bufferForPathSync('sample.js');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              resultsView.scrollToBottom();
              expect(resultsView.find("li > ul > li")).toHaveLength(13);
              expect(resultsPaneView.previewCount.text()).toBe("13 results found in 2 files for items");
              resultsView.selectFirstResult();
              _.times(7, function() {
                return resultsView.selectNextResult();
              });
              expect(resultsView.find("li > ul:eq(1) > li:eq(0)")).toHaveClass('selected');
              buffer.setText('there is one "items" in this file');
              advanceClock(buffer.stoppedChangingDelay);
              expect(resultsView.find("li > ul > li")).toHaveLength(8);
              expect(resultsPaneView.previewCount.text()).toBe("8 results found in 2 files for items");
              expect(resultsView.find("li > ul:eq(1) > li:eq(0)")).toHaveClass('selected');
              buffer.setText('no matches in this file');
              advanceClock(buffer.stoppedChangingDelay);
              expect(resultsView.find("li > ul > li")).toHaveLength(7);
              return expect(resultsPaneView.previewCount.text()).toBe("7 results found in 1 file for items");
            });
          });
        });
        return describe("when no results exist", function() {
          beforeEach(function() {
            projectFindView.findEditor.setText('notintheprojectbro');
            return spyOn(atom.workspace, 'scan').andCallFake(function() {
              return Q();
            });
          });
          return it("displays no errors and no results", function() {
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            return runs(function() {
              var resultsView;
              resultsView = getExistingResultsPane().resultsView;
              expect(projectFindView.errorMessages).not.toBeVisible();
              expect(resultsView).toBeVisible();
              return expect(resultsView.find("li > ul > li")).toHaveLength(0);
            });
          });
        });
      });
      describe("history", function() {
        beforeEach(function() {
          atom.commands.dispatch(workspaceElement, 'project-find:show');
          spyOn(atom.workspace, 'scan').andCallFake(function() {
            var promise;
            promise = Q();
            promise.cancel = function() {};
            return promise;
          });
          projectFindView.findEditor.setText('sort');
          projectFindView.replaceEditor.setText('bort');
          projectFindView.pathsEditor.setText('abc');
          atom.commands.dispatch(projectFindView.findEditor[0], 'core:confirm');
          projectFindView.findEditor.setText('items');
          projectFindView.replaceEditor.setText('eyetims');
          projectFindView.pathsEditor.setText('def');
          return atom.commands.dispatch(projectFindView.findEditor[0], 'core:confirm');
        });
        return it("can navigate the entire history stack", function() {
          expect(projectFindView.findEditor.getText()).toEqual('items');
          atom.commands.dispatch(projectFindView.findEditor[0], 'core:move-up');
          expect(projectFindView.findEditor.getText()).toEqual('sort');
          atom.commands.dispatch(projectFindView.findEditor[0], 'core:move-down');
          expect(projectFindView.findEditor.getText()).toEqual('items');
          atom.commands.dispatch(projectFindView.findEditor[0], 'core:move-down');
          expect(projectFindView.findEditor.getText()).toEqual('');
          expect(projectFindView.pathsEditor.getText()).toEqual('def');
          atom.commands.dispatch(projectFindView.pathsEditor[0], 'core:move-up');
          expect(projectFindView.pathsEditor.getText()).toEqual('abc');
          atom.commands.dispatch(projectFindView.pathsEditor[0], 'core:move-down');
          expect(projectFindView.pathsEditor.getText()).toEqual('def');
          atom.commands.dispatch(projectFindView.pathsEditor[0], 'core:move-down');
          expect(projectFindView.pathsEditor.getText()).toEqual('');
          expect(projectFindView.replaceEditor.getText()).toEqual('eyetims');
          atom.commands.dispatch(projectFindView.replaceEditor[0], 'core:move-up');
          expect(projectFindView.replaceEditor.getText()).toEqual('bort');
          atom.commands.dispatch(projectFindView.replaceEditor[0], 'core:move-down');
          expect(projectFindView.replaceEditor.getText()).toEqual('eyetims');
          atom.commands.dispatch(projectFindView.replaceEditor[0], 'core:move-down');
          return expect(projectFindView.replaceEditor.getText()).toEqual('');
        });
      });
      describe("when find-and-replace:set-find-pattern is triggered", function() {
        it("places the selected text into the find editor", function() {
          editor.setSelectedBufferRange([[1, 6], [1, 10]]);
          atom.commands.dispatch(workspaceElement, 'find-and-replace:use-selection-as-find-pattern');
          expect(projectFindView.findEditor.getText()).toBe('sort');
          editor.setSelectedBufferRange([[1, 13], [1, 21]]);
          atom.commands.dispatch(workspaceElement, 'find-and-replace:use-selection-as-find-pattern');
          return expect(projectFindView.findEditor.getText()).toBe('function');
        });
        return it("places the previously selected text into the find editor if no selection", function() {
          editor.setSelectedBufferRange([[1, 13], [1, 21]]);
          atom.commands.dispatch(workspaceElement, 'find-and-replace:use-selection-as-find-pattern');
          expect(projectFindView.findEditor.getText()).toBe('function');
          editor.setSelectedBufferRange([[1, 30], [1, 30]]);
          atom.commands.dispatch(workspaceElement, 'find-and-replace:use-selection-as-find-pattern');
          return expect(projectFindView.findEditor.getText()).toBe('');
        });
      });
      describe("when there is an error searching", function() {
        return it("displays the errors in the results pane", function() {
          var callback, called, deferred, errorList, resultsPaneView, _ref2;
          _ref2 = [], callback = _ref2[0], deferred = _ref2[1], called = _ref2[2], resultsPaneView = _ref2[3], errorList = _ref2[4];
          projectFindView.findEditor.setText('items');
          spyOn(atom.workspace, 'scan').andCallFake(function(regex, options, fn) {
            callback = fn;
            deferred = Q.defer();
            called = true;
            return deferred.promise;
          });
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsFor(function() {
            return called;
          });
          runs(function() {
            resultsPaneView = getExistingResultsPane();
            errorList = resultsPaneView.errorList;
            expect(errorList.find("li")).toHaveLength(0);
            callback(null, {
              path: '/some/path.js',
              code: 'ENOENT',
              message: 'Nope'
            });
            expect(errorList).toBeVisible();
            expect(errorList.find("li")).toHaveLength(1);
            callback(null, {
              path: '/some/path.js',
              code: 'ENOENT',
              message: 'Broken'
            });
            return deferred.resolve();
          });
          waitsForPromise(function() {
            return searchPromise;
          });
          return runs(function() {
            expect(errorList).toBeVisible();
            expect(errorList.find("li")).toHaveLength(2);
            expect(errorList.find("li:eq(0)").text()).toBe('Nope');
            return expect(errorList.find("li:eq(1)").text()).toBe('Broken');
          });
        });
      });
      return describe("buffer search sharing of the find pattern", function() {
        var getResultDecorations;
        getResultDecorations = function(clazz) {
          var decoration, decorations, markerId, markerIdForDecorations, resultDecorations, _i, _len;
          markerIdForDecorations = editor.decorationsForScreenRowRange(0, editor.getLineCount());
          resultDecorations = [];
          for (markerId in markerIdForDecorations) {
            decorations = markerIdForDecorations[markerId];
            for (_i = 0, _len = decorations.length; _i < _len; _i++) {
              decoration = decorations[_i];
              if (decoration.getProperties()["class"] === clazz) {
                resultDecorations.push(decoration);
              }
            }
          }
          return resultDecorations;
        };
        return it('highlights the search results in the selected file', function() {
          waitsForPromise(function() {
            return atom.workspace.open('sample.js');
          });
          runs(function() {
            editor = atom.workspace.getActiveTextEditor();
            return expect(getResultDecorations('find-result')).toHaveLength(0);
          });
          runs(function() {
            projectFindView.findEditor.setText('items');
            return atom.commands.dispatch(projectFindView[0], 'core:confirm');
          });
          waitsForPromise(function() {
            return searchPromise;
          });
          runs(function() {
            var resultsPaneView, resultsView;
            resultsPaneView = getExistingResultsPane();
            resultsView = resultsPaneView.resultsView;
            resultsView.scrollToBottom();
            expect(resultsView).toBeVisible();
            expect(resultsView.find("li > ul > li")).toHaveLength(13);
            resultsView.selectFirstResult();
            _.times(10, function() {
              return atom.commands.dispatch(resultsView[0], 'core:move-down');
            });
            return atom.commands.dispatch(resultsView[0], 'core:confirm');
          });
          waits(0);
          return runs(function() {
            var initialSelectedRange;
            expect(getResultDecorations('find-result')).toHaveLength(5);
            expect(getResultDecorations('current-result')).toHaveLength(1);
            expect(workspaceElement).toHaveClass('find-visible');
            initialSelectedRange = editor.getSelectedBufferRange();
            atom.commands.dispatch(atom.views.getView(editor), 'find-and-replace:find-next');
            return expect(editor.getSelectedBufferRange()).not.toEqual(initialSelectedRange);
          });
        });
      });
    });
    describe("replacing", function() {
      var replacePromise, sampleCoffee, sampleJs, testDir, _ref2;
      _ref2 = [], testDir = _ref2[0], sampleJs = _ref2[1], sampleCoffee = _ref2[2], replacePromise = _ref2[3];
      beforeEach(function() {
        testDir = path.join(os.tmpdir(), "atom-find-and-replace");
        fs.makeTreeSync(testDir);
        sampleJs = path.join(testDir, 'sample.js');
        sampleCoffee = path.join(testDir, 'sample.coffee');
        fs.writeFileSync(sampleCoffee, fs.readFileSync(require.resolve('./fixtures/sample.coffee')));
        fs.writeFileSync(sampleJs, fs.readFileSync(require.resolve('./fixtures/sample.js')));
        atom.commands.dispatch(workspaceElement, 'project-find:show');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var spy;
          atom.project.setPaths([testDir]);
          return spy = spyOn(projectFindView, 'replaceAll').andCallFake(function() {
            return replacePromise = spy.originalValue.call(projectFindView);
          });
        });
      });
      afterEach(function() {
        var activePane, item, success, _i, _len, _ref3;
        activePane = atom.workspace.getActivePane();
        _ref3 = (activePane != null ? activePane.getItems() : void 0) || [];
        for (_i = 0, _len = _ref3.length; _i < _len; _i++) {
          item = _ref3[_i];
          if (item.shouldPromptToSave != null) {
            spyOn(item, 'shouldPromptToSave').andReturn(false);
          }
          activePane.destroyItem(item);
        }
        success = false;
        runs(function() {
          var retry;
          return retry = setInterval(function() {
            var e;
            try {
              fs.removeSync(testDir);
              success = true;
              return clearInterval(retry);
            } catch (_error) {
              e = _error;
              return success = false;
            }
          }, 50);
        });
        return waitsFor(function() {
          return success;
        });
      });
      describe("when the replace string contains an escaped char", function() {
        var filePath;
        filePath = null;
        beforeEach(function() {
          var projectPath;
          projectPath = temp.mkdirSync("atom");
          filePath = path.join(projectPath, "tabs.txt");
          fs.writeFileSync(filePath, "a\nb\na");
          atom.project.setPaths([projectPath]);
          return atom.commands.dispatch(workspaceElement, 'project-find:show');
        });
        describe("when the regex option is chosen", function() {
          beforeEach(function() {
            return atom.commands.dispatch(projectFindView[0], 'project-find:toggle-regex-option');
          });
          it("finds the escape char", function() {
            projectFindView.findEditor.setText('a');
            projectFindView.replaceEditor.setText('\\t');
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              var fileContent;
              fileContent = fs.readFileSync(filePath, 'utf8');
              return expect(fileContent).toBe("\t\nb\n\t");
            });
          });
          return it("doesn't insert a escaped char if there are multiple backslashs in front of the char", function() {
            projectFindView.findEditor.setText('a');
            projectFindView.replaceEditor.setText('\\\\t');
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              var fileContent;
              fileContent = fs.readFileSync(filePath, 'utf8');
              return expect(fileContent).toBe("\\t\nb\n\\t");
            });
          });
        });
        return describe("when regex option is not set", function() {
          return it("finds the escape char", function() {
            projectFindView.findEditor.setText('a');
            projectFindView.replaceEditor.setText('\\t');
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              var fileContent;
              fileContent = fs.readFileSync(filePath, 'utf8');
              return expect(fileContent).toBe("\\t\nb\n\\t");
            });
          });
        });
      });
      describe("when the replace button is pressed", function() {
        it("runs the search, and replaces all the matches", function() {
          projectFindView.findEditor.setText('items');
          atom.commands.dispatch(projectFindView[0], 'core:confirm');
          waitsForPromise(function() {
            return searchPromise;
          });
          runs(function() {
            projectFindView.replaceEditor.setText('sunshine');
            return projectFindView.replaceAllButton.click();
          });
          waitsForPromise(function() {
            return replacePromise;
          });
          return runs(function() {
            var sampleCoffeeContent, sampleJsContent;
            expect(projectFindView.errorMessages).not.toBeVisible();
            expect(projectFindView.descriptionLabel.text()).toContain('Replaced');
            sampleJsContent = fs.readFileSync(sampleJs, 'utf8');
            expect(sampleJsContent.match(/items/g)).toBeFalsy();
            expect(sampleJsContent.match(/sunshine/g)).toHaveLength(6);
            sampleCoffeeContent = fs.readFileSync(sampleCoffee, 'utf8');
            expect(sampleCoffeeContent.match(/items/g)).toBeFalsy();
            return expect(sampleCoffeeContent.match(/sunshine/g)).toHaveLength(7);
          });
        });
        return describe("when there are search results after a replace", function() {
          return it("runs the search after the replace", function() {
            projectFindView.findEditor.setText('items');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            waitsForPromise(function() {
              return searchPromise;
            });
            runs(function() {
              projectFindView.replaceEditor.setText('items-123');
              return projectFindView.replaceAllButton.click();
            });
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              expect(projectFindView.errorMessages).not.toBeVisible();
              expect(getExistingResultsPane().previewCount.text()).toContain('13 results found in 2 files for items');
              expect(projectFindView.descriptionLabel.text()).toContain('Replaced items with items-123 13 times in 2 files');
              projectFindView.replaceEditor.setText('cats');
              advanceClock(projectFindView.replaceEditor.getModel().getBuffer().stoppedChangingDelay);
              expect(projectFindView.descriptionLabel.text()).not.toContain('Replaced items');
              return expect(projectFindView.descriptionLabel.text()).toContain("13 results found in 2 files for items");
            });
          });
        });
      });
      describe("when the project-find:replace-all is triggered", function() {
        describe("when there are no results", function() {
          return it("doesnt replace anything", function() {
            projectFindView.findEditor.setText('nopenotinthefile');
            projectFindView.replaceEditor.setText('sunshine');
            spyOn(atom.workspace, 'scan').andCallThrough();
            spyOn(atom, 'beep');
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              expect(atom.workspace.scan).toHaveBeenCalled();
              expect(atom.beep).toHaveBeenCalled();
              return expect(projectFindView.descriptionLabel.text()).toContain("Nothing replaced");
            });
          });
        });
        describe("when no search has been run", function() {
          return it("runs the search then replaces everything", function() {
            projectFindView.findEditor.setText('items');
            projectFindView.replaceEditor.setText('sunshine');
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              return expect(projectFindView.descriptionLabel.text()).toContain("Replaced items with sunshine 13 times in 2 files");
            });
          });
        });
        describe("when the search text has changed since that last search", function() {
          beforeEach(function() {
            projectFindView.findEditor.setText('items');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            return waitsForPromise(function() {
              return searchPromise;
            });
          });
          return it("clears the search results and does another replace", function() {
            spyOn(atom.workspace, 'scan').andCallThrough();
            spyOn(atom, 'beep');
            projectFindView.findEditor.setText('sort');
            projectFindView.replaceEditor.setText('ok');
            expect(projectFindView.resultsView).not.toBeVisible();
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              expect(atom.workspace.scan).toHaveBeenCalled();
              expect(atom.beep).not.toHaveBeenCalled();
              return expect(projectFindView.descriptionLabel.text()).toContain("Replaced sort with ok 10 times in 2 files");
            });
          });
        });
        return describe("when the text in the search box triggered the results", function() {
          beforeEach(function() {
            projectFindView.findEditor.setText('items');
            atom.commands.dispatch(projectFindView[0], 'core:confirm');
            return waitsForPromise(function() {
              return searchPromise;
            });
          });
          return it("runs the search, and replaces all the matches", function() {
            projectFindView.replaceEditor.setText('sunshine');
            atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
            expect(projectFindView.errorMessages).not.toBeVisible();
            waitsForPromise(function() {
              return replacePromise;
            });
            return runs(function() {
              var resultsPaneView, resultsView, sampleCoffeeContent, sampleJsContent;
              resultsPaneView = getExistingResultsPane();
              resultsView = resultsPaneView.resultsView;
              expect(resultsView).toBeVisible();
              expect(resultsView.find("li > ul > li")).toHaveLength(0);
              expect(projectFindView.descriptionLabel.text()).toContain("Replaced items with sunshine 13 times in 2 files");
              sampleJsContent = fs.readFileSync(sampleJs, 'utf8');
              expect(sampleJsContent.match(/items/g)).toBeFalsy();
              expect(sampleJsContent.match(/sunshine/g)).toHaveLength(6);
              sampleCoffeeContent = fs.readFileSync(sampleCoffee, 'utf8');
              expect(sampleCoffeeContent.match(/items/g)).toBeFalsy();
              return expect(sampleCoffeeContent.match(/sunshine/g)).toHaveLength(7);
            });
          });
        });
      });
      return describe("when there is an error replacing", function() {
        return it("displays the errors in the results pane", function() {
          var callback, called, deferred, errorList, resultsPaneView, _ref3;
          _ref3 = [], callback = _ref3[0], deferred = _ref3[1], called = _ref3[2], resultsPaneView = _ref3[3], errorList = _ref3[4];
          projectFindView.findEditor.setText('items');
          projectFindView.replaceEditor.setText('sunshine');
          spyOn(atom.workspace, 'replace').andCallFake(function(regex, replacement, paths, fn) {
            callback = fn;
            deferred = Q.defer();
            called = true;
            return deferred.promise;
          });
          atom.commands.dispatch(projectFindView[0], 'project-find:replace-all');
          waitsFor(function() {
            return called;
          });
          runs(function() {
            resultsPaneView = getExistingResultsPane();
            errorList = resultsPaneView.errorList;
            expect(errorList.find("li")).toHaveLength(0);
            callback(null, {
              path: '/some/path.js',
              code: 'ENOENT',
              message: 'Nope'
            });
            expect(errorList).toBeVisible();
            expect(errorList.find("li")).toHaveLength(1);
            callback(null, {
              path: '/some/path.js',
              code: 'ENOENT',
              message: 'Broken'
            });
            return deferred.resolve();
          });
          waitsForPromise(function() {
            return replacePromise;
          });
          return runs(function() {
            expect(errorList).toBeVisible();
            expect(errorList.find("li")).toHaveLength(2);
            expect(errorList.find("li:eq(0)").text()).toBe('Nope');
            return expect(errorList.find("li:eq(1)").text()).toBe('Broken');
          });
        });
      });
    });
    return describe("panel focus", function() {
      beforeEach(function() {
        atom.commands.dispatch(workspaceElement, 'project-find:show');
        return waitsForPromise(function() {
          return activationPromise;
        });
      });
      it("focuses the find editor when the panel gets focus", function() {
        projectFindView.replaceEditor.focus();
        expect(projectFindView.replaceEditor).toHaveFocus();
        projectFindView.focus();
        return expect(projectFindView.findEditor).toHaveFocus();
      });
      return it("moves focus between editors with find-and-replace:focus-next", function() {
        projectFindView.findEditor.focus();
        expect(projectFindView.findEditor).toHaveClass('is-focused');
        expect(projectFindView.replaceEditor).not.toHaveClass('is-focused');
        expect(projectFindView.pathsEditor).not.toHaveClass('is-focused');
        atom.commands.dispatch(projectFindView.findEditor.element, 'find-and-replace:focus-next');
        expect(projectFindView.findEditor).not.toHaveClass('is-focused');
        expect(projectFindView.replaceEditor).toHaveClass('is-focused');
        expect(projectFindView.pathsEditor).not.toHaveClass('is-focused');
        atom.commands.dispatch(projectFindView.replaceEditor.element, 'find-and-replace:focus-next');
        expect(projectFindView.findEditor).not.toHaveClass('is-focused');
        expect(projectFindView.replaceEditor).not.toHaveClass('is-focused');
        expect(projectFindView.pathsEditor).toHaveClass('is-focused');
        atom.commands.dispatch(projectFindView.replaceEditor.element, 'find-and-replace:focus-next');
        expect(projectFindView.findEditor).toHaveClass('is-focused');
        expect(projectFindView.replaceEditor).not.toHaveClass('is-focused');
        expect(projectFindView.pathsEditor).not.toHaveClass('is-focused');
        atom.commands.dispatch(projectFindView.replaceEditor.element, 'find-and-replace:focus-previous');
        expect(projectFindView.findEditor).not.toHaveClass('is-focused');
        expect(projectFindView.replaceEditor).not.toHaveClass('is-focused');
        expect(projectFindView.pathsEditor).toHaveClass('is-focused');
        atom.commands.dispatch(projectFindView.replaceEditor.element, 'find-and-replace:focus-previous');
        expect(projectFindView.findEditor).not.toHaveClass('is-focused');
        expect(projectFindView.replaceEditor).toHaveClass('is-focused');
        return expect(projectFindView.pathsEditor).not.toHaveClass('is-focused');
      });
    });
  });

}).call(this);
