(function() {
  var SelectNext, path;

  path = require('path');

  SelectNext = require('../lib/select-next');

  describe("SelectNext", function() {
    var editor, editorElement, promise, workspaceElement, _ref;
    _ref = [], workspaceElement = _ref[0], editorElement = _ref[1], editor = _ref[2], promise = _ref[3];
    beforeEach(function() {
      workspaceElement = atom.views.getView(atom.workspace);
      atom.project.setPaths([path.join(__dirname, 'fixtures')]);
      waitsForPromise(function() {
        return atom.workspace.open('sample.js');
      });
      runs(function() {
        jasmine.attachToDOM(workspaceElement);
        editor = atom.workspace.getActiveTextEditor();
        editorElement = atom.views.getView(editor);
        promise = atom.packages.activatePackage("find-and-replace");
        return atom.commands.dispatch(editorElement, 'find-and-replace:show');
      });
      return waitsForPromise(function() {
        return promise;
      });
    });
    describe("find-and-replace:select-next", function() {
      describe("when nothing is selected", function() {
        return it("selects the word under the cursor", function() {
          editor.setCursorBufferPosition([1, 3]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[1, 2], [1, 5]]]);
        });
      });
      describe("when a word is selected", function() {
        return it("selects the next occurrence of the selected word skipping any non-word matches", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[0, 0], [0, 3]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[3, 8], [3, 11]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[3, 8], [3, 11]], [[5, 6], [5, 9]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[3, 8], [3, 11]], [[5, 6], [5, 9]]]);
          editor.setText("Testing reallyTesting");
          editor.setCursorBufferPosition([0, 0]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 7]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 7]]]);
        });
      });
      describe("when part of a word is selected", function() {
        return it("selects the next occurrence of the selected text", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[1, 2], [1, 5]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[1, 2], [1, 5]], [[2, 0], [2, 3]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[1, 2], [1, 5]], [[2, 0], [2, 3]], [[3, 8], [3, 11]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[1, 2], [1, 5]], [[2, 0], [2, 3]], [[3, 8], [3, 11]], [[4, 0], [4, 3]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[1, 2], [1, 5]], [[2, 0], [2, 3]], [[3, 8], [3, 11]], [[4, 0], [4, 3]], [[5, 6], [5, 9]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[1, 2], [1, 5]], [[2, 0], [2, 3]], [[3, 8], [3, 11]], [[4, 0], [4, 3]], [[5, 6], [5, 9]], [[0, 0], [0, 3]]]);
        });
      });
      describe("when a non-word is selected", function() {
        return it("selects the next occurrence of the selected text", function() {
          editor.setText("<!\n<a");
          editor.setSelectedBufferRange([[0, 0], [0, 1]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 1]], [[1, 0], [1, 1]]]);
        });
      });
      return describe("when the word is at a line boundary", function() {
        return it("does not select the newlines", function() {
          editor.setText("a\n\na");
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 1]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 1]], [[2, 0], [2, 1]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 1]], [[2, 0], [2, 1]]]);
        });
      });
    });
    describe("find-and-replace:select-all", function() {
      describe("when there is no selection", function() {
        return it("find and selects all occurrences of the word under the cursor", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          atom.commands.dispatch(editorElement, 'find-and-replace:select-all');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[3, 8], [3, 11]], [[5, 6], [5, 9]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-all');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[3, 8], [3, 11]], [[5, 6], [5, 9]]]);
        });
      });
      describe("when a word is selected", function() {
        return it("find and selects all occurrences", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[3, 8], [3, 11]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-all');
          expect(editor.getSelectedBufferRanges()).toEqual([[[3, 8], [3, 11]], [[0, 0], [0, 3]], [[5, 6], [5, 9]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-all');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[3, 8], [3, 11]], [[0, 0], [0, 3]], [[5, 6], [5, 9]]]);
        });
      });
      return describe("when a non-word is selected", function() {
        return it("selects the next occurrence of the selected text", function() {
          editor.setText("<!\n<a");
          editor.setSelectedBufferRange([[0, 0], [0, 1]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-all');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 1]], [[1, 0], [1, 1]]]);
        });
      });
    });
    describe("find-and-replace:select-undo", function() {
      describe("when there is no selection", function() {
        return it("does nothing", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 0]]]);
        });
      });
      describe("when a word is selected", function() {
        return it("unselects current word", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[3, 8], [3, 11]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[3, 11], [3, 11]]]);
        });
      });
      describe("when two words are selected", function() {
        return it("unselects words in order", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[3, 8], [3, 11]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          expect(editor.getSelectedBufferRanges()).toEqual([[[3, 8], [3, 11]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[3, 11], [3, 11]]]);
        });
      });
      describe("when three words are selected", function() {
        return it("unselects words in order", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[0, 0], [0, 3]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[3, 8], [3, 11]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]]]);
        });
      });
      return describe("when starting at the bottom word", function() {
        it("unselects words in order", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[5, 6], [5, 9]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          expect(editor.getSelectedBufferRanges()).toEqual([[[5, 6], [5, 9]], [[0, 0], [0, 3]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[5, 6], [5, 9]]]);
        });
        return it("doesn't stack previously selected", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[5, 6], [5, 9]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-undo');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[5, 6], [5, 9]], [[0, 0], [0, 3]]]);
        });
      });
    });
    return describe("find-and-replace:select-skip", function() {
      describe("when there is no selection", function() {
        return it("does nothing", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          atom.commands.dispatch(editorElement, 'find-and-replace:select-skip');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 0]]]);
        });
      });
      describe("when a word is selected", function() {
        return it("unselects current word and selects next match", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[3, 8], [3, 11]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-skip');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[5, 6], [5, 9]]]);
        });
      });
      describe("when two words are selected", function() {
        return it("unselects second word and selects next match", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[0, 0], [0, 3]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-skip');
          expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]], [[5, 6], [5, 9]]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-skip');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[0, 0], [0, 3]]]);
        });
      });
      return describe("when starting at the bottom word", function() {
        return it("unselects second word and selects next match", function() {
          editor.setText("for\ninformation\nformat\nanother for\nfork\na 3rd for is here");
          editor.setSelectedBufferRange([[5, 6], [5, 9]]);
          atom.commands.dispatch(editorElement, 'find-and-replace:select-next');
          atom.commands.dispatch(editorElement, 'find-and-replace:select-skip');
          return expect(editor.getSelectedBufferRanges()).toEqual([[[5, 6], [5, 9]], [[3, 8], [3, 11]]]);
        });
      });
    });
  });

}).call(this);
