(function() {
  var fs, path, temp;

  path = require('path');

  fs = require('fs-plus');

  temp = require('temp');

  describe("GitDiff package", function() {
    var editor, editorView, projectPath, _ref;
    _ref = [], editor = _ref[0], editorView = _ref[1], projectPath = _ref[2];
    beforeEach(function() {
      var otherPath;
      spyOn(window, 'setImmediate').andCallFake(function(fn) {
        return fn();
      });
      projectPath = temp.mkdirSync('git-diff-spec-');
      otherPath = temp.mkdirSync('some-other-path-');
      fs.copySync(path.join(__dirname, 'fixtures', 'working-dir'), projectPath);
      fs.moveSync(path.join(projectPath, 'git.git'), path.join(projectPath, '.git'));
      atom.project.setPaths([otherPath, projectPath]);
      jasmine.attachToDOM(atom.views.getView(atom.workspace));
      waitsForPromise(function() {
        return atom.workspace.open(path.join(projectPath, 'sample.js'));
      });
      runs(function() {
        editor = atom.workspace.getActiveTextEditor();
        return editorView = atom.views.getView(editor);
      });
      return waitsForPromise(function() {
        return atom.packages.activatePackage('git-diff');
      });
    });
    describe("when the editor has modified lines", function() {
      return it("highlights the modified lines", function() {
        expect(editorView.rootElement.querySelectorAll('.git-line-modified').length).toBe(0);
        editor.insertText('a');
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        expect(editorView.rootElement.querySelectorAll('.git-line-modified').length).toBe(1);
        return expect(editorView.rootElement.querySelector('.git-line-modified')).toHaveClass('line-number-0');
      });
    });
    describe("when the editor has added lines", function() {
      return it("highlights the added lines", function() {
        expect(editorView.rootElement.querySelectorAll('.git-line-added').length).toBe(0);
        editor.moveToEndOfLine();
        editor.insertNewline();
        editor.insertText('a');
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        expect(editorView.rootElement.querySelectorAll('.git-line-added').length).toBe(1);
        return expect(editorView.rootElement.querySelector('.git-line-added')).toHaveClass('line-number-1');
      });
    });
    describe("when the editor has removed lines", function() {
      return it("highlights the line preceeding the deleted lines", function() {
        expect(editorView.rootElement.querySelectorAll('.git-line-added').length).toBe(0);
        editor.setCursorBufferPosition([5]);
        editor.deleteLine();
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        expect(editorView.rootElement.querySelectorAll('.git-line-removed').length).toBe(1);
        return expect(editorView.rootElement.querySelector('.git-line-removed')).toHaveClass('line-number-4');
      });
    });
    describe("when a modified line is restored to the HEAD version contents", function() {
      return it("removes the diff highlight", function() {
        expect(editorView.rootElement.querySelectorAll('.git-line-modified').length).toBe(0);
        editor.insertText('a');
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        expect(editorView.rootElement.querySelectorAll('.git-line-modified').length).toBe(1);
        editor.backspace();
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        return expect(editorView.rootElement.querySelectorAll('.git-line-modified').length).toBe(0);
      });
    });
    describe("when a modified file is opened", function() {
      return it("highlights the changed lines", function() {
        var nextTick;
        fs.writeFileSync(path.join(projectPath, 'sample.txt'), "Some different text.");
        nextTick = false;
        waitsForPromise(function() {
          return atom.workspace.open(path.join(projectPath, 'sample.txt'));
        });
        runs(function() {
          return editorView = atom.views.getView(atom.workspace.getActiveTextEditor());
        });
        setImmediate(function() {
          return nextTick = true;
        });
        waitsFor(function() {
          return nextTick;
        });
        return runs(function() {
          expect(editorView.rootElement.querySelectorAll('.git-line-modified').length).toBe(1);
          return expect(editorView.rootElement.querySelector('.git-line-modified')).toHaveClass('line-number-0');
        });
      });
    });
    describe("when the project paths change", function() {
      return it("doesn't try to use the destroyed git repository", function() {
        editor.deleteLine();
        atom.project.setPaths([temp.mkdirSync("no-repository")]);
        return advanceClock(editor.getBuffer().stoppedChangingDelay);
      });
    });
    describe("move-to-next-diff/move-to-previous-diff events", function() {
      it("moves the cursor to first character of the next/previous diff line", function() {
        editor.insertText('a');
        editor.setCursorBufferPosition([5]);
        editor.deleteLine();
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        editor.setCursorBufferPosition([0]);
        atom.commands.dispatch(editorView, 'git-diff:move-to-next-diff');
        expect(editor.getCursorBufferPosition()).toEqual([4, 4]);
        atom.commands.dispatch(editorView, 'git-diff:move-to-previous-diff');
        return expect(editor.getCursorBufferPosition()).toEqual([0, 0]);
      });
      return it("wraps around to the first/last diff in the file", function() {
        editor.insertText('a');
        editor.setCursorBufferPosition([5]);
        editor.deleteLine();
        advanceClock(editor.getBuffer().stoppedChangingDelay);
        editor.setCursorBufferPosition([0]);
        atom.commands.dispatch(editorView, 'git-diff:move-to-next-diff');
        expect(editor.getCursorBufferPosition()).toEqual([4, 4]);
        atom.commands.dispatch(editorView, 'git-diff:move-to-next-diff');
        expect(editor.getCursorBufferPosition()).toEqual([0, 0]);
        atom.commands.dispatch(editorView, 'git-diff:move-to-previous-diff');
        return expect(editor.getCursorBufferPosition()).toEqual([4, 4]);
      });
    });
    return describe("when the showIconsInEditorGutter config option is true", function() {
      beforeEach(function() {
        return atom.config.set('git-diff.showIconsInEditorGutter', true);
      });
      it("the gutter has a git-diff-icon class", function() {
        return expect(editorView.rootElement.querySelector('.gutter')).toHaveClass('git-diff-icon');
      });
      it("keeps the git-diff-icon class when editor.showLineNumbers is toggled", function() {
        atom.config.set('editor.showLineNumbers', false);
        expect(editorView.rootElement.querySelector('.gutter')).not.toHaveClass('git-diff-icon');
        atom.config.set('editor.showLineNumbers', true);
        return expect(editorView.rootElement.querySelector('.gutter')).toHaveClass('git-diff-icon');
      });
      return it("removes the git-diff-icon class when the showIconsInEditorGutter config option set to false", function() {
        atom.config.set('git-diff.showIconsInEditorGutter', false);
        return expect(editorView.rootElement.querySelector('.gutter')).not.toHaveClass('git-diff-icon');
      });
    });
  });

}).call(this);
