(function() {
  describe('CSS grammar', function() {
    var grammar;
    grammar = null;
    beforeEach(function() {
      waitsForPromise(function() {
        return atom.packages.activatePackage('language-css');
      });
      return runs(function() {
        return grammar = atom.grammars.grammarForScopeName('source.css');
      });
    });
    it('parses the grammar', function() {
      expect(grammar).toBeTruthy();
      return expect(grammar.scopeName).toBe('source.css');
    });
    describe('custom elements', function() {
      it('tokenizes them as tags', function() {
        var tokens;
        tokens = grammar.tokenizeLine('very-custom { color: red; }').tokens;
        expect(tokens[0]).toEqual({
          value: 'very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
        tokens = grammar.tokenizeLine('very-very-custom { color: red; }').tokens;
        return expect(tokens[0]).toEqual({
          value: 'very-very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
      });
      it('tokenizes them with pseudo selectors', function() {
        var tokens;
        tokens = grammar.tokenizeLine('very-custom:hover { color: red; }').tokens;
        expect(tokens[0]).toEqual({
          value: 'very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
        expect(tokens[1]).toEqual({
          value: ':',
          scopes: ['source.css', 'meta.selector.css', 'entity.other.attribute-name.pseudo-class.css', 'punctuation.definition.entity.css']
        });
        expect(tokens[2]).toEqual({
          value: 'hover',
          scopes: ['source.css', 'meta.selector.css', 'entity.other.attribute-name.pseudo-class.css']
        });
        tokens = grammar.tokenizeLine('very-custom::shadow { color: red; }').tokens;
        expect(tokens[0]).toEqual({
          value: 'very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
        expect(tokens[1]).toEqual({
          value: '::',
          scopes: ['source.css', 'meta.selector.css', 'entity.other.attribute-name.pseudo-class.css', 'punctuation.definition.entity.css']
        });
        return expect(tokens[2]).toEqual({
          value: 'shadow',
          scopes: ['source.css', 'meta.selector.css', 'entity.other.attribute-name.pseudo-class.css']
        });
      });
      it('tokenizes them with class selectors', function() {
        var tokens;
        tokens = grammar.tokenizeLine('very-custom.class { color: red; }').tokens;
        expect(tokens[0]).toEqual({
          value: 'very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
        expect(tokens[1]).toEqual({
          value: '.',
          scopes: ['source.css', 'meta.selector.css', 'entity.other.attribute-name.class.css', 'punctuation.definition.entity.css']
        });
        return expect(tokens[2]).toEqual({
          value: 'class',
          scopes: ['source.css', 'meta.selector.css', 'entity.other.attribute-name.class.css']
        });
      });
      return it('does not confuse them with properties', function() {
        var lines;
        lines = grammar.tokenizeLines("body {\n  border-width: 2;\n  font-size : 2;\n  background-image  : none;\n}");
        expect(lines[1][0]).toEqual({
          value: '  ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(lines[1][1]).toEqual({
          value: 'border-width',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-name.css', 'support.type.property-name.css']
        });
        expect(lines[1][2]).toEqual({
          value: ':',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'punctuation.separator.key-value.css']
        });
        expect(lines[1][3]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css']
        });
        expect(lines[1][4]).toEqual({
          value: '2',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'constant.numeric.css']
        });
        expect(lines[2][0]).toEqual({
          value: '  ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(lines[2][1]).toEqual({
          value: 'font-size',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-name.css', 'support.type.property-name.css']
        });
        expect(lines[2][2]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(lines[2][3]).toEqual({
          value: ':',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'punctuation.separator.key-value.css']
        });
        expect(lines[2][4]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css']
        });
        expect(lines[2][5]).toEqual({
          value: '2',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'constant.numeric.css']
        });
        expect(lines[3][0]).toEqual({
          value: '  ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(lines[3][1]).toEqual({
          value: 'background-image',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-name.css', 'support.type.property-name.css']
        });
        expect(lines[3][2]).toEqual({
          value: '  ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(lines[3][3]).toEqual({
          value: ':',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'punctuation.separator.key-value.css']
        });
        expect(lines[3][4]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css']
        });
        return expect(lines[3][5]).toEqual({
          value: 'none',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'support.constant.property-value.css']
        });
      });
    });
    return describe('character escapes', function() {
      it('can handle long hexadecimal escape sequences in single-quoted strings', function() {
        var tokens;
        tokens = grammar.tokenizeLine("very-custom { content: '\\c0ffee' }").tokens;
        expect(tokens[0]).toEqual({
          value: 'very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
        expect(tokens[1]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.selector.css']
        });
        expect(tokens[2]).toEqual({
          value: '{',
          scopes: ['source.css', 'meta.property-list.css', 'punctuation.section.property-list.begin.css']
        });
        expect(tokens[3]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(tokens[4]).toEqual({
          value: 'content',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-name.css', 'support.type.property-name.css']
        });
        expect(tokens[5]).toEqual({
          value: ':',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'punctuation.separator.key-value.css']
        });
        expect(tokens[6]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css']
        });
        expect(tokens[7]).toEqual({
          value: "'",
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'string.quoted.single.css', 'punctuation.definition.string.begin.css']
        });
        return expect(tokens[8]).toEqual({
          value: '\\c0ffee',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'string.quoted.single.css', 'constant.character.escape.css']
        });
      });
      return it('can handle long hexadecimal escape sequences in double-quoted strings', function() {
        var tokens;
        tokens = grammar.tokenizeLine('very-custom { content: "\\c0ffee" }').tokens;
        expect(tokens[0]).toEqual({
          value: 'very-custom',
          scopes: ['source.css', 'meta.selector.css', 'entity.name.tag.custom.css']
        });
        expect(tokens[1]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.selector.css']
        });
        expect(tokens[2]).toEqual({
          value: '{',
          scopes: ['source.css', 'meta.property-list.css', 'punctuation.section.property-list.begin.css']
        });
        expect(tokens[3]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css']
        });
        expect(tokens[4]).toEqual({
          value: 'content',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-name.css', 'support.type.property-name.css']
        });
        expect(tokens[5]).toEqual({
          value: ':',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'punctuation.separator.key-value.css']
        });
        expect(tokens[6]).toEqual({
          value: ' ',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css']
        });
        expect(tokens[7]).toEqual({
          value: '"',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'string.quoted.double.css', 'punctuation.definition.string.begin.css']
        });
        return expect(tokens[8]).toEqual({
          value: '\\c0ffee',
          scopes: ['source.css', 'meta.property-list.css', 'meta.property-value.css', 'string.quoted.double.css', 'constant.character.escape.css']
        });
      });
    });
  });

}).call(this);
