(function() {
  var __indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  describe('Go grammar', function() {
    var grammar;
    grammar = null;
    beforeEach(function() {
      waitsForPromise(function() {
        return atom.packages.activatePackage('language-go');
      });
      return runs(function() {
        return grammar = atom.grammars.grammarForScopeName('source.go');
      });
    });
    it('parses the grammar', function() {
      expect(grammar).toBeTruthy();
      return expect(grammar.scopeName).toBe('source.go');
    });
    it('tokenizes comments', function() {
      var tokens;
      tokens = grammar.tokenizeLine('// I am a comment').tokens;
      expect(tokens[0].value).toEqual('//');
      expect(tokens[0].scopes).toEqual(['source.go', 'comment.line.double-slash.go', 'punctuation.definition.comment.go']);
      expect(tokens[1].value).toEqual(' I am a comment');
      expect(tokens[1].scopes).toEqual(['source.go', 'comment.line.double-slash.go']);
      tokens = grammar.tokenizeLines('/*\nI am a comment\n*/');
      expect(tokens[0][0].value).toEqual('/*');
      expect(tokens[0][0].scopes).toEqual(['source.go', 'comment.block.go', 'punctuation.definition.comment.go']);
      expect(tokens[1][0].value).toEqual('I am a comment');
      expect(tokens[1][0].scopes).toEqual(['source.go', 'comment.block.go']);
      expect(tokens[2][0].value).toEqual('*/');
      return expect(tokens[2][0].scopes).toEqual(['source.go', 'comment.block.go', 'punctuation.definition.comment.go']);
    });
    it('tokenizes strings', function() {
      var delim, delims, scope, tokens, _results;
      delims = {
        'string.quoted.double.go': '"',
        'string.quoted.raw.go': '`'
      };
      _results = [];
      for (scope in delims) {
        delim = delims[scope];
        tokens = grammar.tokenizeLine(delim + 'I am a string' + delim).tokens;
        expect(tokens[0].value).toEqual(delim);
        expect(tokens[0].scopes).toEqual(['source.go', scope, 'punctuation.definition.string.begin.go']);
        expect(tokens[1].value).toEqual('I am a string');
        expect(tokens[1].scopes).toEqual(['source.go', scope]);
        expect(tokens[2].value).toEqual(delim);
        _results.push(expect(tokens[2].scopes).toEqual(['source.go', scope, 'punctuation.definition.string.end.go']));
      }
      return _results;
    });
    it('tokenizes Printf verbs in strings', function() {
      var tokens, verb, verbs, _i, _len, _results;
      verbs = ['%# x', '%-5s', '%5s', '%05s', '%.5s', '%10.1q', '%10v', '%-10v', '%.0d', '%.d', '%+07.2f', '%0100d', '%0.100f', '%#064x', '%+.3F', '%-#20.8x', '%[1]d', '%[2]*[1]d', '%[3]*.[2]*[1]f', '%[3]*.[2]f', '%3.[2]d', '%.[2]d', '%-+[1]x', '%d', '%-d', '%+d', '%#d', '% d', '%0d', '%1.2d', '%-1.2d', '%+1.2d', '%-+1.2d', '%*d', '%.*d', '%*.*d', '%0*d', '%-*d'];
      _results = [];
      for (_i = 0, _len = verbs.length; _i < _len; _i++) {
        verb = verbs[_i];
        tokens = grammar.tokenizeLine('"' + verb + '"').tokens;
        expect(tokens[0].value).toEqual('"', expect(tokens[0].scopes).toEqual(['source.go', 'string.quoted.double.go', 'punctuation.definition.string.begin.go']));
        expect(tokens[1].value).toEqual(verb);
        expect(tokens[1].scopes).toEqual(['source.go', 'string.quoted.double.go', 'constant.escape.format-verb.go']);
        _results.push(expect(tokens[2].value).toEqual('"', expect(tokens[2].scopes).toEqual(['source.go', 'string.quoted.double.go', 'punctuation.definition.string.end.go'])));
      }
      return _results;
    });
    it('tokenizes character escapes in strings', function() {
      var escape, escapes, tokens, _i, _len;
      escapes = ['\\a', '\\b', '\\f', '\\n', '\\r', '\\t', '\\v', '\\\\', '\\000', '\\007', '\\377', '\\x07', '\\xff', '\\u12e4', '\\U00101234'];
      for (_i = 0, _len = escapes.length; _i < _len; _i++) {
        escape = escapes[_i];
        tokens = grammar.tokenizeLine('"' + escape + '"').tokens;
        expect(tokens[1].value).toEqual(escape);
        expect(tokens[1].scopes).toEqual(['source.go', 'string.quoted.double.go', 'constant.character.escape.go']);
      }
      tokens = grammar.tokenizeLine('"\\""').tokens;
      expect(tokens[1].value).toEqual('\\"');
      return expect(tokens[1].scopes).toEqual(['source.go', 'string.quoted.double.go', 'constant.character.escape.go']);
    });
    it('tokenizes Printf verbs in raw strings', function() {
      var tokens, verb, verbs, _i, _len, _results;
      verbs = ['%# x', '%-5s', '%5s', '%05s', '%.5s', '%10.1q', '%10v', '%-10v', '%.0d', '%.d', '%+07.2f', '%0100d', '%0.100f', '%#064x', '%+.3F', '%-#20.8x', '%[1]d', '%[2]*[1]d', '%[3]*.[2]*[1]f', '%[3]*.[2]f', '%3.[2]d', '%.[2]d', '%-+[1]x', '%d', '%-d', '%+d', '%#d', '% d', '%0d', '%1.2d', '%-1.2d', '%+1.2d', '%-+1.2d', '%*d', '%.*d', '%*.*d', '%0*d', '%-*d'];
      _results = [];
      for (_i = 0, _len = verbs.length; _i < _len; _i++) {
        verb = verbs[_i];
        tokens = grammar.tokenizeLine('`' + verb + '`').tokens;
        expect(tokens[0].value).toEqual('`', expect(tokens[0].scopes).toEqual(['source.go', 'string.quoted.raw.go', 'punctuation.definition.string.begin.go']));
        expect(tokens[1].value).toEqual(verb);
        expect(tokens[1].scopes).toEqual(['source.go', 'string.quoted.raw.go', 'constant.escape.format-verb.go']);
        _results.push(expect(tokens[2].value).toEqual('`', expect(tokens[2].scopes).toEqual(['source.go', 'string.quoted.raw.go', 'punctuation.definition.string.end.go'])));
      }
      return _results;
    });
    it('tokenizes runes', function() {
      var tokens, verb, verbs, _i, _len, _results;
      verbs = ['u', 'X', '$', ':', '(', '.', '2', '=', '!', '@', '\\a', '\\b', '\\f', '\\n', '\\r', '\\t', '\\v', '\\\\', '\\000', '\\007', '\\377', '\\x07', '\\xff', '\\u12e4', '\\U00101234'];
      _results = [];
      for (_i = 0, _len = verbs.length; _i < _len; _i++) {
        verb = verbs[_i];
        tokens = grammar.tokenizeLine('\'' + verb + '\'').tokens;
        _results.push(expect(tokens[0].value).toEqual('\'' + verb + '\'', expect(tokens[0].scopes).toEqual(['source.go', 'constant.rune.go'])));
      }
      return _results;
    });
    it('tokenizes invalid whitespace around chan annotations', function() {
      var expr, invalid, invalids, tokens, _results;
      invalids = {
        'chan <- sendonly': ' ',
        '<- chan recvonly': ' '
      };
      _results = [];
      for (expr in invalids) {
        invalid = invalids[expr];
        tokens = grammar.tokenizeLine(expr).tokens;
        expect(tokens[1].value).toEqual(invalid);
        _results.push(expect(tokens[1].scopes).toEqual(['source.go', 'invalid.illegal.go']));
      }
      return _results;
    });
    it('tokenizes keywords', function() {
      var keyword, keywordLists, list, scope, tokens, _results;
      keywordLists = {
        'keyword.go': ['func', 'var', 'const', 'type', 'struct', 'interface', 'case', 'default'],
        'keyword.directive.go': ['package', 'import'],
        'keyword.statement.go': ['defer', 'go', 'goto', 'return', 'break', 'continue', 'fallthrough'],
        'keyword.conditional.go': ['if', 'else', 'switch', 'select'],
        'keyword.repeat.go': ['for', 'range']
      };
      _results = [];
      for (scope in keywordLists) {
        list = keywordLists[scope];
        _results.push((function() {
          var _i, _len, _results1;
          _results1 = [];
          for (_i = 0, _len = list.length; _i < _len; _i++) {
            keyword = list[_i];
            tokens = grammar.tokenizeLine(keyword).tokens;
            expect(tokens[0].value).toEqual(keyword);
            _results1.push(expect(tokens[0].scopes).toEqual(['source.go', scope]));
          }
          return _results1;
        })());
      }
      return _results;
    });
    it('tokenizes types', function() {
      var tokens, type, types, _i, _len, _results;
      types = ['chan', 'map', 'bool', 'string', 'error', 'int', 'int8', 'int16', 'int32', 'int64', 'rune', 'byte', 'uint', 'uint8', 'uint16', 'uint32', 'uint64', 'uintptr', 'float32', 'float64', 'complex64', 'complex128'];
      _results = [];
      for (_i = 0, _len = types.length; _i < _len; _i++) {
        type = types[_i];
        tokens = grammar.tokenizeLine(type).tokens;
        expect(tokens[0].value).toEqual(type);
        _results.push(expect(tokens[0].scopes).toEqual(['source.go', 'storage.type.go']));
      }
      return _results;
    });
    it('tokenizes "func" as a keyword regardless of the context', function() {
      var funcKeyword, funcType, line, next, relevantToken, t, tokens, _i, _j, _len, _len1, _results;
      funcKeyword = ['func f()', 'func (x) f()', 'func(x) f()', 'func'];
      for (_i = 0, _len = funcKeyword.length; _i < _len; _i++) {
        line = funcKeyword[_i];
        tokens = grammar.tokenizeLine(line).tokens;
        expect(tokens[0].value).toEqual('func');
        expect(tokens[0].scopes).toEqual(['source.go', 'keyword.go']);
      }
      funcType = [
        {
          'line': 'var f1 func(',
          'tokenPos': 4
        }, {
          'line': 'f2 :=func()',
          'tokenPos': 3
        }, {
          'line': '\tfunc(',
          'tokenPos': 1
        }, {
          'line': 'type HandlerFunc func(',
          'tokenPos': 4
        }
      ];
      _results = [];
      for (_j = 0, _len1 = funcType.length; _j < _len1; _j++) {
        t = funcType[_j];
        tokens = grammar.tokenizeLine(t.line).tokens;
        relevantToken = tokens[t.tokenPos];
        expect(relevantToken.value).toEqual('func');
        expect(relevantToken.scopes).toEqual(['source.go', 'keyword.go']);
        next = tokens[t.tokenPos + 1];
        expect(next.value).toEqual('(');
        _results.push(expect(next.scopes).toEqual(['source.go', 'keyword.operator.bracket.go']));
      }
      return _results;
    });
    it('tokenizes func names in their declarations', function() {
      var next, relevantToken, t, tests, tokens, _i, _len, _results;
      tests = [
        {
          'line': 'func f()',
          'tokenPos': 2
        }, {
          'line': 'func (T) f()',
          'tokenPos': 6
        }, {
          'line': 'func (t T) f()',
          'tokenPos': 8
        }, {
          'line': 'func (t *T) f()',
          'tokenPos': 9
        }
      ];
      _results = [];
      for (_i = 0, _len = tests.length; _i < _len; _i++) {
        t = tests[_i];
        tokens = grammar.tokenizeLine(t.line).tokens;
        expect(tokens[0].value).toEqual('func');
        expect(tokens[0].scopes).toEqual(['source.go', 'keyword.go']);
        relevantToken = tokens[t.tokenPos];
        expect(relevantToken).toBeDefined();
        expect(relevantToken.value).toEqual('f');
        expect(relevantToken.scopes).toEqual(['source.go', 'support.function.decl.go']);
        next = tokens[t.tokenPos + 1];
        expect(next.value).toEqual('(');
        _results.push(expect(next.scopes).toEqual(['source.go', 'keyword.operator.bracket.go']));
      }
      return _results;
    });
    it('tokenizes receiver types in method declarations', function() {
      var next, relevantToken, t, tests, tokens, _i, _len, _results;
      tests = [
        {
          'line': 'func (T) f()',
          'tokenPos': 3
        }, {
          'line': 'func (t T) f()',
          'tokenPos': 5
        }, {
          'line': 'func (t *T) f()',
          'tokenPos': 6
        }
      ];
      _results = [];
      for (_i = 0, _len = tests.length; _i < _len; _i++) {
        t = tests[_i];
        tokens = grammar.tokenizeLine(t.line).tokens;
        expect(tokens[0].value).toEqual('func');
        expect(tokens[0].scopes).toEqual(['source.go', 'keyword.go']);
        relevantToken = tokens[t.tokenPos];
        expect(relevantToken).toBeDefined();
        expect(relevantToken.value).toEqual('T');
        expect(relevantToken.scopes).toEqual(['source.go', 'variable.go']);
        next = tokens[t.tokenPos + 1];
        expect(next.value).toEqual(')');
        _results.push(expect(next.scopes).toEqual(['source.go', 'keyword.operator.bracket.go']));
      }
      return _results;
    });
    it('tokenizes numerics', function() {
      var invalidOctals, num, numerics, tokens, _i, _j, _len, _len1, _results;
      numerics = ['42', '0600', '0xBadFace', '170141183460469231731687303715884105727', '0.', '72.40', '072.40', '2.71828', '1.e+0', '6.67428e-11', '1E6', '.25', '.12345E+5', '0i', '011i', '0.i', '2.71828i', '1.e+0i', '6.67428e-11i', '1E6i', '.25i', '.12345E+5i'];
      for (_i = 0, _len = numerics.length; _i < _len; _i++) {
        num = numerics[_i];
        tokens = grammar.tokenizeLine(num).tokens;
        expect(tokens[0].value).toEqual(num);
        expect(tokens[0].scopes).toEqual(['source.go', 'constant.numeric.go']);
      }
      invalidOctals = ['08', '039', '0995'];
      _results = [];
      for (_j = 0, _len1 = invalidOctals.length; _j < _len1; _j++) {
        num = invalidOctals[_j];
        tokens = grammar.tokenizeLine(num).tokens;
        expect(tokens[0].value).toEqual(num);
        _results.push(expect(tokens[0].scopes).toEqual(['source.go', 'invalid.illegal.numeric.go']));
      }
      return _results;
    });
    it('tokenizes language constants', function() {
      var constant, constants, tokens, _i, _len, _results;
      constants = ['iota', 'true', 'false', 'nil'];
      _results = [];
      for (_i = 0, _len = constants.length; _i < _len; _i++) {
        constant = constants[_i];
        tokens = grammar.tokenizeLine(constant).tokens;
        expect(tokens[0].value).toEqual(constant);
        _results.push(expect(tokens[0].scopes).toEqual(['source.go', 'constant.language.go']));
      }
      return _results;
    });
    it('tokenizes built-in functions', function() {
      var func, funcVal, funcVals, funcs, tokens, _i, _len, _results;
      funcs = ['append(x)', 'cap(x)', 'close(x)', 'complex(x)', 'copy(x)', 'delete(x)', 'imag(x)', 'len(x)', 'make(x)', 'new(x)', 'panic(x)', 'print(x)', 'println(x)', 'real(x)', 'recover(x)'];
      funcVals = ['append', 'cap', 'close', 'complex', 'copy', 'delete', 'imag', 'len', 'make', 'new', 'panic', 'print', 'println', 'real', 'recover'];
      _results = [];
      for (_i = 0, _len = funcs.length; _i < _len; _i++) {
        func = funcs[_i];
        funcVal = funcVals[funcs.indexOf(func)];
        tokens = grammar.tokenizeLine(func).tokens;
        expect(tokens[0].value).toEqual(funcVal);
        _results.push(expect(tokens[0].scopes).toEqual(['source.go', 'support.function.built-in.go']));
      }
      return _results;
    });
    it('tokenizes operators', function() {
      var allKeywords, fullOp, op, opers, scopes, tokens, _i, _len, _results;
      opers = ['+', '&', '+=', '&=', '&&', '==', '!=', '-', '|', '-=', '|=', '||', '<', '<=', '*', '^', '*=', '^=', '<-', '>', '>=', '/', '<<', '/=', '<<=', '++', '=', ':=', ';', '%', '>>', '%=', '>>=', '--', '!', '...', '&^', '&^='];
      _results = [];
      for (_i = 0, _len = opers.length; _i < _len; _i++) {
        op = opers[_i];
        tokens = grammar.tokenizeLine(op).tokens;
        fullOp = tokens.map(function(tok) {
          return tok.value;
        }).join('');
        expect(fullOp).toEqual(op);
        scopes = tokens.map(function(tok) {
          return tok.scopes;
        });
        allKeywords = scopes.every(function(scope) {
          return __indexOf.call(scope, 'keyword.operator.go') >= 0;
        });
        _results.push(expect(allKeywords).toBe(true));
      }
      return _results;
    });
    it('tokenizes bracket operators', function() {
      var allKeywords, fullOp, op, opers, scopes, tokens, _i, _len, _results;
      opers = ['[', ']', '(', ')', '{', '}'];
      _results = [];
      for (_i = 0, _len = opers.length; _i < _len; _i++) {
        op = opers[_i];
        tokens = grammar.tokenizeLine(op).tokens;
        fullOp = tokens.map(function(tok) {
          return tok.value;
        }).join('');
        expect(fullOp).toEqual(op);
        scopes = tokens.map(function(tok) {
          return tok.scopes;
        });
        allKeywords = scopes.every(function(scope) {
          return __indexOf.call(scope, 'keyword.operator.bracket.go') >= 0;
        });
        _results.push(expect(allKeywords).toBe(true));
      }
      return _results;
    });
    it('tokenizes punctuation operators', function() {
      var allKeywords, fullOp, op, opers, scopes, tokens, _i, _len, _results;
      opers = ['.', ',', ':'];
      _results = [];
      for (_i = 0, _len = opers.length; _i < _len; _i++) {
        op = opers[_i];
        tokens = grammar.tokenizeLine(op).tokens;
        fullOp = tokens.map(function(tok) {
          return tok.value;
        }).join('');
        expect(fullOp).toEqual(op);
        scopes = tokens.map(function(tok) {
          return tok.scopes;
        });
        allKeywords = scopes.every(function(scope) {
          return __indexOf.call(scope, 'keyword.operator.punctuation.go') >= 0;
        });
        _results.push(expect(allKeywords).toBe(true));
      }
      return _results;
    });
    it('tokenizes func names in calls to them', function() {
      var next, relevantToken, t, tests, tokens, want, _i, _len, _results;
      tests = [
        {
          'line': 'a.b()',
          'name': 'b',
          'tokenPos': 2,
          'isFunc': true
        }, {
          'line': 'pkg.Func1(',
          'name': 'Func1',
          'tokenPos': 2,
          'isFunc': true
        }, {
          'line': 'pkg.Func1().Func2(',
          'name': 'Func2',
          'tokenPos': 6,
          'isFunc': true
        }, {
          'line': 'pkg.var',
          'name': 'var',
          'tokenPos': 2,
          'isFunc': false
        }, {
          'line': 'doWork(ch)',
          'name': 'doWork',
          'tokenPos': 0,
          'isFunc': true
        }, {
          'line': 'f1()',
          'name': 'f1',
          'tokenPos': 0,
          'isFunc': true
        }
      ];
      want = ['source.go', 'support.function.go'];
      _results = [];
      for (_i = 0, _len = tests.length; _i < _len; _i++) {
        t = tests[_i];
        tokens = grammar.tokenizeLine(t.line).tokens;
        relevantToken = tokens[t.tokenPos];
        if (t.isFunc) {
          expect(relevantToken).not.toBeNull();
          expect(relevantToken.value).toEqual(t.name);
          expect(relevantToken.scopes).toEqual(want);
          next = tokens[t.tokenPos + 1];
          expect(next.value).toEqual('(');
          _results.push(expect(next.scopes).toEqual(['source.go', 'keyword.operator.bracket.go']));
        } else {
          _results.push(expect(relevantToken.scopes).not.toEqual(want));
        }
      }
      return _results;
    });
    return describe('in variable declarations', function() {
      var testName, testNum, testOp, testOpBracket, testOpPunctuation, testType, testVar, wantedScope;
      testVar = function(token) {
        expect(token.value).toBe('var');
        return expect(token.scopes).toEqual(['source.go', 'keyword.go']);
      };
      wantedScope = ['source.go', 'variable.go'];
      testName = function(token, name) {
        expect(token.value).toBe(name);
        return expect(token.scopes).toEqual(wantedScope);
      };
      testOp = function(token, op) {
        expect(token.value).toBe(op);
        return expect(token.scopes).toEqual(['source.go', 'keyword.operator.go']);
      };
      testOpBracket = function(token, op) {
        expect(token.value).toBe(op);
        return expect(token.scopes).toEqual(['source.go', 'keyword.operator.bracket.go']);
      };
      testOpPunctuation = function(token, op) {
        expect(token.value).toBe(op);
        return expect(token.scopes).toEqual(['source.go', 'keyword.operator.punctuation.go']);
      };
      testType = function(token, name) {
        expect(token.value).toBe(name);
        return expect(token.scopes).toEqual(['source.go', 'storage.type.go']);
      };
      testNum = function(token, value) {
        expect(token.value).toBe(value);
        return expect(token.scopes).toEqual(['source.go', 'constant.numeric.go']);
      };
      return describe('in "var" statements', function() {
        it('tokenizes a single name and a type', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var i int').tokens;
          testVar(tokens[0]);
          testName(tokens[2], 'i');
          return testType(tokens[4], 'int');
        });
        it('tokenizes a single name and its initialization', function() {
          var tokens;
          tokens = grammar.tokenizeLine(' var k =  0').tokens;
          testVar(tokens[1]);
          testName(tokens[3], 'k');
          testOp(tokens[5], '=');
          return testNum(tokens[7], '0');
        });
        it('tokenizes a single name, a type, and an initialization', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var z blub = 7').tokens;
          testVar(tokens[0]);
          testName(tokens[2], 'z');
          testName(tokens[4], 'blub');
          testOp(tokens[6], '=');
          return testNum(tokens[8], '7');
        });
        it('tokenizes multiple names and a type', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var U, V,  W  float64').tokens;
          testVar(tokens[0]);
          testName(tokens[2], 'U');
          testOpPunctuation(tokens[3], ',');
          testName(tokens[5], 'V');
          testOpPunctuation(tokens[6], ',');
          testName(tokens[8], 'W');
          return testType(tokens[10], 'float64');
        });
        it('tokenizes multiple names and initialization expressions', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var x, y, z = 1, 2, 3').tokens;
          testVar(tokens[0]);
          testName(tokens[2], 'x');
          testOpPunctuation(tokens[3], ',');
          testName(tokens[5], 'y');
          testOpPunctuation(tokens[6], ',');
          testName(tokens[8], 'z');
          testOp(tokens[10], '=');
          testNum(tokens[12], '1');
          testOpPunctuation(tokens[13], ',');
          testNum(tokens[15], '2');
          testOpPunctuation(tokens[16], ',');
          return testNum(tokens[18], '3');
        });
        it('tokenizes multiple names, a type, and initialization expressions', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var x, y float32 = float, thirtytwo').tokens;
          testVar(tokens[0]);
          testName(tokens[2], 'x');
          testOpPunctuation(tokens[3], ',');
          testName(tokens[5], 'y');
          testType(tokens[7], 'float32');
          testOp(tokens[9], '=');
          testName(tokens[11], 'float');
          testOpPunctuation(tokens[12], ',');
          return testName(tokens[14], 'thirtytwo');
        });
        it('tokenizes multiple names and a function call', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var re, im = complexSqrt(-1)').tokens;
          testVar(tokens[0]);
          testName(tokens[2], 're');
          testName(tokens[5], 'im');
          return testOp(tokens[7], '=');
        });
        it('tokenizes with a placeholder', function() {
          var tokens;
          tokens = grammar.tokenizeLine('var _, found = entries[name]').tokens;
          testVar(tokens[0]);
          testName(tokens[2], '_');
          testName(tokens[5], 'found');
          return testOp(tokens[7], '=');
        });
        describe('in "var" statement blocks', function() {
          it('tokenizes single names with a type', function() {
            var closing, decl, kwd, _ref;
            _ref = grammar.tokenizeLines('\tvar (\n\t\tfoo *bar\n\t)'), kwd = _ref[0], decl = _ref[1], closing = _ref[2];
            testVar(kwd[1]);
            testOpBracket(kwd[3], '(');
            testName(decl[1], 'foo');
            testOp(decl[3], '*');
            testName(decl[4], 'bar');
            return testOpBracket(closing[1], ')');
          });
          it('tokenizes single names with an initializer', function() {
            var closing, decl, kwd, _ref;
            _ref = grammar.tokenizeLines('var (\n\tfoo = 42\n)'), kwd = _ref[0], decl = _ref[1], closing = _ref[2];
            testVar(kwd[0], 'var');
            testOpBracket(kwd[2], '(');
            testName(decl[1], 'foo');
            testOp(decl[3], '=');
            testNum(decl[5], '42');
            return testOpBracket(closing[0], ')');
          });
          return it('tokenizes multiple names', function() {
            var closing, decl, kwd, _, _ref;
            _ref = grammar.tokenizeLines('var (\n\n\tfoo, bar = baz, quux\n)'), kwd = _ref[0], _ = _ref[1], decl = _ref[2], closing = _ref[3];
            testVar(kwd[0]);
            testOpBracket(kwd[2], '(');
            testName(decl[1], 'foo');
            testOpPunctuation(decl[2], ',');
            testName(decl[4], 'bar');
            testOp(decl[6], '=');
            testName(decl[8], 'baz');
            testOpPunctuation(decl[9], ',');
            testName(decl[11], 'quux');
            return testOpBracket(closing[0], ')');
          });
        });
        return describe('in shorthand variable declarations', function() {
          it('tokenizes single names', function() {
            var tokens;
            tokens = grammar.tokenizeLine('f := func() int { return 7 }').tokens;
            testName(tokens[0], 'f');
            testOp(tokens[2], ':=');
            tokens = grammar.tokenizeLine('ch := make(chan int)').tokens;
            testName(tokens[0], 'ch');
            return testOp(tokens[2], ':=');
          });
          return xit('tokenizes multiple names', function() {
            var tokens;
            tokens = grammar.tokenizeLine('i, j := 0, 10').tokens;
            testName(tokens[0], 'i');
            testOpPunctuation(tokens[1], ',');
            testName(tokens[3], 'j');
            tokens = grammar.tokenizeLine('if _, y, z := coord(p); z > 0').tokens;
            testName(tokens[2], '_');
            testName(tokens[5], 'y');
            testName(tokens[8], 'z');
            return testOp(tokens[10], ':=');
          });
        });
      });
    });
  });

}).call(this);
