(function() {
  var TextEditor;

  TextEditor = require('atom').TextEditor;

  describe("Javascript grammar", function() {
    var grammar;
    grammar = null;
    beforeEach(function() {
      waitsForPromise(function() {
        return atom.packages.activatePackage("language-javascript");
      });
      return runs(function() {
        return grammar = atom.grammars.grammarForScopeName("source.js");
      });
    });
    it("parses the grammar", function() {
      expect(grammar).toBeTruthy();
      return expect(grammar.scopeName).toBe("source.js");
    });
    describe("strings", function() {
      return it("tokenizes single-line strings", function() {
        var delim, delimsByScope, scope, tokens, _results;
        delimsByScope = {
          "string.quoted.double.js": '"',
          "string.quoted.single.js": "'"
        };
        _results = [];
        for (scope in delimsByScope) {
          delim = delimsByScope[scope];
          tokens = grammar.tokenizeLine(delim + "x" + delim).tokens;
          expect(tokens[0].value).toEqual(delim);
          expect(tokens[0].scopes).toEqual(["source.js", scope, "punctuation.definition.string.begin.js"]);
          expect(tokens[1].value).toEqual("x");
          expect(tokens[1].scopes).toEqual(["source.js", scope]);
          expect(tokens[2].value).toEqual(delim);
          _results.push(expect(tokens[2].scopes).toEqual(["source.js", scope, "punctuation.definition.string.end.js"]));
        }
        return _results;
      });
    });
    describe("keywords", function() {
      return it("tokenizes with as a keyword", function() {
        var tokens;
        tokens = grammar.tokenizeLine('with').tokens;
        return expect(tokens[0]).toEqual({
          value: 'with',
          scopes: ['source.js', 'keyword.control.js']
        });
      });
    });
    describe("regular expressions", function() {
      it("tokenizes regular expressions", function() {
        var tokens;
        tokens = grammar.tokenizeLine('/test/').tokens;
        expect(tokens[0]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.begin.js']
        });
        expect(tokens[1]).toEqual({
          value: 'test',
          scopes: ['source.js', 'string.regexp.js']
        });
        expect(tokens[2]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.end.js']
        });
        tokens = grammar.tokenizeLine('foo + /test/').tokens;
        expect(tokens[0]).toEqual({
          value: 'foo ',
          scopes: ['source.js']
        });
        expect(tokens[1]).toEqual({
          value: '+',
          scopes: ['source.js', 'keyword.operator.js']
        });
        expect(tokens[2]).toEqual({
          value: ' ',
          scopes: ['source.js', 'string.regexp.js']
        });
        expect(tokens[3]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.begin.js']
        });
        expect(tokens[4]).toEqual({
          value: 'test',
          scopes: ['source.js', 'string.regexp.js']
        });
        return expect(tokens[5]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.end.js']
        });
      });
      return it("tokenizes regular expressions inside arrays", function() {
        var tokens;
        tokens = grammar.tokenizeLine('[/test/]').tokens;
        expect(tokens[0]).toEqual({
          value: '[',
          scopes: ['source.js', 'meta.brace.square.js']
        });
        expect(tokens[1]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.begin.js']
        });
        expect(tokens[2]).toEqual({
          value: 'test',
          scopes: ['source.js', 'string.regexp.js']
        });
        expect(tokens[3]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.end.js']
        });
        expect(tokens[4]).toEqual({
          value: ']',
          scopes: ['source.js', 'meta.brace.square.js']
        });
        tokens = grammar.tokenizeLine('[1, /test/]').tokens;
        expect(tokens[0]).toEqual({
          value: '[',
          scopes: ['source.js', 'meta.brace.square.js']
        });
        expect(tokens[1]).toEqual({
          value: '1',
          scopes: ['source.js', 'constant.numeric.js']
        });
        expect(tokens[2]).toEqual({
          value: ',',
          scopes: ['source.js', 'meta.delimiter.object.comma.js']
        });
        expect(tokens[3]).toEqual({
          value: ' ',
          scopes: ['source.js', 'string.regexp.js']
        });
        expect(tokens[4]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.begin.js']
        });
        expect(tokens[5]).toEqual({
          value: 'test',
          scopes: ['source.js', 'string.regexp.js']
        });
        expect(tokens[6]).toEqual({
          value: '/',
          scopes: ['source.js', 'string.regexp.js', 'punctuation.definition.string.end.js']
        });
        expect(tokens[7]).toEqual({
          value: ']',
          scopes: ['source.js', 'meta.brace.square.js']
        });
        tokens = grammar.tokenizeLine('0x1D306').tokens;
        expect(tokens[0]).toEqual({
          value: '0x1D306',
          scopes: ['source.js', 'constant.numeric.js']
        });
        tokens = grammar.tokenizeLine('0X1D306').tokens;
        expect(tokens[0]).toEqual({
          value: '0X1D306',
          scopes: ['source.js', 'constant.numeric.js']
        });
        tokens = grammar.tokenizeLine('0b011101110111010001100110').tokens;
        expect(tokens[0]).toEqual({
          value: '0b011101110111010001100110',
          scopes: ['source.js', 'constant.numeric.js']
        });
        tokens = grammar.tokenizeLine('0B011101110111010001100110').tokens;
        expect(tokens[0]).toEqual({
          value: '0B011101110111010001100110',
          scopes: ['source.js', 'constant.numeric.js']
        });
        tokens = grammar.tokenizeLine('0o1411').tokens;
        expect(tokens[0]).toEqual({
          value: '0o1411',
          scopes: ['source.js', 'constant.numeric.js']
        });
        tokens = grammar.tokenizeLine('0O1411').tokens;
        return expect(tokens[0]).toEqual({
          value: '0O1411',
          scopes: ['source.js', 'constant.numeric.js']
        });
      });
    });
    describe("operators", function() {
      it("tokenizes void correctly", function() {
        var tokens;
        tokens = grammar.tokenizeLine('void').tokens;
        return expect(tokens[0]).toEqual({
          value: 'void',
          scopes: ['source.js', 'keyword.operator.js']
        });
      });
      it("tokenizes the / arithmetic operator when separated by newlines", function() {
        var lines;
        lines = grammar.tokenizeLines("1\n/ 2");
        expect(lines[0][0]).toEqual({
          value: '1',
          scopes: ['source.js', 'constant.numeric.js']
        });
        expect(lines[1][0]).toEqual({
          value: '/ ',
          scopes: ['source.js']
        });
        return expect(lines[1][1]).toEqual({
          value: '2',
          scopes: ['source.js', 'constant.numeric.js']
        });
      });
      it("should tokenizes = correctly", function() {
        var tokens;
        tokens = grammar.tokenizeLine('test = 2').tokens;
        expect(tokens[0]).toEqual({
          value: 'test ',
          scopes: ['source.js']
        });
        expect(tokens[1]).toEqual({
          value: '=',
          scopes: ['source.js', 'keyword.operator.js']
        });
        expect(tokens[2]).toEqual({
          value: ' ',
          scopes: ['source.js']
        });
        return expect(tokens[3]).toEqual({
          value: '2',
          scopes: ['source.js', 'constant.numeric.js']
        });
      });
      it("should tokenizes + correctly", function() {
        var tokens;
        tokens = grammar.tokenizeLine('test + 2').tokens;
        expect(tokens[0]).toEqual({
          value: 'test ',
          scopes: ['source.js']
        });
        expect(tokens[1]).toEqual({
          value: '+',
          scopes: ['source.js', 'keyword.operator.js']
        });
        expect(tokens[2]).toEqual({
          value: ' ',
          scopes: ['source.js']
        });
        return expect(tokens[3]).toEqual({
          value: '2',
          scopes: ['source.js', 'constant.numeric.js']
        });
      });
      return describe("operators with 2 characters", function() {
        it("should tokenizes += correctly", function() {
          var tokens;
          tokens = grammar.tokenizeLine('test += 2').tokens;
          expect(tokens[0]).toEqual({
            value: 'test ',
            scopes: ['source.js']
          });
          expect(tokens[1]).toEqual({
            value: '+=',
            scopes: ['source.js', 'keyword.operator.js']
          });
          expect(tokens[2]).toEqual({
            value: ' ',
            scopes: ['source.js']
          });
          return expect(tokens[3]).toEqual({
            value: '2',
            scopes: ['source.js', 'constant.numeric.js']
          });
        });
        it("should tokenizes -= correctly", function() {
          var tokens;
          tokens = grammar.tokenizeLine('test -= 2').tokens;
          expect(tokens[0]).toEqual({
            value: 'test ',
            scopes: ['source.js']
          });
          expect(tokens[1]).toEqual({
            value: '-=',
            scopes: ['source.js', 'keyword.operator.js']
          });
          expect(tokens[2]).toEqual({
            value: ' ',
            scopes: ['source.js']
          });
          return expect(tokens[3]).toEqual({
            value: '2',
            scopes: ['source.js', 'constant.numeric.js']
          });
        });
        it("should tokenizes *= correctly", function() {
          var tokens;
          tokens = grammar.tokenizeLine('test *= 2').tokens;
          expect(tokens[0]).toEqual({
            value: 'test ',
            scopes: ['source.js']
          });
          expect(tokens[1]).toEqual({
            value: '*=',
            scopes: ['source.js', 'keyword.operator.js']
          });
          expect(tokens[2]).toEqual({
            value: ' ',
            scopes: ['source.js']
          });
          return expect(tokens[3]).toEqual({
            value: '2',
            scopes: ['source.js', 'constant.numeric.js']
          });
        });
        return it("should tokenizes /= correctly", function() {
          var tokens;
          tokens = grammar.tokenizeLine('test /= 2').tokens;
          expect(tokens[0]).toEqual({
            value: 'test ',
            scopes: ['source.js']
          });
          expect(tokens[1]).toEqual({
            value: '/=',
            scopes: ['source.js', 'keyword.operator.js']
          });
          expect(tokens[2]).toEqual({
            value: ' ',
            scopes: ['source.js']
          });
          return expect(tokens[3]).toEqual({
            value: '2',
            scopes: ['source.js', 'constant.numeric.js']
          });
        });
      });
    });
    describe("ES6 string templates", function() {
      return it("tokenizes them as strings", function() {
        var tokens;
        tokens = grammar.tokenizeLine('`hey ${name}`').tokens;
        expect(tokens[0]).toEqual({
          value: '`',
          scopes: ['source.js', 'string.quoted.template.js', 'punctuation.definition.string.begin.js']
        });
        expect(tokens[1]).toEqual({
          value: 'hey ',
          scopes: ['source.js', 'string.quoted.template.js']
        });
        expect(tokens[2]).toEqual({
          value: '${',
          scopes: ['source.js', 'string.quoted.template.js', 'source.js.embedded.source', 'punctuation.section.embedded.js']
        });
        expect(tokens[3]).toEqual({
          value: 'name',
          scopes: ['source.js', 'string.quoted.template.js', 'source.js.embedded.source']
        });
        expect(tokens[4]).toEqual({
          value: '}',
          scopes: ['source.js', 'string.quoted.template.js', 'source.js.embedded.source', 'punctuation.section.embedded.js']
        });
        return expect(tokens[5]).toEqual({
          value: '`',
          scopes: ['source.js', 'string.quoted.template.js', 'punctuation.definition.string.end.js']
        });
      });
    });
    describe("ES6 class", function() {
      it("tokenizes class", function() {
        var tokens;
        tokens = grammar.tokenizeLine('class MyClass').tokens;
        expect(tokens[0]).toEqual({
          value: 'class',
          scopes: ['source.js', 'meta.class.js', 'storage.type.class.js']
        });
        return expect(tokens[2]).toEqual({
          value: 'MyClass',
          scopes: ['source.js', 'meta.class.js', 'entity.name.type.js']
        });
      });
      it("tokenizes class...extends", function() {
        var tokens;
        tokens = grammar.tokenizeLine('class MyClass extends SomeClass').tokens;
        expect(tokens[0]).toEqual({
          value: 'class',
          scopes: ['source.js', 'meta.class.js', 'storage.type.class.js']
        });
        expect(tokens[2]).toEqual({
          value: 'MyClass',
          scopes: ['source.js', 'meta.class.js', 'entity.name.type.js']
        });
        expect(tokens[4]).toEqual({
          value: 'extends',
          scopes: ['source.js', 'meta.class.js', 'storage.modifier.js']
        });
        return expect(tokens[6]).toEqual({
          value: 'SomeClass',
          scopes: ['source.js', 'meta.class.js', 'entity.name.type.js']
        });
      });
      return it("tokenizes anonymous class", function() {
        var tokens;
        tokens = grammar.tokenizeLine('class extends SomeClass').tokens;
        expect(tokens[0]).toEqual({
          value: 'class',
          scopes: ['source.js', 'meta.class.js', 'storage.type.class.js']
        });
        expect(tokens[2]).toEqual({
          value: 'extends',
          scopes: ['source.js', 'meta.class.js', 'storage.modifier.js']
        });
        return expect(tokens[4]).toEqual({
          value: 'SomeClass',
          scopes: ['source.js', 'meta.class.js', 'entity.name.type.js']
        });
      });
    });
    describe("ES6 import", function() {
      it("Tokenizes import ... as", function() {
        var tokens;
        tokens = grammar.tokenizeLine('import \'react\' as React').tokens;
        expect(tokens[0]).toEqual({
          value: 'import',
          scopes: ['source.js', 'meta.import.js', 'keyword.control.js']
        });
        return expect(tokens[6]).toEqual({
          value: 'as',
          scopes: ['source.js', 'meta.import.js', 'keyword.control.js']
        });
      });
      return it("Tokenizes import ... from", function() {
        var tokens;
        tokens = grammar.tokenizeLine('import React from \'react\'').tokens;
        expect(tokens[0]).toEqual({
          value: 'import',
          scopes: ['source.js', 'meta.import.js', 'keyword.control.js']
        });
        expect(tokens[4]).toEqual({
          value: 'from',
          scopes: ['source.js', 'meta.import.js', 'keyword.control.js']
        });
        tokens = grammar.tokenizeLine('import {React} from \'react\'').tokens;
        expect(tokens[0]).toEqual({
          value: 'import',
          scopes: ['source.js', 'meta.import.js', 'keyword.control.js']
        });
        return expect(tokens[6]).toEqual({
          value: 'from',
          scopes: ['source.js', 'meta.import.js', 'keyword.control.js']
        });
      });
    });
    describe("ES6 yield", function() {
      it("Tokenizes yield", function() {
        var tokens;
        tokens = grammar.tokenizeLine('yield next').tokens;
        return expect(tokens[0]).toEqual({
          value: 'yield',
          scopes: ['source.js', 'meta.control.yield.js', 'keyword.control.js']
        });
      });
      return it("Tokenizes yield*", function() {
        var tokens;
        tokens = grammar.tokenizeLine('yield * next').tokens;
        expect(tokens[0]).toEqual({
          value: 'yield',
          scopes: ['source.js', 'meta.control.yield.js', 'keyword.control.js']
        });
        return expect(tokens[2]).toEqual({
          value: '*',
          scopes: ['source.js', 'meta.control.yield.js', 'storage.modifier.js']
        });
      });
    });
    it("doesn't confuse strings and functions (regression)", function() {
      var tokens;
      tokens = grammar.tokenizeLine("'a'.b(':c(d)')").tokens;
      expect(tokens[0]).toEqual({
        value: "'",
        scopes: ['source.js', 'string.quoted.single.js', 'punctuation.definition.string.begin.js']
      });
      expect(tokens[1]).toEqual({
        value: "a",
        scopes: ['source.js', 'string.quoted.single.js']
      });
      expect(tokens[2]).toEqual({
        value: "'",
        scopes: ['source.js', 'string.quoted.single.js', 'punctuation.definition.string.end.js']
      });
      expect(tokens[3]).toEqual({
        value: ".",
        scopes: ['source.js', 'meta.delimiter.method.period.js']
      });
      expect(tokens[4]).toEqual({
        value: "b",
        scopes: ['source.js']
      });
      expect(tokens[5]).toEqual({
        value: "(",
        scopes: ['source.js', 'meta.brace.round.js']
      });
      expect(tokens[6]).toEqual({
        value: "'",
        scopes: ['source.js', 'string.quoted.single.js', 'punctuation.definition.string.begin.js']
      });
      expect(tokens[7]).toEqual({
        value: ":c(d)",
        scopes: ['source.js', 'string.quoted.single.js']
      });
      expect(tokens[8]).toEqual({
        value: "'",
        scopes: ['source.js', 'string.quoted.single.js', 'punctuation.definition.string.end.js']
      });
      return expect(tokens[9]).toEqual({
        value: ")",
        scopes: ['source.js', 'meta.brace.round.js']
      });
    });
    describe("default: in a switch statement", function() {
      return it("tokenizes it as a keyword", function() {
        var tokens;
        tokens = grammar.tokenizeLine('default: ').tokens;
        return expect(tokens[0]).toEqual({
          value: 'default',
          scopes: ['source.js', 'keyword.control.js']
        });
      });
    });
    it("tokenizes comments in function params", function() {
      var tokens;
      tokens = grammar.tokenizeLine('foo: function (/**Bar*/bar){').tokens;
      expect(tokens[4]).toEqual({
        value: '(',
        scopes: ['source.js', 'meta.function.json.js', 'punctuation.definition.parameters.begin.js']
      });
      expect(tokens[5]).toEqual({
        value: '/**',
        scopes: ['source.js', 'meta.function.json.js', 'comment.block.documentation.js', 'punctuation.definition.comment.js']
      });
      expect(tokens[6]).toEqual({
        value: 'Bar',
        scopes: ['source.js', 'meta.function.json.js', 'comment.block.documentation.js']
      });
      expect(tokens[7]).toEqual({
        value: '*/',
        scopes: ['source.js', 'meta.function.json.js', 'comment.block.documentation.js', 'punctuation.definition.comment.js']
      });
      return expect(tokens[8]).toEqual({
        value: 'bar',
        scopes: ['source.js', 'meta.function.json.js', 'variable.parameter.function.js']
      });
    });
    describe("non-anonymous functions", function() {
      it("tokenizes methods", function() {
        var tokens;
        tokens = grammar.tokenizeLine('Foo.method = function nonAnonymous(').tokens;
        expect(tokens[0]).toEqual({
          value: 'Foo',
          scopes: ['source.js', 'meta.function.js', 'support.class.js']
        });
        expect(tokens[2]).toEqual({
          value: 'method',
          scopes: ['source.js', 'meta.function.js', 'entity.name.function.js']
        });
        expect(tokens[4]).toEqual({
          value: '=',
          scopes: ['source.js', 'meta.function.js', 'keyword.operator.js']
        });
        expect(tokens[6]).toEqual({
          value: 'function',
          scopes: ['source.js', 'meta.function.js', 'storage.type.function.js']
        });
        expect(tokens[8]).toEqual({
          value: 'nonAnonymous',
          scopes: ['source.js', 'meta.function.js', 'entity.name.function.js']
        });
        return expect(tokens[9]).toEqual({
          value: '(',
          scopes: ['source.js', 'meta.function.js', 'punctuation.definition.parameters.begin.js']
        });
      });
      it("tokenizes methods", function() {
        var tokens;
        tokens = grammar.tokenizeLine('f(a, b) {}').tokens;
        expect(tokens[0]).toEqual({
          value: 'f',
          scopes: ['source.js', 'meta.method.js', 'entity.name.function.js']
        });
        expect(tokens[1]).toEqual({
          value: '(',
          scopes: ['source.js', 'meta.method.js', 'punctuation.definition.parameters.begin.js']
        });
        expect(tokens[2]).toEqual({
          value: 'a',
          scopes: ['source.js', 'meta.method.js', 'variable.parameter.function.js']
        });
        expect(tokens[4]).toEqual({
          value: 'b',
          scopes: ['source.js', 'meta.method.js', 'variable.parameter.function.js']
        });
        return expect(tokens[5]).toEqual({
          value: ')',
          scopes: ['source.js', 'meta.method.js', 'punctuation.definition.parameters.end.js']
        });
      });
      it("tokenizes functions", function() {
        var tokens;
        tokens = grammar.tokenizeLine('var func = function nonAnonymous(').tokens;
        expect(tokens[0]).toEqual({
          value: 'var',
          scopes: ['source.js', 'storage.modifier.js']
        });
        expect(tokens[2]).toEqual({
          value: 'func',
          scopes: ['source.js', 'meta.function.js', 'entity.name.function.js']
        });
        expect(tokens[4]).toEqual({
          value: '=',
          scopes: ['source.js', 'meta.function.js', 'keyword.operator.js']
        });
        expect(tokens[6]).toEqual({
          value: 'function',
          scopes: ['source.js', 'meta.function.js', 'storage.type.function.js']
        });
        expect(tokens[8]).toEqual({
          value: 'nonAnonymous',
          scopes: ['source.js', 'meta.function.js', 'entity.name.function.js']
        });
        return expect(tokens[9]).toEqual({
          value: '(',
          scopes: ['source.js', 'meta.function.js', 'punctuation.definition.parameters.begin.js']
        });
      });
      it("tokenizes object functions", function() {
        var tokens;
        tokens = grammar.tokenizeLine('foo: function nonAnonymous(').tokens;
        expect(tokens[0]).toEqual({
          value: 'foo',
          scopes: ['source.js', 'meta.function.json.js', 'entity.name.function.js']
        });
        expect(tokens[2]).toEqual({
          value: 'function',
          scopes: ['source.js', 'meta.function.json.js', 'storage.type.function.js']
        });
        expect(tokens[4]).toEqual({
          value: 'nonAnonymous',
          scopes: ['source.js', 'meta.function.json.js', 'entity.name.function.js']
        });
        return expect(tokens[5]).toEqual({
          value: '(',
          scopes: ['source.js', 'meta.function.json.js', 'punctuation.definition.parameters.begin.js']
        });
      });
      it("tokenizes quoted object functions", function() {
        var tokens;
        tokens = grammar.tokenizeLine('"foo": function nonAnonymous(').tokens;
        expect(tokens[1]).toEqual({
          value: 'foo',
          scopes: ['source.js', 'meta.function.json.js', 'string.quoted.double.js', 'entity.name.function.js']
        });
        expect(tokens[4]).toEqual({
          value: 'function',
          scopes: ['source.js', 'meta.function.json.js', 'storage.type.function.js']
        });
        expect(tokens[6]).toEqual({
          value: 'nonAnonymous',
          scopes: ['source.js', 'meta.function.json.js', 'entity.name.function.js']
        });
        return expect(tokens[7]).toEqual({
          value: '(',
          scopes: ['source.js', 'meta.function.json.js', 'punctuation.definition.parameters.begin.js']
        });
      });
      return it("tokenizes async functions", function() {
        var tokens;
        tokens = grammar.tokenizeLine('async function f(){}').tokens;
        expect(tokens[0]).toEqual({
          value: 'async',
          scopes: ['source.js', 'meta.function.js', 'storage.modifier.js']
        });
        expect(tokens[2]).toEqual({
          value: 'function',
          scopes: ['source.js', 'meta.function.js', 'storage.type.function.js']
        });
        return expect(tokens[4]).toEqual({
          value: 'f',
          scopes: ['source.js', 'meta.function.js', 'entity.name.function.js']
        });
      });
    });
    it("tokenizes /* */ comments", function() {
      var tokens;
      tokens = grammar.tokenizeLine('/**/').tokens;
      expect(tokens[0]).toEqual({
        value: '/*',
        scopes: ['source.js', 'comment.block.js', 'punctuation.definition.comment.js']
      });
      expect(tokens[1]).toEqual({
        value: '*/',
        scopes: ['source.js', 'comment.block.js', 'punctuation.definition.comment.js']
      });
      tokens = grammar.tokenizeLine('/* foo */').tokens;
      expect(tokens[0]).toEqual({
        value: '/*',
        scopes: ['source.js', 'comment.block.js', 'punctuation.definition.comment.js']
      });
      expect(tokens[1]).toEqual({
        value: ' foo ',
        scopes: ['source.js', 'comment.block.js']
      });
      return expect(tokens[2]).toEqual({
        value: '*/',
        scopes: ['source.js', 'comment.block.js', 'punctuation.definition.comment.js']
      });
    });
    it("tokenizes /** */ comments", function() {
      var tokens;
      tokens = grammar.tokenizeLine('/***/').tokens;
      expect(tokens[0]).toEqual({
        value: '/**',
        scopes: ['source.js', 'comment.block.documentation.js', 'punctuation.definition.comment.js']
      });
      expect(tokens[1]).toEqual({
        value: '*/',
        scopes: ['source.js', 'comment.block.documentation.js', 'punctuation.definition.comment.js']
      });
      tokens = grammar.tokenizeLine('/** foo */').tokens;
      expect(tokens[0]).toEqual({
        value: '/**',
        scopes: ['source.js', 'comment.block.documentation.js', 'punctuation.definition.comment.js']
      });
      expect(tokens[1]).toEqual({
        value: ' foo ',
        scopes: ['source.js', 'comment.block.documentation.js']
      });
      return expect(tokens[2]).toEqual({
        value: '*/',
        scopes: ['source.js', 'comment.block.documentation.js', 'punctuation.definition.comment.js']
      });
    });
    return describe("indentation", function() {
      var editor, expectPreservedIndentation;
      editor = null;
      beforeEach(function() {
        editor = new TextEditor({});
        return editor.setGrammar(grammar);
      });
      expectPreservedIndentation = function(text) {
        var actualLine, actualLines, expectedLines, i, _i, _len, _results;
        editor.setText(text);
        editor.autoIndentBufferRows(0, editor.getLineCount() - 1);
        expectedLines = text.split("\n");
        actualLines = editor.getText().split("\n");
        _results = [];
        for (i = _i = 0, _len = actualLines.length; _i < _len; i = ++_i) {
          actualLine = actualLines[i];
          _results.push(expect([actualLine, editor.indentLevelForLine(actualLine)]).toEqual([expectedLines[i], editor.indentLevelForLine(expectedLines[i])]));
        }
        return _results;
      };
      it("indents allman-style curly braces", function() {
        return expectPreservedIndentation("if (true)\n{\n  for (;;)\n  {\n    while (true)\n    {\n      x();\n    }\n  }\n}\nelse\n{\n  do\n  {\n    y();\n  } while (true);\n}");
      });
      it("indents non-allman-style curly braces", function() {
        return expectPreservedIndentation("if (true) {\n  for (;;) {\n    while (true) {\n      x();\n    }\n  }\n} else {\n  do {\n    y();\n  } while (true);\n}");
      });
      it("doesn't indent case statements, because it wouldn't know when to outdent", function() {
        return expectPreservedIndentation("switch (e) {\n  case 5:\n  something();\n  case 6:\n  somethingElse();\n}");
      });
      it("indents collection literals", function() {
        return expectPreservedIndentation("[\n  {\n    a: b,\n    c: d\n  },\n  e,\n  f\n]");
      });
      return it("indents function arguments", function() {
        return expectPreservedIndentation("f(\n  g(\n    h,\n    i\n  ),\n  j\n);");
      });
    });
  });

}).call(this);
