(function() {
  describe("YAML grammar", function() {
    var grammar;
    grammar = null;
    beforeEach(function() {
      waitsForPromise(function() {
        return atom.packages.activatePackage("language-yaml");
      });
      return runs(function() {
        return grammar = atom.grammars.grammarForScopeName('source.yaml');
      });
    });
    it("parses the grammar", function() {
      expect(grammar).toBeTruthy();
      return expect(grammar.scopeName).toBe("source.yaml");
    });
    it("selects the grammar for cloud config files", function() {
      waitsForPromise(function() {
        return atom.workspace.open('cloud.config');
      });
      return runs(function() {
        return expect(atom.workspace.getActiveTextEditor().getGrammar()).toBe(grammar);
      });
    });
    describe("strings", function() {
      describe("double quoted", function() {
        return it("parses escaped quotes", function() {
          var tokens;
          tokens = grammar.tokenizeLine("\"I am \\\"escaped\\\"\"").tokens;
          expect(tokens[0]).toEqual({
            value: "\"",
            scopes: ["source.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
          });
          expect(tokens[1]).toEqual({
            value: "I am ",
            scopes: ["source.yaml", "string.quoted.double.yaml"]
          });
          expect(tokens[2]).toEqual({
            value: "\\\"",
            scopes: ["source.yaml", "string.quoted.double.yaml", "constant.character.escape.yaml"]
          });
          expect(tokens[3]).toEqual({
            value: "escaped",
            scopes: ["source.yaml", "string.quoted.double.yaml"]
          });
          expect(tokens[4]).toEqual({
            value: "\\\"",
            scopes: ["source.yaml", "string.quoted.double.yaml", "constant.character.escape.yaml"]
          });
          expect(tokens[5]).toEqual({
            value: "\"",
            scopes: ["source.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
          });
          tokens = grammar.tokenizeLine("key: \"I am \\\"escaped\\\"\"").tokens;
          expect(tokens[0]).toEqual({
            value: "key",
            scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
          });
          expect(tokens[1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(tokens[2]).toEqual({
            value: " ",
            scopes: ["source.yaml", "string.unquoted.yaml"]
          });
          expect(tokens[3]).toEqual({
            value: "\"",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
          });
          expect(tokens[4]).toEqual({
            value: "I am ",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
          });
          expect(tokens[5]).toEqual({
            value: "\\\"",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "constant.character.escape.yaml"]
          });
          expect(tokens[6]).toEqual({
            value: "escaped",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
          });
          expect(tokens[7]).toEqual({
            value: "\\\"",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "constant.character.escape.yaml"]
          });
          return expect(tokens[8]).toEqual({
            value: "\"",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
          });
        });
      });
      describe("single quoted", function() {
        return it("parses escaped quotes", function() {
          var tokens;
          tokens = grammar.tokenizeLine("'I am \\'escaped\\''").tokens;
          expect(tokens[0]).toEqual({
            value: "'",
            scopes: ["source.yaml", "string.quoted.single.yaml", "punctuation.definition.string.begin.yaml"]
          });
          expect(tokens[1]).toEqual({
            value: "I am ",
            scopes: ["source.yaml", "string.quoted.single.yaml"]
          });
          expect(tokens[2]).toEqual({
            value: "\\'",
            scopes: ["source.yaml", "string.quoted.single.yaml", "constant.character.escape.yaml"]
          });
          expect(tokens[3]).toEqual({
            value: "escaped",
            scopes: ["source.yaml", "string.quoted.single.yaml"]
          });
          expect(tokens[4]).toEqual({
            value: "\\'",
            scopes: ["source.yaml", "string.quoted.single.yaml", "constant.character.escape.yaml"]
          });
          expect(tokens[5]).toEqual({
            value: "'",
            scopes: ["source.yaml", "string.quoted.single.yaml", "punctuation.definition.string.end.yaml"]
          });
          tokens = grammar.tokenizeLine("key: 'I am \\'escaped\\''").tokens;
          expect(tokens[0]).toEqual({
            value: "key",
            scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
          });
          expect(tokens[1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(tokens[2]).toEqual({
            value: " ",
            scopes: ["source.yaml", "string.unquoted.yaml"]
          });
          expect(tokens[3]).toEqual({
            value: "'",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml", "punctuation.definition.string.begin.yaml"]
          });
          expect(tokens[4]).toEqual({
            value: "I am ",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml"]
          });
          expect(tokens[5]).toEqual({
            value: "\\'",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml", "constant.character.escape.yaml"]
          });
          expect(tokens[6]).toEqual({
            value: "escaped",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml"]
          });
          expect(tokens[7]).toEqual({
            value: "\\'",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml", "constant.character.escape.yaml"]
          });
          return expect(tokens[8]).toEqual({
            value: "'",
            scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml", "punctuation.definition.string.end.yaml"]
          });
        });
      });
      return describe("text blocks", function() {
        it("parses simple content", function() {
          var lines;
          lines = grammar.tokenizeLines("key: |\n  content here\n  second line");
          expect(lines[0][0]).toEqual({
            value: "key",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
          });
          expect(lines[0][1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(lines[1][0]).toEqual({
            value: "  content here",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
          return expect(lines[2][0]).toEqual({
            value: "  second line",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
        });
        it("parses content with empty lines", function() {
          var lines;
          lines = grammar.tokenizeLines("key: |\n  content here\n\n  second line");
          expect(lines[0][0]).toEqual({
            value: "key",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
          });
          expect(lines[0][1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(lines[1][0]).toEqual({
            value: "  content here",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
          expect(lines[2][0]).toEqual({
            value: "",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
          return expect(lines[3][0]).toEqual({
            value: "  second line",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
        });
        it("parses keys with decimals", function() {
          var lines;
          lines = grammar.tokenizeLines("2.0: |\n  content here\n  second line");
          expect(lines[0][0]).toEqual({
            value: "2.0",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
          });
          expect(lines[0][1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(lines[1][0]).toEqual({
            value: "  content here",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
          return expect(lines[2][0]).toEqual({
            value: "  second line",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
        });
        it("properly parses comments in blocks", function() {
          var lines;
          lines = grammar.tokenizeLines("key: |\n  # this is a legit comment\n  no highlights\nkey: |\n  ### this is just a markdown header\n  second line");
          expect(lines[0][0]).toEqual({
            value: "key",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
          });
          expect(lines[0][1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(lines[1][0]).toEqual({
            value: "  ",
            scopes: ["source.yaml", "punctuation.whitespace.comment.leading.yaml"]
          });
          expect(lines[1][1]).toEqual({
            value: "#",
            scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
          });
          expect(lines[1][2]).toEqual({
            value: " this is a legit comment",
            scopes: ["source.yaml", "comment.line.number-sign.yaml"]
          });
          expect(lines[2][0]).toEqual({
            value: "  no highlights",
            scopes: ["source.yaml"]
          });
          expect(lines[3][0]).toEqual({
            value: "key",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
          });
          expect(lines[3][1]).toEqual({
            value: ":",
            scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
          });
          expect(lines[4][0]).toEqual({
            value: "  ### this is just a markdown header",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
          return expect(lines[5][0]).toEqual({
            value: "  second line",
            scopes: ["source.yaml", "string.unquoted.block.yaml"]
          });
        });
        return describe("parses content with unindented empty lines", function() {
          it("ending the content", function() {
            var lines;
            lines = grammar.tokenizeLines("key: |\n  content here\n\n  second line");
            expect(lines[0][0]).toEqual({
              value: "key",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[0][1]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[0][2]).toEqual({
              value: " |",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[1][0]).toEqual({
              value: "  content here",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[2][0]).toEqual({
              value: "",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            return expect(lines[3][0]).toEqual({
              value: "  second line",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
          });
          it("ending with new element", function() {
            var lines;
            lines = grammar.tokenizeLines("key: |\n  content here\n\n  second line\nother: hi");
            expect(lines[0][0]).toEqual({
              value: "key",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[0][1]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[0][2]).toEqual({
              value: " |",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[1][0]).toEqual({
              value: "  content here",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[2][0]).toEqual({
              value: "",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[3][0]).toEqual({
              value: "  second line",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[4][0]).toEqual({
              value: "other",
              scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[4][1]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[4][2]).toEqual({
              value: " ",
              scopes: ["source.yaml", "string.unquoted.yaml"]
            });
            return expect(lines[4][3]).toEqual({
              value: "hi",
              scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
            });
          });
          it("ending with new element, part of list", function() {
            var lines;
            lines = grammar.tokenizeLines("- key: |\n    content here\n\n    second line\n- other: hi");
            expect(lines[0][0]).toEqual({
              value: "- ",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "punctuation.definition.entry.yaml"]
            });
            expect(lines[0][1]).toEqual({
              value: "key",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[0][2]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[0][3]).toEqual({
              value: " |",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[1][0]).toEqual({
              value: "    content here",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[2][0]).toEqual({
              value: "",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[3][0]).toEqual({
              value: "    second line",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[4][0]).toEqual({
              value: "- ",
              scopes: ["source.yaml", "string.unquoted.yaml", "punctuation.definition.entry.yaml"]
            });
            expect(lines[4][1]).toEqual({
              value: "other",
              scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[4][2]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[4][3]).toEqual({
              value: " ",
              scopes: ["source.yaml", "string.unquoted.yaml"]
            });
            return expect(lines[4][4]).toEqual({
              value: "hi",
              scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
            });
          });
          it("ending with twice unindented new element", function() {
            var lines;
            lines = grammar.tokenizeLines("root:\n  key: |\n    content here\n\n    second line\nother: hi");
            expect(lines[1][1]).toEqual({
              value: "key",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[1][2]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[1][3]).toEqual({
              value: " |",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[2][0]).toEqual({
              value: "    content here",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[3][0]).toEqual({
              value: "",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[4][0]).toEqual({
              value: "    second line",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[5][0]).toEqual({
              value: "other",
              scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[5][1]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[5][2]).toEqual({
              value: " ",
              scopes: ["source.yaml", "string.unquoted.yaml"]
            });
            return expect(lines[5][3]).toEqual({
              value: "hi",
              scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
            });
          });
          return it("ending with an indented comment", function() {
            var lines;
            lines = grammar.tokenizeLines("root:\n  key: |\n    content here\n\n    second line\n  # hi");
            expect(lines[1][1]).toEqual({
              value: "key",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml"]
            });
            expect(lines[1][2]).toEqual({
              value: ":",
              scopes: ["source.yaml", "string.unquoted.block.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
            });
            expect(lines[1][3]).toEqual({
              value: " |",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[2][0]).toEqual({
              value: "    content here",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[3][0]).toEqual({
              value: "",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[4][0]).toEqual({
              value: "    second line",
              scopes: ["source.yaml", "string.unquoted.block.yaml"]
            });
            expect(lines[5][0]).toEqual({
              value: "  ",
              scopes: ["source.yaml", "punctuation.whitespace.comment.leading.yaml"]
            });
            expect(lines[5][1]).toEqual({
              value: "#",
              scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
            });
            return expect(lines[5][2]).toEqual({
              value: " hi",
              scopes: ["source.yaml", "comment.line.number-sign.yaml"]
            });
          });
        });
      });
    });
    it("parses the leading ! before values", function() {
      var tokens;
      tokens = grammar.tokenizeLine("key: ! 'hi'").tokens;
      expect(tokens[0]).toEqual({
        value: "key",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(tokens[1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(tokens[2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(tokens[3]).toEqual({
        value: "! ",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
      expect(tokens[4]).toEqual({
        value: "'",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(tokens[5]).toEqual({
        value: "hi",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml"]
      });
      return expect(tokens[6]).toEqual({
        value: "'",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.single.yaml", "punctuation.definition.string.end.yaml"]
      });
    });
    it("parses nested keys", function() {
      var lines;
      lines = grammar.tokenizeLines("first:\n  second:\n    third: 3\n    fourth: \"4th\"");
      expect(lines[0][0]).toEqual({
        value: "first",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[0][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[1][0]).toEqual({
        value: "  ",
        scopes: ["source.yaml"]
      });
      expect(lines[1][1]).toEqual({
        value: "second",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[1][2]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[2][0]).toEqual({
        value: "    ",
        scopes: ["source.yaml"]
      });
      expect(lines[2][1]).toEqual({
        value: "third",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[2][2]).toEqual({
        value: ":",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[2][3]).toEqual({
        value: " 3",
        scopes: ["source.yaml", "constant.numeric.yaml"]
      });
      expect(lines[3][0]).toEqual({
        value: "    ",
        scopes: ["source.yaml"]
      });
      expect(lines[3][1]).toEqual({
        value: "fourth",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[3][2]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[3][3]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[3][4]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(lines[3][5]).toEqual({
        value: "4th",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
      });
      return expect(lines[3][6]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
      });
    });
    it("parses keys and values", function() {
      var lines;
      lines = grammar.tokenizeLines("first: 1st\nsecond: 2nd\nthird: th{ree}");
      expect(lines[0][0]).toEqual({
        value: "first",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[0][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[0][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[0][3]).toEqual({
        value: "1st",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
      expect(lines[1][0]).toEqual({
        value: "second",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[1][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[1][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[1][3]).toEqual({
        value: "2nd",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
      expect(lines[2][0]).toEqual({
        value: "third",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[2][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[2][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      return expect(lines[2][3]).toEqual({
        value: "th{ree}",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
    });
    it("parses comments at the beginning of lines", function() {
      var lines;
      lines = grammar.tokenizeLines("# first: 1\n  # second\n##");
      expect(lines[0][0]).toEqual({
        value: "#",
        scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
      });
      expect(lines[0][1]).toEqual({
        value: " first: 1",
        scopes: ["source.yaml", "comment.line.number-sign.yaml"]
      });
      expect(lines[1][0]).toEqual({
        value: "  ",
        scopes: ["source.yaml", "punctuation.whitespace.comment.leading.yaml"]
      });
      expect(lines[1][1]).toEqual({
        value: "#",
        scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
      });
      expect(lines[1][2]).toEqual({
        value: " second",
        scopes: ["source.yaml", "comment.line.number-sign.yaml"]
      });
      expect(lines[2][0]).toEqual({
        value: "#",
        scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
      });
      return expect(lines[2][1]).toEqual({
        value: "#",
        scopes: ["source.yaml", "comment.line.number-sign.yaml"]
      });
    });
    it("parses comments at the end of lines", function() {
      var lines;
      lines = grammar.tokenizeLines("first: 1 # foo\nsecond: 2nd  #bar\nthird: \"3\"\nfourth: four#");
      expect(lines[0][0]).toEqual({
        value: "first",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[0][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[0][2]).toEqual({
        value: " 1 ",
        scopes: ["source.yaml", "constant.numeric.yaml"]
      });
      expect(lines[0][3]).toEqual({
        value: "#",
        scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
      });
      expect(lines[0][4]).toEqual({
        value: " foo",
        scopes: ["source.yaml", "comment.line.number-sign.yaml"]
      });
      expect(lines[1][0]).toEqual({
        value: "second",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[1][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[1][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[1][3]).toEqual({
        value: "2nd  ",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
      expect(lines[1][4]).toEqual({
        value: "#",
        scopes: ["source.yaml", "comment.line.number-sign.yaml", "punctuation.definition.comment.yaml"]
      });
      expect(lines[1][5]).toEqual({
        value: "bar",
        scopes: ["source.yaml", "comment.line.number-sign.yaml"]
      });
      expect(lines[2][0]).toEqual({
        value: "third",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[2][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[2][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[2][3]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(lines[2][4]).toEqual({
        value: "3",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
      });
      expect(lines[2][5]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
      });
      expect(lines[3][0]).toEqual({
        value: "fourth",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[3][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[3][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      return expect(lines[3][3]).toEqual({
        value: "four#",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
    });
    it("parses colons in key names", function() {
      var lines;
      lines = grammar.tokenizeLines("colon::colon: 1\ncolon::colon: 2nd\ncolon::colon: \"3\"\ncolon: \"this is another : colon\"\ncolon: \"this is another :colon\"");
      expect(lines[0][0]).toEqual({
        value: "colon::colon",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[0][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[0][2]).toEqual({
        value: " 1",
        scopes: ["source.yaml", "constant.numeric.yaml"]
      });
      expect(lines[1][0]).toEqual({
        value: "colon::colon",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[1][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[1][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[1][3]).toEqual({
        value: "2nd",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
      expect(lines[2][0]).toEqual({
        value: "colon::colon",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[2][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[2][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[2][3]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(lines[2][4]).toEqual({
        value: "3",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
      });
      expect(lines[2][5]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
      });
      expect(lines[3][0]).toEqual({
        value: "colon",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[3][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[3][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[3][3]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(lines[3][4]).toEqual({
        value: "this is another : colon",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
      });
      expect(lines[3][5]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
      });
      expect(lines[4][0]).toEqual({
        value: "colon",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[4][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[4][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[4][3]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(lines[4][4]).toEqual({
        value: "this is another :colon",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
      });
      return expect(lines[4][5]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
      });
    });
    return it("parses spaces in key names", function() {
      var lines;
      lines = grammar.tokenizeLines("spaced out: 1\nmore        spaces: 2nd\nwith quotes: \"3\"");
      expect(lines[0][0]).toEqual({
        value: "spaced out",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[0][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "constant.numeric.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[0][2]).toEqual({
        value: " 1",
        scopes: ["source.yaml", "constant.numeric.yaml"]
      });
      expect(lines[1][0]).toEqual({
        value: "more        spaces",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[1][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[1][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[1][3]).toEqual({
        value: "2nd",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.unquoted.yaml"]
      });
      expect(lines[2][0]).toEqual({
        value: "with quotes",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml"]
      });
      expect(lines[2][1]).toEqual({
        value: ":",
        scopes: ["source.yaml", "string.unquoted.yaml", "entity.name.tag.yaml", "punctuation.separator.key-value.yaml"]
      });
      expect(lines[2][2]).toEqual({
        value: " ",
        scopes: ["source.yaml", "string.unquoted.yaml"]
      });
      expect(lines[2][3]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.begin.yaml"]
      });
      expect(lines[2][4]).toEqual({
        value: "3",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml"]
      });
      return expect(lines[2][5]).toEqual({
        value: "\"",
        scopes: ["source.yaml", "string.unquoted.yaml", "string.quoted.double.yaml", "punctuation.definition.string.end.yaml"]
      });
    });
  });

}).call(this);
