(function() {
  var $, BufferedProcess, fs, os, plist, semver;

  $ = require('jquery');

  os = require('os');

  fs = require('fs');

  plist = require('plist');

  semver = require('semver');

  BufferedProcess = require('atom').BufferedProcess;


  /*
  A collection of methods for retrieving information about the user's system for
  bug report purposes.
   */

  module.exports = {

    /*
    Section: System Information
     */
    getPlatform: function() {
      return os.platform();
    },
    getOSVersion: function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          switch (_this.getPlatform()) {
            case 'darwin':
              return resolve(_this.macVersionText());
            case 'win32':
              return resolve(_this.winVersionText());
            default:
              return resolve("" + (os.platform()) + " " + (os.release()));
          }
        };
      })(this));
    },
    macVersionText: function() {
      return this.macVersionInfo().then(function(info) {
        if (!(info.ProductName && info.ProductVersion)) {
          return 'Unknown OS X version';
        }
        return "" + info.ProductName + " " + info.ProductVersion;
      });
    },
    macVersionInfo: function() {
      return new Promise(function(resolve, reject) {
        var e;
        try {
          return fs.readFile('/System/Library/CoreServices/SystemVersion.plist', 'utf8', function(error, text) {
            return resolve(plist.parse(text));
          });
        } catch (_error) {
          e = _error;
          return resolve('Unknown OSX version');
        }
      });
    },
    winVersionText: function() {
      return new Promise(function(resolve, reject) {
        var data, systemInfo;
        data = [];
        systemInfo = new BufferedProcess({
          command: 'systeminfo',
          stdout: function(oneLine) {
            return data.push(oneLine);
          },
          exit: (function(_this) {
            return function() {
              var info, res;
              info = data.join('\n');
              info = (res = /OS.Name.\s+(.*)$/im.exec(info)) ? res[1] : 'Unknown Windows Version';
              return resolve(info);
            };
          })(this)
        });
        return systemInfo.onWillThrowError(function(_arg) {
          var handle;
          handle = _arg.handle;
          handle();
          return resolve('Unknown Windows Version');
        });
      });
    },

    /*
    Section: Config Values
     */
    getConfigForPackage: function(packageName) {
      var config;
      config = {
        core: atom.config.settings.core
      };
      if (packageName != null) {
        config[packageName] = atom.config.settings[packageName];
      } else {
        config.editor = atom.config.settings.editor;
      }
      return config;
    },

    /*
    Section: Installed Packages
     */
    getInstalledPackages: function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var data;
          data = [];
          return new BufferedProcess({
            command: atom.packages.getApmPath(),
            args: ['ls', '--json', '--no-color'],
            stdout: function(oneLine) {
              return data.push(oneLine);
            },
            exit: function() {
              var packages, stdout;
              stdout = data.join('\n');
              packages = JSON.parse(stdout);
              return resolve({
                dev: _this.filterActivePackages(packages.dev),
                user: _this.filterActivePackages(packages.user)
              });
            }
          });
        };
      })(this));
    },
    filterActivePackages: function(packages) {
      var pack, _i, _len, _ref, _results;
      _ref = packages != null ? packages : [];
      _results = [];
      for (_i = 0, _len = _ref.length; _i < _len; _i++) {
        pack = _ref[_i];
        if (atom.packages.getActivePackage(pack.name) != null) {
          _results.push("" + pack.name + ", v" + pack.version);
        }
      }
      return _results;
    },
    getLatestAtomData: function() {
      var atomUrl;
      atomUrl = 'https://atom.io/api/updates';
      return new Promise(function(resolve, reject) {
        return $.ajax(atomUrl, {
          accept: 'application/vnd.github.v3+json',
          contentType: "application/json",
          success: function(data) {
            return resolve(data);
          },
          error: function(error) {
            return reject(error);
          }
        });
      });
    },
    checkAtomUpToDate: function() {
      return this.getLatestAtomData().then((function(_this) {
        return function(latestAtomData) {
          var installedVersion, latestVersion, upToDate;
          installedVersion = atom.getVersion();
          latestVersion = latestAtomData.name;
          upToDate = (installedVersion != null) && semver.gte(installedVersion, latestVersion);
          return {
            upToDate: upToDate,
            latestVersion: latestVersion,
            installedVersion: installedVersion
          };
        };
      })(this));
    },
    getPackageVersion: function(packageName) {
      var pack;
      pack = atom.packages.getLoadedPackage(packageName);
      return pack != null ? pack.metadata.version : void 0;
    },
    getPackageVersionShippedWithAtom: function(packageName) {
      return require(path.join(atom.getLoadSettings().resourcePath, 'package.json')).packageDependencies[packageName];
    },
    getLatestPackageData: function(packageName) {
      var packagesUrl;
      packagesUrl = 'https://atom.io/api/packages';
      return new Promise(function(resolve, reject) {
        return $.ajax("" + packagesUrl + "/" + packageName, {
          accept: 'application/vnd.github.v3+json',
          contentType: "application/json",
          success: function(data) {
            return resolve(data);
          },
          error: function(error) {
            return reject(error);
          }
        });
      });
    },
    checkPackageUpToDate: function(packageName) {
      return this.getLatestPackageData(packageName).then((function(_this) {
        return function(latestPackageData) {
          var installedVersion, isCore, latestVersion, upToDate, versionShippedWithAtom;
          installedVersion = _this.getPackageVersion(packageName);
          upToDate = (installedVersion != null) && semver.gte(installedVersion, latestPackageData.releases.latest);
          latestVersion = latestPackageData.releases.latest;
          isCore = latestPackageData.repository.url.startsWith('https://github.com/atom/');
          if (isCore) {
            versionShippedWithAtom = _this.getPackageVersionShippedWithAtom(packageName);
            upToDate = (installedVersion != null) && semver.gte(installedVersion, versionShippedWithAtom);
          }
          return {
            isCore: isCore,
            upToDate: upToDate,
            latestVersion: latestVersion,
            installedVersion: installedVersion,
            versionShippedWithAtom: versionShippedWithAtom
          };
        };
      })(this));
    }
  };

}).call(this);
