(function() {
  var CommandLogger;

  CommandLogger = require('../lib/command-logger');

  describe('CommandLogger', function() {
    var dispatch, logger, removeScrollbarClass, workspaceElement, _ref;
    _ref = [], workspaceElement = _ref[0], logger = _ref[1];
    dispatch = function(command) {
      return atom.commands.dispatch(workspaceElement, command);
    };
    removeScrollbarClass = function(str) {
      return str.replace(/\.scrollbars-visible-(when-scrolling|always)/g, '');
    };
    beforeEach(function() {
      workspaceElement = atom.views.getView(atom.workspace);
      logger = new CommandLogger;
      return logger.start();
    });
    describe('logging of commands', function() {
      it('catches the name of the command', function() {
        dispatch('foo:bar');
        return expect(logger.latestEvent().name).toBe('foo:bar');
      });
      it('catches the source of the command', function() {
        dispatch('foo:bar');
        return expect(logger.latestEvent().source).toBeDefined();
      });
      it('logs repeat commands as one command', function() {
        dispatch('foo:bar');
        dispatch('foo:bar');
        expect(logger.latestEvent().name).toBe('foo:bar');
        return expect(logger.latestEvent().count).toBe(2);
      });
      it('ignores show.bs.tooltip commands', function() {
        dispatch('show.bs.tooltip');
        return expect(logger.latestEvent().name).not.toBe('show.bs.tooltip');
      });
      it('ignores editor:display-updated commands', function() {
        dispatch('editor:display-updated');
        return expect(logger.latestEvent().name).not.toBe('editor:display-updated');
      });
      it('ignores mousewheel commands', function() {
        dispatch('mousewheel');
        return expect(logger.latestEvent().name).not.toBe('mousewheel');
      });
      return it('only logs up to `logSize` commands', function() {
        var char, _i;
        for (char = _i = 'a'; 'a' <= 'z' ? _i <= 'z' : _i >= 'z'; char = 'a' <= 'z' ? ++_i : --_i) {
          dispatch(char);
        }
        return expect(logger.eventLog.length).toBe(logger.logSize);
      });
    });
    return describe('formatting of text log', function() {
      it('does not output empty log items', function() {
        return expect(logger.getText()).toBe("```\n```");
      });
      it('formats commands with the time, name and source', function() {
        atom.commands.dispatch(workspaceElement, 'foo:bar');
        return expect(removeScrollbarClass(logger.getText())).toBe("```\n     -0:00.0 foo:bar (atom-workspace.workspace)\n```");
      });
      it('formats commands in chronological order', function() {
        dispatch('foo:first');
        dispatch('foo:second');
        dispatch('foo:third');
        return expect(removeScrollbarClass(logger.getText())).toBe("```\n     -0:00.0 foo:first (atom-workspace.workspace)\n     -0:00.0 foo:second (atom-workspace.workspace)\n     -0:00.0 foo:third (atom-workspace.workspace)\n```");
      });
      it('displays a multiplier for repeated commands', function() {
        dispatch('foo:bar');
        dispatch('foo:bar');
        return expect(removeScrollbarClass(logger.getText())).toBe("```\n  2x -0:00.0 foo:bar (atom-workspace.workspace)\n```");
      });
      it('logs the external data event as the last event', function() {
        var event;
        dispatch('foo:bar');
        event = {
          time: Date.now(),
          title: 'bummer'
        };
        return expect(removeScrollbarClass(logger.getText(event))).toBe("```\n     -0:00.0 foo:bar (atom-workspace.workspace)\n     -0:00.0 bummer\n```");
      });
      it('does not report anything older than ten minutes', function() {
        logger.logCommand({
          type: 'foo:bar',
          time: Date.now() - 11 * 60 * 1000,
          target: {
            nodeName: 'DIV'
          }
        });
        logger.logCommand({
          type: 'wow:bummer',
          target: {
            nodeName: 'DIV'
          }
        });
        return expect(logger.getText()).toBe("```\n     -0:00.0 wow:bummer (div)\n```");
      });
      return it('does not report commands that have no name', function() {
        dispatch('');
        return expect(logger.getText()).toBe("```\n```");
      });
    });
  });

}).call(this);
