(function() {
  var $, Notification, NotificationElement, fs, generateException, generateFakeAjaxResponses, temp;

  $ = require('jquery');

  fs = require('fs-plus');

  temp = require('temp').track();

  Notification = require('atom').Notification;

  NotificationElement = require('../lib/notification-element');

  describe("Notifications", function() {
    var activationPromise, workspaceElement, _ref;
    _ref = [], workspaceElement = _ref[0], activationPromise = _ref[1];
    beforeEach(function() {
      workspaceElement = atom.views.getView(atom.workspace);
      atom.notifications.clear();
      activationPromise = atom.packages.activatePackage('notifications');
      return waitsForPromise(function() {
        return activationPromise;
      });
    });
    describe("when the package is activated", function() {
      return it("attaches an atom-notifications element to the dom", function() {
        return expect(workspaceElement.querySelector('atom-notifications')).toBeDefined();
      });
    });
    describe("when there are notifications before activation", function() {
      beforeEach(function() {
        return atom.packages.deactivatePackage('notifications');
      });
      return it("displays all non displayed notifications", function() {
        var error, warning;
        warning = new Notification('warning', 'Un-displayed warning');
        error = new Notification('error', 'Displayed error');
        error.setDisplayed(true);
        atom.notifications.addNotification(error);
        atom.notifications.addNotification(warning);
        activationPromise = atom.packages.activatePackage('notifications');
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var notification, notificationContainer;
          notificationContainer = workspaceElement.querySelector('atom-notifications');
          notification = notificationContainer.querySelector('atom-notification.warning');
          expect(notification).toExist();
          notification = notificationContainer.querySelector('atom-notification.error');
          return expect(notification).not.toExist();
        });
      });
    });
    return describe("when notifications are added to atom.notifications", function() {
      var notificationContainer;
      notificationContainer = null;
      beforeEach(function() {
        var enableInitNotification;
        enableInitNotification = atom.notifications.addSuccess('A message to trigger initialization', {
          dismissable: true
        });
        enableInitNotification.dismiss();
        advanceClock(NotificationElement.prototype.visibilityDuration);
        advanceClock(NotificationElement.prototype.animationDuration);
        notificationContainer = workspaceElement.querySelector('atom-notifications');
        return spyOn($, 'ajax');
      });
      it("adds an atom-notification element to the container with a class corresponding to the type", function() {
        var notification;
        expect(notificationContainer.childNodes.length).toBe(0);
        atom.notifications.addSuccess('A message');
        notification = notificationContainer.querySelector('atom-notification.success');
        expect(notificationContainer.childNodes.length).toBe(1);
        expect(notification).toHaveClass('success');
        expect(notification.querySelector('.message').textContent.trim()).toBe('A message');
        atom.notifications.addInfo('A message');
        expect(notificationContainer.childNodes.length).toBe(2);
        expect(notificationContainer.querySelector('atom-notification.info')).toBeDefined();
        atom.notifications.addWarning('A message');
        expect(notificationContainer.childNodes.length).toBe(3);
        expect(notificationContainer.querySelector('atom-notification.warning')).toBeDefined();
        atom.notifications.addError('A message');
        expect(notificationContainer.childNodes.length).toBe(4);
        expect(notificationContainer.querySelector('atom-notification.error')).toBeDefined();
        atom.notifications.addFatalError('A message');
        expect(notificationContainer.childNodes.length).toBe(5);
        return expect(notificationContainer.querySelector('atom-notification.fatal')).toBeDefined();
      });
      describe("when a dismissable notification is added", function() {
        it("is removed when Notification::dismiss() is called", function() {
          var notification, notificationElement;
          notification = atom.notifications.addSuccess('A message', {
            dismissable: true
          });
          notificationElement = notificationContainer.querySelector('atom-notification.success');
          expect(notificationContainer.childNodes.length).toBe(1);
          notification.dismiss();
          advanceClock(NotificationElement.prototype.visibilityDuration);
          expect(notificationElement).toHaveClass('remove');
          advanceClock(NotificationElement.prototype.animationDuration);
          return expect(notificationContainer.childNodes.length).toBe(0);
        });
        it("is removed when the close icon is clicked", function() {
          jasmine.attachToDOM(workspaceElement);
          waitsForPromise(function() {
            return atom.workspace.open();
          });
          return runs(function() {
            var notification, notificationElement;
            notification = atom.notifications.addSuccess('A message', {
              dismissable: true
            });
            notificationElement = notificationContainer.querySelector('atom-notification.success');
            expect(notificationContainer.childNodes.length).toBe(1);
            notificationElement.focus();
            notificationElement.querySelector('.close.icon').click();
            expect(atom.views.getView(atom.workspace.getActiveTextEditor())).toHaveFocus();
            advanceClock(NotificationElement.prototype.visibilityDuration);
            expect(notificationElement).toHaveClass('remove');
            advanceClock(NotificationElement.prototype.animationDuration);
            return expect(notificationContainer.childNodes.length).toBe(0);
          });
        });
        return it("is removed when core:cancel is triggered", function() {
          var notification, notificationElement;
          notification = atom.notifications.addSuccess('A message', {
            dismissable: true
          });
          notificationElement = notificationContainer.querySelector('atom-notification.success');
          expect(notificationContainer.childNodes.length).toBe(1);
          atom.commands.dispatch(workspaceElement, 'core:cancel');
          advanceClock(NotificationElement.prototype.visibilityDuration * 3);
          expect(notificationElement).toHaveClass('remove');
          advanceClock(NotificationElement.prototype.animationDuration * 3);
          return expect(notificationContainer.childNodes.length).toBe(0);
        });
      });
      describe("when an autoclose notification is added", function() {
        return it("closes and removes the message after a given amount of time", function() {
          var notification;
          atom.notifications.addSuccess('A message');
          notification = notificationContainer.querySelector('atom-notification.success');
          expect(notification).not.toHaveClass('remove');
          advanceClock(NotificationElement.prototype.visibilityDuration);
          expect(notification).toHaveClass('remove');
          expect(notificationContainer.childNodes.length).toBe(1);
          advanceClock(NotificationElement.prototype.animationDuration);
          return expect(notificationContainer.childNodes.length).toBe(0);
        });
      });
      return describe("when an exception is thrown", function() {
        var fatalError, issueBody, _ref1;
        _ref1 = [], notificationContainer = _ref1[0], fatalError = _ref1[1], issueBody = _ref1[2];
        describe("when the editor is in dev mode", function() {
          beforeEach(function() {
            spyOn(atom, 'inDevMode').andReturn(true);
            generateException();
            notificationContainer = workspaceElement.querySelector('atom-notifications');
            return fatalError = notificationContainer.querySelector('atom-notification.fatal');
          });
          return it("does not display a notification", function() {
            expect(notificationContainer.childNodes.length).toBe(0);
            return expect(fatalError).toBe(null);
          });
        });
        describe("when there are multiple packages in the stack trace", function() {
          beforeEach(function() {
            var detail, stack;
            stack = "TypeError: undefined is not a function\n  at Object.module.exports.Pane.promptToSaveItem [as defaultSavePrompt] (/Applications/Atom.app/Contents/Resources/app/src/pane.js:490:23)\n  at Pane.promptToSaveItem (/Users/someguy/.atom/packages/save-session/lib/save-prompt.coffee:21:15)\n  at Pane.module.exports.Pane.destroyItem (/Applications/Atom.app/Contents/Resources/app/src/pane.js:442:18)\n  at HTMLDivElement.<anonymous> (/Applications/Atom.app/Contents/Resources/app/node_modules/tabs/lib/tab-bar-view.js:174:22)\n  at space-pen-ul.jQuery.event.dispatch (/Applications/Atom.app/Contents/Resources/app/node_modules/archive-view/node_modules/atom-space-pen-views/node_modules/space-pen/vendor/jquery.js:4676:9)\n  at space-pen-ul.elemData.handle (/Applications/Atom.app/Contents/Resources/app/node_modules/archive-view/node_modules/atom-space-pen-views/node_modules/space-pen/vendor/jquery.js:4360:46)";
            detail = 'ok';
            atom.notifications.addFatalError('TypeError: undefined', {
              detail: detail,
              stack: stack
            });
            notificationContainer = workspaceElement.querySelector('atom-notifications');
            fatalError = notificationContainer.querySelector('atom-notification.fatal');
            spyOn(require('fs'), 'realpathSync').andCallFake(function(p) {
              return p;
            });
            return spyOn(fatalError.issue, 'getPackagePathsByPackageName').andCallFake(function() {
              return {
                'save-session': '/Users/someguy/.atom/packages/save-session',
                'tabs': '/Applications/Atom.app/Contents/Resources/app/node_modules/tabs'
              };
            });
          });
          return it("chooses the first package in the trace", function() {
            return expect(fatalError.issue.getPackageName()).toBe('save-session');
          });
        });
        describe("when an exception is thrown from a package", function() {
          beforeEach(function() {
            issueBody = null;
            spyOn(atom, 'inDevMode').andReturn(false);
            generateFakeAjaxResponses();
            generateException();
            notificationContainer = workspaceElement.querySelector('atom-notifications');
            return fatalError = notificationContainer.querySelector('atom-notification.fatal');
          });
          it("displays a fatal error with the package name in the error", function() {
            waitsForPromise(function() {
              return fatalError.getRenderPromise().then(function() {
                return issueBody = fatalError.issue.issueBody;
              });
            });
            return runs(function() {
              var button;
              expect(notificationContainer.childNodes.length).toBe(1);
              expect(fatalError).toHaveClass('has-close');
              expect(fatalError.innerHTML).toContain('ReferenceError: a is not defined');
              expect(fatalError.innerHTML).toContain("<a href=\"https://github.com/atom/notifications\">notifications package</a>");
              expect(fatalError.issue.getPackageName()).toBe('notifications');
              button = fatalError.querySelector('.btn');
              expect(button.textContent).toContain('Create issue on the notifications package');
              expect(button.getAttribute('href')).toContain('atom/notifications/issues/new');
              expect(issueBody).toMatch(/Atom Version\*\*: [0-9].[0-9]+.[0-9]+/ig);
              expect(issueBody).not.toMatch(/Unknown/ig);
              expect(issueBody).toContain('ReferenceError: a is not defined');
              expect(issueBody).toContain('Thrown From**: [notifications](https://github.com/atom/notifications) package, v');
              return expect(issueBody).toContain('# User');
            });
          });
          return it("contains core and notifications config values", function() {
            atom.config.set('notifications.something', 10);
            waitsForPromise(function() {
              return fatalError.getRenderPromise().then(function() {
                return issueBody = fatalError.issue.issueBody;
              });
            });
            return runs(function() {
              expect(issueBody).toContain('"core":');
              expect(issueBody).toContain('"notifications":');
              return expect(issueBody).not.toContain('"editor":');
            });
          });
        });
        describe("when an exception is thrown from an unloaded package", function() {
          beforeEach(function() {
            var detail, message, packageDir, packagesDir, stack;
            spyOn(atom, 'inDevMode').andReturn(false);
            generateFakeAjaxResponses();
            packagesDir = temp.mkdirSync('atom-packages-');
            atom.packages.packageDirPaths.push(path.join(packagesDir, '.atom', 'packages'));
            packageDir = path.join(packagesDir, '.atom', 'packages', 'unloaded');
            fs.writeFileSync(path.join(packageDir, 'package.json'), "{\n  \"name\": \"unloaded\",\n  \"version\": \"1.0.0\",\n  \"repository\": \"https://github.com/atom/notifications\"\n}");
            stack = "Error\n  at " + (path.join(packageDir, 'index.js')) + ":1:1";
            detail = 'ReferenceError: unloaded error';
            message = "Error";
            atom.notifications.addFatalError(message, {
              stack: stack,
              detail: detail,
              dismissable: true
            });
            notificationContainer = workspaceElement.querySelector('atom-notifications');
            return fatalError = notificationContainer.querySelector('atom-notification.fatal');
          });
          return it("displays a fatal error with the package name in the error", function() {
            waitsForPromise(function() {
              return fatalError.getRenderPromise();
            });
            return runs(function() {
              expect(notificationContainer.childNodes.length).toBe(1);
              expect(fatalError).toHaveClass('has-close');
              expect(fatalError.innerHTML).toContain('ReferenceError: unloaded error');
              expect(fatalError.innerHTML).toContain("<a href=\"https://github.com/atom/notifications\">unloaded package</a>");
              return expect(fatalError.issue.getPackageName()).toBe('unloaded');
            });
          });
        });
        describe("when an exception is thrown from a package without a trace, but with a URL", function() {
          beforeEach(function() {
            var e, filePath;
            issueBody = null;
            spyOn(atom, 'inDevMode').andReturn(false);
            generateFakeAjaxResponses();
            try {
              a + 1;
            } catch (_error) {
              e = _error;
              filePath = e.stack.split('\n')[1].match(/\(([^:]+)/)[1];
              window.onerror.call(window, e.toString(), filePath, 2, 3, {
                message: e.toString(),
                stack: void 0
              });
            }
            notificationContainer = workspaceElement.querySelector('atom-notifications');
            return fatalError = notificationContainer.querySelector('atom-notification.fatal');
          });
          return it("detects the package name from the URL", function() {
            waitsForPromise(function() {
              return fatalError.getRenderPromise();
            });
            return runs(function() {
              expect(fatalError.innerHTML).toContain('ReferenceError: a is not defined');
              expect(fatalError.innerHTML).toContain("<a href=\"https://github.com/atom/notifications\">notifications package</a>");
              return expect(fatalError.issue.getPackageName()).toBe('notifications');
            });
          });
        });
        describe("when an exception is thrown from core", function() {
          beforeEach(function() {
            var e;
            atom.commands.dispatch(workspaceElement, 'some-package:a-command');
            atom.commands.dispatch(workspaceElement, 'some-package:a-command');
            atom.commands.dispatch(workspaceElement, 'some-package:a-command');
            spyOn(atom, 'inDevMode').andReturn(false);
            generateFakeAjaxResponses();
            try {
              a + 1;
            } catch (_error) {
              e = _error;
              e.stack = e.stack.replace(/notifications/g, 'core');
              window.onerror.call(window, e.toString(), '/dev/null', 2, 3, e);
            }
            notificationContainer = workspaceElement.querySelector('atom-notifications');
            fatalError = notificationContainer.querySelector('atom-notification.fatal');
            return waitsForPromise(function() {
              return fatalError.getRenderPromise().then(function() {
                return issueBody = fatalError.issue.issueBody;
              });
            });
          });
          it("displays a fatal error with the package name in the error", function() {
            var button;
            expect(notificationContainer.childNodes.length).toBe(1);
            expect(fatalError).toBeDefined();
            expect(fatalError).toHaveClass('has-close');
            expect(fatalError.innerHTML).toContain('ReferenceError: a is not defined');
            expect(fatalError.innerHTML).toContain('bug in Atom');
            expect(fatalError.issue.getPackageName()).toBeUndefined();
            button = fatalError.querySelector('.btn');
            expect(button.textContent).toContain('Create issue on atom/atom');
            expect(button.getAttribute('href')).toContain('atom/atom/issues/new');
            expect(issueBody).toContain('ReferenceError: a is not defined');
            return expect(issueBody).toContain('**Thrown From**: Atom Core');
          });
          it("contains core and editor config values", function() {
            expect(issueBody).toContain('"core":');
            expect(issueBody).toContain('"editor":');
            return expect(issueBody).not.toContain('"notifications":');
          });
          it("contains the commands that the user run in the issue body", function() {
            return expect(issueBody).toContain('some-package:a-command');
          });
          return it("allows the user to toggle the stack trace", function() {
            var stackContainer, stackToggle;
            stackToggle = fatalError.querySelector('.stack-toggle');
            stackContainer = fatalError.querySelector('.stack-container');
            expect(stackToggle).toExist();
            expect(stackContainer.style.display).toBe('none');
            stackToggle.click();
            expect(stackContainer.style.display).toBe('block');
            stackToggle.click();
            return expect(stackContainer.style.display).toBe('none');
          });
        });
        describe("when the there is an error searching for the issue", function() {
          beforeEach(function() {
            spyOn(atom, 'inDevMode').andReturn(false);
            generateFakeAjaxResponses({
              issuesErrorResponse: '403'
            });
            generateException();
            fatalError = notificationContainer.querySelector('atom-notification.fatal');
            return waitsForPromise(function() {
              return fatalError.getRenderPromise().then(function() {
                return issueBody = fatalError.issue.issueBody;
              });
            });
          });
          return it("asks the user to create an issue", function() {
            var button, fatalNotification;
            button = fatalError.querySelector('.btn');
            fatalNotification = fatalError.querySelector('.fatal-notification');
            expect(button.textContent).toContain('Create issue');
            expect(fatalNotification.textContent).toContain('You can help by creating an issue');
            return expect(button.getAttribute('href')).toContain('github.com/atom/notifications/issues/new');
          });
        });
        describe("when the error has not been reported", function() {
          beforeEach(function() {
            return spyOn(atom, 'inDevMode').andReturn(false);
          });
          describe("when the system is darwin", function() {
            beforeEach(function() {
              generateFakeAjaxResponses();
              generateException();
              fatalError = notificationContainer.querySelector('atom-notification.fatal');
              return waitsForPromise(function() {
                return fatalError.getRenderPromise().then(function() {
                  return issueBody = fatalError.issue.issueBody;
                });
              });
            });
            return it("asks the user to create an issue", function() {
              var button, fatalNotification;
              button = fatalError.querySelector('.btn');
              fatalNotification = fatalError.querySelector('.fatal-notification');
              expect(button.textContent).toContain('Create issue');
              expect(fatalNotification.textContent).toContain('You can help by creating an issue');
              return expect(button.getAttribute('href')).toContain('github.com/atom/notifications/issues/new');
            });
          });
          return describe("when the system is win32", function() {
            beforeEach(function() {
              var UserUtilities;
              UserUtilities = require('../lib/user-utilities');
              spyOn(UserUtilities, 'getPlatform').andReturn('win32');
              generateFakeAjaxResponses();
              generateException();
              fatalError = notificationContainer.querySelector('atom-notification.fatal');
              return waitsForPromise(function() {
                return fatalError.getRenderPromise().then(function() {
                  return issueBody = fatalError.issue.issueBody;
                });
              });
            });
            return it("uses a shortened url via git.io", function() {
              var button;
              button = fatalError.querySelector('.btn');
              expect(button.textContent).toContain('Create issue');
              return expect(button.getAttribute('href')).toContain('git.io');
            });
          });
        });
        describe("when the package is out of date", function() {
          beforeEach(function() {
            var UserUtilities, installedVersion;
            installedVersion = '0.9.0';
            UserUtilities = require('../lib/user-utilities');
            spyOn(UserUtilities, 'getPackageVersion').andCallFake(function() {
              return installedVersion;
            });
            return spyOn(atom, 'inDevMode').andReturn(false);
          });
          describe("when the package is a non-core package", function() {
            beforeEach(function() {
              generateFakeAjaxResponses({
                packageResponse: {
                  repository: {
                    url: 'https://github.com/someguy/notifications'
                  },
                  releases: {
                    latest: '0.10.0'
                  }
                }
              });
              generateException();
              fatalError = notificationContainer.querySelector('atom-notification.fatal');
              return waitsForPromise(function() {
                return fatalError.getRenderPromise().then(function() {
                  return issueBody = fatalError.issue.issueBody;
                });
              });
            });
            return it("asks the user to update their packages", function() {
              var button, fatalNotification;
              fatalNotification = fatalError.querySelector('.fatal-notification');
              button = fatalError.querySelector('.btn');
              expect(button.textContent).toContain('Check for package updates');
              expect(fatalNotification.textContent).toContain('Upgrading to the latest');
              return expect(button.getAttribute('href')).toBe('#');
            });
          });
          return describe("when the package is a core package", function() {
            beforeEach(function() {
              return generateFakeAjaxResponses({
                packageResponse: {
                  repository: {
                    url: 'https://github.com/atom/notifications'
                  },
                  releases: {
                    latest: '0.11.0'
                  }
                }
              });
            });
            describe("when the locally installed version is lower than Atom's version", function() {
              beforeEach(function() {
                var UserUtilities, versionShippedWithAtom;
                versionShippedWithAtom = '0.10.0';
                UserUtilities = require('../lib/user-utilities');
                spyOn(UserUtilities, 'getPackageVersionShippedWithAtom').andCallFake(function() {
                  return versionShippedWithAtom;
                });
                generateException();
                fatalError = notificationContainer.querySelector('atom-notification.fatal');
                return waitsForPromise(function() {
                  return fatalError.getRenderPromise().then(function() {
                    return issueBody = fatalError.issue.issueBody;
                  });
                });
              });
              it("doesn't show the Create Issue button", function() {
                var button;
                button = fatalError.querySelector('.btn-issue');
                return expect(button).not.toExist();
              });
              return it("tells the user that the package is a locally installed core package and out of date", function() {
                var fatalNotification;
                fatalNotification = fatalError.querySelector('.fatal-notification');
                expect(fatalNotification.textContent).toContain('Locally installed core Atom package');
                return expect(fatalNotification.textContent).toContain('is out of date');
              });
            });
            return describe("when the locally installed version matches Atom's version", function() {
              beforeEach(function() {
                var UserUtilities, versionShippedWithAtom;
                versionShippedWithAtom = '0.9.0';
                UserUtilities = require('../lib/user-utilities');
                spyOn(UserUtilities, 'getPackageVersionShippedWithAtom').andCallFake(function() {
                  return versionShippedWithAtom;
                });
                generateException();
                fatalError = notificationContainer.querySelector('atom-notification.fatal');
                return waitsForPromise(function() {
                  return fatalError.getRenderPromise().then(function() {
                    return issueBody = fatalError.issue.issueBody;
                  });
                });
              });
              return it("ignores the out of date package because they cant upgrade it without upgrading atom", function() {
                var button;
                fatalError = notificationContainer.querySelector('atom-notification.fatal');
                button = fatalError.querySelector('.btn');
                return expect(button.textContent).toContain('Create issue');
              });
            });
          });
        });
        describe("when Atom is out of date", function() {
          beforeEach(function() {
            var installedVersion;
            installedVersion = '0.179.0';
            spyOn(atom, 'getVersion').andCallFake(function() {
              return installedVersion;
            });
            spyOn(atom, 'inDevMode').andReturn(false);
            generateFakeAjaxResponses({
              atomResponse: {
                name: '0.180.0'
              }
            });
            generateException();
            fatalError = notificationContainer.querySelector('atom-notification.fatal');
            return waitsForPromise(function() {
              return fatalError.getRenderPromise().then(function() {
                return issueBody = fatalError.issue.issueBody;
              });
            });
          });
          it("doesn't show the Create Issue button", function() {
            var button;
            button = fatalError.querySelector('.btn-issue');
            return expect(button).not.toExist();
          });
          it("tells the user that Atom is out of date", function() {
            var fatalNotification;
            fatalNotification = fatalError.querySelector('.fatal-notification');
            return expect(fatalNotification.textContent).toContain('Atom is out of date');
          });
          return it("provides a link to the latest released version", function() {
            var fatalNotification;
            fatalNotification = fatalError.querySelector('.fatal-notification');
            return expect(fatalNotification.innerHTML).toContain('<a href="https://github.com/atom/atom/releases/tag/v0.180.0">latest version</a>');
          });
        });
        describe("when the error has been reported", function() {
          beforeEach(function() {
            return spyOn(atom, 'inDevMode').andReturn(false);
          });
          describe("when the issue is open", function() {
            beforeEach(function() {
              generateFakeAjaxResponses({
                issuesResponse: {
                  items: [
                    {
                      title: 'ReferenceError: a is not defined',
                      html_url: 'http://url.com/ok',
                      state: 'open'
                    }
                  ]
                }
              });
              generateException();
              fatalError = notificationContainer.querySelector('atom-notification.fatal');
              return waitsForPromise(function() {
                return fatalError.getRenderPromise().then(function() {
                  return issueBody = fatalError.issue.issueBody;
                });
              });
            });
            return it("shows the user a view issue button", function() {
              var button, fatalNotification;
              fatalNotification = fatalError.querySelector('.fatal-notification');
              button = fatalError.querySelector('.btn');
              expect(button.textContent).toContain('View Issue');
              expect(button.getAttribute('href')).toBe('http://url.com/ok');
              expect(fatalNotification.textContent).toContain('already been reported');
              return expect($.ajax.calls[0].args[0]).toContain('atom/notifications');
            });
          });
          return describe("when the issue is closed", function() {
            beforeEach(function() {
              generateFakeAjaxResponses({
                issuesResponse: {
                  items: [
                    {
                      title: 'ReferenceError: a is not defined',
                      html_url: 'http://url.com/closed',
                      state: 'closed'
                    }
                  ]
                }
              });
              generateException();
              fatalError = notificationContainer.querySelector('atom-notification.fatal');
              return waitsForPromise(function() {
                return fatalError.getRenderPromise().then(function() {
                  return issueBody = fatalError.issue.issueBody;
                });
              });
            });
            return it("shows the user a view issue button", function() {
              var button;
              button = fatalError.querySelector('.btn');
              expect(button.textContent).toContain('View Issue');
              return expect(button.getAttribute('href')).toBe('http://url.com/closed');
            });
          });
        });
        describe("when a BufferedProcessError is thrown", function() {
          return it("adds an error to the notifications", function() {
            var error;
            expect(notificationContainer.querySelector('atom-notification.error')).not.toExist();
            window.onerror('Uncaught BufferedProcessError: Failed to spawn command `bad-command`', 'abc', 2, 3, {
              name: 'BufferedProcessError'
            });
            error = notificationContainer.querySelector('atom-notification.error');
            expect(error).toExist();
            expect(error.innerHTML).toContain('Failed to spawn command');
            return expect(error.innerHTML).not.toContain('BufferedProcessError');
          });
        });
        return describe("when a spawn ENOENT error is thrown", function() {
          beforeEach(function() {
            return spyOn(atom, 'inDevMode').andReturn(false);
          });
          describe("when the binary has no path", function() {
            beforeEach(function() {
              var e, message;
              try {
                return a + 1;
              } catch (_error) {
                e = _error;
                e.code = 'ENOENT';
                message = 'Error: spawn some_binary ENOENT';
                return window.onerror.call(window, message, 'abc', 2, 3, e);
              }
            });
            return it("displays a dismissable error without the stack trace", function() {
              var error;
              notificationContainer = workspaceElement.querySelector('atom-notifications');
              error = notificationContainer.querySelector('atom-notification.error');
              return expect(error.textContent).toContain("'some_binary' could not be spawned");
            });
          });
          return describe("when the binary has /atom in the path", function() {
            beforeEach(function() {
              var e, message;
              try {
                return a + 1;
              } catch (_error) {
                e = _error;
                e.code = 'ENOENT';
                message = 'Error: spawn /opt/atom/Atom Helper (deleted) ENOENT';
                return window.onerror.call(window, message, 'abc', 2, 3, e);
              }
            });
            return it("displays a fatal error", function() {
              var error;
              notificationContainer = workspaceElement.querySelector('atom-notifications');
              error = notificationContainer.querySelector('atom-notification.fatal');
              return expect(error).toExist();
            });
          });
        });
      });
    });
  });

  generateException = function() {
    var e;
    try {
      return a + 1;
    } catch (_error) {
      e = _error;
      return window.onerror.call(window, e.toString(), '/dev/null', 2, 3, e);
    }
  };

  generateFakeAjaxResponses = function(options) {
    return $.ajax.andCallFake(function(url, settings) {
      var response, _ref, _ref1, _ref2, _ref3;
      if (url.indexOf('git.io') > -1) {
        response = (_ref = options != null ? options.shortenerResponse : void 0) != null ? _ref : [
          '--', '201', {
            getResponseHeader: function() {
              return 'http://git.io/cats';
            }
          }
        ];
        return settings.success.apply(settings, response);
      } else if (url.indexOf('atom.io/api/packages') > -1) {
        response = (_ref1 = options != null ? options.packageResponse : void 0) != null ? _ref1 : {
          repository: {
            url: 'https://github.com/atom/notifications'
          },
          releases: {
            latest: '0.0.0'
          }
        };
        return settings.success(response);
      } else if (url.indexOf('atom.io/api/updates') > -1) {
        response = (_ref2 = options != null ? options.atomResponse : void 0) != null ? _ref2 : {
          name: atom.getVersion()
        };
        return settings.success(response);
      } else {
        if ((options != null ? options.issuesErrorResponse : void 0) != null) {
          return typeof settings.error === "function" ? settings.error({}, options.issuesErrorResponse, null) : void 0;
        } else {
          response = (_ref3 = options != null ? options.issuesResponse : void 0) != null ? _ref3 : {
            items: []
          };
          return settings.success(response);
        }
      }
    });
  };

  window.waitsForPromise = function(fn) {
    var promise;
    promise = fn();
    return window.waitsFor(5000, function(moveOn) {
      promise.then(moveOn);
      return promise["catch"](function(error) {
        jasmine.getEnv().currentSpec.fail("Expected promise to be resolved, but it was rejected with " + (jasmine.pp(error)));
        return moveOn();
      });
    });
  };

}).call(this);
