(function() {
  var GitHubFile, fs, os, path;

  GitHubFile = require('../lib/github-file');

  fs = require('fs-plus');

  path = require('path');

  os = require('os');

  describe("GitHubFile", function() {
    var editor, githubFile, _ref;
    _ref = [], githubFile = _ref[0], editor = _ref[1];
    describe("commands", function() {
      var filePathRelativeToWorkingDir, fixturePath, setupGithubFile, setupWorkingDir, teardownWorkingDirAndRestoreFixture, workingDirPath;
      workingDirPath = path.join(os.tmpdir(), 'open-on-github-working-dir');
      filePathRelativeToWorkingDir = 'some-dir/some-file.md';
      fixturePath = function(fixtureName) {
        return path.join(__dirname, "fixtures", "" + fixtureName + ".git");
      };
      setupWorkingDir = function(fixtureName) {
        var filePath, subdirectoryPath;
        fs.makeTreeSync(workingDirPath);
        fs.copySync(fixturePath(fixtureName), path.join(workingDirPath, '.git'));
        subdirectoryPath = path.join(workingDirPath, 'some-dir');
        fs.makeTreeSync(subdirectoryPath);
        filePath = path.join(subdirectoryPath, 'some-file.md');
        return fs.writeFileSync(filePath, 'some file content');
      };
      setupGithubFile = function() {
        atom.project.setPaths([workingDirPath]);
        waitsForPromise(function() {
          return atom.workspace.open(filePathRelativeToWorkingDir);
        });
        return runs(function() {
          editor = atom.workspace.getActiveTextEditor();
          return githubFile = GitHubFile.fromPath(editor.getPath());
        });
      };
      teardownWorkingDirAndRestoreFixture = function(fixtureName) {
        var success;
        success = null;
        runs(function() {
          var repeat;
          atom.project.setPaths([]);
          return repeat = setInterval(function() {
            var e;
            try {
              fs.removeSync(workingDirPath);
              clearInterval(repeat);
              return success = true;
            } catch (_error) {
              e = _error;
              return success = false;
            }
          }, 50);
        });
        return waitsFor(function() {
          return success;
        });
      };
      describe("open", function() {
        describe("when the file is openable on GitHub.com", function() {
          var fixtureName;
          fixtureName = 'github-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          it("opens the GitHub.com blob URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.open();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/master/some-dir/some-file.md');
          });
          describe("when text is selected", function() {
            return it("opens the GitHub.com blob URL for the file with the selection range in the hash", function() {
              atom.config.set('open-on-github.includeLineNumbersInUrls', true);
              spyOn(githubFile, 'openUrlInBrowser');
              githubFile.open([[0, 0], [1, 1]]);
              return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/master/some-dir/some-file.md#L1-L2');
            });
          });
          return describe("when the file has a '#' in its name", function() {
            return it("opens the GitHub.com blob URL for the file", function() {
              waitsForPromise(function() {
                return atom.workspace.open('a/b#/test#hash.md');
              });
              return runs(function() {
                editor = atom.workspace.getActiveTextEditor();
                githubFile = GitHubFile.fromPath(editor.getPath());
                spyOn(githubFile, 'openUrlInBrowser');
                githubFile.open();
                return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/master/a/b%23/test%23hash.md');
              });
            });
          });
        });
        describe("when the branch has a '/' in its name", function() {
          var fixtureName;
          fixtureName = 'branch-with-slash-in-name';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com blob URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.open();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/foo/bar/some-dir/some-file.md');
          });
        });
        describe("when the branch has a '#' in its name", function() {
          var fixtureName;
          fixtureName = 'branch-with-hash-in-name';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com blob URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.open();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/a%23b%23c/some-dir/some-file.md');
          });
        });
        describe("when the remote has a '/' in its name", function() {
          var fixtureName;
          fixtureName = 'remote-with-slash-in-name';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com blob URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.open();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/baz/some-dir/some-file.md');
          });
        });
        describe("when the local branch is not tracked", function() {
          var fixtureName;
          fixtureName = 'non-tracked-branch';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com blob URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.open();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blob/non-tracked-branch/some-dir/some-file.md');
          });
        });
        describe("when there is no remote", function() {
          var fixtureName;
          fixtureName = 'no-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("logs an error", function() {
            spyOn(atom.notifications, 'addWarning');
            githubFile.open();
            return expect(atom.notifications.addWarning).toHaveBeenCalledWith('No URL defined for remote: null');
          });
        });
        describe("when the root directory doesn't have a git repo", function() {
          beforeEach(function() {
            teardownWorkingDirAndRestoreFixture();
            fs.mkdirSync(workingDirPath);
            return setupGithubFile();
          });
          return it("does nothing", function() {
            spyOn(atom.notifications, 'addWarning');
            githubFile.open();
            expect(atom.notifications.addWarning).toHaveBeenCalled();
            return expect(atom.notifications.addWarning.mostRecentCall.args[0]).toContain("No repository found");
          });
        });
        return describe("when the remote repo is not hosted on github.com", function() {
          var fixtureName;
          fixtureName = 'github-enterprise-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return githubFile = setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens a GitHub enterprise style blob URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.open();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://git.enterprize.me/some-user/some-repo/blob/master/some-dir/some-file.md');
          });
        });
      });
      describe("blame", function() {
        return describe("when the file is openable on GitHub.com", function() {
          var fixtureName;
          fixtureName = 'github-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          it("opens the GitHub.com blame URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.blame();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blame/master/some-dir/some-file.md');
          });
          return describe("when text is selected", function() {
            return it("opens the GitHub.com blame URL for the file with the selection range in the hash", function() {
              atom.config.set('open-on-github.includeLineNumbersInUrls', true);
              spyOn(githubFile, 'openUrlInBrowser');
              githubFile.blame([[0, 0], [1, 1]]);
              return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/blame/master/some-dir/some-file.md#L1-L2');
            });
          });
        });
      });
      describe("branchCompare", function() {
        return describe("when the file is openable on GitHub.com", function() {
          var fixtureName;
          fixtureName = 'github-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com branch compare URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.openBranchCompare();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/compare/master');
          });
        });
      });
      describe("history", function() {
        return describe("when the file is openable on GitHub.com", function() {
          var fixtureName;
          fixtureName = 'github-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com history URL for the file", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.history();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo/commits/master/some-dir/some-file.md');
          });
        });
      });
      describe("copyUrl", function() {
        var fixtureName;
        fixtureName = 'github-remote';
        beforeEach(function() {
          setupWorkingDir(fixtureName);
          atom.config.set('open-on-github.includeLineNumbersInUrls', true);
          return setupGithubFile();
        });
        afterEach(function() {
          return teardownWorkingDirAndRestoreFixture(fixtureName);
        });
        describe("when text is selected", function() {
          return it("copies the URL to the clipboard with the selection range in the hash", function() {
            githubFile.copyUrl([[0, 0], [1, 1]]);
            return expect(atom.clipboard.read()).toBe('https://github.com/some-user/some-repo/blob/master/some-dir/some-file.md#L1-L2');
          });
        });
        return describe("when no text is selected", function() {
          return it("copies the URL to the clipboard with the cursor location in the hash", function() {
            githubFile.copyUrl([[2, 1], [2, 1]]);
            return expect(atom.clipboard.read()).toBe('https://github.com/some-user/some-repo/blob/master/some-dir/some-file.md#L3');
          });
        });
      });
      return describe("openRepository", function() {
        return describe("when the file is openable on GitHub.com", function() {
          var fixtureName;
          fixtureName = 'github-remote';
          beforeEach(function() {
            setupWorkingDir(fixtureName);
            return setupGithubFile();
          });
          afterEach(function() {
            return teardownWorkingDirAndRestoreFixture(fixtureName);
          });
          return it("opens the GitHub.com repository URL", function() {
            spyOn(githubFile, 'openUrlInBrowser');
            githubFile.openRepository();
            return expect(githubFile.openUrlInBrowser).toHaveBeenCalledWith('https://github.com/some-user/some-repo');
          });
        });
      });
    });
    describe("githubRepoUrl", function() {
      beforeEach(function() {
        return githubFile = new GitHubFile();
      });
      it("returns the GitHub.com URL for an HTTP remote URL", function() {
        githubFile.gitUrl = function() {
          return "https://github.com/foo/bar.git";
        };
        return expect(githubFile.githubRepoUrl()).toBe("https://github.com/foo/bar");
      });
      it("returns the GitHub.com URL for an SSH remote URL", function() {
        githubFile.gitUrl = function() {
          return "git@github.com:foo/bar.git";
        };
        return expect(githubFile.githubRepoUrl()).toBe("http://github.com/foo/bar");
      });
      it("returns a GitHub enterprise URL for a non-Github.com remote URL", function() {
        githubFile.gitUrl = function() {
          return "https://git.enterprize.me/foo/bar.git";
        };
        expect(githubFile.githubRepoUrl()).toBe("https://git.enterprize.me/foo/bar");
        githubFile.gitUrl = function() {
          return "git@git.enterprize.me:foo/bar.git";
        };
        return expect(githubFile.githubRepoUrl()).toBe("http://git.enterprize.me/foo/bar");
      });
      it("returns the GitHub.com URL for a git:// URL", function() {
        githubFile.gitUrl = function() {
          return "git://github.com/foo/bar.git";
        };
        return expect(githubFile.githubRepoUrl()).toBe("http://github.com/foo/bar");
      });
      it("returns the GitHub.com URL for a ssh:// URL", function() {
        githubFile.gitUrl = function() {
          return "ssh://git@github.com/foo/bar.git";
        };
        return expect(githubFile.githubRepoUrl()).toBe("http://github.com/foo/bar");
      });
      it("returns undefined for Bitbucket URLs", function() {
        githubFile.gitUrl = function() {
          return "https://bitbucket.org/somebody/repo.git";
        };
        expect(githubFile.githubRepoUrl()).toBeUndefined();
        githubFile.gitUrl = function() {
          return "https://bitbucket.org/somebody/repo";
        };
        expect(githubFile.githubRepoUrl()).toBeUndefined();
        githubFile.gitUrl = function() {
          return "git@bitbucket.org:somebody/repo.git";
        };
        expect(githubFile.githubRepoUrl()).toBeUndefined();
        githubFile.gitUrl = function() {
          return "git@bitbucket.org:somebody/repo";
        };
        return expect(githubFile.githubRepoUrl()).toBeUndefined();
      });
      return it("removes leading and trailing slashes", function() {
        githubFile.gitUrl = function() {
          return "https://github.com/foo/bar/";
        };
        expect(githubFile.githubRepoUrl()).toBe("https://github.com/foo/bar");
        githubFile.gitUrl = function() {
          return "https://github.com/foo/bar//////";
        };
        expect(githubFile.githubRepoUrl()).toBe("https://github.com/foo/bar");
        githubFile.gitUrl = function() {
          return "git@github.com:/foo/bar.git";
        };
        return expect(githubFile.githubRepoUrl()).toBe("http://github.com/foo/bar");
      });
    });
    return it("activates when a command is triggered on the active editor", function() {
      var activationPromise;
      activationPromise = atom.packages.activatePackage('open-on-github');
      waitsForPromise(function() {
        return atom.workspace.open();
      });
      runs(function() {
        return atom.commands.dispatch(atom.views.getView(atom.workspace.getActivePane()), 'open-on-github:file');
      });
      return waitsForPromise(function() {
        return activationPromise;
      });
    });
  });

}).call(this);
