(function() {
  var $, fs, path, temp;

  path = require('path');

  fs = require('fs-plus');

  temp = require('temp');

  $ = require('atom-space-pen-views').$;

  describe('Package Generator', function() {
    var activationPromise, getEditorView, getWorkspaceView;
    activationPromise = [][0];
    getWorkspaceView = function() {
      return atom.views.getView(atom.workspace);
    };
    getEditorView = function() {
      return atom.views.getView(atom.workspace.getActiveTextEditor());
    };
    beforeEach(function() {
      waitsForPromise(function() {
        return atom.workspace.open('sample.js');
      });
      return runs(function() {
        return activationPromise = atom.packages.activatePackage("package-generator");
      });
    });
    describe("when package-generator:generate-package is triggered", function() {
      return it("displays a miniEditor with the correct text and selection", function() {
        atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-package");
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var base, fullPath, packageGeneratorView, packageName;
          packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
          packageName = packageGeneratorView.miniEditor.getModel().getSelectedText();
          expect(packageName).toEqual('my-package');
          fullPath = packageGeneratorView.miniEditor.getModel().getText();
          base = atom.config.get('core.projectHome');
          return expect(fullPath).toEqual(path.join(base, 'my-package'));
        });
      });
    });
    describe("when package-generator:generate-syntax-theme is triggered", function() {
      return it("displays a miniEditor with correct text and selection", function() {
        atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-syntax-theme");
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var base, fullPath, packageGeneratorView, themeName;
          packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
          themeName = packageGeneratorView.miniEditor.getModel().getSelectedText();
          expect(themeName).toEqual('my-theme');
          fullPath = packageGeneratorView.miniEditor.getModel().getText();
          base = atom.config.get('core.projectHome');
          return expect(fullPath).toEqual(path.join(base, 'my-theme-syntax'));
        });
      });
    });
    describe("when core:cancel is triggered", function() {
      return it("detaches from the DOM and focuses the the previously focused element", function() {
        jasmine.attachToDOM(getWorkspaceView());
        atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-package");
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var packageGeneratorView;
          packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
          expect(packageGeneratorView.miniEditor.element).toBe(document.activeElement);
          atom.commands.dispatch(packageGeneratorView.element, "core:cancel");
          expect(packageGeneratorView.panel.isVisible()).toBeFalsy();
          return expect(getEditorView()).toBe(document.activeElement);
        });
      });
    });
    return describe("when a package is generated", function() {
      var packageName, packagePath, packageRoot, _ref;
      _ref = [], packageName = _ref[0], packagePath = _ref[1], packageRoot = _ref[2];
      beforeEach(function() {
        spyOn(atom, "open");
        packageRoot = temp.mkdirSync('atom');
        packageName = "sweet-package-dude";
        packagePath = path.join(packageRoot, packageName);
        return fs.removeSync(packageRoot);
      });
      afterEach(function() {
        return fs.removeSync(packageRoot);
      });
      it("forces the package's name to be lowercase with dashes", function() {
        packageName = "CamelCaseIsForTheBirds";
        packagePath = path.join(path.dirname(packagePath), packageName);
        atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-package");
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var apmExecute, packageGeneratorView;
          packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
          packageGeneratorView.miniEditor.setText(packagePath);
          apmExecute = spyOn(packageGeneratorView, 'runCommand');
          atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
          expect(apmExecute).toHaveBeenCalled();
          expect(apmExecute.mostRecentCall.args[0]).toBe(atom.packages.getApmPath());
          return expect(apmExecute.mostRecentCall.args[1]).toEqual(['init', '--package', "" + (path.join(path.dirname(packagePath), "camel-case-is-for-the-birds"))]);
        });
      });
      describe('when creating a package', function() {
        beforeEach(function() {
          atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-package");
          return waitsForPromise(function() {
            return activationPromise;
          });
        });
        describe("when the package is created outside of the packages directory", function() {
          var apmExecute, generateOutside;
          apmExecute = [][0];
          generateOutside = function(callback) {
            var packageGeneratorView;
            packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
            expect(packageGeneratorView.hasParent()).toBeTruthy();
            packageGeneratorView.miniEditor.setText(packagePath);
            apmExecute = spyOn(packageGeneratorView, 'runCommand').andCallFake(function(command, args, exit) {
              return process.nextTick(function() {
                return exit();
              });
            });
            atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
            waitsFor(function() {
              return atom.open.callCount === 1;
            });
            return runs(callback);
          };
          it("calls `apm init` and `apm link`", function() {
            atom.config.set('package-generator.createInDevMode', false);
            return generateOutside(function() {
              expect(apmExecute.argsForCall[0][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[0][1]).toEqual(['init', '--package', "" + packagePath]);
              expect(apmExecute.argsForCall[1][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[1][1]).toEqual(['link', "" + packagePath]);
              return expect(atom.open.argsForCall[0][0].pathsToOpen[0]).toBe(packagePath);
            });
          });
          return it("calls `apm init` and `apm link --dev`", function() {
            atom.config.set('package-generator.createInDevMode', true);
            return generateOutside(function() {
              expect(apmExecute.argsForCall[0][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[0][1]).toEqual(['init', '--package', "" + packagePath]);
              expect(apmExecute.argsForCall[1][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[1][1]).toEqual(['link', '--dev', "" + packagePath]);
              return expect(atom.open.argsForCall[0][0].pathsToOpen[0]).toBe(packagePath);
            });
          });
        });
        return describe("when the package is created inside the packages directory", function() {
          return it("calls `apm init`", function() {
            var apmExecute, packageGeneratorView;
            packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
            spyOn(packageGeneratorView, 'isStoredInDotAtom').andReturn(true);
            expect(packageGeneratorView.hasParent()).toBeTruthy();
            packageGeneratorView.miniEditor.setText(packagePath);
            apmExecute = spyOn(packageGeneratorView, 'runCommand').andCallFake(function(command, args, exit) {
              return process.nextTick(function() {
                return exit();
              });
            });
            atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
            waitsFor(function() {
              return atom.open.callCount;
            });
            return runs(function() {
              expect(apmExecute.argsForCall[0][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[0][1]).toEqual(['init', '--package', "" + packagePath]);
              expect(atom.open.argsForCall[0][0].pathsToOpen[0]).toBe(packagePath);
              return expect(apmExecute.argsForCall[1]).toBeUndefined();
            });
          });
        });
      });
      describe('when creating a theme', function() {
        beforeEach(function() {
          atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-syntax-theme");
          return waitsForPromise(function() {
            return activationPromise;
          });
        });
        describe("when the theme is created outside of the packages directory", function() {
          return it("calls `apm init` and `apm link`", function() {
            var apmExecute, packageGeneratorView;
            packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
            expect(packageGeneratorView.hasParent()).toBeTruthy();
            packageGeneratorView.miniEditor.setText(packagePath);
            apmExecute = spyOn(packageGeneratorView, 'runCommand').andCallFake(function(command, args, exit) {
              return process.nextTick(function() {
                return exit();
              });
            });
            atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
            waitsFor(function() {
              return atom.open.callCount === 1;
            });
            return runs(function() {
              expect(apmExecute.argsForCall[0][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[0][1]).toEqual(['init', '--theme', "" + packagePath]);
              expect(apmExecute.argsForCall[1][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[1][1]).toEqual(['link', "" + packagePath]);
              return expect(atom.open.argsForCall[0][0].pathsToOpen[0]).toBe(packagePath);
            });
          });
        });
        return describe("when the theme is created inside of the packages directory", function() {
          return it("calls `apm init`", function() {
            var apmExecute, packageGeneratorView;
            packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
            spyOn(packageGeneratorView, 'isStoredInDotAtom').andReturn(true);
            expect(packageGeneratorView.hasParent()).toBeTruthy();
            packageGeneratorView.miniEditor.setText(packagePath);
            apmExecute = spyOn(packageGeneratorView, 'runCommand').andCallFake(function(command, args, exit) {
              return process.nextTick(function() {
                return exit();
              });
            });
            atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
            waitsFor(function() {
              return atom.open.callCount === 1;
            });
            return runs(function() {
              expect(apmExecute.argsForCall[0][0]).toBe(atom.packages.getApmPath());
              expect(apmExecute.argsForCall[0][1]).toEqual(['init', '--theme', "" + packagePath]);
              expect(atom.open.argsForCall[0][0].pathsToOpen[0]).toBe(packagePath);
              return expect(apmExecute.argsForCall[1]).toBeUndefined();
            });
          });
        });
      });
      it("displays an error when the package path already exists", function() {
        jasmine.attachToDOM(getWorkspaceView());
        fs.makeTreeSync(packagePath);
        atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-package");
        waitsForPromise(function() {
          return activationPromise;
        });
        return runs(function() {
          var packageGeneratorView;
          packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
          expect(packageGeneratorView.hasParent()).toBeTruthy();
          expect(packageGeneratorView.error).not.toBeVisible();
          packageGeneratorView.miniEditor.setText(packagePath);
          atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
          expect(packageGeneratorView.hasParent()).toBeTruthy();
          return expect(packageGeneratorView.error).toBeVisible();
        });
      });
      return it("opens the package", function() {
        atom.commands.dispatch(getWorkspaceView(), "package-generator:generate-package");
        waitsForPromise(function() {
          return activationPromise;
        });
        runs(function() {
          var apmExecute, loadPackage, packageGeneratorView;
          packageGeneratorView = $(getWorkspaceView()).find(".package-generator").view();
          packageGeneratorView.miniEditor.setText(packagePath);
          apmExecute = spyOn(packageGeneratorView, 'runCommand').andCallFake(function(command, args, exit) {
            return process.nextTick(function() {
              return exit();
            });
          });
          loadPackage = spyOn(atom.packages, 'loadPackage');
          return atom.commands.dispatch(packageGeneratorView.element, "core:confirm");
        });
        waitsFor(function() {
          return atom.open.callCount === 1;
        });
        return runs(function() {
          return expect(atom.open).toHaveBeenCalledWith({
            pathsToOpen: [packagePath]
          });
        });
      });
    });
  });

}).call(this);
