(function() {
  var fs, path, temp;

  path = require('path');

  fs = require('fs-plus');

  temp = require('temp').track();

  describe("Snippet Loading", function() {
    var activateSnippetsPackage, configDirPath;
    configDirPath = null;
    beforeEach(function() {
      configDirPath = temp.mkdirSync('atom-config-dir-');
      spyOn(atom, 'getConfigDirPath').andReturn(configDirPath);
      spyOn(console, 'warn');
      if (atom.notifications != null) {
        spyOn(atom.notifications, 'addError');
      }
      return spyOn(atom.packages, 'getLoadedPackages').andReturn([atom.packages.loadPackage(path.join(__dirname, 'fixtures', 'package-with-snippets')), atom.packages.loadPackage(path.join(__dirname, 'fixtures', 'package-with-broken-snippets'))]);
    });
    afterEach(function() {
      return jasmine.unspy(atom.packages, 'getLoadedPackages');
    });
    activateSnippetsPackage = function() {
      var module;
      module = null;
      waitsForPromise(function() {
        return atom.packages.activatePackage("snippets").then(function(_arg) {
          var mainModule;
          mainModule = _arg.mainModule;
          module = mainModule;
          return module.loaded = false;
        });
      });
      return waitsFor("all snippets to load", 3000, function() {
        return module.loaded;
      });
    };
    it("loads the bundled snippet template snippets", function() {
      activateSnippetsPackage();
      return runs(function() {
        var csonSnippet, jsonSnippet;
        jsonSnippet = atom.config.get('snippets.snip', {
          scope: ['.source.json']
        });
        expect(jsonSnippet.name).toBe('Atom Snippet');
        expect(jsonSnippet.prefix).toBe('snip');
        expect(jsonSnippet.body).toContain('"prefix":');
        expect(jsonSnippet.body).toContain('"body":');
        expect(jsonSnippet.tabStops.length).toBeGreaterThan(0);
        csonSnippet = atom.config.get('snippets.snip', {
          scope: ['.source.coffee']
        });
        expect(csonSnippet.name).toBe('Atom Snippet');
        expect(csonSnippet.prefix).toBe('snip');
        expect(csonSnippet.body).toContain("'prefix':");
        expect(csonSnippet.body).toContain("'body':");
        return expect(csonSnippet.tabStops.length).toBeGreaterThan(0);
      });
    });
    it("loads non-hidden snippet files from atom packages with snippets directories", function() {
      activateSnippetsPackage();
      return runs(function() {
        var snippet;
        snippet = atom.config.get('snippets.test', {
          scope: ['.test']
        });
        expect(snippet.prefix).toBe('test');
        return expect(snippet.body).toBe('testing 123');
      });
    });
    it("logs a warning if package snippets files cannot be parsed", function() {
      activateSnippetsPackage();
      return runs(function() {
        expect(console.warn.calls.length).toBe(1);
        return expect(console.warn.mostRecentCall.args[0]).toMatch(/Error reading.*package-with-broken-snippets/);
      });
    });
    describe("::onDidLoadSnippets(callback)", function() {
      return it("invokes listeners when all snippets are loaded", function() {
        var loadedCallback;
        loadedCallback = null;
        waitsFor("package to activate", function(done) {
          return atom.packages.activatePackage("snippets").then(function(_arg) {
            var mainModule;
            mainModule = _arg.mainModule;
            mainModule.onDidLoadSnippets(loadedCallback = jasmine.createSpy('onDidLoadSnippets callback'));
            return done();
          });
        });
        return waitsFor("onDidLoad callback to be called", function() {
          return loadedCallback.callCount > 0;
        });
      });
    });
    describe("when ~/.atom/snippets.json exists", function() {
      beforeEach(function() {
        fs.writeFileSync(path.join(configDirPath, 'snippets.json'), "{\n  \".foo\": {\n    \"foo snippet\": {\n      \"prefix\": \"foo\",\n      \"body\": \"bar1\"\n    }\n  }\n}");
        return activateSnippetsPackage();
      });
      it("loads the snippets from that file", function() {
        var snippet;
        snippet = atom.config.get('snippets.foo', {
          scope: ['.foo']
        });
        expect(snippet.name).toBe('foo snippet');
        expect(snippet.prefix).toBe("foo");
        return expect(snippet.body).toBe("bar1");
      });
      return describe("when that file changes", function() {
        return it("reloads the snippets", function() {
          fs.writeFileSync(path.join(configDirPath, 'snippets.json'), "{\n  \".foo\": {\n    \"foo snippet\": {\n      \"prefix\": \"foo\",\n      \"body\": \"bar2\"\n    }\n  }\n}");
          waitsFor("snippets to be changed", function() {
            var _ref;
            return ((_ref = atom.config.get('snippets.foo', {
              scope: ['.foo']
            })) != null ? _ref.body : void 0) === 'bar2';
          });
          runs(function() {
            return fs.writeFileSync(path.join(configDirPath, 'snippets.json'), "");
          });
          return waitsFor("snippets to be removed", function() {
            return atom.config.get('snippets.foo', {
              scope: ['.foo']
            }) == null;
          });
        });
      });
    });
    describe("when ~/.atom/snippets.cson exists", function() {
      beforeEach(function() {
        fs.writeFileSync(path.join(configDirPath, 'snippets.cson'), "\".foo\":\n  \"foo snippet\":\n    \"prefix\": \"foo\"\n    \"body\": \"bar1\"");
        return activateSnippetsPackage();
      });
      it("loads the snippets from that file", function() {
        var snippet;
        snippet = atom.config.get('snippets.foo', {
          scope: ['.foo']
        });
        expect(snippet.name).toBe('foo snippet');
        expect(snippet.prefix).toBe("foo");
        return expect(snippet.body).toBe("bar1");
      });
      return describe("when that file changes", function() {
        return it("reloads the snippets", function() {
          fs.writeFileSync(path.join(configDirPath, 'snippets.cson'), "\".foo\":\n  \"foo snippet\":\n    \"prefix\": \"foo\"\n    \"body\": \"bar2\"");
          waitsFor("snippets to be changed", function() {
            var _ref;
            return ((_ref = atom.config.get('snippets.foo', {
              scope: ['.foo']
            })) != null ? _ref.body : void 0) === 'bar2';
          });
          runs(function() {
            return fs.writeFileSync(path.join(configDirPath, 'snippets.cson'), "");
          });
          return waitsFor("snippets to be removed", function() {
            return atom.config.get('snippets.foo', {
              scope: ['.foo']
            }) == null;
          });
        });
      });
    });
    return it("notifies the user when the user snippets file cannot be loaded", function() {
      fs.writeFileSync(path.join(configDirPath, 'snippets.cson'), "\".junk\":::");
      activateSnippetsPackage();
      return runs(function() {
        expect(console.warn).toHaveBeenCalled();
        if (atom.notifications != null) {
          return expect(atom.notifications.addError).toHaveBeenCalled();
        }
      });
    });
  });

}).call(this);
