(function() {
  var Snippets, path, temp;

  path = require('path');

  temp = require('temp').track();

  Snippets = require('../lib/snippets');

  describe("Snippets extension", function() {
    var editor, editorElement, simulateTabKeyEvent, _ref;
    _ref = [], editorElement = _ref[0], editor = _ref[1];
    simulateTabKeyEvent = function(_arg) {
      var event, shift;
      shift = (_arg != null ? _arg : {}).shift;
      event = atom.keymaps.constructor.buildKeydownEvent('tab', {
        shift: shift,
        target: editorElement
      });
      return atom.keymaps.handleKeyboardEvent(event);
    };
    beforeEach(function() {
      spyOn(Snippets, 'loadAll');
      waitsForPromise(function() {
        return atom.workspace.open('sample.js');
      });
      waitsForPromise(function() {
        return atom.packages.activatePackage('language-javascript');
      });
      waitsForPromise(function() {
        return atom.packages.activatePackage("snippets");
      });
      return runs(function() {
        editor = atom.workspace.getActiveTextEditor();
        return editorElement = atom.views.getView(editor);
      });
    });
    describe("provideSnippets interface", function() {
      var snippetsInterface;
      snippetsInterface = null;
      beforeEach(function() {
        return snippetsInterface = Snippets.provideSnippets();
      });
      return describe("bundledSnippetsLoaded", function() {
        it("indicates the loaded state of the bundled snippets", function() {
          Snippets.loaded = false;
          expect(snippetsInterface.bundledSnippetsLoaded()).toBe(false);
          Snippets.doneLoading();
          return expect(snippetsInterface.bundledSnippetsLoaded()).toBe(true);
        });
        return it("can insert a snippet", function() {
          editor.setSelectedBufferRange([[0, 4], [0, 13]]);
          snippetsInterface.insertSnippet("hello ${1:world}", editor);
          return expect(editor.lineTextForBufferRow(0)).toBe("var hello world = function () {");
        });
      });
    });
    describe("when 'tab' is triggered on the editor", function() {
      beforeEach(function() {
        return Snippets.add(__filename, {
          ".source.js": {
            "without tab stops": {
              prefix: "t1",
              body: "this is a test"
            },
            "overlapping prefix": {
              prefix: "tt1",
              body: "this is another test"
            },
            "special chars": {
              prefix: "@unique",
              body: "@unique see"
            },
            "tab stops": {
              prefix: "t2",
              body: "go here next:($2) and finally go here:($0)\ngo here first:($1)\n"
            },
            "indented second line": {
              prefix: "t3",
              body: "line 1\n\tline 2$1\n"
            },
            "nested tab stops": {
              prefix: "t5",
              body: '${1:"${2:key}"}: ${3:value}'
            },
            "caused problems with undo": {
              prefix: "t6",
              body: "first line$1\n  ${2:placeholder ending second line}"
            },
            "contains empty lines": {
              prefix: "t7",
              body: "first line $1\n\n\nfourth line after blanks $2"
            },
            "with/without placeholder": {
              prefix: "t8",
              body: "with placeholder ${1:test}\nwithout placeholder ${2}"
            },
            "multi-caret": {
              prefix: "t9",
              body: "with placeholder ${1:test}\nwithout placeholder $1"
            },
            "multi-caret-multi-tabstop": {
              prefix: "t9b",
              body: "with placeholder ${1:test}\nwithout placeholder $1\nsecond tabstop $2\nthird tabstop $3"
            },
            "large indices": {
              prefix: "t10",
              body: "hello${10} ${11:large} indices${1}"
            },
            "no body": {
              prefix: "bad1"
            },
            "number body": {
              prefix: "bad2",
              body: 100
            },
            "many tabstops": {
              prefix: "t11",
              body: "$0one${1} ${2:two} three${3}"
            }
          }
        });
      });
      describe("when the snippet body is invalid or missing", function() {
        return it("does not register the snippet", function() {
          editor.setText('');
          editor.insertText('bad1');
          atom.commands.dispatch(editorElement, 'snippets:expand');
          expect(editor.getText()).toBe('bad1');
          editor.setText('');
          editor.setText('bad2');
          atom.commands.dispatch(editorElement, 'snippets:expand');
          return expect(editor.getText()).toBe('bad2');
        });
      });
      describe("when the letters preceding the cursor trigger a snippet", function() {
        describe("when the snippet contains no tab stops", function() {
          it("replaces the prefix with the snippet text and places the cursor at its end", function() {
            editor.insertText("t1");
            expect(editor.getCursorScreenPosition()).toEqual([0, 2]);
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("this is a testvar quicksort = function () {");
            return expect(editor.getCursorScreenPosition()).toEqual([0, 14]);
          });
          return it("inserts a real tab the next time a tab is pressed after the snippet is expanded", function() {
            editor.insertText("t1");
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("this is a testvar quicksort = function () {");
            simulateTabKeyEvent();
            return expect(editor.lineTextForBufferRow(0)).toBe("this is a test  var quicksort = function () {");
          });
        });
        describe("when the snippet contains tab stops", function() {
          it("places the cursor at the first tab-stop, and moves the cursor in response to 'next-tab-stop' events", function() {
            var markerCountBefore;
            markerCountBefore = editor.getMarkerCount();
            editor.setCursorScreenPosition([2, 0]);
            editor.insertText('t2');
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(2)).toBe("go here next:() and finally go here:()");
            expect(editor.lineTextForBufferRow(3)).toBe("go here first:()");
            expect(editor.lineTextForBufferRow(4)).toBe("    if (items.length <= 1) return items;");
            expect(editor.getSelectedBufferRange()).toEqual([[3, 15], [3, 15]]);
            simulateTabKeyEvent();
            expect(editor.getSelectedBufferRange()).toEqual([[2, 14], [2, 14]]);
            editor.insertText('abc');
            simulateTabKeyEvent();
            expect(editor.getSelectedBufferRange()).toEqual([[2, 40], [2, 40]]);
            simulateTabKeyEvent({
              shift: true
            });
            expect(editor.getSelectedBufferRange()).toEqual([[2, 14], [2, 17]]);
            simulateTabKeyEvent({
              shift: true
            });
            expect(editor.getSelectedBufferRange()).toEqual([[3, 15], [3, 15]]);
            simulateTabKeyEvent({
              shift: true
            });
            expect(editor.getCursorScreenPosition()).toEqual([3, 15]);
            simulateTabKeyEvent();
            simulateTabKeyEvent();
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(2)).toBe("go here next:(abc) and finally go here:(  )");
            return expect(editor.getMarkerCount()).toBe(markerCountBefore);
          });
          describe("when tab stops are nested", function() {
            return it("destroys the inner tab stop if the outer tab stop is modified", function() {
              editor.setText('');
              editor.insertText('t5');
              atom.commands.dispatch(editorElement, 'snippets:expand');
              expect(editor.lineTextForBufferRow(0)).toBe('"key": value');
              expect(editor.getSelectedBufferRange()).toEqual([[0, 0], [0, 5]]);
              editor.insertText("foo");
              simulateTabKeyEvent();
              return expect(editor.getSelectedBufferRange()).toEqual([[0, 5], [0, 10]]);
            });
          });
          describe("when tab stops are separated by blank lines", function() {
            return it("correctly places the tab stops (regression)", function() {
              editor.setText('');
              editor.insertText('t7');
              atom.commands.dispatch(editorElement, 'snippets:expand');
              atom.commands.dispatch(editorElement, 'snippets:next-tab-stop');
              return expect(editor.getCursorBufferPosition()).toEqual([3, 25]);
            });
          });
          describe("when the cursor is moved beyond the bounds of the current tab stop", function() {
            return it("terminates the snippet", function() {
              editor.setCursorScreenPosition([2, 0]);
              editor.insertText('t2');
              simulateTabKeyEvent();
              editor.moveUp();
              editor.moveLeft();
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(2)).toBe("go here next:(  ) and finally go here:()");
              expect(editor.getCursorBufferPosition()).toEqual([2, 16]);
              editor.setCursorScreenPosition([4, 0]);
              editor.insertText('t2');
              simulateTabKeyEvent();
              simulateTabKeyEvent();
              editor.moveRight();
              simulateTabKeyEvent({
                shift: true
              });
              return expect(editor.getCursorBufferPosition()).toEqual([4, 15]);
            });
          });
          describe("when the cursor is moved within the bounds of the current tab stop", function() {
            return it("should not terminate the snippet", function() {
              editor.setCursorScreenPosition([0, 0]);
              editor.insertText('t8');
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(0)).toBe("with placeholder test");
              editor.moveRight();
              editor.moveLeft();
              editor.insertText("foo");
              expect(editor.lineTextForBufferRow(0)).toBe("with placeholder tesfoot");
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(1)).toBe("without placeholder var quicksort = function () {");
              editor.insertText("test");
              expect(editor.lineTextForBufferRow(1)).toBe("without placeholder testvar quicksort = function () {");
              editor.moveLeft();
              editor.insertText("foo");
              return expect(editor.lineTextForBufferRow(1)).toBe("without placeholder tesfootvar quicksort = function () {");
            });
          });
          return describe("when the backspace is press within the bounds of the current tab stop", function() {
            return it("should not terminate the snippet", function() {
              editor.setCursorScreenPosition([0, 0]);
              editor.insertText('t8');
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(0)).toBe("with placeholder test");
              editor.moveRight();
              editor.backspace();
              editor.insertText("foo");
              expect(editor.lineTextForBufferRow(0)).toBe("with placeholder tesfoo");
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(1)).toBe("without placeholder var quicksort = function () {");
              editor.insertText("test");
              expect(editor.lineTextForBufferRow(1)).toBe("without placeholder testvar quicksort = function () {");
              editor.backspace();
              editor.insertText("foo");
              return expect(editor.lineTextForBufferRow(1)).toBe("without placeholder tesfoovar quicksort = function () {");
            });
          });
        });
        describe("when the snippet contains hard tabs", function() {
          describe("when the edit session is in soft-tabs mode", function() {
            return it("translates hard tabs in the snippet to the appropriate number of spaces", function() {
              expect(editor.getSoftTabs()).toBeTruthy();
              editor.insertText("t3");
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(1)).toBe("  line 2");
              return expect(editor.getCursorBufferPosition()).toEqual([1, 8]);
            });
          });
          return describe("when the edit session is in hard-tabs mode", function() {
            return it("inserts hard tabs in the snippet directly", function() {
              editor.setSoftTabs(false);
              editor.insertText("t3");
              simulateTabKeyEvent();
              expect(editor.lineTextForBufferRow(1)).toBe("\tline 2");
              return expect(editor.getCursorBufferPosition()).toEqual([1, 7]);
            });
          });
        });
        describe("when the snippet prefix is indented", function() {
          describe("when the snippet spans a single line", function() {
            return it("does not indent the next line", function() {
              editor.setCursorScreenPosition([2, Infinity]);
              editor.insertText(' t1');
              atom.commands.dispatch(editorElement, 'snippets:expand');
              return expect(editor.lineTextForBufferRow(3)).toBe("    var pivot = items.shift(), current, left = [], right = [];");
            });
          });
          return describe("when the snippet spans multiple lines", function() {
            return it("indents the subsequent lines of the snippet to be even with the start of the first line", function() {
              expect(editor.getSoftTabs()).toBeTruthy();
              editor.setCursorScreenPosition([2, Infinity]);
              editor.insertText(' t3');
              atom.commands.dispatch(editorElement, 'snippets:expand');
              expect(editor.lineTextForBufferRow(2)).toBe("    if (items.length <= 1) return items; line 1");
              expect(editor.lineTextForBufferRow(3)).toBe("      line 2");
              return expect(editor.getCursorBufferPosition()).toEqual([3, 12]);
            });
          });
        });
        return describe("when multiple snippets match the prefix", function() {
          return it("expands the snippet that is the longest match for the prefix", function() {
            editor.insertText('t113');
            expect(editor.getCursorScreenPosition()).toEqual([0, 4]);
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("t113  var quicksort = function () {");
            expect(editor.getCursorScreenPosition()).toEqual([0, 6]);
            editor.undo();
            editor.undo();
            editor.insertText("tt1");
            expect(editor.getCursorScreenPosition()).toEqual([0, 3]);
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("this is another testvar quicksort = function () {");
            expect(editor.getCursorScreenPosition()).toEqual([0, 20]);
            editor.undo();
            editor.undo();
            editor.insertText("@t1");
            expect(editor.getCursorScreenPosition()).toEqual([0, 3]);
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("@this is a testvar quicksort = function () {");
            return expect(editor.getCursorScreenPosition()).toEqual([0, 15]);
          });
        });
      });
      describe("when the word preceding the cursor ends with a snippet prefix", function() {
        return it("inserts a tab as normal", function() {
          editor.insertText("t1t1t1");
          simulateTabKeyEvent();
          return expect(editor.lineTextForBufferRow(0)).toBe("t1t1t1  var quicksort = function () {");
        });
      });
      describe("when the letters preceding the cursor don't match a snippet", function() {
        return it("inserts a tab as normal", function() {
          editor.insertText("xxte");
          expect(editor.getCursorScreenPosition()).toEqual([0, 4]);
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("xxte  var quicksort = function () {");
          return expect(editor.getCursorScreenPosition()).toEqual([0, 6]);
        });
      });
      describe("when text is selected", function() {
        return it("inserts a tab as normal", function() {
          editor.insertText("t1");
          editor.setSelectedBufferRange([[0, 0], [0, 2]]);
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("  t1var quicksort = function () {");
          return expect(editor.getSelectedBufferRange()).toEqual([[0, 0], [0, 4]]);
        });
      });
      describe("when a previous snippet expansion has just been undone", function() {
        return it("expands the snippet based on the current prefix rather than jumping to the old snippet's tab stop", function() {
          editor.insertText('t6\n');
          editor.setCursorBufferPosition([0, 2]);
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("first line");
          editor.undo();
          expect(editor.lineTextForBufferRow(0)).toBe("t6");
          simulateTabKeyEvent();
          return expect(editor.lineTextForBufferRow(0)).toBe("first line");
        });
      });
      describe("when the prefix contains non-word characters", function() {
        it("selects the non-word characters as part of the prefix", function() {
          editor.insertText("@unique");
          expect(editor.getCursorScreenPosition()).toEqual([0, 7]);
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("@unique seevar quicksort = function () {");
          expect(editor.getCursorScreenPosition()).toEqual([0, 11]);
          editor.setCursorBufferPosition([10, 0]);
          editor.insertText("'@unique");
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(10)).toBe("'@unique see");
          return expect(editor.getCursorScreenPosition()).toEqual([10, 12]);
        });
        return it("does not select the whitespace before the prefix", function() {
          editor.insertText("a; @unique");
          expect(editor.getCursorScreenPosition()).toEqual([0, 10]);
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("a; @unique seevar quicksort = function () {");
          return expect(editor.getCursorScreenPosition()).toEqual([0, 14]);
        });
      });
      describe("when snippet contains tabstops with or without placeholder", function() {
        return it("should create two markers", function() {
          var markerCountBefore;
          markerCountBefore = editor.getMarkerCount();
          editor.setCursorScreenPosition([0, 0]);
          editor.insertText('t8');
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("with placeholder test");
          expect(editor.lineTextForBufferRow(1)).toBe("without placeholder var quicksort = function () {");
          expect(editor.getMarkerCount()).toBe(3);
          expect(editor.getSelectedBufferRange()).toEqual([[0, 17], [0, 21]]);
          simulateTabKeyEvent();
          return expect(editor.getSelectedBufferRange()).toEqual([[1, 20], [1, 20]]);
        });
      });
      describe("when snippet contains multi-caret tabstops with or without placeholder", function() {
        it("should create two markers", function() {
          editor.setCursorScreenPosition([0, 0]);
          editor.insertText('t9');
          simulateTabKeyEvent();
          expect(editor.lineTextForBufferRow(0)).toBe("with placeholder test");
          expect(editor.lineTextForBufferRow(1)).toBe("without placeholder var quicksort = function () {");
          editor.insertText('hello');
          expect(editor.lineTextForBufferRow(0)).toBe("with placeholder hello");
          return expect(editor.lineTextForBufferRow(1)).toBe("without placeholder hellovar quicksort = function () {");
        });
        it("terminates the snippet when cursors are destroyed", function() {
          editor.setCursorScreenPosition([0, 0]);
          editor.insertText('t9b');
          simulateTabKeyEvent();
          editor.consolidateSelections();
          simulateTabKeyEvent();
          return expect(editor.lineTextForBufferRow(1)).toEqual("without placeholder   ");
        });
        it("terminates the snippet expansion if a new cursor moves outside the bounds of the tab stops", function() {
          editor.setCursorScreenPosition([0, 0]);
          editor.insertText('t9b');
          simulateTabKeyEvent();
          editor.insertText('test');
          editor.consolidateSelections();
          editor.moveDown();
          editor.moveToBeginningOfLine();
          simulateTabKeyEvent();
          return expect(editor.lineTextForBufferRow(2).indexOf("  second")).toBe(0);
        });
        return it("moves to the second tabstop after a multi-caret tabstop", function() {
          editor.setCursorScreenPosition([0, 0]);
          editor.insertText('t9b');
          simulateTabKeyEvent();
          editor.insertText('line 1');
          simulateTabKeyEvent();
          editor.insertText('line 2');
          simulateTabKeyEvent();
          editor.insertText('line 3');
          return expect(editor.lineTextForBufferRow(2).indexOf("line 2 ")).toBe(-1);
        });
      });
      describe("when the snippet contains tab stops with an index >= 10", function() {
        return it("parses and orders the indices correctly", function() {
          editor.setText('t10');
          editor.setCursorScreenPosition([0, 3]);
          simulateTabKeyEvent();
          expect(editor.getText()).toBe("hello large indices");
          expect(editor.getCursorBufferPosition()).toEqual([0, 19]);
          simulateTabKeyEvent();
          expect(editor.getCursorBufferPosition()).toEqual([0, 5]);
          simulateTabKeyEvent();
          return expect(editor.getSelectedBufferRange()).toEqual([[0, 6], [0, 11]]);
        });
      });
      return describe("when there are multiple cursors", function() {
        describe("when the cursors share a common snippet prefix", function() {
          it("expands the snippet for all cursors and allows simultaneous editing", function() {
            editor.insertText('t9');
            editor.setCursorBufferPosition([12, 2]);
            editor.insertText(' t9');
            editor.addCursorAtBufferPosition([0, 2]);
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("with placeholder test");
            expect(editor.lineTextForBufferRow(1)).toBe("without placeholder var quicksort = function () {");
            expect(editor.lineTextForBufferRow(13)).toBe("}; with placeholder test");
            expect(editor.lineTextForBufferRow(14)).toBe("without placeholder ");
            editor.insertText('hello');
            expect(editor.lineTextForBufferRow(0)).toBe("with placeholder hello");
            expect(editor.lineTextForBufferRow(1)).toBe("without placeholder hellovar quicksort = function () {");
            expect(editor.lineTextForBufferRow(13)).toBe("}; with placeholder hello");
            return expect(editor.lineTextForBufferRow(14)).toBe("without placeholder hello");
          });
          return describe("when there are many tabstops", function() {
            return it("moves the cursors between the tab stops for their corresponding snippet when tab and shift-tab are pressed", function() {
              var cursors;
              editor.addCursorAtBufferPosition([7, 5]);
              editor.addCursorAtBufferPosition([12, 2]);
              editor.insertText('t11');
              simulateTabKeyEvent();
              cursors = editor.getCursors();
              expect(cursors.length).toEqual(3);
              expect(cursors[0].getBufferPosition()).toEqual([0, 3]);
              expect(cursors[1].getBufferPosition()).toEqual([7, 8]);
              expect(cursors[2].getBufferPosition()).toEqual([12, 5]);
              expect(cursors[0].selection.isEmpty()).toBe(true);
              expect(cursors[1].selection.isEmpty()).toBe(true);
              expect(cursors[2].selection.isEmpty()).toBe(true);
              simulateTabKeyEvent();
              expect(cursors[0].getBufferPosition()).toEqual([0, 7]);
              expect(cursors[1].getBufferPosition()).toEqual([7, 12]);
              expect(cursors[2].getBufferPosition()).toEqual([12, 9]);
              expect(cursors[0].selection.isEmpty()).toBe(false);
              expect(cursors[1].selection.isEmpty()).toBe(false);
              expect(cursors[2].selection.isEmpty()).toBe(false);
              expect(cursors[0].selection.getText()).toEqual('two');
              expect(cursors[1].selection.getText()).toEqual('two');
              expect(cursors[2].selection.getText()).toEqual('two');
              simulateTabKeyEvent();
              expect(cursors[0].getBufferPosition()).toEqual([0, 13]);
              expect(cursors[1].getBufferPosition()).toEqual([7, 18]);
              expect(cursors[2].getBufferPosition()).toEqual([12, 15]);
              expect(cursors[0].selection.isEmpty()).toBe(true);
              expect(cursors[1].selection.isEmpty()).toBe(true);
              expect(cursors[2].selection.isEmpty()).toBe(true);
              simulateTabKeyEvent();
              expect(cursors[0].getBufferPosition()).toEqual([0, 0]);
              expect(cursors[1].getBufferPosition()).toEqual([7, 5]);
              expect(cursors[2].getBufferPosition()).toEqual([12, 2]);
              expect(cursors[0].selection.isEmpty()).toBe(true);
              expect(cursors[1].selection.isEmpty()).toBe(true);
              return expect(cursors[2].selection.isEmpty()).toBe(true);
            });
          });
        });
        describe("when the cursors do not share common snippet prefixes", function() {
          return it("inserts tabs as normal", function() {
            editor.insertText('t9');
            editor.setCursorBufferPosition([12, 2]);
            editor.insertText(' t8');
            editor.addCursorAtBufferPosition([0, 2]);
            simulateTabKeyEvent();
            expect(editor.lineTextForBufferRow(0)).toBe("t9  var quicksort = function () {");
            return expect(editor.lineTextForBufferRow(12)).toBe("}; t8 ");
          });
        });
        return describe("when a snippet is triggered within an existing snippet expansion", function() {
          return it("ignores the snippet expansion and goes to the next tab stop", function() {
            var cursors;
            editor.addCursorAtBufferPosition([7, 5]);
            editor.addCursorAtBufferPosition([12, 2]);
            editor.insertText('t11');
            simulateTabKeyEvent();
            simulateTabKeyEvent();
            editor.insertText('t1');
            simulateTabKeyEvent();
            cursors = editor.getCursors();
            expect(cursors.length).toEqual(3);
            expect(cursors[0].getBufferPosition()).toEqual([0, 12]);
            expect(cursors[1].getBufferPosition()).toEqual([7, 17]);
            expect(cursors[2].getBufferPosition()).toEqual([12, 14]);
            expect(cursors[0].selection.isEmpty()).toBe(true);
            expect(cursors[1].selection.isEmpty()).toBe(true);
            expect(cursors[2].selection.isEmpty()).toBe(true);
            expect(editor.lineTextForBufferRow(0)).toBe("one t1 threevar quicksort = function () {");
            expect(editor.lineTextForBufferRow(7)).toBe("    }one t1 three");
            return expect(editor.lineTextForBufferRow(12)).toBe("};one t1 three");
          });
        });
      });
    });
    describe("when atom://.atom/snippets is opened", function() {
      return it("opens ~/.atom/snippets.cson", function() {
        var configDirPath;
        atom.workspace.destroyActivePaneItem();
        configDirPath = temp.mkdirSync('atom-config-dir-');
        spyOn(atom, 'getConfigDirPath').andReturn(configDirPath);
        atom.workspace.open('atom://.atom/snippets');
        waitsFor(function() {
          return atom.workspace.getActiveTextEditor() != null;
        });
        return runs(function() {
          return expect(atom.workspace.getActiveTextEditor().getURI()).toBe(path.join(configDirPath, 'snippets.cson'));
        });
      });
    });
    describe("snippet insertion API", function() {
      return it("will automatically parse snippet definition and replace selection", function() {
        editor.setSelectedBufferRange([[0, 4], [0, 13]]);
        Snippets.insert("hello ${1:world}", editor);
        expect(editor.lineTextForBufferRow(0)).toBe("var hello world = function () {");
        expect(editor.getMarkerCount()).toBe(2);
        return expect(editor.getSelectedBufferRange()).toEqual([[0, 10], [0, 15]]);
      });
    });
    return describe("when the 'snippets:available' command is triggered", function() {
      var availableSnippetsView;
      availableSnippetsView = null;
      beforeEach(function() {
        Snippets.add(__filename, {
          ".source.js": {
            "test": {
              prefix: "test",
              body: "${1:Test pass you will}, young "
            },
            "challenge": {
              prefix: "chal",
              body: "$1: ${2:To pass this challenge}"
            }
          }
        });
        delete Snippets.availableSnippetsView;
        atom.commands.dispatch(editorElement, "snippets:available");
        return availableSnippetsView = atom.workspace.getModalPanels()[0].getItem();
      });
      it("renders a select list of all available snippets", function() {
        expect(availableSnippetsView.getSelectedItem().prefix).toBe('test');
        expect(availableSnippetsView.getSelectedItem().name).toBe('test');
        expect(availableSnippetsView.getSelectedItem().bodyText).toBe('${1:Test pass you will}, young ');
        atom.commands.dispatch(availableSnippetsView.filterEditorView[0], 'core:move-down');
        expect(availableSnippetsView.getSelectedItem().prefix).toBe('chal');
        expect(availableSnippetsView.getSelectedItem().name).toBe('challenge');
        return expect(availableSnippetsView.getSelectedItem().bodyText).toBe('$1: ${2:To pass this challenge}');
      });
      it("writes the selected snippet to the editor as snippet", function() {
        atom.commands.dispatch(availableSnippetsView.filterEditorView[0], 'core:confirm');
        expect(editor.getCursorScreenPosition()).toEqual([0, 18]);
        expect(editor.getSelectedText()).toBe('Test pass you will');
        return expect(editor.lineTextForBufferRow(0)).toBe('Test pass you will, young var quicksort = function () {');
      });
      return it("closes the dialog when triggered again", function() {
        atom.commands.dispatch(availableSnippetsView.filterEditorView[0], 'snippets:available');
        return expect(atom.workspace.getModalPanels().length).toBe(0);
      });
    });
  });

}).call(this);
