(function() {
  var Grim, fs, os, path;

  Grim = require('grim');

  fs = require('fs-plus');

  path = require('path');

  os = require('os');

  describe("Built-in Status Bar Tiles", function() {
    var dummyView, statusBar, workspaceElement, _ref;
    _ref = [], statusBar = _ref[0], workspaceElement = _ref[1], dummyView = _ref[2];
    beforeEach(function() {
      workspaceElement = atom.views.getView(atom.workspace);
      dummyView = document.createElement("div");
      statusBar = null;
      waitsForPromise(function() {
        return atom.packages.activatePackage('status-bar');
      });
      return runs(function() {
        return statusBar = workspaceElement.querySelector("status-bar");
      });
    });
    describe("the file info, cursor and selection tiles", function() {
      var buffer, cursorPosition, editor, fileInfo, selectionCount, _ref1;
      _ref1 = [], editor = _ref1[0], buffer = _ref1[1], fileInfo = _ref1[2], cursorPosition = _ref1[3], selectionCount = _ref1[4];
      beforeEach(function() {
        waitsForPromise(function() {
          return atom.workspace.open('sample.js');
        });
        return runs(function() {
          var launchMode, _ref2;
          _ref2 = statusBar.getLeftTiles().map(function(tile) {
            return tile.getItem();
          }), launchMode = _ref2[0], fileInfo = _ref2[1], cursorPosition = _ref2[2], selectionCount = _ref2[3];
          editor = atom.workspace.getActiveTextEditor();
          return buffer = editor.getBuffer();
        });
      });
      describe("when associated with an unsaved buffer", function() {
        return it("displays 'untitled' instead of the buffer's path, but still displays the buffer position", function() {
          waitsForPromise(function() {
            return atom.workspace.open();
          });
          return runs(function() {
            expect(fileInfo.currentPath.textContent).toBe('untitled');
            expect(cursorPosition.textContent).toBe('1,1');
            return expect(selectionCount).toBeHidden();
          });
        });
      });
      describe("when the associated editor's path changes", function() {
        return it("updates the path in the status bar", function() {
          waitsForPromise(function() {
            return atom.workspace.open('sample.txt');
          });
          return runs(function() {
            return expect(fileInfo.currentPath.textContent).toBe('sample.txt');
          });
        });
      });
      describe("when the associated editor's buffer's content changes", function() {
        return it("enables the buffer modified indicator", function() {
          expect(fileInfo.bufferModified.textContent).toBe('');
          editor.insertText("\n");
          advanceClock(buffer.stoppedChangingDelay);
          expect(fileInfo.bufferModified.textContent).toBe('*');
          return editor.backspace();
        });
      });
      describe("when the buffer content has changed from the content on disk", function() {
        it("disables the buffer modified indicator on save", function() {
          var filePath;
          filePath = path.join(os.tmpdir(), "atom-whitespace.txt");
          fs.writeFileSync(filePath, "");
          waitsForPromise(function() {
            return atom.workspace.open(filePath);
          });
          return runs(function() {
            editor = atom.workspace.getActiveTextEditor();
            expect(fileInfo.bufferModified.textContent).toBe('');
            editor.insertText("\n");
            advanceClock(buffer.stoppedChangingDelay);
            expect(fileInfo.bufferModified.textContent).toBe('*');
            editor.getBuffer().save();
            return expect(fileInfo.bufferModified.textContent).toBe('');
          });
        });
        it("disables the buffer modified indicator if the content matches again", function() {
          expect(fileInfo.bufferModified.textContent).toBe('');
          editor.insertText("\n");
          advanceClock(buffer.stoppedChangingDelay);
          expect(fileInfo.bufferModified.textContent).toBe('*');
          editor.backspace();
          advanceClock(buffer.stoppedChangingDelay);
          return expect(fileInfo.bufferModified.textContent).toBe('');
        });
        return it("disables the buffer modified indicator when the change is undone", function() {
          expect(fileInfo.bufferModified.textContent).toBe('');
          editor.insertText("\n");
          advanceClock(buffer.stoppedChangingDelay);
          expect(fileInfo.bufferModified.textContent).toBe('*');
          editor.undo();
          advanceClock(buffer.stoppedChangingDelay);
          return expect(fileInfo.bufferModified.textContent).toBe('');
        });
      });
      describe("when the buffer changes", function() {
        it("updates the buffer modified indicator for the new buffer", function() {
          expect(fileInfo.bufferModified.textContent).toBe('');
          waitsForPromise(function() {
            return atom.workspace.open('sample.txt');
          });
          return runs(function() {
            editor = atom.workspace.getActiveTextEditor();
            editor.insertText("\n");
            advanceClock(buffer.stoppedChangingDelay);
            return expect(fileInfo.bufferModified.textContent).toBe('*');
          });
        });
        return it("doesn't update the buffer modified indicator for the old buffer", function() {
          var oldBuffer;
          oldBuffer = editor.getBuffer();
          expect(fileInfo.bufferModified.textContent).toBe('');
          waitsForPromise(function() {
            return atom.workspace.open('sample.txt');
          });
          return runs(function() {
            oldBuffer.setText("new text");
            advanceClock(buffer.stoppedChangingDelay);
            return expect(fileInfo.bufferModified.textContent).toBe('');
          });
        });
      });
      describe("when the associated editor's cursor position changes", function() {
        return it("updates the cursor position in the status bar", function() {
          jasmine.attachToDOM(workspaceElement);
          editor.setCursorScreenPosition([1, 2]);
          return expect(cursorPosition.textContent).toBe('2,3');
        });
      });
      describe("when the associated editor's selection changes", function() {
        return it("updates the selection count in the status bar", function() {
          jasmine.attachToDOM(workspaceElement);
          editor.setSelectedBufferRange([[0, 0], [0, 2]]);
          return expect(selectionCount.textContent).toBe('(2)');
        });
      });
      describe("when the active pane item does not implement getCursorBufferPosition()", function() {
        return it("hides the cursor position view", function() {
          jasmine.attachToDOM(workspaceElement);
          atom.workspace.getActivePane().activateItem(dummyView);
          return expect(cursorPosition).toBeHidden();
        });
      });
      describe("when the active pane item implements getTitle() but not getPath()", function() {
        return it("displays the title", function() {
          jasmine.attachToDOM(workspaceElement);
          dummyView.getTitle = (function(_this) {
            return function() {
              return 'View Title';
            };
          })(this);
          atom.workspace.getActivePane().activateItem(dummyView);
          expect(fileInfo.currentPath.textContent).toBe('View Title');
          return expect(fileInfo.currentPath).toBeVisible();
        });
      });
      describe("when the active pane item neither getTitle() nor getPath()", function() {
        return it("hides the path view", function() {
          jasmine.attachToDOM(workspaceElement);
          atom.workspace.getActivePane().activateItem(dummyView);
          return expect(fileInfo.currentPath).toBeHidden();
        });
      });
      return describe("when the active pane item's title changes", function() {
        return it("updates the path view with the new title", function() {
          var callback, callbacks, _i, _len;
          jasmine.attachToDOM(workspaceElement);
          callbacks = [];
          dummyView.onDidChangeTitle = function(fn) {
            callbacks.push(fn);
            return {
              dispose: function() {}
            };
          };
          dummyView.getTitle = function() {
            return 'View Title';
          };
          atom.workspace.getActivePane().activateItem(dummyView);
          expect(fileInfo.currentPath.textContent).toBe('View Title');
          dummyView.getTitle = function() {
            return 'New Title';
          };
          for (_i = 0, _len = callbacks.length; _i < _len; _i++) {
            callback = callbacks[_i];
            callback();
          }
          return expect(fileInfo.currentPath.textContent).toBe('New Title');
        });
      });
    });
    return describe("the git tile", function() {
      var gitView;
      gitView = null;
      beforeEach(function() {
        var _ref1;
        return _ref1 = statusBar.getRightTiles().map(function(tile) {
          return tile.getItem();
        }), gitView = _ref1[0], _ref1;
      });
      describe("the git branch label", function() {
        beforeEach(function() {
          fs.removeSync(path.join(os.tmpdir(), '.git'));
          return jasmine.attachToDOM(workspaceElement);
        });
        it("displays the current branch for files in repositories", function() {
          atom.project.setPaths([atom.project.getDirectories()[0].resolve('git/master.git')]);
          waitsForPromise(function() {
            return atom.workspace.open('HEAD');
          });
          runs(function() {
            var currentBranch;
            currentBranch = atom.project.getRepositories()[0].getShortHead();
            expect(gitView.branchArea).toBeVisible();
            expect(gitView.branchLabel.textContent).toBe(currentBranch);
            atom.workspace.getActivePane().destroyItems();
            expect(gitView.branchArea).toBeVisible();
            return expect(gitView.branchLabel.textContent).toBe(currentBranch);
          });
          atom.workspace.getActivePane().activateItem(dummyView);
          return expect(gitView.branchArea).not.toBeVisible();
        });
        it("doesn't display the current branch for a file not in a repository", function() {
          atom.project.setPaths([os.tmpdir()]);
          waitsForPromise(function() {
            return atom.workspace.open(path.join(os.tmpdir(), 'temp.txt'));
          });
          return runs(function() {
            return expect(gitView.branchArea).toBeHidden();
          });
        });
        return it("doesn't display the current branch for a file outside the current project", function() {
          waitsForPromise(function() {
            return atom.workspace.open(path.join(os.tmpdir(), 'atom-specs', 'not-in-project.txt'));
          });
          return runs(function() {
            return expect(gitView.branchArea).toBeHidden();
          });
        });
      });
      return describe("the git status label", function() {
        var filePath, ignorePath, ignoredPath, newPath, originalPathText, projectPath, repo, _ref1;
        _ref1 = [], repo = _ref1[0], filePath = _ref1[1], originalPathText = _ref1[2], newPath = _ref1[3], ignorePath = _ref1[4], ignoredPath = _ref1[5], projectPath = _ref1[6];
        beforeEach(function() {
          projectPath = atom.project.getDirectories()[0].resolve('git/working-dir');
          fs.moveSync(path.join(projectPath, 'git.git'), path.join(projectPath, '.git'));
          atom.project.setPaths([projectPath]);
          filePath = atom.project.getDirectories()[0].resolve('a.txt');
          newPath = atom.project.getDirectories()[0].resolve('new.txt');
          fs.writeFileSync(newPath, "I'm new here");
          ignorePath = path.join(projectPath, '.gitignore');
          fs.writeFileSync(ignorePath, 'ignored.txt');
          ignoredPath = path.join(projectPath, 'ignored.txt');
          fs.writeFileSync(ignoredPath, '');
          atom.project.getRepositories()[0].getPathStatus(filePath);
          atom.project.getRepositories()[0].getPathStatus(newPath);
          originalPathText = fs.readFileSync(filePath, 'utf8');
          return jasmine.attachToDOM(workspaceElement);
        });
        afterEach(function() {
          fs.writeFileSync(filePath, originalPathText);
          fs.removeSync(newPath);
          fs.removeSync(ignorePath);
          fs.removeSync(ignoredPath);
          return fs.moveSync(path.join(projectPath, '.git'), path.join(projectPath, 'git.git'));
        });
        it("displays the modified icon for a changed file", function() {
          fs.writeFileSync(filePath, "i've changed for the worse");
          atom.project.getRepositories()[0].getPathStatus(filePath);
          waitsForPromise(function() {
            return atom.workspace.open(filePath);
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toHaveClass('icon-diff-modified');
          });
        });
        it("doesn't display the modified icon for an unchanged file", function() {
          waitsForPromise(function() {
            return atom.workspace.open(filePath);
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toHaveText('');
          });
        });
        it("displays the new icon for a new file", function() {
          waitsForPromise(function() {
            return atom.workspace.open(newPath);
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toHaveClass('icon-diff-added');
          });
        });
        it("displays the ignored icon for an ignored file", function() {
          waitsForPromise(function() {
            return atom.workspace.open(ignoredPath);
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toHaveClass('icon-diff-ignored');
          });
        });
        it("updates when a status-changed event occurs", function() {
          fs.writeFileSync(filePath, "i've changed for the worse");
          atom.project.getRepositories()[0].getPathStatus(filePath);
          waitsForPromise(function() {
            return atom.workspace.open(filePath);
          });
          return runs(function() {
            expect(gitView.gitStatusIcon).toHaveClass('icon-diff-modified');
            fs.writeFileSync(filePath, originalPathText);
            atom.project.getRepositories()[0].getPathStatus(filePath);
            return expect(gitView.gitStatusIcon).not.toHaveClass('icon-diff-modified');
          });
        });
        it("displays the diff stat for modified files", function() {
          fs.writeFileSync(filePath, "i've changed for the worse");
          atom.project.getRepositories()[0].getPathStatus(filePath);
          waitsForPromise(function() {
            return atom.workspace.open(filePath);
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toHaveText('+1');
          });
        });
        it("displays the diff stat for new files", function() {
          waitsForPromise(function() {
            return atom.workspace.open(newPath);
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toHaveText('+1');
          });
        });
        return it("does not display for files not in the current project", function() {
          waitsForPromise(function() {
            return atom.workspace.open('/tmp/atom-specs/not-in-project.txt');
          });
          return runs(function() {
            return expect(gitView.gitStatusIcon).toBeHidden();
          });
        });
      });
    });
  });

}).call(this);
