//
//  Account.h
//  EconomacsSDK
//
//  Created by Erik Aderstedt on 2008-04-27.
//  Copyright 2008 Aderstedt Software AB. All rights reserved.
//
//	The Account entity encapsulates an account. 
// 
//	An account can act as a special account set up in the singleton Company instance.
//	To this end, the account has a number of bidirectional relationships, that either
//	point to the Company object or are nil:
//	currencyGains
//	currencyLosses
//	incomeAccount,
//	isResultAccount,
//	customerAssets,
//	roundingAccount
//
//	Note that the account may also figure as the vat or sales account for one or more products.
//	That relationship is unidirectional from the Article entity and we cannot get these products
//	without performing a fetch. The method -(void)nullifyLinkedProducts removes these couplings
//	in preparation for account removal.

#if defined(IFAKTURA)
#import <Foundation/Foundation.h>
#import <CoreData/CoreData.h>
#else
#import <Cocoa/Cocoa.h>
#endif

@class FiscalYear, Tag, VatInfo;

#define KTYP_ASSET		0
#define KTYP_DEBT		1
#define KTYP_INCOME		2
#define KTYP_EXPENSE	3
#define KTYP_EQUITY		4

enum accountTypes {
	AS_KTYP_ASSET='AtyA',
	AS_KTYP_DEBT='AtyD',
	AS_KTYP_INCOME='AtyI',
	AS_KTYP_EXPENSE='AtyE',
	AS_KTYP_EQUITY='AtyQ'
} ;

enum ASAccountType {
    kASAccountTypeAsset,
    kASAccountTypeDebt,
    kASAccountTypeIncome,
    kASAccountTypeExpense,
    kASAccountTypeEquity
};

#define USAGE_CUSTOMER_ASSET	1
#define USAGE_CURRENCY_GAIN		2
#define USAGE_CURRENCY_LOSS		3
#define USAGE_ROUNDING			4

@interface Account : NSManagedObject {
    
}
@property(strong) NSDecimalNumber *cachedBalance1;
@property(strong) NSDecimalNumber *cachedBalance2;
@property(strong) NSDecimalNumber *cachedBalance3;

@property(strong) NSString *name;
@property(retain) NSNumber *number;
@property(retain) NSNumber *type;
@property(retain) NSString *taxCode;
@property(retain) NSString *vatCode;
@property(retain) NSSet *startingBalances;
@property(retain) NSSet *rows;
@property(retain) VatInfo *vatInfo;
@property(assign) NSInteger vatLevel;
@property(retain) NSNumber *vatIndex;

+ (NSPredicate *)balanceAccountPredicate;
+ (NSPredicate *)balanceAccountPredicateForRows;
+ (NSPredicate *)resultAccountPredicate;
+ (Account *)locateAccountWithNumber:(NSNumber *)a inContext:(NSManagedObjectContext *)moc;
+ (Account *)locateAccountWithName:(NSString *)s inContext:(NSManagedObjectContext *)moc;
+ (void)removeEmptyAccountsInContext:(NSManagedObjectContext *)moc;

- (BOOL)balanceAccount;
- (enum ASAccountType)accountType;
- (void)setTypeDescription:(NSString *)t;
- (NSString *)typeDescription;
+ (NSNumber *)typeFromDescription:(NSString *)typeDescription;
- (NSString *)vatLevelDescription;
- (NSString *)numberDescription;
- (NSString *)accountDescription;
- (NSDecimalNumber *)balance;
- (NSDecimalNumber *)balanceByDateIndex:(NSNumber *)d;
- (NSDecimalNumber *)balanceByDateIndex:(NSNumber *)d inYear:(FiscalYear *)fY;
- (NSDecimalNumber *)balanceByDateIndex:(NSNumber *)d inYear:(FiscalYear *)fY forTag:(Tag *)t;
- (NSDecimalNumber *)balanceBeforeDateIndex:(NSNumber *)d inYear:(FiscalYear *)fY;
- (NSDecimalNumber *)balanceBeforeDateIndex:(NSNumber *)d inYear:(FiscalYear *)fY forTag:(Tag *)t;
- (NSNumber *)numberOfRows;
- (NSNumber *)used;
- (BOOL)isNormallyOnlyDebited;
- (BOOL)isNormallyOnlyCredited;
- (NSInteger)depth;
- (NSDecimalNumber *)difference;

- (NSComparisonResult)compareAccountNumber:(Account *)a2;
+ (void)constructHierarchyInAccounts:(NSArray *)accounts;

@end
