//
//  GSInstance.h
//  Glyphs
//
//  Created by Georg Seifert on 5.2.08.
//  Copyright 2008 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>

@class GSFont;
@class GSCustomProperty;

#import <GlyphsCore/GSContainerProtocol.h>

/** The class defining the instance object
*/

@interface GSInstance : NSObject <NSCoding, NSCopying, GSContainerProtocol> {
	GSFont* __unsafe_unretained _font;
	NSString* _name;
	NSString* _linkStyle;
	float _interpolationWeight;
	float _interpolationWidth;
	float _interpolationCustom;
	NSString* _weightClass;
	NSString* _widthClass;
	BOOL _active;
	BOOL _isItalic;
	BOOL _isBold;
	NSDictionary* _instanceInterpolations;
	NSMutableDictionary* _tempData;
	NSMutableArray* _customParameters;
	BOOL _manualInterpolation;
}

/** reference to the containing font */
@property (unsafe_unretained, nonatomic) GSFont* font;

/** @name info */

/** The Intances name

 This is used as the style name.
 */
@property (copy, nonatomic) NSString* name;

/** The style to use as a the regular for this style.

 if styleName is `Medium`, you could link to `Light`. So this instance becomes the Bold stil for the Light.
 */
@property (copy, nonatomic) NSString* linkStyle;

/// The interpolation Weight position.
@property (nonatomic) float interpolationWeight;

/// The interpolation Width position
@property (nonatomic) float interpolationWidth;

/// The interpolation Cutom position
@property (nonatomic) float interpolationCustom;

/** Weight Class

 e.g. Regular or Bold
 */
@property (strong, nonatomic) NSString* weightClass;

/// returnes the actual value as set by the weightClass or custom parameter "weightClass"
@property (nonatomic) NSUInteger weightClassValue;

/** Width Class

 e.g. Medium (normal)
 */
@property (strong, nonatomic) NSString* widthClass;

/// returnes the actual value as set by the widthClass
@property (nonatomic) NSUInteger widthClassValue;

/// If it should be exported.
@property (nonatomic) BOOL active;

/// the Style linking Italic bit
@property (nonatomic) BOOL isItalic;

/// the Style linking Bold bit
@property (nonatomic) BOOL isBold;

/// The content of the instance to store it in a pList.
@property (unsafe_unretained) NSDictionary* instanceDict;

/** To store stuff temporarily
 
 This is not saved to file
 */
@property (strong, nonatomic) NSMutableDictionary* tempData;

- (instancetype)initWithInstance:(GSInstance*)Instance;

#ifndef GLYPHS_VIEWER

- (BOOL)saveToFile:(FILE*)File error:(NSError**)error;

/** A dict that contains the interpolation coefficents for each master.
 
 This is automatcially updated if you change interpolationWeight, interpolationWidth, interpolationCustom. It contains FontMaster IDs as keys and coeffients for that master as values.
 
 Or, you can set it maually if you set manualInterpolation to true. There is no UI for this, so you need to do that with a script
 */
@property (strong, nonatomic) NSDictionary* instanceInterpolations;

/** Disables automatic calculation of instanceInterpolations
 
 This allowes to manually setting instanceInterpolations.
 */
@property (nonatomic) BOOL manualInterpolation;

/** Calculates the coefficients for the given masters
 
 This is used to automatically calculate the instanceInterpolations
 
 @param masters A list of GSFontMaster objects
 
 @return a dict of FontMaster IDs as keys and coeffients as values.
 */
- (NSDictionary *)instanceInterpolations:(NSArray *)masters;

- (NSDictionary *)instanceInterpolationsWithUpdates:(BOOL)UpdateKVO;

/** Call it if you have made changes to the instance to force an update of instanceInterpolations. */
- (void) updateInterpolationValues ;

/** @name customParameters */

/// A list of GSCustomProperty objects
@property (strong, nonatomic) NSMutableArray* customParameters;

/** The count of custom parameters */
- (NSUInteger)countOfCustomParameters;

- (GSCustomProperty*)objectInCustomParametersAtIndex:(NSUInteger)theIndex;

/** The value of the custom parameter where name == Key
 
 @param Key The name of the paremter
 
 @return the value of the first matching parameter
 */
- (id)customValueForKey:(NSString*)Key;

/** The custom parameter where name == Key
 
 @param Key The name of the paremter
 
 @return the first matching paramter
 */
- (GSCustomProperty*)customPropertyForKey:(NSString*)Key;

/** Set the value for key
 
 It will look for an existing paramter with the name an overwrite its value, or adds a new parameter
 @param value The value to add
 @param key   The name of the paramter
 */
- (void)setCustomParameter:(id)value forKey:(NSString*)key;

/** Removes the first paramter with then Name
 
 @param Key The name
 */
- (void)removeObjectFromCustomParametersForKey:(NSString*)Key;

- (void)insertObject:(GSCustomProperty*)aCustomProperty inCustomParametersAtIndex:(NSUInteger)idx;

- (void)removeObjectFromCustomParametersAtIndex:(NSUInteger)idx;

- (void)replaceObjectInCustomParametersAtIndex:(NSUInteger)idx withObject:(GSCustomProperty*)aCustomProperty;

/** Scales the instance object.
 
 @param Scale Scale value
 */
- (void)scaleBy:(CGFloat)Scale;
#endif
@end
