//
//  GSPenProtocol.h
//  GlyphsCore
//
//  Created by Georg Seifert on 16.11.11.
//  Copyright 2011 schriftgestaltung.de. All rights reserved.
//

#import <Cocoa/Cocoa.h>
#import "GSNode.h"

@class GSLayer;

@protocol GSPenProtocol

/** Begin a new sub path, set the current point to 'pt'. You must
end each sub path with a call to pen.closePath() or pen.endPath().
*/
- (void)moveTo:(NSPoint)pt;

/** Draw a straight line from the current point to 'pt'.*/
- (void)lineTo:(NSPoint)pt;

/** Draw a cubic bezier with an arbitrary number of control points.

The last point specified is on-curve, all others are off-curve
(control) points. If the number of control points is > 2, the
segment is split into multiple bezier segments. This works
like this:

Let n be the number of control points (which is the number of
arguments to this call minus 1). If n==2, a plain vanilla cubic
bezier is drawn. If n==1, we fall back to a quadratic segment and
if n==0 we draw a straight line. It gets interesting when n>2:
n-1 PostScript-style cubic segments will be drawn as if it were
one curve. See decomposeSuperBezierSegment().

The conversion algorithm used for n>2 is inspired by NURB
splines, and is conceptually equivalent to the TrueType "implied
points" principle. See also decomposeQuadraticSegment().
*/
- (void)curveTo:(NSPoint)pt off1:(NSPoint)off1 off2:(NSPoint)off2;

/** Draw a whole string of quadratic curve segments.

The last point specified is on-curve, all others are off-curve
points.

This method implements TrueType-style curves, breaking up curves
using 'implied points': between each two consequtive off-curve points,
there is one implied point exactly in the middle between them. See
also decomposeQuadraticSegment().

The last argument (normally the on-curve point) may be None.
This is to support contours that have NO on-curve points (a rarely
seen feature of TrueType outlines).
*/
- (void)qCurveTo:(NSArray*)Points;

/** Close the current sub path. You must call either pen.closePath()
or pen.endPath() after each sub path.
*/
- (void)closePath;

/** End the current sub path, but don't close it. You must call
either pen.closePath() or pen.endPath() after each sub path.
*/
- (void)endPath;

- (void)addComponent:(id)Base transformation:(NSAffineTransform*)Transformation;

@optional
/**
To initialize stuff
*/
- (void)startGlyph:(GSLayer*)Layer;

- (void)moveTo:(NSPoint)pt connection:(GSNodeType)connection selected:(BOOL)selected;

- (void)lineTo:(NSPoint)pt connection:(GSNodeType)connection selected:(BOOL)selected;

- (void)curveTo:(NSPoint)pt off1:(NSPoint)off1 off2:(NSPoint)off2 connection:(GSNodeType)connection selected:(BOOL)selected;

@end

@interface GSSegmentPen : NSObject <GSPenProtocol> {
	NSMutableArray* _contour;
}

@end
