module CAUL_HOG
module HideOverlappingGeometry
  
  ### Params ###
  @layer = nil
  @smooth_angle = Math::PI / 8.0
  @munique = false
  @hide_within = false
  ##############
  
  @NO_LAYER = "-N O N E-"
  @EPSILON = 0.001
  @E_VE_POS = Geom::Vector3d.new(@EPSILON, @EPSILON, @EPSILON)
  @E_VE_NEG = Geom::Vector3d.new(-@EPSILON, -@EPSILON, -@EPSILON)
  @VERBOSE = false
  
  ####################################
  ########## UI ######################
  ####################################
  
  def self.getLayerList(mod)
    list = []
    s = ""
    mod.layers.each { |l| list << l.name }
    list.sort!
    list.insert(0, @NO_LAYER)
    list.each_index { |i| s += list[i].to_s + ((i == list.length-1) ? "" : "|") }
    return s
  end

  #Read settings from database
  def self.readSettings(mod)
    @smooth_angle = Sketchup.read_default("CAUL_HideOverlapping","smooth")
    @layer = Sketchup.read_default("CAUL_HideOverlapping","layer")
    @munique = Sketchup.read_default("CAUL_HideOverlapping","munique")
    @hide_within = Sketchup.read_default("CAUL_HideOverlapping","hide_within")
    if @smooth_angle === nil || @munique === nil || @hide_within === nil
      UI.messagebox("No settings stored")
      return false
    end
    
    if @layer != nil && (mod.layers.find { |l| l.name == @layer } == nil)
      UI.messagebox("The stored layer does not exist in the model")
      return false
    end
    return true
  end
  
  #Write settings to database
  def self.writeSettings()
    Sketchup.write_default("CAUL_HideOverlapping","smooth", @smooth_angle)
    Sketchup.write_default("CAUL_HideOverlapping","layer", @layer)
    Sketchup.write_default("CAUL_HideOverlapping","munique", @munique)
    Sketchup.write_default("CAUL_HideOverlapping","hide_within", @hide_within)
  end
  
  #Input box
  def self.prompt(mod)
    prompts = ["Smooth", "Layer", "Make unique", "Hide within nesting"]
    defaults = ["22.5", @NO_LAYER, "No", "No"]
    list = getLayerList(mod)
    alts = ["", list.to_s, "Yes|No", "Yes|No"]
    input = UI.inputbox(prompts, defaults, alts, "Hide Overlapping")
    return false unless input
    @smooth_angle = input[0].to_f * (Math::PI / 180)
    @layer = (input[1] == @NO_LAYER) ? nil : input[1]
    @munique = (input[2] == "Yes") ? true : false
    @hide_within = (input[3] == "Yes") ? true : false
    writeSettings()
  end
  
  def self.printParams()
    puts "smooth: "+@smooth_angle.to_s
    puts "layer: "+@layer.to_s
    puts "unique: "+@munique.to_s
    puts "hide nested: "+@hide_within.to_s
  end
  
  
  ####################################
  ####### PARSE INPUT ################
  ####################################
  
  #set up the groups and components in clusters related to the outmost group or comp
  #The result is an array of clusters. A cluster is an array with representation
  #for each group or component on the form: [id, ents, transformation, bb]
  def self.parse(sel)
    cluster_id = 0
    clusters = []
    gs = sel.grep(Sketchup::Group) + sel.grep(Sketchup::ComponentInstance)
    
    gs.each { |g|
      cluster = []
      g.make_unique if @munique
      ents = g.is_a?(Sketchup::ComponentInstance) ? g.definition.entities : g.entities
      tr = g.transformation
      bb = getBB(ents, tr)
      cluster << [cluster_id, ents, tr, bb] if bb != nil
      cluster_id += 1
      getNested(cluster_id, cluster, ents, tr)
      clusters << cluster
    }
    
    return @hide_within ? getAllPairs(clusters) : getPairs(clusters) 
  end
  
  def self.getNested(cluster_id, cluster, ents, tr)
    gs = ents.grep(Sketchup::Group) + ents.grep(Sketchup::ComponentInstance)
    gs.each { |g|
      g.make_unique if @munique
      ents = g.is_a?(Sketchup::ComponentInstance) ? g.definition.entities : g.entities
      tr2 = tr * g.transformation 
      bb = getBB(ents, tr2)
      cluster << [cluster_id, ents, tr2, bb] if bb != nil
      getNested(cluster_id, cluster, ents, tr2)
    }
  end
  
  #we need a bb for the outer context. This bb is constructed for vertices,
  #if the group or component contains no vertices, then the bb is nil.
  def self.getBB(ents, tr)
    vs = ents.grep(Sketchup::Edge).map{|e| e.vertices}
    return nil if vs.length == 0
    vs.flatten!.uniq!
    bb = Geom::BoundingBox.new
    vs.each { |v| bb.add(v.position.transform(tr)) }
    bb_min = bb.min.offset(@E_VE_NEG)
    bb_max = bb.max.offset(@E_VE_POS)
    bb2 = Geom::BoundingBox.new
    bb2.add(bb_min); bb2.add(bb_max)
    return bb2
  end
  
  #Get pairs ONLY between clusters
  def self.getPairs(clusters)
    res = []
    (0..clusters.length - 2).each { |i|
      (i+1..clusters.length-1).each { |j|
         c0 = clusters[i]
         c1 = clusters[j]
         (0..c0.length-1).each { |k|
           (0..c1.length-1).each { |m|
             bb_inter = c0[k][3].intersect(c1[m][3])
             res << [c0[k], c1[m], bb_inter] if bb_inter.max.x >= bb_inter.min.x
    }}}}
    return res
  end
  
  #Get pairs BOTH between and within clusters
  def self.getAllPairs(clusters)
    clusters.flatten!(1)
    res = []
    (0..clusters.length-2).each { |i|
      (i+1..clusters.length-1).each { |j|
        bb_inter = clusters[i][3].intersect(clusters[j][3])
        res << [clusters[i], clusters[j], bb_inter] if bb_inter.max.x >= bb_inter.min.x
    }}
    return res
  end
  
  #############################################
  ### EXTRACT POTENTIALLY OVERLAPPING FACES ###
  #############################################
  
  #Given a pair of groups (or components), extract face pairs 
  #potentially overlapping each other. g is in this context
  #an array of [[id, ents, tr, bb], [id, ents, tr, bb], bb_inter]
  @id_count = 0
  @X_DIV = @Y_DIV = @Z_DIV = 8
  
  def self.getBBFaces(gp)
    s0 = Time.now
    g0 = gp[0] #g0 -> [id, ents, tr, bb]
    g1 = gp[1] #g1 -> [id, ents, tr, bb]
    bb_min = gp[2].min
    bb_max = gp[2].max
    g0bb = getBBArray(g0)
    g1bb = getBBArray(g1)   
    g0bb_int = g0bb.select { |b| bbIntersect?(bb_min, bb_max, b[0], b[1]) }
    g1bb_int = g1bb.select { |b| bbIntersect?(bb_min, bb_max, b[0], b[1]) }    
    
    if @VERBOSE
      s1 = Time.now
      puts formatString('Extract BB:', 20)+(s1 - s0).to_s+' s'
      puts '   #g0 faces: '+g0bb.length.to_s
      puts '   #g0 in BB: '+g0bb_int.length.to_s
      puts '   #g1 faces: '+g1bb.length.to_s
      puts '   #g1 in BB: '+g1bb_int.length.to_s
    end
    
    return g0bb_int, g1bb_int
  end
  
  #g has the form [id, ents, tr, bb]
  def self.getBBArray(g)
    tr = g[2]
    return g[1].grep(Sketchup::Face).map { |f|
      bb = Geom::BoundingBox.new      
      bb.add(f.outer_loop.vertices.map { |v| v.position.transform tr })
      @id_count += 1
      p = f.outer_loop.vertices[0].position.transform tr
      norm = f.normal.transform tr
      [bb.min.offset(@E_VE_NEG), bb.max.offset(@E_VE_POS), f, @id_count, p, norm, tr, 0] }
  end
   
  def self.bbIntersect?(a_min, a_max, b_min, b_max)
    return false if b_min.x > a_max.x || b_max.x < a_min.x
    return false if b_min.y > a_max.y || b_max.y < a_min.y
    return false if b_min.z > a_max.z || b_max.z < a_min.z
    return true
  end  
    
  #Extract potentially intersecting face pairs with a space hash
  def self.extractFacePairs(gp)
    bb = gp[2]
    g0bb, g1bb = getBBFaces(gp)
  
    s0 = Time.now
    #compute the bounds for the space hash
    dx_inv = 1.0 / ((bb.max.x - bb.min.x) / @X_DIV)
    dy_inv = 1.0 / ((bb.max.y - bb.min.y) / @Y_DIV)
    dz_inv = 1.0 / ((bb.max.z - bb.min.z) / @Z_DIV)
    
    #create the space hash
    sh = []
    (0..@X_DIV - 1).each { |i| sh << []
      (0..@Y_DIV - 1).each { |j| sh[i] << []
        (0..@Z_DIV - 1).each { |k| sh[i][j] << []
    }}}
    
    #add g0bb to space hash
    g0bb.each { |a| 
      inds = getBounds(a, bb, dx_inv, dy_inv, dz_inv)
      (inds[0]..inds[1]).each { |i|  
        (inds[2]..inds[3]).each { |j| 
          (inds[4]..inds[5]).each { |k|
            sh[i][j][k] << a
    }}}}
    
    res_h = {}
    res_a = []
    
    #find face pairsmm whose bounding boxes intersect
    g1bb.each { |b|
      b_id = (b[3] << 32)
      inds = getBounds(b, bb, dx_inv, dy_inv, dz_inv)
      (inds[0]..inds[1]).each { |i|  
        (inds[2]..inds[3]).each { |j| 
          (inds[4]..inds[5]).each { |k|
            sh[i][j][k].each { |a| 
              next if res_h.has_key?(b_id + a[3])
              res_h[b_id + a[3]] = nil
              #We have three filtering criteria:
              #1) Bounding boxes intersect
              #2) The faces have exactly opposit normals ??
              #3) The faces are on the same plane
              if bbIntersect?(a[0], a[1], b[0], b[1])
                if a[5] == b[5].reverse #|| a[5] == b[5]
                  if a[4].on_plane? [b[4], b[5]]
                    res_a << [a, b]
                  end
                end
              end
            }
      }}}
    }
    
    if @VERBOSE
      s1 = Time.now
      puts formatString('Extract Pairs:', 20)+(s1 - s0).to_s+' s'
      puts '   #Pairs: ' +res_a.length.to_s
    end
    
    return res_a
  end
  
  def self.getBounds(b, bb, dx_inv, dy_inv, dz_inv)
    return [ [((b[0].x - bb.min.x) * dx_inv).floor, 0].max,
             [((b[1].x - bb.min.x) * dx_inv).floor, @X_DIV - 1].min,
             [((b[0].y - bb.min.y) * dy_inv).floor, 0].max,
             [((b[1].y - bb.min.y) * dy_inv).floor, @Y_DIV - 1].min,
             [((b[0].z - bb.min.z) * dz_inv).floor, 0].max,
             [((b[1].z - bb.min.z) * dz_inv).floor, @Z_DIV - 1].min ]
  end
  
  ##################################
  ### DETERMINE OVERLAPPING FACES ##
  ##################################
  
  #Given a list of potentially overlapping face pairs, conclude whether
  #they really overlap
  @ng0; @ng1; @res_g
  
  def self.getOverlappingFaces(pairs)
    np = organizeFacePairs(pairs)
    res = []
    idt = Geom::Transformation.new
    np.each { |n|
      fa0 = n[0]
      copyFaceToGroup(@ng0, fa0[2], fa0[6])
      f0 = @ng0.entities.find {|ent| ent.is_a?(Sketchup::Face) }
    
      (1..n.length - 1).each { |i|
        fa1 = n[i]
        @ng1.entities.clear!
        @res_g.entities.clear!
        copyFaceToGroup(@ng1, fa1[2], fa1[6])
        f1 = @ng1.entities.find {|ent| ent.is_a?(Sketchup::Face) }
        et = @ng0.entities.intersect_with(false, idt, @res_g, idt , true, @ng1)
        if et.length != 0
          res << [fa0, fa1] if commonInteriorPoint?(et[0], f0, f1)
          next
        end
        @res_g.entities.clear!
        et = @ng1.entities.intersect_with(false, idt, @res_g, idt , true, @ng0)
        if et.length != 0
          res << [fa0, fa1] if commonInteriorPoint?(et[0], f0, f1)
      
          next
        else
          #No intersection edges may mean that we have either no overlap or
          #a complete overlap. e is an arbitrary edge which will suffice in case
          #we have a complete overlap.
          e = @ng0.entities.find {|ent| ent.is_a?(Sketchup::Edge)}
          res << [fa0, fa1] if commonInteriorPoint?(e, f0, f1)
        end
      }
      @ng0.entities.clear!
    }
    return res
  end
    
  #e is an intersection edge, that is, e is on both f0 and f1. e is used to
  #construct test points (p1, p2) on either side of e on the face plane.
  #If p1 or p2 is interior to both faces then the faces overlap.
  def self.commonInteriorPoint?(e, f0, f1)
    v0 = (e.end.position - e.start.position)
    v0.length = v0.length / 2.0
    #p0 is the midpoint of the edge
    p0 = e.start.position.offset(v0)
    v1 = v0.cross(f0.normal)
    v1.length = 0.011 #the points(p1, p2) are constructed close to the tolerance
    p1 = p0.offset(v1)
    return true if f0.classify_point(p1) == 1 &&  f1.classify_point(p1) == 1 
    p2 = p0.offset!(v1.reverse!)
    return true if f0.classify_point(p2) == 1 && f1.classify_point(p2) == 1
    return false
  end
  
  def self.copyFaceToGroup(g, f, tr)
    f0 = g.entities.add_face(f.outer_loop.vertices.map { |v| v.position.transform tr })
    f.loops.each { |l|
      next if l.outer?
      f1 = g.entities.add_face(l.vertices.map { |v| v.position.transform tr })
      f1.erase! if f1 != nil
    }
  end
  
  def self.organizeFacePairs(pairs)
    hash = {}
    pairs.each { |pair|
      if hash.has_key?(pair[0][2])
        hash[pair[0][2]] << pair[1]
      else
        hash[pair[0][2]] = [pair[0], pair[1]]
      end
    }
    res = hash.values
    return res
  end
  
  #############################
  ###### PROCESS EDGES ########
  #############################
  
  #Given a list of overlapping face pairs, compute the status of all
  #edges connected to the faces.
  def self.getOverlappingEdges(fps)
    res = []
    fps.each { |fp|
      a = processFacePairEdges(fp)
      a.each { |pair| res << pair }
    }
    return res
  end
  
  def self.processFacePairEdges(fp)
    res = []
    es0 = getFaceEdges(fp[0][2], fp[0][6])
    es1 = getFaceEdges(fp[1][2], fp[1][6])
    (0..es0.length-1).each { |i|
      (0..es1.length-1).each { |j|
        res << [es0[i], es1[j]] if overlap?(es0[i], es1[j])
    }}
    return res
  end
  
  def self.classifyPoint(p0, e)
    v = (p0 - e[0]) 
    len = v.dot(e[2])
    return 0 if len.abs <= v.length - 0.0001 #the edges are not on the same line
    return 2 if len >= e[3] #to the right
    return 1 if len >= 0 #on the edge
    return -1 #to the left
  end
  
  def self.overlap?(e0, e1)
    if (e0[2] == e1[2]) #compare the edge vectors
      return true if e0[0] == e1[0] #take care of border cases
      return false if e0[1] == e1[0] 
      res = classifyPoint(e1[0], e0)
      return false if res == 0 || res == 2
      return true if res == 1
      return classifyPoint(e0[0], e1) == 1
    elsif (e0[2] == e1[2].reverse)
      return true if e0[0] == e1[1]
      return false if e0[1] == e1[1] 
      res = classifyPoint(e1[1], e0)
      return false if res == 0 || res == 2
      return true if res == 1
      return classifyPoint(e0[0], e1) == 1
    end
    return false
  end
    
  def self.getFaceEdges(f, tr)
    es = []
    f.edges.each { |e|
      p0 = e.start.position.transform(tr)
      p1 = e.end.position.transform(tr)
      v = (p1 - p0)
      len = v.length
      es << [p0, p1, v.normalize!, len, e]
    }
    return es
  end
  
  def self.getHiddenEdges(eps, fh, eh, tr0, tr1) #send in a global face hash
    eps.each { |es|
      fs0 = es[0][4].faces
      fs1 = es[1][4].faces
      fhs0, fvs0 = getFaceSets(fs0, fh)
      fhs1, fvs1 = getFaceSets(fs1, fh)
      
      #we determine if an edge is hidden by the following rules
      #1) if all faces are in the hidden face hash, then hide the edge as well
      if fvs0.length == 0 && fvs1.length == 0
        eh[es[0][4]] = nil
        eh[es[1][4]] = nil
      #2) If the edges are connected to visible faces, then hide based on smooth
      elsif fvs0.length > 0 && fvs1.length > 0
        #compare the angle netween the normals
        n0 = fvs0[0].normal.transform(tr0)
        n1 = fvs1[0].normal.transform(tr1)
        #comapre the angls between the normals, if it is less than
        a = n0.angle_between(n1)
        if a < @smooth_angle
          eh[es[0][4]] = nil
          eh[es[1][4]] = nil
        end
      else
        eh[es[0][4]] = nil if fvs0.length == 0
        eh[es[0][4]] = nil if fvs1.length == 0
      end
    }
  end
    
  #divide the faces in fs into two sets, hidden faces and visible faces
  def self.getFaceSets(fs, fh)
    fhs = []; fvs = []
    fs.each { |f|
      if fh.has_key?(f)
        fhs << f
      else
        fvs << f
      end
    }
    return fhs, fvs
  end
  
  ####################################
  ##### PROCESS A GROUP PAIR #########
  ####################################
  
  #return an array of group pairs and fill in the global face hash..
  def self.processGroupPair(gp, fh)
    #Extract potentially overlapping faces (based on bb_intersect and normals)
    fps0 = extractFacePairs(gp)
    #filter out the overlapping face pairs from the list of potentially overlapping faces
    fps1 = getOverlappingFaces(fps0)
    #get the overlapping edges from the overlapping faces
    eps = getOverlappingEdges(fps1)
    #add faces to the global face hash
    fps1.each { |fp|
      fh[fp[0][2]] = nil unless fh.has_key?(fp[0][2])
      fh[fp[1][2]] = nil unless fh.has_key?(fp[1][2])
    }
    return eps
  end
  
  ####################
  ####### AUX ########
  ####################
  
  def self.formatString(s, len) 
    sp = ''
    (0..(len - s.length - 1)).each { |i| sp += ' ' }
    return s + sp
  end
    
  def self.init0(ent)
    @id_count = 1
    @ng0 = ent.add_group
    @ng1 = ent.add_group
    @res_g = ent.add_group
  end
  
  def self.cleanUp()
    @ng0.erase!
    @ng1.erase!
    @res_g.erase!
  end
  
  ####################
  ####### MAIN #######
  ####################
  
  def self.hideOverlapping(sel, ent)
    init0(ent) #create some aux groups
    
    pairs = parse(sel)
    fh = {}
    pairs.each { |gp|
      #hidden faces are added to the hash fh. A list of overlapping
      #edge pairs is returned for later processing.
      eps = processGroupPair(gp, fh)
      gp << eps
    }
    
    #Determine which edges to hide (this requires the global face hash)
    eh = {}
    pairs.each { |gp|
       getHiddenEdges(gp[3], fh, eh, gp[0][2], gp[1][2])
    }
    
    fh.each_key { |f| hideEntity(f) }
    eh.each_key { |e| hideEntity(e) }  
    cleanUp() #delete the aux groups
  end
  
  def self.hideEntity(elem)
    if @layer == nil
      elem.hidden = true
    else
      elem.layer = @layer
    end
  end

  def self.main(mode)
    mod = Sketchup.active_model # Open model
    ent = mod.entities # All entities in model
    sel = mod.selection # Current selection

    
    if(mode == 0) #prompt
      return unless prompt(mod)
    else
      return unless readSettings(mod)
    end
    #printParams()
    mod.start_operation("Hide Overlapping", true) 
    t0 = Time.now   
    hideOverlapping(sel, ent)
    mod.commit_operation() ; t1 = Time.now
    puts 'TOTAL: '+(t1 - t0).to_s+' s'
  end
  
  unless file_loaded?( __FILE__ )
    menu = UI.menu( 'Plugins' )
    sub = menu.add_submenu('Hide Overlapping')
    sub.add_item('Hide Overlapping') {self.main 0}
    sub.add_item('Hide with last settings') {self.main 1}
  end
  
 

end
end
