﻿--RazerOrbweaver, a World of Warcraft® user interface addon.

local R, DB, PEW = RazerOrbweaver

local BINDER = setmetatable({}, { __index = CreateFrame("Button") })

local L = LibStub("AceLocale-3.0"):GetLocale("RazerOrbweaver")

local KeyMap = R.KeyMap

local RazerActions = R.RazerActions

local BTNIndex = R.BTNIndex

local BINDIndex = R.BINDIndex

local editmode = false

local sIndex = R.sIndex
local cIndex = R.cIndex

function BINDER:GetModifier()

	local modifier

	if (IsAltKeyDown()) then
		modifier = "ALT-"
	end

	if (IsControlKeyDown()) then
		if (modifier) then
			modifier = modifier.."CTRL-";
		else
			modifier = "CTRL-";
		end
	end

	if (IsShiftKeyDown()) then
		if (modifier) then
			modifier = modifier.."SHIFT-";
		else
			modifier = "SHIFT-";
		end
	end

	return modifier
end


function BINDER:GetBindkeyList(button)

	if (not button.data) then return L.KEYBIND_NONE end

	local bindkeys = button.keys.hotKeyText:gsub(":", ", ")

	bindkeys = bindkeys:gsub("^, ", "")
	bindkeys = bindkeys:gsub(", $", "")

	if (strlen(bindkeys) < 1) then
		bindkeys = L.KEYBIND_NONE
	end

	return bindkeys
end

function BINDER:GetKeyText(key)

	local keytext

	if (key:find("Button")) then

		keytext = key:gsub("([Bb][Uu][Tt][Tt][Oo][Nn])(%d+)","m%2")

	elseif (key:find("NUMPAD")) then

		keytext = key:gsub("NUMPAD","n")
		keytext = keytext:gsub("DIVIDE","/")
		keytext = keytext:gsub("MULTIPLY","*")
		keytext = keytext:gsub("MINUS","-")
		keytext = keytext:gsub("PLUS","+")
		keytext = keytext:gsub("DECIMAL",".")

	elseif (key:find("MOUSEWHEEL")) then

		keytext = key:gsub("MOUSEWHEEL","mw")
		keytext = keytext:gsub("UP","U")
		keytext = keytext:gsub("DOWN","D")
	else
		keytext = key
	end

	keytext = keytext:gsub("ALT%-","a")
	keytext = keytext:gsub("CTRL%-","c")
	keytext = keytext:gsub("SHIFT%-","s")
	keytext = keytext:gsub("INSERT","Ins")
	keytext = keytext:gsub("DELETE","Del")
	keytext = keytext:gsub("HOME","Home")
	keytext = keytext:gsub("END","End")
	keytext = keytext:gsub("PAGEUP","PgUp")
	keytext = keytext:gsub("PAGEDOWN","PgDn")
	keytext = keytext:gsub("BACKSPACE","Bksp")
	keytext = keytext:gsub("SPACE","Spc")

	return keytext
end


function BINDER:ClearBindings(button, key)

	if (key) then

		SetOverrideBinding(button, true, key, nil)

		local newkey = key:gsub("%-", "%%-")

		button.keys.hotKeys = button.keys.hotKeys:gsub(newkey..":", "")

		local keytext = self:GetKeyText(key)

		button.keys.hotKeyText = button.keys.hotKeyText:gsub(keytext..":", "")

	else
		local bindkey = "CLICK "..button:GetName()..":LeftButton"

		while (GetBindingKey(bindkey)) do

			SetBinding(GetBindingKey(bindkey), nil)

		end

		ClearOverrideBindings(button)

		button.keys.hotKeys = ":"

		button.keys.hotKeyText = ":"
	end

	self:ApplyBindings(button)

end

function BINDER:SetRazerBinding(button, key)

	local found

	gsub(button.keys.hotKeys, "[^:]+", function(binding) if(binding == key) then found = true end end)

	if (not found) then

		local keytext = self:GetKeyText(key)

		button.keys.hotKeys = button.keys.hotKeys..key..":"
		button.keys.hotKeyText = button.keys.hotKeyText..keytext..":"
	end

	self:ApplyBindings(button)
end

function BINDER:ApplyBindings(button)

	if (button.sdata.move) then

		local index = tonumber(button.sdata.move)
		local action = RazerActions[index][2]

		if (action) then

			ClearOverrideBindings(button)

			local key = button.keys.hotKeys:match("^:([^:]+)") or ""

			if (key and #key > 0) then
				SetBinding(key, action)
			end

			local keytext = self:GetKeyText(key)

			button:SetAttribute("hotkeys", button.keys.hotKeys)

			button.hotkey:SetText(keytext)

		end
	else

		if (button:IsShown()) then
			gsub(button.keys.hotKeys, "[^:]+", function(key) SetOverrideBindingClick(button, button.keys.hotKeyPri, key, button:GetName()) end)
		end

		button:SetAttribute("hotkeys", button.keys.hotKeys)

		button.hotkey:SetText(button.keys.hotKeyText:match("^:([^:]+)") or "")

	end

	if (GetCurrentBindingSet() > 0 and GetCurrentBindingSet() < 3) then SaveBindings(GetCurrentBindingSet()) end
end


function BINDER:ProcessBinding(key, button)

	if (button and button.keys and button.keys.hotKeyLock) then
		UIErrorsFrame:AddMessage(L.BINDINGS_LOCKED, 1.0, 1.0, 1.0, 1.0, UIERRORS_HOLD_TIME)
		return
	end

	if (key == "ESCAPE") then

		self:ClearBindings(button)

	elseif (key) then

		for index,btn in pairs(BTNIndex) do
			if (button ~= btn and btn.keys and not btn.keys.hotKeyLock) then
				btn.keys.hotKeys:gsub("[^:]+", function(binding) if (key == binding) then self:ClearBindings(btn, binding) self:ApplyBindings(btn) end end)
			end
		end

		self:SetRazerBinding(button, key)

	end

	if (self:IsVisible()) then
		self:OnEnter()
	end

	button:SaveData()

end

function BINDER:OnShow()

	if (self.button.bar) then
		self:SetFrameLevel(self.button.bar:GetFrameLevel()+1)
		self.button.hotkey:Show()
	end

end

function BINDER:OnHide()
	self.button:SetData(self.button.bar)
end

function BINDER:OnEnter()

	local button = self.button

	self.select:Show()

	GameTooltip:SetOwner(self, "ANCHOR_RIGHT")

	local spell, item, link = button.macrospell, button.macroitem

	if (spell) then

		if (sIndex[spell:lower()]) then

			GameTooltip:SetSpellBookItem(sIndex[spell:lower()].index, sIndex[spell:lower()].booktype)

		elseif (type(button.data.macro_Icon) == "table") then

			GameTooltip:SetHyperlink("spell:"..button.data.macro_Icon[5])
		end

	elseif (item) then

		_, link = GetItemInfo(item)

		if (link) then
			GameTooltip:SetHyperlink(link)
		end
	else
		if (strlen(button.data.macro_Text) > 0) then
			GameTooltip:SetText(button.data.macro_Text)
		else
			GameTooltip:SetText(L.EMPTY_BUTTON)
		end
	end

	GameTooltip:AddLine(format(L.KEYBIND_TOOLTIP1, self.bindType:gsub("^%l", string.upper).." "..button.id).."|r", 1.0, 1.0, 1.0)
	GameTooltip:AddLine(format(L.KEYBIND_TOOLTIP2, self.bindType, self.bindType), 1.0, 1.0, 1.0)
	GameTooltip:AddLine(L.KEYBIND_TOOLTIP3..self:GetBindkeyList(button).."|r")

	GameTooltip:Show()

end

function BINDER:OnLeave()

	self.select:Hide()

	GameTooltip:Hide()

end

function BINDER:OnUpdate()

	if (self:IsMouseOver()) then
		self:EnableKeyboard(true)
	else
		self:EnableKeyboard(false)
	end
end

function BINDER:OnClick(button)

	if (button == "LeftButton" or button == "RightButton") then
		return
	end

	local modifier, key = self:GetModifier()

	if (button == "MiddleButton") then
		key = "Button3"
	else
		key = button
	end

	if (modifier) then
		key = modifier..key
	end

	self:ProcessBinding(key, self.button)

end

function BINDER:OnKeyDown(key)

	if (key:find("ALT") or key:find("SHIFT") or key:find("CTRL") or key:find("PRINTSCREEN")) then
		return
	end

	local modifier = self:GetModifier()

	if (modifier) then
		key = modifier..key
	end

	self:ProcessBinding(key, self.button)

end

function BINDER:OnMouseWheel(delta)

	local modifier, key, action = self:GetModifier()

	if (delta > 0) then
		key = "MOUSEWHEELUP"
		action = "MousewheelUp"
	else
		key = "MOUSEWHEELDOWN"
		action = "MousewheelDown"
	end

	if (modifier) then
		key = modifier..key
	end

	self:ProcessBinding(key, self.button)

end


local BINDER_MT = { __index = BINDER }

local function CreateBindFrame(index, button)

	local binder = CreateFrame("Button", button:GetName().."BindFrame", button, "RazerOrbweaverBindFrameTemplate")

	setmetatable(binder, BINDER_MT)

	binder:EnableMouseWheel(true)
	binder:RegisterForClicks("AnyDown")
	binder:SetAllPoints(button)
	binder:SetScript("OnShow", BINDER.OnShow)
	binder:SetScript("OnHide", BINDER.OnHide)
	binder:SetScript("OnEnter", BINDER.OnEnter)
	binder:SetScript("OnLeave", BINDER.OnLeave)
	binder:SetScript("OnClick", BINDER.OnClick)
	binder:SetScript("OnKeyDown", BINDER.OnKeyDown)
	binder:SetScript("OnMouseWheel", BINDER.OnMouseWheel)
	binder:SetScript("OnUpdate", BINDER.OnUpdate)

	binder.type:SetText(L.BINDFRAME_BIND)
	binder.button = button
	button.binder = binder
	binder.bindType = "button"

	BINDIndex[index] = binder

	binder:Hide()

end

function R.BindingsWarning()

	StaticPopup_Show("RAZERONE_CONFIRM_BIND_MANUALLY")

	if (R.GUI and R.GUI:IsVisible()) then
		R.ManualEdit_OnShow(R.GUI.manualBindings)
	end

end

function R.BindingsResetWarning()

	StaticPopup_Show("RAZERONE_RESET_BINDINGS_TO_DEFAULT")

	if (R.GUI and R.GUI:IsVisible()) then
		R.ManualEdit_OnShow(R.GUI.manualBindings)
	end

end

function R.ToggleBindings(show, hide)

	if (DB.razerBindings) then
		if (not hide) then
			R.BindingsWarning(); return
		end
	end

	if (editmode or hide) then

		editmode = false

		for index, binder in pairs(BINDIndex) do
			binder:Hide(); binder.button.editmode = editmode
		end

		if (R.GUI and R.GUI.cTab and R.GUI.cTab == R.GUI.tab3) then
			--do nothing
		else
			R.ToggleBars(true)
		end

	else

		editmode = true

		for index, binder in pairs(BINDIndex) do
			binder:Show(); binder.button.editmode = editmode
		end

		R.ToggleBars(nil, true)

	end

	if (R.GUI and R.GUI:IsVisible()) then
		R.ManualEdit_OnShow(R.GUI.manualBindings)
	end

end

local function controlOnEvent(self, event, ...)

	if (event == "ADDON_LOADED" and ... == "RazerOrbweaver") then

		DB = RazerOrbweaverDB

		StaticPopupDialogs["RAZERONE_CONFIRM_BIND_MANUALLY"] = {
			text = L.CONFIRM_BIND_MANUALLY,
			button1 = YES,
			button2 = NO,
			OnAccept = function() RazerOrbweaverDB.razerBindings = false; R:ToggleBindings() end,
			OnCancel = function() end,
			hideOnEscape = 1,
			timeout = 0,
		}

		StaticPopupDialogs["RAZERONE_RESET_BINDINGS_TO_DEFAULT"] = {
			text = L.BINDINGS_RESET,
			button1 = OKAY,
			button2 = CANCEL,
			timeout = 0,
			OnAccept = function() RazerOrbweaverDB.razerBindings = true; R.ToggleBindings(nil, true); for _, btn in pairs(BTNIndex) do btn.binder:ClearBindings(btn) btn:SetBindingDefaults() btn.binder:ApplyBindings(btn) btn:SaveData() end end,
			OnCancel = function() end,
			hideOnEscape = 1,
		}

	elseif (event == "PLAYER_LOGIN") then

		--local frame = CreateFrame("Button", nil, R.GamePad, "RazerOrbweaverOptionButtonTemplate")
		--frame:SetWidth(120)
		--frame:SetHeight(25)
		--frame:SetPoint("BOTTOM", R.GamePad, "BOTTOM", 0, 8)
		--frame:SetScript("OnClick", function(self) R.ToggleBindings() end)
		--frame:SetScript("OnHide", function(self) end)
		--frame.text:SetText(L.EDIT_BINDINGS)

		for index,button in pairs(BTNIndex) do
			CreateBindFrame(index, button)
		end

	elseif (event == "PLAYER_ENTERING_WORLD" and not PEW) then

		for index,button in pairs(BTNIndex) do
			if (button.keys) then
				button.binder:ApplyBindings(button)
			end
		end

		PEW = true; self.elapsed = 0

	end
end

local frame = CreateFrame("Frame", nil, UIParent)
frame:SetScript("OnEvent", controlOnEvent)
frame:SetScript("OnUpdate", controlOnUpdate)
frame:RegisterEvent("ADDON_LOADED")
frame:RegisterEvent("PLAYER_LOGIN")
frame:RegisterEvent("PLAYER_ENTERING_WORLD")
frame.elapsed = 0