/* Copyright (c) 2013 Dropbox, Inc. All rights reserved. */

#import "DBAccountManager.h"
#import "DBDatastoreStatus.h"
#import "DBError.h"
#import "DBTable.h"
#import "DBUtil.h"

/** Enum giving the possible roles a principal can have with respect to a `DBDatastore`. */
typedef NS_ENUM(NSInteger, DBRole) {
    DBRoleNone,  /** No permission */
    DBRoleViewer,  /** Read-only permission */
    DBRoleEditor,  /** Read-write permission (includes ACL changes) */
    DBRoleOwner,  /** Owner (same as read-write, cannot be kicked out) */
};

/** The principal used to set or retrieve the role for DfB team members. */
extern NSString * const DBPrincipalTeam;

/** The principal used to set or retrieve the role for the general public. */
extern NSString * const DBPrincipalPublic;

@class DBDatastoreManager;

/** The maximum size in bytes of a datastore. */
extern const NSUInteger DBDatastoreSizeLimit;

/** The maximum size in bytes of changes that can be queued up between calls to <sync:> */
extern const NSUInteger DBDatastoreUnsyncedChangesSizeLimit;

/** The maximum number of records in a datastore. */
extern const NSUInteger DBDatastoreRecordCountLimit;

/** The size in bytes of a datastore before accounting for the size of its records. */
extern const NSUInteger DBDatastoreBaseSize;

/** The size in bytes of unsynced changes before accounting for the size of each change. */
extern const NSUInteger DBDatastoreBaseUnsyncedChangesSize;

/** The size in bytes of a change before accounting for the size of its values. */
extern const NSUInteger DBDatastoreBaseChangeSize;

/** A datastore is a simple, syncable database for app data. You can open the default datastore
 using <openDefaultStoreForAccount:error:> and open or create other datastores using a
 <DBDatastoreManager>.
 <!-- paragraph separator in class docs for appledoc bug -->
 You interact with data in the datastore using tables. You can call <getTable:> to get a table,
 or [getTables:](#DBDatastore.getTables:) to list all tables in a datastore containing
 records.
 <!-- paragraph separator in class docs for appledoc bug -->
 Changes you make to the datastore will be visible immediately, and calling <sync:> will persist
 outgoing changes and queue them to be uploaded to the server. While a datastore is open, it will
 monitor for remote changes and download them when possible. When there are remote changes waiting
 to be incorporated, the <DBDatastoreStatus> will have its `incoming` property set to `YES`, and calling
 <sync:> will also apply those changes to your view of the datastore, resolving any conflicts along
 the way.
 <!-- paragraph separator in class docs for appledoc bug -->
 To find out when there are changes ready to be synced, add an observer using <addObserver:block:>
 to register a block that will be called every time <status> changes. */
@interface DBDatastore : NSObject

/** Returns `YES` if `datastoreId` is a valid ID for a `DBDatastore`, or `NO` otherwise.
 Datastore IDs come in 2 forms:
 
 Private datastores (such as the default datastore) use IDs which
 can be 1-64 characters long, must not begin or end with a '.', and may contain lower-case
 letters, digits, and these punctuation characters: . - _  (Note that older SDKs limited these
 to 32 characters, so take care if your datastore needs to be accessed by legacy clients.)

 Shareable datastore IDs (generated by `-[DBDatastoreManager createDatastore:]) always
 begin with a '.' and can contain 1-63 additional characters which can be upper-case, lower-case,
 digits, and these punctuation characters: . - */
+ (BOOL)isValidId:(NSString *)datastoreId;

/** Returns `YES` if `datastoreId` is a valid ID for a shareable `DBDatastore`, or `NO` otherwise.
 This is a valid ID (see `isValidId`) that starts with a '.'.
 */
+ (BOOL)isValidShareableId:(NSString *)datastoreId;

/** Opens the default datastore for this account. */
+ (DBDatastore *)openDefaultStoreForAccount:(DBAccount *)account error:(DBError **)error;

/** Opens the local default datastore for this account manager. */
+ (DBDatastore *)openDefaultLocalStoreForAccountManager:(DBAccountManager *)accountManager error:(DBError **)error;

/** Close a datastore when you're done using it to indicate that you are no longer
 interested in receiving updates for this datastore.

 Any changes made since the last call to <sync:> will be discarded on close. If the account is
 unlinked remotely, the datastore will close automatically.
 */
- (void)close;

/** Get all the tables in this datastore that contain records. */
- (NSArray *)getTables:(DBError **)error;

/** Get a table with the specified ID, which can be used to insert or query records. If this is a
 new table ID, the table will not be visible until a record is inserted. */
- (DBTable *)getTable:(NSString *)tableId;

/** Set the title for this datastore.  Will be `nil` if no title is set.
 Setting it to `nil` will delete the title field. */
@property (nonatomic, copy) NSString *title;

/** The last modified time for this datastore, or nil if no data has been
 synced yet.

 The last modified time is automatically updated on each call to <sync:>
 which commits local changes, or incorporates remote changes. The timestamp is based
 on the local clock of the device where the change is made. */
@property (nonatomic, readonly) NSDate *mtime;

/** The current size of this datastore in bytes.  The overall size
 of a datastore is calculated by summing the size of all records,
 plus the base size of an empty datastore itself. */
@property (nonatomic, readonly) NSUInteger size;

/** The total number of records in this datastore. */
@property (nonatomic, readonly) NSUInteger recordCount;

/** The size in bytes of changes that will be queued for upload
 * by the next call to <sync:>. */
@property (nonatomic, readonly) NSUInteger unsyncedChangesSize;

/** Apply all outstanding changes to the datastore, and also incorporate remote changes in.

 @returns A dictionary mapping of `tableId` to a set of <DBRecord> objects if the call was
 successful, or `nil` if an error occurred. The table IDs and records in the dictionary correspond
 to the tables and records that changed due to remote changes applied during this sync. If records
 are deleted by the sync, the `DBRecord`s will have the `deleted` property set, but no fields. */
- (NSDictionary *)sync:(DBError **)error;

/** Whether the datastore is currently open. */
@property (nonatomic, readonly, getter=isOpen) BOOL open;

/** The current sync status of the datastore. */
@property (nonatomic, readonly) DBDatastoreStatus *status;

/** Add `block` as an observer when the status of the datastore changes. */
- (void)addObserver:(id)observer block:(DBObserver)block;

/** Remove all blocks registered for the given `observer`. */
- (void)removeObserver:(id)observer;

/** The ID for this datastore. */
@property (nonatomic, readonly) NSString *datastoreId;

/** The datastore manager for this datastore. */
@property (nonatomic, readonly) DBDatastoreManager *manager;

/** The effective role the current user has for this datastore. */
@property (nonatomic, readonly) DBRole effectiveRole;

/** Whether this datastore can be written (i.e., role is owner or editor). */
@property (nonatomic, readonly) BOOL isWritable;

/** Whether this datastore can be shared. */
@property (nonatomic, readonly) BOOL isShareable;

/** Get the role specified by the ACL for a principal (shareable datastores only). */
- (DBRole) getRoleForPrincipal:(NSString *)principal;

/** Assign a role to a principal in the ACL (shareable datastores only). */
- (void)setRoleForPrincipal:(NSString *)principal to:(DBRole)role;

/** Delete any role for a principal from the ACL(shareable datastores only). */
- (void)deleteRoleForPrincipal:(NSString *)principal;

/** Return the ACL in the form of a mapping from principals to roles (as NSIntegers). */
- (NSDictionary *)listRoles;

@end
