"use strict";

var util = require('util');
var EventEmitter = require('events');
var http = require('http');
var https = require('https');
var URL = require('url');
var assign = require('lodash.assign');
var path = require('path');
var fs = require('fs');

require('@ali/ssl-root-cas')
    .inject();

var supportedProtocol = ['https:', 'http:'];
const MAX_REDIRECT_TIME = 10;
const MAX_ERR_TIME = 10;
const REQ_TIME_OUT = 10*1000;

var Download = function(url, destPath, headers){
    EventEmitter.call(this);
    this.urlObj = URL.parse(url);
    this.ErrorCode = require('./ErrorCode');
    this.receivedBytes = 0;
    this.errorTime = 0;
    this.emit(this.EventsName.info, 'start download :' + url);
    if(supportedProtocol.indexOf(this.urlObj.protocol) === -1){
        var err = new Error('Unknown protocol');
        err.code = this.ErrorCode.UNKNOWN_PROTOCOL;
        this.emit(this.EventsName.info, 'Unknown protocol')
        throw err;
    }
    if(!destPath){
        var err = new Error('destPath is null');
        err.code = this.ErrorCode.DEST_PATH_IS_NULL;
        this.emit(this.EventsName.info, 'destPath is null')
        throw err;
    }
    this.headers = assign({}, headers);
    if(!this.headers['user-agent']){
        this.headers['user-agent'] = '@ali-ding-download'
    }
    this.destPath = path.resolve(destPath);
    this.resultFileDir = path.dirname(this.destPath);
    this.redirectTime = 0;
    this.contentSize = 0;
    this.state = this.stateCode.normal;
    this._process();

}

util.inherits(Download, EventEmitter);

Download.prototype._process = function(){
    return this._getTmpPath().then(function(tmpPath){
        this.emit(this.EventsName.info, 'tmpPath is :' + tmpPath)
        this.tmpPath = tmpPath;
        return this._requestAndWrite();
    }.bind(this)).then(function(result){
        this.emit(this.EventsName.info,'download success , start change name')
        return this._changeName();
    }.bind(this)).then(function(result){
        this.emit(this.EventsName.info,'change name success , finish')
        this.emit(this.EventsName.finish);
    }.bind(this)).catch(function(err){
        this._cleanTmpFile();
        this.emit(this.EventsName.error, err);
    }.bind(this))
}

Download.prototype._requestAndWrite = function(){
    //var writeStream = this._getWriteStream();
    return this._getWriteStream().then(function(writeStream){
        return this._request(writeStream);
    }.bind(this));
}

Download.prototype._cleanTmpFile = function(){
    try{
        fs.unlinkSync(this.tmpPath);
    }catch(e){
        this.emit(this.EventsName.info, 'unlink sync failure :' + e.toString());
    }
}

Download.prototype._changeName = function(){
    return this._hasPermission(this.destPath, fs.F_OK).then(function(result){
        if(result === true){
            this.emit(this.EventsName.info, 'destPath is exist : '+ this.destPath);
            return new Promise(function(resolve, reject){
                fs.unlink(this.destPath, function(err){
                    if(err){
                        err.code = this.ErrorCode.CHANGE_NAME_ERR;
                        reject(err);
                    }else{
                        this.emit(this.EventsName.info, 'delete old file success :' + this.destPath)
                        resolve(true);
                    }
                }.bind(this))
            }.bind(this))
        }else{
            return true;
        }
    }.bind(this)).then(function(){
        return new Promise(function(resolve, reject){
            fs.rename(this.tmpPath, this.destPath, function(err){
                if(err) {
                    err.code = this.ErrorCode.CHANGE_NAME_ERR;
                    reject(err);
                }else{
                    this.emit(this.EventsName.progress, 1, this.contentSize, this.contentSize);
                    resolve(true);
                }
            }.bind(this))
        }.bind(this));
    }.bind(this))
}

Download.prototype._request = function(writeStream){

    if(this.urlObj.protocol === 'http:'){
        var h = http;
    }else if(this.urlObj.protocol === 'https:'){
        var h = https;
    }

    var option = {
        protocol: this.urlObj.protocol,
        hostname: this.urlObj.hostname,
        port: this.urlObj.port || (this.urlObj.protocol === 'https:' ? 443: 80),
        path: this.urlObj.path,
        headers: assign({}, this.headers, {"Range": "bytes=" + this.receivedBytes + '-'})
    }

    return new Promise(function(resolve, reject){

        if(this.state === this.stateCode.start_pause){
            this.state = this.stateCode.paused;
            this.once(this.EventsName._resume, function(){
                writeStream.end(function(){
                    resolve(this._requestAndWrite());
                }.bind(this));
            })
            this.emit(this.EventsName.pause);
            return;
        }

        if(this.state === this.stateCode.aborted){
            var err = new Error('user aborted');
            err.code = this.ErrorCode.USER_ABORTED;
            reject(err);
            return;
        }

        var finish = function(){
            var stat = fs.statSync(this.tmpPath);
            if(this.state === this.stateCode.normal){
                if(!this.contentSize){ // no content length, no retry , just return true
                    resolve(true);
                    return;
                }
                if(stat.size === this.contentSize){
                    resolve(true);
                }else if(stat.size > this.contentSize){
                    this.emit(this.EventsName.info, 'file size greater than content size , throw error')
                    var err = new Error('file size not equal content size');
                    err.code = this.ErrorCode.SIZE_NOT_EQUAL;
                    reject(err);
                }else{
                    this.errorTime ++;
                    this.emit(this.EventsName.info, 'write stream is finish, but file size is not equal content size')
                    if(this.errorTime > MAX_ERR_TIME){
                        this.emit(this.EventsName.info, 'error time is greater than MAX_ERR_TIME, throw error : '+ this.errorTime);
                        var err = new Error('download error');
                        err.code = this.ErrorCode.DOWNLOAD_ERROR;
                        reject(err);
                    }else{
                        this.receivedBytes = stat.size || 0;
                        this.emit(this.EventsName.info, 'retry download from size :'+ this.receivedBytes);
                        resolve(this._requestAndWrite());
                    }
                }
            }else if(this.state === this.stateCode.start_pause){
                this.receivedBytes = stat.size || 0;
                this.state = this.stateCode.paused;
                this.emit(this.EventsName.info, 'paused by user, wait for resume');
                this.once(this.EventsName._resume, function(){
                    this.emit(this.EventsName.info, 'resume download');
                    this.emit(this.EventsName.resume);
                    resolve(this._requestAndWrite());
                }.bind(this))
                this.emit(this.EventsName.pause);
            }else if(this.state === this.stateCode.aborted){
                var err = new Error('user aborted');
                this.emit(this.EventsName.info, 'user aborted');
                err.code = this.ErrorCode.USER_ABORTED;
                reject(err);
            }
        }.bind(this);

        var req = this.req = h.get(option, function(res){

            this.emit(this.EventsName.info, 'request option is: '+ JSON.stringify(option));

            var statusCode = res.statusCode;
            this.emit(this.EventsName.info, 'receive status code : ' + statusCode);
            if(statusCode === 301 || statusCode === 302){
                this.redirectTime ++;
                this.emit(this.EventsName.info, 'redirect to : '+ res.headers.location + ', statusCode : '+ statusCode);
                if(this.redirectTime > MAX_REDIRECT_TIME){
                    this.emit(this.EventsName.info, 'redirect too many times, throw error');
                    var err = new Error('Too many redirect');
                    err.code = this.ErrorCode.TOO_MANY_REDIRECT;
                    reject(err);
                    return;
                }
                var redirectUrlObj = URL.parse(res.headers.location);
                if(supportedProtocol.indexOf(redirectUrlObj.protocol) === -1){
                    var err = new Error('Unknown protocol');
                    this.emit(this.EventsName.info, 'redirect to unknown protocol : '+ redirectUrlObj.protocol);
                    err.code = this.ErrorCode.UNKNOWN_PROTOCOL;
                    err.status = statusCode;
                    writeStream.removeListener('finish', finish);
                    writeStream.end(function(){
                        reject(err);
                    });
                }else{
                    this.urlObj = redirectUrlObj;
                    resolve(this._request(writeStream));
                }
            }else if(statusCode === 200 || statusCode === 206){
                this.redirectTime = 0;
                if(this.receivedBytes === 0){
                    this.contentSize = parseInt(res.headers['content-length']);
                }

                res.pipe(writeStream);
                writeStream.on('finish', finish);
                res.on('error', function(err){
                    this.emit(this.EventsName.info, 'res emit error : '+ err.message);
                    this.errorTime ++;
                    if(this.errorTime > MAX_ERR_TIME){
                        err.code = this.ErrorCode.DOWNLOAD_ERROR;
                        reject(err);
                        this.emit(this.EventsName.info, 'error time is greater then MAX_ERR_TIME');
                        return;
                    }else{
                        writeStream.end(function(){
                        }.bind(this));
                    }
                }.bind(this))
                var receivedSize = this.receivedBytes;
                res.on('data', function(chunk){
                    receivedSize += chunk.length;
                    this.emit(this.EventsName.progress, (receivedSize * 0.99/ this.contentSize), receivedSize, this.contentSize);
                }.bind(this));
            }else{
                // other status
                var err = new Error('Unknown status');
                err.code = this.ErrorCode.UNKNOWN_STATUS;
                err.status = statusCode;
                this.emit(this.EventsName.info, 'unknown status :'+ statusCode);
                reject(err);
            }
        }.bind(this));

        req.on('error', function(err){
            this.errorTime ++;
            this.emit(this.EventsName.info, 'req error: ' + err.toString());
            if(this.errorTime > MAX_ERR_TIME){
                var err = new Error('Too many HTTP error : '+ this.errorTime);
                err.code = this.ErrorCode.TOO_MANY_ERR;
                reject(err);
                return;
            }else{
                writeStream.removeListener('finish', finish);
                resolve(this._request(writeStream))
            }
        }.bind(this));

        req.setTimeout(REQ_TIME_OUT, function(){
            this.emit(this.EventsName.info, 'req timeout');
            writeStream.end(function(){
            }.bind(this));

            this.emit(this.EventsName.info, 'request timeout:'+new Date().getTime());
            var err = new Error('request timeout');
            err.code = this.ErrorCode.REQUEST_TIME_OUT;
            reject(err);
        }.bind(this))

    }.bind(this));


}

Download.prototype._hasPermission = function(p, type){
    return new Promise(function(resolve, reject){
        fs.access(p, type, function(err){
            if(err){
                resolve(false);
            }else{
                resolve(true);
            }
        }.bind(this));
    }.bind(this));
}

Download.prototype._getTmpPath = function(){

    var isFileExist = function(fpath){
        return new Promise(function(resolve, reject){
            fs.access(fpath, fs.F_OK, function(err){
                if(err && err.code === "ENOENT"){
                    resolve(false);
                }else{
                    resolve(true);
                }
            })
        })
    }

    var count = 0;

    var generateTmpFile = function(){
        if(count < 10){
            count ++;
            return this.destPath + '.dingdownload' + (count > 1 ? count: '');
        }else{
            var err = new Error('get tmp file error')
            err.code = this.ErrorCode.GENERATE_TMP_ERROR;
            throw err;
        }
    }.bind(this);

    return new Promise(function(resolve, reject){
        var tryGenerate = function(){
            var t = generateTmpFile();
            isFileExist(t).then(function(result){
                if(result === true){
                    tryGenerate();
                }else{
                    resolve(t);
                }
            })
        }
        tryGenerate();
    }.bind(this));
}

Download.prototype._getWriteStream = function(){
    return new Promise(function(resolve, reject){
        var result = fs.createWriteStream(this.tmpPath, {
            flags: 'a+',
            start: this.receivedBytes
        })
        result.on('open', function(){
            resolve(result);
        })
        result.on('error', function(e){
            this.emit(this.EventsName.info, 'writeStream error:'+new Date().getTime());
            if(e.code === 'EACCES'){
                e.code = this.ErrorCode.NO_PERMISSION_WRITE;
            }
            reject(e);
        }.bind(this))
    }.bind(this))
}

Download.prototype.pause = function(){
    if(this.state === this.stateCode.normal && !!this.contentSize){
        this.state = this.stateCode.start_pause;
        this.req && this.req.abort();
        return true;
    }else{
        return false;
    }
}

Download.prototype.resume = function(){
    if(this.state !== this.stateCode.paused){
        return false;
    }
    this.state = this.stateCode.normal;
    this.emit(this.EventsName._resume);
    return true;
}

Download.prototype.abort = function(){
    this.state = this.stateCode.aborted;
    this.req && this.req.abort();
    this.emit(this.EventsName.error, {code:this.ErrorCode.USER_ABORTED});

}

Download.prototype.EventsName = {
    'progress': 'progress',
    'finish': 'finish',
    'pause': 'pause',
    'resume': 'resume',
    'cancel': 'cancel',
    'error': 'error',
    'info': 'info',
    '_resume': '_resume',
    '_abort': '_aborted'
}

Download.prototype.stateCode = {
    'normal': 0,
    'start_pause': 1,
    'paused': 2,
    'aborted': 3
}


module.exports = Download;
